//**************************************************************************
//*                     This file is part of the                           *
//*                      Mpxplay - audio player.                           *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2007 by PDSoft (Attila Padar)                *
//*                    http://mpxplay.cjb.net                              *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: SPEEX (SPX) stream decoding (in DLL)

#include "in_file.h"

#include "speex/speex.h"
#include "speex/misc.h"
#include "speex/speex_he.h"
#include "speex/speex_st.h"
#include "speex/speex_cb.h"

#include <string.h>

#define SPX_MAXFRAMESIZE_PCM 2000

typedef struct spx_decoder_data_s{
 unsigned int current_decoder_part;
 unsigned int packet_count;
 spx_int32_t enh_enabled;
 spx_int32_t frame_size;
 int granule_frame_size;
 spx_int32_t rate;
 int nframes,frame_counter;
 int channels;
 int extra_headers;
 void *st;

 SpeexStereoState stereo;
 SpeexCallback callback;
 SpeexBits bits;
}spx_decoder_data_s;

#ifdef __DOS__
struct mpxplay_resource_s *mrs;
#define SPX_PRINTF(s) mrs->pds_textdisplay_printf(s)
#else
#include <share.h>
void SPX_PRINTF(char *s)
{
 printf("%s\n",s);
}

int matherr(struct _exception *a)
{
 a->retval=1.0;
 return 1;
}
#endif

static const SpeexStereoState spx_stereo_init=SPEEX_STEREO_STATE_INIT;

static unsigned int spx_process_header(struct spx_decoder_data_s *spxd,struct mpxplay_audio_decoder_info_s *adi,struct mpxplay_streampacket_info_s *spi)
{
 const SpeexMode *mode;
 SpeexHeader *header;
 //char sout[100];

 header = speex_packet_to_header(spi->bitstreambuf,spi->bs_leftbytes);
 spi->bs_usedbytes=spi->bs_leftbytes;
 if(!header)
  return 0;
 if((header->mode>=SPEEX_NB_MODES) || (header->mode<0))
  goto err_out_hdr;

 mode = speex_lib_get_mode(header->mode);

 if(header->speex_version_id > 1)
  goto err_out_hdr;
 if(mode->bitstream_version < header->mode_bitstream_version)
  goto err_out_hdr;
 if(mode->bitstream_version > header->mode_bitstream_version)
  goto err_out_hdr;

 spxd->st = speex_decoder_init(mode);
 if(!spxd->st)
  goto err_out_hdr;

 speex_decoder_ctl(spxd->st, SPEEX_SET_ENH, &spxd->enh_enabled);
 speex_decoder_ctl(spxd->st, SPEEX_GET_FRAME_SIZE, &spxd->frame_size);
 spxd->granule_frame_size = spxd->frame_size;

 if(!(spxd->channels==1)){
  spxd->callback.callback_id = SPEEX_INBAND_STEREO;
  spxd->callback.func = speex_std_stereo_request_handler;
  spxd->callback.data = &spxd->stereo;
  speex_decoder_ctl(spxd->st, SPEEX_SET_HANDLER, &spxd->callback);
 }

 if(!spxd->rate)
  spxd->rate = header->rate;

 speex_decoder_ctl(spxd->st, SPEEX_SET_SAMPLING_RATE, &spxd->rate);

 spxd->nframes = header->frames_per_packet;

 if(spxd->channels==-1)
  spxd->channels = header->nb_channels;

 spxd->extra_headers = header->extra_headers;

 //sprintf(sout,"f:%d c:%d n:%d s:%d",spxd->rate,spxd->channels,spxd->nframes,spxd->frame_size);
 //SPX_PRINTF(sout);

 free(header);
 return 1;

err_out_hdr:
 free(header);
 return 0;
}

static int AD_SPX_open(struct mpxplay_audio_decoder_info_s *adi,struct mpxplay_streampacket_info_s *spi)
{
 struct spx_decoder_data_s *spxd;

 spxd=(struct spx_decoder_data_s *)calloc(1,sizeof(*spxd));
 if(!spxd)
  return MPXPLAY_ERROR_INFILE_MEMORY;
 adi->private_data=spxd;

 spxd->packet_count=0;
 spxd->enh_enabled=1;
 spxd->nframes=2;
 spxd->channels=-1;
 memcpy((void *)&spxd->stereo,(void *)&spx_stereo_init,sizeof(SpeexStereoState));

 speex_bits_init(&spxd->bits);

 adi->bits=16;
#ifndef FIXED_POINT
 funcbit_enable(adi->infobits,ADI_FLAG_FLOATOUT);
#endif

 return MPXPLAY_ERROR_INFILE_OK;
}

static void AD_SPX_close(struct mpxplay_audio_decoder_info_s *adi)
{
 struct spx_decoder_data_s *spxd=(struct spx_decoder_data_s *)adi->private_data;
 if(spxd){
  if(spxd->st)
   speex_decoder_destroy(spxd->st);
  free(spxd);
 }
}

static int AD_SPX_parse_frame(struct mpxplay_audio_decoder_info_s *adi,struct mpxplay_streampacket_info_s *spi)
{
 struct spx_decoder_data_s *spxd=(struct spx_decoder_data_s *)adi->private_data;

 if(spxd->packet_count==0){
  if(spx_process_header(spxd,adi,spi)){
   adi->freq=spxd->rate;
   adi->filechannels=adi->outchannels=spxd->channels;
   adi->pcm_framelen=spxd->frame_size;
   return MPXPLAY_ERROR_INFILE_OK;
  }
 }

 spxd->packet_count++;

 return MPXPLAY_ERROR_INFILE_CANTOPEN;
}

static int AD_SPX_decode(struct mpxplay_audio_decoder_info_s *adi,struct mpxplay_streampacket_info_s *spi)
{
 struct spx_decoder_data_s *spxd=(struct spx_decoder_data_s *)adi->private_data;
#ifdef FIXED_POINT
 MPXPLAY_PCMOUT_INT16_T *pcmout=(MPXPLAY_PCMOUT_INT16_T *)adi->pcm_bufptr;
#else
 MPXPLAY_PCMOUT_FLOAT_T *pcmout=(MPXPLAY_PCMOUT_FLOAT_T *)adi->pcm_bufptr;
#endif
 int retcode;

 spxd->packet_count++;
 if(spxd->packet_count<=(spxd->extra_headers+1))
  return MPXPLAY_ERROR_INFILE_RESYNC;

 switch(spxd->current_decoder_part){
  case 0:speex_bits_read_from(&spxd->bits, spi->bitstreambuf, spi->bs_leftbytes);
         spi->bs_usedbytes=spi->bs_leftbytes;
         spxd->frame_counter=0;
         spxd->current_decoder_part=1;
  case 1:
#ifdef FIXED_POINT
         retcode=speex_decode_int(spxd->st, &spxd->bits, pcmout);
#else
         retcode=speex_decode(spxd->st, &spxd->bits, pcmout);
#endif
         if(retcode<0){
          spxd->current_decoder_part=0;
          break;
         }
         if(spxd->channels==2)
#ifdef FIXED_POINT
          speex_decode_stereo_int(pcmout, spxd->frame_size, &spxd->stereo);
#else
          speex_decode_stereo(pcmout, spxd->frame_size, &spxd->stereo);
#endif
         adi->pcm_samplenum=spxd->frame_size*spxd->channels;
         spxd->frame_counter++;
         if(spxd->frame_counter>=spxd->nframes)
          spxd->current_decoder_part=0;

 }
 return MPXPLAY_ERROR_INFILE_OK;
}

static void AD_SPX_clearbuff(struct mpxplay_audio_decoder_info_s *adi,unsigned int seektype)
{
 struct spx_decoder_data_s *spxd=(struct spx_decoder_data_s *)adi->private_data;
 spx_int32_t tmp;
 if(seektype&MPX_SEEKTYPE_BOF)
  spxd->packet_count=0;
 if(seektype&(MPX_SEEKTYPE_BOF|MPX_SEEKTYPE_PAUSE)){
  speex_decoder_ctl(spxd->st, SPEEX_RESET_STATE, &tmp);
  spxd->current_decoder_part=0;
 }
}

static struct mpxplay_audio_decoder_func_s AD_SPX_funcs={
 0,
 NULL,
 NULL,
 NULL,
 &AD_SPX_open,
 &AD_SPX_close,
 NULL,
 &AD_SPX_parse_frame,
 &AD_SPX_decode,
 &AD_SPX_clearbuff,
 NULL,
 NULL,
 0,
 SPX_MAXFRAMESIZE_PCM,
 {{MPXPLAY_WAVEID_SPEEX,"SPX"},{0,NULL}}
};

static mpxplay_module_entry_s adspx_module_entry={
 MPXPLAY_DLLMODULETYPE_DECODER_AUDIO,
 0,
 "AdSPX",
 MPXPLAY_DLLMODULEVER_DECODER_AUDIO,
 (void *)&AD_SPX_funcs
};

static mpxplay_dll_entry_s mpxplay_dll_entry_structure={
 MPXPLAY_DLLENTRY_STRUCTURE_VERSION,
 {
  &adspx_module_entry,
  NULL
 }
};

#ifdef __DOS__
extern void dllstrtr_update_crwdata(unsigned long *cwd);

long __export mpxplay_dll_entrypoint(struct mpxplay_resource_s *p_mrs,unsigned long *crwdata_begin)
{
 mrs=p_mrs;
 dllstrtr_update_crwdata(crwdata_begin);
 return (long)(&mpxplay_dll_entry_structure);
}

#else

__declspec( dllexport ) mpxplay_dll_entry_s *mpxplay_dll_entrypoint(void)
{
 return (&mpxplay_dll_entry_structure);
}

#endif
