/*
 * CMCOMPRE.C
 *
 *  This file is part of DOSZIP
 *  Copyright (c) 1996 Hjort Nidudsson.
 */

#include <dzmain.h>
#include <errno.h>
#include <arch.h>
#include <string.h>
#include <io.h>
#include <progress.h>

#define MAXPACKERS	16
enum PackerConfig {
	P00_ADD,
	P01_ADD_PATH,
	P02_EXTRACT,
	P03_EXTRACT_PATH,
	P04_DELETE,
	P05_MAX,
	P06_MIN,
	P07_PASSWORD_PREFIX,
	P08_OUTPATH_PREFIX,
	P09_RESPONSE_PREFIX,
};

extern char *envtemp;
int recursive(char *, char *, char *);
int cpy_setselectedcnt(void);

const char INI_Packer  	[] = "Packer";
const char INI_00      	[] = "00";
const char cp_newline  	[] = "\r\n";
const char cp_ziplst   	[] = "ziplst";
const char
	cp_linkerror	[] = "You can't do this in a link or .ZIP panel.",
	cp_badentry	[] = "Bad or missing Entry in DZ.INI",
	cp_badentrymsg	[] = "Section: [%s]\nEntry: [%s]\n",
	cp_enterpassword[] = "Enter password",
	cp_skippassword	[] = "Skip Password?",
	cp_badpasswordsw[] = "07=(Password switch)",
	cp_compressto	[] = "%s to",
	cp_compressusing[] = "%s using %s to",
	cp_missingTEMP	[] = "Bad or missing TEMP directory",
	cp_nothingtodo	[] = "Nothing to do..";

#define MAXZCMD		64
#define MAXZSWITCH	32

typedef struct {
	char 	section[32];
	char	command[128];
	char	A00[MAXZCMD];	 /* Add a file to a compressed file        */
	char	A01[MAXZCMD];	 /* - and store the file's full pathname   */
	char	E02[MAXZCMD];    /* Extract a file from a compressed file  */
	char	E03[MAXZCMD];	 /* - and use the file's full pathname     */
	char	D04[MAXZCMD];	 /* Delete a file from a compressed file   */
	char	M05[MAXZSWITCH]; /* Create the smallest compressed file    */
	char	M06[MAXZSWITCH]; /* Select the best method supported       */
	char	P07[MAXZSWITCH]; /* Password switch                        */
	char	O08[MAXZSWITCH]; /* Char(s) that precede the target path   */
	char	R09[MAXZSWITCH]; /* Char(s) that precede the response file */
      } COMP;

static COMP *cpi;
static DOBJ *dialog;

enum {	ID1_OUTDIR,
	ID1_INCDIR,
	ID1_EDITCMD,
	ID1_COMPRESS,
	ID1_OPTIONS,
	ID1_METHOD,
	ID1_FILTER,
	ID1_CANCEL	};

enum {	ID2_OUTDIR,
	ID2_INCDIR,
	ID2_EDITCMD,
	ID2_COMPRESS,
	ID2_OPTIONS,
	ID2_METHOD,
	ID2_CANCEL	};


static int response_open(void);
static int response_make(void);
static int response_addfile(const char *);
static int response_dofile(FBLK *);
static int response_scan_dofile(const char *, wfblk *);

static int ini_ermsg(const char *__section, const char *__entry);
static int ini_packerinfo(void);
static int ini_initcommand(void);
static int method_print(void);
static int method_select(void);
static int event_compress(void);
static int event_filter(void);

/* Make Response File */

static long fcount;

int response_open(void)
{
	char b[WMAXPATH];

	strfcat(b, envtemp, cp_ziplst);
	arc_outhnd = osopen(b, 0, M_WRONLY, A_CREATE | A_TRUNC);
	if (arc_outhnd == -1)
		return stderror(b, EMERRNO, 0);
	cflag |= _C_DELZIPLST;
	return 1;
}

int response_addfile(const char *name)
{
	int size;

	fcount++;
	strcpy(__srcfile, name);
	strcat(__srcfile, cp_newline);
	size = strlen(__srcfile);
	return (oswrite(arc_outhnd, __srcfile, size) != size);
}

int response_dofile(FBLK *b)
{
	char fb[256];

	if (ISSUBDIR(b->flag)) {
		if (progress_set(NULL, b->name, 0))
			return _ER_USERABORT;
		if (zflag & _C_ZINCDIR) {
			strfcat(fb, __srcpath, b->name);
			return scansub(fb, cp_stdmask, 0);
		}
	} else {
		if (progress_set(NULL, b->name, 1))
			return _ER_USERABORT;
		if (filter_fblk(filter, b))
			return response_addfile(b->name);
	}
	return 0;
}

int response_scan_dofile(const char *p, wfblk *w)
{
	char *q;

	if (progress_set(NULL, w->name, 1))
		return _ER_USERABORT;
	if (filter_wblk(filter, w) == 0)
		return 0;
	strfcat(__srcfile, p, w->name);
	q = __srcfile + strlen(__srcpath);
	if (*q == '\\')
		q++;
	return response_addfile(q);
}

int response_make(void)
{
	FBLK *b;
	int result;

	fcount = 0L;
	b = cpanel_findfirst();
	fp_fileblock = response_scan_dofile;
	fp_directory = scan_files;
	progress_open(cpi->section, NULL);
	if (ISSELECTED(b->flag)) {
		do {
			if ((result = response_dofile(b)) != 0)
				break;
			b->flag &= ~(_A_SELECTED);
		} while ((b = panel_ffselected(cpanel)) != NULL);
	} else {
		result = response_dofile(b);
	}
	progress_close();
	return result;
}

/* Read Config File --> [Packer] */

int ini_ermsg(const char *section, const char *entry)
{
	zflag &= ~_C_ZPACKERID;
	return ermsg(cp_badentry, cp_badentrymsg, section, entry);
}

int ini_packerinfo(void)
{
	char *p,*s;
	char entry[16];

	sprintf(entry, "%02u", zflag & _C_ZPACKERID);
	if ((p = inientry(INI_Packer, entry)) == NULL)
		return ini_ermsg(INI_Packer, entry);
	s = cpi->section;
	strncpy(s, p, 32);
	strcpy(entry, INI_00);
	if (*cpanel->flag & _W_LONGNAME)
		entry[0]++;
	if ((p = inientry(s, entry)) != NULL)
		strncpy(cpi->A00, p, MAXZCMD);
	entry[1]++; /* 01 */
	if ((p = inientry(s, entry)) != NULL)
		strncpy(cpi->A01, p, MAXZCMD);
	entry[1]++; /* 02 */
	if ((p = inientry(s, entry)) != NULL)
		strncpy(cpi->E02, p, MAXZCMD);
	entry[1]++; /* 03 */
	if ((p = inientry(s, entry)) != NULL)
		strncpy(cpi->E03, p, MAXZCMD);
	entry[1]++; /* 04 */
/*
 *	if ((p = inientry(s, entry)) != NULL)
 *		strncpy(cpi->D04, p, MAXZCMD);
 */
	entry[1]++; /* 05 */
	if ((p = inientry(s, entry)) != NULL)
		strncpy(cpi->M05, p, MAXZSWITCH);
	entry[1]++; /* 06 */
	if ((p = inientry(s, entry)) != NULL)
		strncpy(cpi->M06, p, MAXZSWITCH);
	entry[1]++; /* 07 */
	if ((p = inientry(s, entry)) != NULL)
		strncpy(cpi->P07, p, MAXZSWITCH);
	entry[1]++; /* 08 */
	if ((p = inientry(s, entry)) != NULL)
		strncpy(cpi->O08, p, MAXZSWITCH);
	entry[1]++; /* 09 */
	if ((p = inientry(s, entry)) != NULL)
		strncpy(cpi->R09, p, MAXZSWITCH);
	return 1;
}

/* Init command-string */

#define PWLEN	80

int ini_initcommand(void)
{
	int x;
	char *p;
	char *q;
	char password[PWLEN];

	q = cpi->command;
	if (zflag & _C_ZDECOMP) {
		if (zflag & _C_ZINCDIR)
			strcpy(q, cpi->E03);
		else
			strcpy(q, cpi->E02);
	} else {
		if (zflag & _C_ZINCDIR)
			strcpy(q, cpi->A01);
		else
			strcpy(q, cpi->A00);
	}
	if (q[0] == 0)
		return ini_ermsg(cpi->section, "00..03");
	strcat(q, " ");
	if ((zflag & _C_ZDECOMP) == 0) {
		p = NULL;
		x = (zflag & _C_ZBEST);
		if (x == _C_ZSMALLEST)
			p = cpi->M05;
		else if (x == _C_ZFASTEST)
			p = cpi->M06;
		if (p != NULL && *p) {
			strcat(q, p);
			strcat(q, " ");
		}
	}
	if (zflag & _C_ZPASSWORD) {
		if (cpi->P07[0] == 0)
			return ini_ermsg(cpi->section, cp_badpasswordsw);
		password[0] = 0;
		x = tgetline(cp_enterpassword, password, 32, PWLEN);
		if (x == 0 || password[0] == 0)
			return confirm_continue(cp_skippassword);
		strcat(q, cpi->P07);
		strcat(q, password);
		strcat(q, " ");
	}
	return 1;
}

/* Method Selection Dialog box */

int method_print(void)
{
	int x,y,z;

	z = (zflag & _C_ZDECOMP);
	y = dialog->rc.y + 2;
	x = dialog->rc.x + 14;
	if (z) {
		scputc(x, y, 0, 53, ' ');
		scputf(x, y, 0, 53, cp_compressusing,
			cpanel_findfirst()->name, cpi->section);
	} else {
		x += 4;
		scputc(x, y, 0, 32, ' ');
		scputf(x, y, 0, 32, cp_compressto, cpi->section);
	}
	return z;
}

int method_select(void)
{
	int x;
	DOBJ *d;
	TOBJ *o;
	RECT rc;
	char *p;
	char entry[32];

	memzero(&cpi->section, sizeof(COMP) - 10);
	if ((d = rsopen(IDD_Archive)) == NULL)
		return _C_ESCAPE;
	o = d->object;
	d->index = (zflag & _C_ZPACKERID);
	strcpy(entry, INI_00);
	for (x = 0; x < MAXPACKERS; x++) {
		if ((p = inientry(INI_Packer, entry)) == NULL)
			break;
		strcpy(o[x].data, ".");
		strcat(o[x].data, p);
		wcputs(rcbprc(d->wp + 2, o[x].rc, d->rc.col), 0, 13, o[x].data);
		o[x].flag &= ~_O_STATE;
		if (entry[1] == '9') {
			entry[1] = '0';
			entry[0]++;
		} else {
			entry[1]++;
		}
	}
	d->count = x;
	if (d->index >= x)
		d->index = 0;
	d->rc.row = x + 2;
	rc = d->rc;
	rc.x = 0;
	rc.y = 0;
	rcframe(rc, d->wp, 17, 0);
	if (x == 0 || (x = tdevent(d)) == 0) {
		tdclose(d);
		return _C_NORMAL;
	}
	tdclose(d);
	zflag &= ~_C_ZPACKERID;
	zflag |= (x - 1);
	if (ini_packerinfo() == 0)
		return _C_ESCAPE;
	method_print();
	while (mousep());
	return _C_NORMAL;
}

/* Compress Dialog box */

int event_compress(void)
{
	char *p,*q;
	char tmp[WMAXPATH];
	int result;

	if (envtemp == NULL)
		return stderror(cp_missingTEMP, EMUSER, _C_ESCAPE);
	if (ini_packerinfo() == 0)
		return _C_NORMAL;
	if (ini_initcommand() == 0)
		return _C_NORMAL;
	if (response_open() == 0)
		return _C_NORMAL;

	q = cpi->command;
	p = dialog->object[0].data;
	strcat(q, p);
	tmp[0] = 0;
	strcpy(tmp, cpi->R09);
	strfcat(tmp + strlen(tmp), envtemp, cp_ziplst);
	//tmp[0] = cpi->R09[0];
	//strfcat(&tmp[1], envtemp, cp_ziplst);
	strcat(q, " ");
	//if (tmp[0])
		strcat(q, tmp);
	//else
		//strcat(q, tmp + 1);
	result = response_make();
	close(arc_outhnd);
	if (result)
		return _C_NORMAL;
	if (fcount == 0L)
		return ermsg(cp_compress, cp_nothingtodo);
	if (dialog->object[ID1_EDITCMD].flag & _O_FLAGB) {
		if (!tgetline(cp_compress, q, 40, 128))
			return _C_NORMAL;
	}
	return ex_command(q);
}

/* Operation Filters */

int event_filter(void)
{
	int x,y;

	cmfilter();
	x = dialog->rc.x + 44;
	y = dialog->rc.y + 7;
	if (filter == NULL)
		scputc(x, y, 0, 1, ' ');
	else
		scputc(x, y, 0, 1, 7);
	return _C_NORMAL;
}

int cmcompress(void)
{
	int 	ifs;
	DOBJ *	d;
	TOBJ *	o;
	FBLK *	b;
	char *	p;
	panel *	t;
	COMP 	zip;

	if (!(b = cpanel_findfirst()))
		return 0;
	if ((p = cpanel_gettarget()) == NULL)
		return stderror(cp_needpanel, EMUSER, 0);
	t = &panela;
	if (cpanel == t)
		t = &panelb;
	if (*t->flag & _W_ARCHIVE || ISARCHIVE(b->flag))
		return stderror(cp_linkerror, EMUSER, 0);
	strcpy(__outpath, wlongpath(p, NULL));
	strcpy(__srcpath, wlongpath(cpanel->ide->path, NULL));
	if (stricmp(__srcpath, __outpath) != 0 &&
	    recursive(b->name, __srcpath, __outpath))
		return stderror(cp_recursive, EMUSER, 0);
	if ((d = rsopen(IDD_Compress)) == NULL)
		return 0;
	zflag &= ~_C_ZDECOMP;
	cpi = &zip;
	memzero(cpi, sizeof(COMP));
	filter = NULL;
	dialog = d;
	o = d->object;
	o[ID1_OPTIONS].proc = cmcompression;
	o[ID1_METHOD].proc = method_select;
	o[ID1_FILTER].proc = event_filter;
	strfcat(o[ID1_OUTDIR].data, __outpath, "default");
	if (zflag & _C_ZINCDIR)
		o[ID1_INCDIR].flag |= _O_FLAGB;
	tdinit(d);
	tdshow(d);
	if (ini_packerinfo() == 0) {
		tdclose(d);
		return 0;
	}
	method_print();
	if (rsevent(IDD_Compress, d)) {
		zflag &= ~_C_ZINCDIR;
		if (o[ID1_INCDIR].flag & _O_FLAGB)
			zflag |= _C_ZINCDIR;
		ifs = _ifsmgr;
		if ((*cpanel->flag & _W_LONGNAME) == 0)
			_ifsmgr = 0;
		event_compress();
		_ifsmgr = ifs;
		tdclose(d);
		return 1;
	}
	tdclose(d);
	return 0;
}

/* Decompress Dialog box */

int cmdecompress(void)
{
	int 	x,y;
	DOBJ *	d;
	TOBJ *	o;
	FBLK *	b;
	COMP 	zip;
	char *	p;
	char *	q;
	char 	entry[32];
	char 	section[32];

	if (!(b = cpanel_findfirst()))
		return 0;
	if (ISSUBDIR(b->flag))
		return 0;
	if (ISARCHIVE(b->flag))
		return ermsg(NULL, cp_linkerror);
	if (ISSELECTED(b->flag))
		return notsup();

	if ((p = strrchr(b->name, '.')) != NULL) {
		strncpy(section, p + 1, 30);
		strupr(section);
		strcpy(entry, INI_00);
		for (x = 0; x < MAXPACKERS; x++) {
			if ((p = inientry(INI_Packer, entry)) == NULL)
				break;
			if (strnicmp(section, p, 12) == 0) {
				zflag &= ~_C_ZPACKERID;
				zflag |= x;
				break;
			}
			entry[1]++;
			if (entry[1] > '9') {
				entry[0]++;
				entry[1] = '0';
			}
		}
	}

	if ((p = cpanel_gettarget()) == NULL)
		return stderror(cp_needpanel, EMUSER, 0);
	zflag |= _C_ZDECOMP;
	cpi = &zip;
	memzero(cpi, sizeof(COMP));
	if ((d = rsopen(IDD_Decompress)) == NULL)
		return 0;
	dialog = d;
	o = d->object;
	strcpy(o[ID2_OUTDIR].data, p);
	if (zflag & _C_ZINCDIR)
		o[ID2_INCDIR].flag |= _O_FLAGB;
	o[ID2_OPTIONS].proc = cmcompression;
	o[ID2_METHOD].proc = method_select;
	tdinit(d);
	o[ID2_OUTDIR].flag |= _O_STATE;
	tdshow(d);
	if (ini_packerinfo() == 0) {
		tdclose(d);
		return 0;
	}
	method_print();
	x = rsevent(IDD_Decompress, d);
	y = (o[ID2_INCDIR].flag & _O_FLAGB);
	tdclose(d);
	if (x == 0)
		return 0;
	x = (o[ID2_EDITCMD].flag & _O_FLAGB);
	zflag &= ~_C_ZINCDIR;
	if (y)
		zflag |= _C_ZINCDIR;
	if (ini_packerinfo() == 0)
		return 0;
	if (ini_initcommand() == 0)
		return 0;
	q = cpi->command;
	strcat(q, wshortname(b->name));
	strcat(q, " ");
	if (cpi->O08[0])
		strcat(q, cpi->O08);
	strcat(q, p);
	if (x) {
		if (!tgetline(cp_decompress, q, 40, 128))
			return _C_NORMAL;
	}
	return ex_command(q);
}

/* Compression Setup Dialog box */

enum {  ID_ZAUTOSELECT,
	ID_ZSMALLEST,
	ID_ZFASTEST,
	ID_ZPASSWORD	};

int cmcompression(void)
{
	DOBJ *d;
	TOBJ *o;

	if ((d = rsopen(IDD_Compression)) == 0)
		return 0;
	o = d->object;
	if (zflag & _C_ZSMALLEST)
		o[ID_ZSMALLEST].flag |= _O_RADIO;
	else if (zflag & _C_ZFASTEST)
		o[ID_ZFASTEST].flag |= _O_RADIO;
	else
		o[ID_ZAUTOSELECT].flag |= _O_RADIO;
	if (zflag & _C_ZPASSWORD)
		o[ID_ZPASSWORD].flag |= _O_FLAGB;
	tdinit(d);
	if (rsevent(IDD_Compression, d)) {
		zflag &= ~(_C_ZPASSWORD|_C_ZSMALLEST|_C_ZFASTEST);
		if (o[ID_ZSMALLEST].flag & _O_RADIO)
			zflag |= _C_ZSMALLEST;
		else if (o[ID_ZFASTEST].flag & _O_RADIO)
			zflag |= _C_ZFASTEST;
		if (o[ID_ZPASSWORD].flag & _O_FLAGB)
			zflag |= _C_ZPASSWORD;
	}
	twclose(d);
	return 1;
}

/******** Resource begin Compression *
	{ 0x041C,   6,   4, {17, 6,46,13} },
	{ 0x0001,   0, 'A', { 5, 3,36, 1} },
	{ 0x0001,   0, 'S', { 5, 4,24, 1} },
	{ 0x0001,   0, 'U', { 5, 5,22, 1} },
	{ 0x0002,   0, 'P', { 5, 8,25, 1} },
	{ 0x0000,   0, 'O', { 4,11, 6, 1} },
	{ 0x0200,   0, 'C', {14,11,10, 1} },
********* Resource data  *******************/
static int Compression_RC[] = {
	0x0562,0x041C,0x0406,0x0611,0x0D2E,0x0001,0x4100,0x0305,
	0x0124,0x0001,0x5300,0x0405,0x0118,0x0001,0x5500,0x0505,
	0x0116,0x0002,0x5000,0x0805,0x0119,0x0000,0x4F00,0x0B04,
	0x0106,0x0200,0x4300,0x0B0E,0x010A,0x2EF0,0xF050,0x2A34,
	0x0EF0,0xF02E,0x2A23,0xF02E,0x2A34,0xF02E,0x2A26,0xF02E,
	0x2A58,0x07F0,0xF02E,0x2A37,0xF02E,0x2A77,0x5050,0xF05C,
	0x5003,0xF020,0x2A03,0x5050,0xF05C,0x5007,0xF020,0x2A1A,
	0x06F0,0xF020,0x2A04,0x0AF0,0xF020,0x2A15,0x11F0,0x4320,
	0x6D6F,0x7270,0x7365,0x6973,0x6E6F,0x43F0,0xDA20,0x2020,
	0x7453,0x726F,0x6761,0x2065,0x656D,0x6874,0x646F,0xF020,
	0xC415,0xF0BF,0x2006,0x20B3,0x2028,0x2029,0x7541,0x6F74,
	0x616D,0x6974,0x6163,0x6C6C,0x2079,0x6573,0x656C,0x7463,
	0x6220,0x7365,0x2074,0x656D,0x6874,0x646F,0xB320,0x06F0,
	0xB320,0x2820,0x2920,0x4320,0x6572,0x7461,0x2065,0x6D73,
	0x6C61,0x656C,0x7473,0x6620,0x6C69,0xF065,0x200D,0xF0B3,
	0x2006,0x20B3,0x2028,0x2029,0x7355,0x2065,0x6166,0x7473,
	0x7365,0x2074,0x656D,0x6874,0x646F,0x0FF0,0xB320,0x06F0,
	0xC020,0x26F0,0xD9C4,0x06F0,0xDA20,0x20C4,0x6547,0x656E,
	0x6172,0x206C,0x1CF0,0xBFC4,0x06F0,0xB320,0x5B20,0x5D20,
	0x4320,0x6E6F,0x6966,0x6D72,0x7720,0x7469,0x2068,0x6170,
	0x7373,0x6F77,0x6472,0x0CF0,0xB320,0x06F0,0xC020,0x26F0,
	0xD9C4,0x37F0,0x4F20,0x206B,0xDC20,0x05F0,0x4320,0x6E61,
	0x6563,0x206C,0xDC20,0x1AF0,0xF020,0xDF06,0x04F0,0xF020,
	0xDF0A,0x15F0,0x2A20,
}; //	374 byte
ROBJ *IDD_Compression = (ROBJ *)&Compression_RC;
/******** Resource end   Compression */

/******** Resource begin Decompress *
	{ 0x0C1C,   7,   3, { 5, 7,70, 9} },
	{ 0x0004,  16, 'P', { 3, 3,64, 1} },
	{ 0x0002,   0, 'E', { 5, 5,33, 1} },
	{ 0x0002,   0, 'L', {41, 5,21, 1} },
	{ 0x0000,   0, 'D', { 3, 7,14, 1} },
	{ 0x4000,   0, 'O', {20, 7,11, 1} },
	{ 0x4000,   0, 'S', {34, 7,17, 1} },
	{ 0x0A00,   0, 'C', {56, 7,10, 1} },
********* Resource data  *******************/
static int Decompress_RC[] = {
	0x06C2,0x0C1C,0x0307,0x0705,0x0946,0x0004,0x5010,0x0303,
	0x0140,0x0002,0x4500,0x0505,0x0121,0x0002,0x4C00,0x0529,
	0x0115,0x0000,0x4400,0x0703,0x010E,0x4000,0x4F00,0x0714,
	0x010B,0x4000,0x5300,0x0722,0x0111,0x0A00,0x4300,0x0738,
	0x010A,0x46F0,0xF050,0x2A4E,0xF02E,0x2A40,0x40F0,0xF007,
	0x2A52,0xF03F,0x2A30,0xF03F,0x2A54,0x5050,0xF05C,0x500B,
	0x2A20,0x502A,0x5C50,0x08F0,0x2050,0x2A2A,0x5050,0xF05C,
	0x500E,0xF020,0x2A04,0x5050,0xF05C,0x5007,0xF020,0x2A07,
	0x0EF0,0xF020,0x2A03,0x0BF0,0xF020,0x2A03,0x11F0,0xF020,
	0x2A05,0x0AF0,0xF020,0x2A03,0x1EF0,0x4420,0x6365,0x6D6F,
	0x7270,0x7365,0xF073,0x2067,0x6544,0x6F63,0x706D,0x6572,
	0x7373,0x3CF0,0xF020,0xFA40,0x06F0,0xDA20,0x3EF0,0xBFC4,
	0x06F0,0xB320,0x5B20,0x5D20,0x4520,0x7478,0x6172,0x7463,
	0x6620,0x6C69,0x7365,0x7720,0x7469,0x2068,0x7566,0x6C6C,
	0x7020,0x7461,0x7368,0x03F0,0x5B20,0x5D20,0x4520,0x6964,
	0x2074,0x6F43,0x6D6D,0x6E61,0x2064,0x694C,0x656E,0x04F0,
	0xB320,0x06F0,0xC020,0x3EF0,0xD9C4,0x08F0,0x4420,0x6365,
	0x6D6F,0x7270,0x7365,0x2073,0xDC20,0x04F0,0x4F20,0x7470,
	0x6F69,0x736E,0x2020,0xF0DC,0x2004,0x6553,0x656C,0x7463,
	0x6D20,0x7465,0x6F68,0x2064,0xDC20,0x06F0,0x4320,0x6E61,
	0x6563,0x206C,0xDC20,0x07F0,0xF020,0xDF0E,0x03F0,0xF020,
	0xDF0B,0x03F0,0xF020,0xDF11,0x05F0,0xF020,0xDF0A,0x03F0,
	0x2A20,
}; //	354 byte
ROBJ *IDD_Decompress = (ROBJ *)&Decompress_RC;
/******** Resource end   Decompress */

/******** Resource begin Archive *
	{ 0x041C,  16,   0, {40, 3,17,18} },
	{ 0x8006,   2,   0, { 2, 1,13, 1} },
	{ 0x8006,   2,   0, { 2, 2,13, 1} },
	{ 0x8006,   2,   0, { 2, 3,13, 1} },
	{ 0x8006,   2,   0, { 2, 4,13, 1} },
	{ 0x8006,   2,   0, { 2, 5,13, 1} },
	{ 0x8006,   2,   0, { 2, 6,13, 1} },
	{ 0x8006,   2,   0, { 2, 7,13, 1} },
	{ 0x8006,   2,   0, { 2, 8,13, 1} },
	{ 0x8006,   2,   0, { 2, 9,13, 1} },
	{ 0x8006,   2,   0, { 2,10,13, 1} },
	{ 0x8006,   2,   0, { 2,11,13, 1} },
	{ 0x8006,   2,   0, { 2,12,13, 1} },
	{ 0x8006,   2,   0, { 2,13,13, 1} },
	{ 0x8006,   2,   0, { 2,14,13, 1} },
	{ 0x8006,   2,   0, { 2,15,13, 1} },
	{ 0x8006,   2,   0, { 2,16,13, 1} },
********* Resource data  *******************/
static int Archive_RC[] = {
	0x05A7,0x041C,0x0010,0x0328,0x1211,0x8006,0x0002,0x0102,
	0x010D,0x8006,0x0002,0x0202,0x010D,0x8006,0x0002,0x0302,
	0x010D,0x8006,0x0002,0x0402,0x010D,0x8006,0x0002,0x0502,
	0x010D,0x8006,0x0002,0x0602,0x010D,0x8006,0x0002,0x0702,
	0x010D,0x8006,0x0002,0x0802,0x010D,0x8006,0x0002,0x0902,
	0x010D,0x8006,0x0002,0x0A02,0x010D,0x8006,0x0002,0x0B02,
	0x010D,0x8006,0x0002,0x0C02,0x010D,0x8006,0x0002,0x0D02,
	0x010D,0x8006,0x0002,0x0E02,0x010D,0x8006,0x0002,0x0F02,
	0x010D,0x8006,0x0002,0x1002,0x010D,0x32F1,0xF12A,0x2032,
	0x2A2A,
}; //	146 byte
ROBJ *IDD_Archive = (ROBJ *)&Archive_RC;
/******** Resource end   Archive */

/******** Resource begin Compress *
	{ 0x0C1C,   8,   3, { 5, 7,70, 9} },
	{ 0x0004,  16, 'U', { 3, 3,64, 1} },
	{ 0x0002,   0, 'R', { 5, 5,25, 1} },
	{ 0x0002,   0, 'E', {36, 5,21, 1} },
	{ 0x0000,   0, 'P', { 3, 7,12, 1} },
	{ 0x4000,   0, 'O', {18, 7,11, 1} },
	{ 0x4000,   0, 'T', {32, 7, 8, 1} },
	{ 0x4000,   0, 'F', {43, 7,10, 1} },
	{ 0x0A00,   0, 'C', {56, 7,10, 1} },
********* Resource data  *******************/
static int Compress_RC[] = {
	0x06D2,0x0C1C,0x0308,0x0705,0x0946,0x0004,0x5510,0x0303,
	0x0140,0x0002,0x5200,0x0505,0x0119,0x0002,0x4500,0x0524,
	0x0115,0x0000,0x5000,0x0703,0x010C,0x4000,0x4F00,0x0712,
	0x010B,0x4000,0x5400,0x0720,0x0108,0x4000,0x4600,0x072B,
	0x010A,0x0A00,0x4300,0x0738,0x010A,0x46F0,0xF050,0x2A52,
	0xF02E,0x2A3C,0x40F0,0xF007,0x2A52,0xF03F,0x2A1E,0xF03F,
	0x2A66,0x05F0,0x5C50,0x06F0,0x2050,0x2A2A,0x5050,0xF05C,
	0x5008,0x2A20,0x502A,0x5C50,0x05F0,0x2050,0x2A2A,0x5050,
	0xF05C,0x5007,0x2A20,0x502A,0x5C50,0x07F0,0x2050,0x07F0,
	0xF02A,0x200C,0x03F0,0xF02A,0x200B,0x03F0,0xF02A,0x2008,
	0x03F0,0xF02A,0x200A,0x03F0,0xF02A,0x200A,0x03F0,0xF02A,
	0x201F,0x6F43,0x706D,0x6572,0x7373,0x68F0,0x4320,0x6D6F,
	0x7270,0x7365,0x2073,0x7375,0x6E69,0xF067,0x2038,0x40F0,
	0xF0FA,0x2006,0xF0DA,0xC43E,0xF0BF,0x2006,0x20B3,0x205B,
	0x205D,0x6552,0x7563,0x7372,0x2065,0x7573,0x6462,0x7269,
	0x6365,0x6F74,0x6972,0x7365,0x05F0,0x5B20,0x5D20,0x4520,
	0x6964,0x2074,0x6F43,0x6D6D,0x6E61,0x2064,0x694C,0x656E,
	0x09F0,0xB320,0x06F0,0xC020,0x3EF0,0xD9C4,0x08F0,0x4320,
	0x6D6F,0x7270,0x7365,0x2073,0xDC20,0x04F0,0x4F20,0x7470,
	0x6F69,0x736E,0x2020,0xF0DC,0x2004,0x7954,0x6570,0x2020,
	0xF0DC,0x2004,0x6946,0x746C,0x7265,0x2020,0xF0DC,0x2004,
	0x6143,0x636E,0x6C65,0x2020,0xF0DC,0x2007,0x0CF0,0xF0DF,
	0x2003,0x0BF0,0xF0DF,0x2003,0x08F0,0xF0DF,0x2003,0x0AF0,
	0xF0DF,0x2003,0x0AF0,0xF0DF,0x2003,0x2A2A,
}; //	380 byte
ROBJ *IDD_Compress = (ROBJ *)&Compress_RC;
/******** Resource end   Compress */
