/* ***************************************************************** *
 * Copyright 1998 International Business Machines Corporation. All   *
 * Rights Reserved.                                                  *
 *                                                                   *
 * Please read this carefully.  Your use of this reference           *
 * implementation of certain of the IETF public-key infrastructure   *
 * specifications ("Software") indicates your acceptance of the      *
 * following.  If you do not agree to the following, do not install  *
 * or use any of the Software.                                       *
 *                                                                   *
 * Permission to use, reproduce, distribute and create derivative    *
 * works from the Software ("Software Derivative Works"), and to     *
 * distribute such Software Derivative Works is hereby granted to    *
 * you by International Business Machines Corporation ("IBM").  This *
 * permission includes a license under the patents of IBM that are   *
 * necessarily infringed by your use of the Software as provided by  *
 * IBM.                                                              *
 *                                                                   *
 * IBM licenses the Software to you on an "AS IS" basis, without     *
 * warranty of any kind.  IBM HEREBY EXPRESSLY DISCLAIMS ALL         *
 * WARRANTIES OR CONDITIONS, EITHER EXPRESS OR IMPLIED, INCLUDING,   *
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OR CONDITIONS OF       *
 * MERCHANTABILITY, NON INFRINGEMENT AND FITNESS FOR A PARTICULAR    *
 * PURPOSE.  You are solely responsible for determining the          *
 * appropriateness of using this Software and assume all risks       *
 * associated with the use of this Software, including but not       *
 * limited to the risks of program errors, damage to or loss of      *
 * data, programs or equipment, and unavailability or interruption   *
 * of operations.                                                    *
 *                                                                   *
 * IBM WILL NOT BE LIABLE FOR ANY DIRECT DAMAGES OR FOR ANY SPECIAL, *
 * INCIDENTAL, OR  INDIRECT DAMAGES OR FOR ANY ECONOMIC              *
 * CONSEQUENTIAL DAMAGES (INCLUDING LOST PROFITS OR SAVINGS), EVEN   *
 * IF IBM HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.  IBM  *
 * will not be liable for the loss of, or damage to, your records or *
 * data, or any damages claimed by you based on a third party claim. *
 *                                                                   *
 * IBM wishes to obtain your feedback to assist in improving the     *
 * Software.  You grant IBM a world-wide, royalty-free right to use, *
 * copy, distribute, sublicense and prepare derivative works based   *
 * upon any feedback, including materials, error corrections,        *
 * Software Derivatives, enhancements, suggestions and the like that *
 * you provide to IBM relating to the Software (this does not        *
 * include products for which you charge a royalty and distribute to *
 * IBM under other terms and conditions).                            *
 *                                                                   *
 * You agree to distribute the Software and any Software Derivatives *
 * under a license agreement that: 1) is sufficient to notify all    *
 * licensees of the Software and Software Derivatives that IBM       *
 * assumes no liability for any claim that may arise regarding the   *
 * Software or Software Derivatives, and 2) that disclaims all       *
 * warranties, both express and implied, from IBM regarding the      *
 * Software and Software Derivatives.  (If you include this          *
 * Agreement with any distribution of the Software or Software       *
 * Derivatives you will have met this requirement.)  You agree that  *
 * you will not delete any copyright notices in the Software.        *
 *                                                                   *
 * This Agreement is the exclusive statement of your rights in the   *
 * Software as provided by IBM.   Except for the rights granted to   *
 * you in the second paragraph above, You are not granted any other  *
 * patent rights, including but not limited to the right to make     *
 * combinations of the Software with products that infringe IBM      *
 * patents. You agree to comply with all applicable laws and         *
 * regulations, including all export and import laws and regulation. *
 * This Agreement is governed by the laws of the State of New York.  *
 * This Agreement supersedes all other communications,               *
 * understandings or agreements we may have had prior to this        *
 * Agreement.                                                        *
 * ***************************************************************** */

#include <platform.h>

#ifdef NEED_XPG_MESSAGES
#include "pkiqdxpg.h"
#include "pkiTask.h"

xpg_catalog_t * xpg_catalog_t::root = NULL;

static pki_mutex_t mutex = PKI_MUTEX_INIT;

static set_header_t null_set_header;
static msg_header_t null_msg_header;
static cat_header_t null_cat_header;

static int cmpstr(const char * s1, const char * s2) {
  while ((*s1 != 0) && (*s2 != 0)) {
    if (tolower(*s1) != tolower(*s2)) return 1;
    s1++, s2++;
  };
  return *s1!=*s2;
}

bool msg_header_t::operator == (const msg_header_t & o) const {
  return (memcmp(this, &o, sizeof(o)) == 0);
};

bool msg_header_t::operator != (const msg_header_t & o) const {
  return !(*this == o);
};

bool set_header_t::operator == (const set_header_t & o) const {
  return (memcmp(this, &o, sizeof(o)) == 0);
};

bool set_header_t::operator != (const set_header_t & o) const {
  return !(*this == o);
};

bool cat_header_t::operator == (const cat_header_t & o) const {
  return (memcmp(this, &o, sizeof(o)) == 0);
};

bool cat_header_t::operator != (const cat_header_t & o) const {
  return !(*this == o);
};


const char * xpg_catalog_t::lookupMessage(pki_msgcat_handle_t h, int theSet, int theMsg) {
  xpg_catalog_t * p = NULL;
  xpg_setlist_t * s = NULL;
  xpg_msglist_t * m = NULL;

  const char * msg = NULL;
  pkiLockMutex(&mutex);
  p = root;
  while ((p != NULL) && (p->id != h)) p=p->next;
  if (p) {
// Now we've found the catalog.  Search for the set, then the message.
    s = p->set;        
    while ((s != NULL) && (s->header.id != theSet)) s=s->next;
    if (s) {
      m = s->msgs;
      while ((m != NULL) && (m->header.id != theMsg)) m=m->next;
      if (m) msg = m->text;
    };
  };
  pkiUnlockMutex(&mutex);
  return msg;
}




xpg_msglist_t::xpg_msglist_t(int id, const char * msgText, const char * mn) {
  next = NULL;

  header.id = id;
  if (mn) strncpy(header.mnemonic, mn, sizeof(header.mnemonic));
  else memset(header.mnemonic, 0, sizeof(header.mnemonic));
  header.length = strlen(msgText);
  text = (char *)malloc(strlen(msgText) + 1);
  strcpy(text, msgText);
}

xpg_msglist_t::xpg_msglist_t(const msg_header_t & h, FILE * f) {
  msg_header_t loc_header;

  next = NULL;
  header = h;
  text = (char *)malloc(header.length + 1);
  text[header.length] = 0;

  if (fread(text, header.length, 1, f) != 1) return;

  if (fread(&loc_header, sizeof(loc_header), 1, f) != 1) return;
  if (loc_header == null_msg_header) return;
  next = new xpg_msglist_t(loc_header, f);
}

xpg_setlist_t::xpg_setlist_t(int id, const char * mn) {

  msgs = NULL;
  next = NULL;
  header.id = id;
  if (mn) strncpy(header.mnemonic, mn, sizeof(header.mnemonic));
  else memset(header.mnemonic, 0, sizeof(header.mnemonic));
   
}

xpg_setlist_t::xpg_setlist_t(const set_header_t & h, FILE * f) {
  msg_header_t msg_header;
  set_header_t loc_header;

  msgs = NULL;
  next = NULL;
  header = h;
  
  if ((fread(&msg_header, sizeof(msg_header), 1, f) == 1) 
    && (msg_header != null_msg_header)) {
      if (msg_header.majic != 27) {
#ifdef DEBUG
        fprintf(stdout,"Error reading meg header.  Majic should be 27, was %d\n", msg_header.majic);
        fflush(stdout);
#endif
        return;
      };
    msgs = new xpg_msglist_t(msg_header, f);
  } else msgs = NULL;

  if (f == NULL) return;
  if (fread(&loc_header, sizeof(loc_header), 1, f) != 1) return;
  if (loc_header == null_set_header) return;
  next = new xpg_setlist_t(loc_header, f);
  
}


 
uint32 xpg_msglist_t::write(FILE * f) {
  uint32 st = 0;

  if (header.majic != 27) {
#ifdef DEBUG
    fprintf(stdout, "Error writing msglist - header corrupt (majic should be 27, was %d)\n", header.majic);
    fflush(stdout);
#endif
    return XPG_CATALOG_CORRUPT;
  };

  
  fwrite(&header, sizeof(header), 1, f);
  fwrite(text, header.length, 1, f);

  if (next) {
    st = next->write(f);
  } else {
    fwrite(&null_msg_header, sizeof(null_msg_header), 1, f);
  };
  return st;
}

uint32 xpg_msglist_t::genHeader(FILE * f, int setId) {
  uint32 st = 0;

  if (header.mnemonic[0] != 0) {
    fprintf(f, "#define %s %d\n", header.mnemonic, pkiMakeSts(setId, header.id));
  } else {
    fprintf(f, "#define UNNAMED_%d %d\n", header.id, pkiMakeSts(setId, header.id));
  };

  if (next) st = next->genHeader(f, setId);
  return st;
}

uint32 xpg_setlist_t::write(FILE * f) {
  uint32 st = 0;


  if (header.majic != 42) {
#ifdef DEBUG
    fprintf(stdout, "Error writing setlist - header corrupt (majic should be 42, was %d)\n", header.majic);
    fflush(stdout);
#endif
    return XPG_CATALOG_CORRUPT;
  };

  fwrite(&header, sizeof(header), 1, f);
  
  if (msgs) {
    st = msgs->write(f);
  } else {
    fwrite(&null_msg_header, sizeof(null_msg_header), 1, f);
  };

  
  if (next) {
    st = next->write(f);
  } else {
    fwrite(&null_set_header, sizeof(null_set_header), 1, f);
  };
  return st;
}

uint32 xpg_setlist_t::genHeader(FILE * f) {
  uint32 st = 0;

  if (header.mnemonic[0] == 0) {
    fprintf(f, "/* Set %d */\n", header.id);
  } else {
    fprintf(f, "/* Set %s (%d) */\n", header.mnemonic, header.id);
  };
  
  if (msgs) {
    st = msgs->genHeader(f, header.id);
  };

  if (next) {
    st = next->genHeader(f);
  };
  return st;
}

uint32 xpg_catalog_t::write(pki_msgcat_handle_t h, FILE * f) {
  uint32 st;
  xpg_catalog_t * p;  
  
  if ((st = pkiCreateMutex(&mutex, true)) != 0) return st;
  try {
// we've now locked the mutex.  Find the specified catalog, then write it...
    p = root;
    while (p != NULL) {
      if (p->id == h) break;
      p=p->next;
    };
    if (p == NULL) {
// The catalog doesn't exist.
      pkiUnlockMutex(&mutex);
      return XPG_CATALOG_NOT_FOUND;
    };

    if (p->header.majic != 35) {
#ifdef DEBUG
      fprintf(stdout, "Error writing catalog - header corrupt (majic should be 35, was %d)\n", p->header.majic);
      fflush(stdout);
#endif
      return XPG_CATALOG_CORRUPT;
    };
    fwrite(&(p->header), sizeof(p->header), 1, f);
    if (p->set) {
      st = p->set->write(f);
    } else {
      fwrite(&null_set_header, sizeof(null_set_header), 1, f);
    };
  } catch(...) {
    pkiUnlockMutex(&mutex);
    throw;
  };
  pkiUnlockMutex(&mutex);
  return st;
}


uint32 xpg_catalog_t::genHeader(pki_msgcat_handle_t h, const char * ident, FILE * f) {
  uint32 st;
  xpg_catalog_t * p;  
  
  if ((st = pkiCreateMutex(&mutex, true)) != 0) return st;
  try {
// we've now locked the mutex.  Find the specified catalog, then write it...
    p = root;
    while (p != NULL) {
      if (p->id == h) break;
      p=p->next;
    };
    if (p == NULL) {
// The catalog doesn't exist.
      pkiUnlockMutex(&mutex);
      return XPG_CATALOG_NOT_FOUND;
    };

    fprintf(f, "#ifndef %s\n", ident);
    fprintf(f, "#define %s\n", ident);
    fprintf(f, "\n/* Automatically-generated header.  Do not edit. */\n\n");
    if (p->set) {
      st = p->set->genHeader(f);
    };

    fprintf(f, "#endif\n");

  } catch(...) {
    pkiUnlockMutex(&mutex);
    throw;
  };
  pkiUnlockMutex(&mutex);
  return st;
}


pki_msgcat_handle_t xpg_catalog_t::read(FILE * f) {
  pki_msgcat_handle_t id;
  cat_header_t header;
  set_header_t set_header;
  xpg_catalog_t * newcat;
  xpg_catalog_t * p;
  xpg_catalog_t * pp;

  if (fread(&header, sizeof(header), 1, f) != 1) return 0;

  if (header.majic != 35) {
#ifdef DEBUG
    fprintf(stdout,"Error reading catalog header.  Majic should be 35, was %d\n", header.majic);
    fflush(stdout);
#endif
    return 0;
  };

  if (pkiCreateMutex(&mutex, true) != 0) return 0;
  
  try {
// we've now locked the mutex.  Look for the specified catalog...
    p = root;
    pp = NULL;
    while (p != NULL) {
      id = p->id;
      if (p->header == header) break;
      pp = p;
      p=p->next;
    };
    if (p == NULL) {
// The catalog was not found.  Load it...
      id++;
      newcat = new xpg_catalog_t;

  
      newcat->header = header;
      newcat->id = id;
      if ((fread(&set_header, sizeof(set_header), 1, f) == 1) 
        && (set_header != null_set_header)) {
        if (set_header.majic != 42) {
#ifdef DEBUG
          fprintf(stdout,"Error reading set header.  Majic should be 42, was %d\n", set_header.majic);
          fflush(stdout);
#endif
          return 0;
        };
        newcat->set = new xpg_setlist_t(set_header, f);
      };
// If pp is null, this is the first catalog we've read, otherwise it points to the
// last catalog.
      if (pp) pp->next = newcat;
      else root = newcat;
    };
  } catch(...) {
    pkiUnlockMutex(&mutex);
    throw;
  };
  pkiUnlockMutex(&mutex);
  return id;
}


pki_msgcat_handle_t xpg_catalog_t::createCatalog(const char * name) {
  pki_msgcat_handle_t id = 0;
  cat_header_t header;
  xpg_catalog_t * p;
  xpg_catalog_t * pp;
  xpg_catalog_t * newcat;

  strncpy(header.name, name, sizeof(header.name));

  if (pkiCreateMutex(&mutex, true) != 0) return 0;
  try {
// Walk the chain and see if we've already got a catalog by this name.
    p = root;
    pp = NULL;
    while (p != NULL) {
      id = p->id;
      if (p->header == header) break;
      pp = p;
      p=p->next;
    };
    if (p == NULL) {
// The catalog was not found.  Create it...
      id++;
      newcat = new xpg_catalog_t;
      newcat->header = header;
      newcat->id = id;
// If pp is null, this is the first catalog we've created, otherwise it points to the
// last catalog.
      if (pp) pp->next = newcat;
      else root = newcat;
    };
  } catch (...) {
    pkiUnlockMutex(&mutex);
    throw;
  };
  pkiUnlockMutex(&mutex);
  return id;
}

static int r40cvt(const char * text) {
  int s = 0;
  if (strlen(text) < 1) return s;
  s = (tolower(text[0]) - 'a' + 1) * 27 * 27;
  if (strlen(text) < 2) return s;
  s += (tolower(text[1]) - 'a' + 1) * 27;
  if (strlen(text) < 3) return s;
  s += (tolower(text[2]) - 'a' + 1);
  return s;
}


uint32 xpg_catalog_t::defineMessage(pki_msgcat_handle_t h, 
                                    const char * setName, 
                                    const char * msgName,
                                    const char * text,
                                    int & theSet,
                                    int & theMsg) {
// Set & msg specified by name
//
// Find the specified set within the catalog, or create it if it doesn't exist.
// Add the specified message to the set if it doesn't already exist.
// If the specified message name is found, an error will be flagged.
// Set names of three of fewer characters will be directly converted to
// a set number; other set names will just get the next allocated number.

  xpg_catalog_t * p;
  xpg_setlist_t * s;  
  xpg_setlist_t * ss;  
  xpg_msglist_t * m;
  xpg_msglist_t * mm;
  int id;
  size_t sl;

  theSet = 0;
  theMsg = 0;
  if (h==0) return XPG_CATALOG_NOT_OPEN;
  sl = strlen(setName);

  if ((sl <= 3) && 
      (sl >= 1) && 
      isalpha(setName[0]) && 
      ((sl < 2) || isalpha(setName[1])) && 
      ((sl < 3) || isalpha(setName[2]))
     ) {
    theSet = r40cvt(setName);
  };

  if (pkiCreateMutex(&mutex, true) != 0) return 0;
  try {
// Walk the chain and see if we've already got a catalog by this name.
    p = root;
    while (p != NULL) {
      if (p->id == h) break;
      p=p->next;
    };
    if (p == NULL) {
// The catalog doesn't exist.
      pkiUnlockMutex(&mutex);
      return XPG_CATALOG_NOT_FOUND;
    };
// We've found the catalog.  See if we can find the set...
    s = p->set;        
    ss = NULL;
    id = 0;
    while ((s != NULL) && 
      (cmpstr(s->header.mnemonic,setName)!=0) &&
      (s->header.id != theSet)) {
      if (s->header.id < 26*26) id = max(id,s->header.id);
      ss = s;
      s=s->next;
    };
    if (s==NULL) {
// We didn't find the set.  Create it...
      if (theSet == 0) theSet = id+1;
      
      s = new xpg_setlist_t(theSet, setName);

      if (ss) ss->next = s;
      else p->set = s;

    } else {
// We found a set that matches either the name or the ID.  Check that it matches both!
#     ifdef DEBUG
      fprintf(stdout, "Defining set %s, id %d\n", setName, theSet);
      fprintf(stdout, "Found set %s, id %d\n", s->header.mnemonic, s->header.id);
#     endif
      if ((cmpstr(s->header.mnemonic,setName)!=0) || (s->header.id != theSet)) {
#       ifdef DEBUG
        fprintf(stdout, "cmpstr returned %d\n", cmpstr(s->header.mnemonic,setName));
#       endif
        pkiUnlockMutex(&mutex);
        return XPG_SET_ALREADY_DEFINED;
      };
    };
    m = s->msgs;
    mm = NULL;
    id = 0;
    while ((m != NULL) && 
      (cmpstr(m->header.mnemonic,msgName)!=0)) {
      id = max(id,m->header.id);
      mm = m;
      m=m->next;
    };
    if (m==NULL) {
// We didn't find the message.  Create it...
      if (mm) mm->next = m = new xpg_msglist_t(id+1, text, msgName);
      else s->msgs = m = new xpg_msglist_t(id+1, text, msgName);
      theMsg = id+1;
    } else {
      theMsg = m->header.id;
      pkiUnlockMutex(&mutex);
      return XPG_MSG_ALREADY_DEFINED;
    };
  } catch (...) {
    pkiUnlockMutex(&mutex);
    throw;
  };
  pkiUnlockMutex(&mutex);
  return 0;


}

uint32 xpg_catalog_t::display(void) {
  xpg_catalog_t * p;
  xpg_setlist_t * s;  
  xpg_msglist_t * m;

  fprintf(stdout, "Dump of catalogs\n");

  if (pkiCreateMutex(&mutex, true) != 0) return 0;
  try {
// Walk the chain, printing as we go...
    p = root;
    while (p != NULL) {
      fprintf(stdout, "Catalog %d:\n", p->id);
      if (p->header.majic != 35) {
        fprintf(stdout, " Error: Catalog majic should be 35, was %d\n", p->header.majic);
        fflush(stdout);
      };
      fprintf(stdout, " Name: %s\n", p->header.name);
 
      if ((s = p->set) == NULL) {
        fprintf(stdout, "  No sets in this catalog\n");
      } else {
        while (s != NULL) {
          fprintf(stdout, "  Set %d\n", s->header.id);
          if (s->header.majic != 42) {
            fprintf(stdout, "   Error: Set majic should be 42, was %d\n", s->header.majic);
            fflush(stdout);
          };
          fprintf(stdout, "   Mnemonic: %s\n", s->header.mnemonic);
          if ((m = s->msgs) == NULL) {
            fprintf(stdout, "    No messages in this set\n");
          } else {
            while (m != NULL) {
              fprintf(stdout, "    Message %d\n", m->header.id);
              if (m->header.majic != 27) {
                fprintf(stdout, "    Error: Message majic should be 27, was %d\n", m->header.majic);
                fflush(stdout);
              };
              fprintf(stdout, "    Mnemonic: %s\n", m->header.mnemonic);
              fprintf(stdout, "    Text: %s", m->text);
              m = m->next;
            };
          };
          s = s->next;
        };
      };
      p = p->next;
    };
  } catch (...) {
    pkiUnlockMutex(&mutex);
    throw;
  };
  pkiUnlockMutex(&mutex);
  return 0;
}


#endif
