/* ***************************************************************** *
 * Copyright 1998 International Business Machines Corporation. All   *
 * Rights Reserved.                                                  *
 *                                                                   *
 * Please read this carefully.  Your use of this reference           *
 * implementation of certain of the IETF public-key infrastructure   *
 * specifications ("Software") indicates your acceptance of the      *
 * following.  If you do not agree to the following, do not install  *
 * or use any of the Software.                                       *
 *                                                                   *
 * Permission to use, reproduce, distribute and create derivative    *
 * works from the Software ("Software Derivative Works"), and to     *
 * distribute such Software Derivative Works is hereby granted to    *
 * you by International Business Machines Corporation ("IBM").  This *
 * permission includes a license under the patents of IBM that are   *
 * necessarily infringed by your use of the Software as provided by  *
 * IBM.                                                              *
 *                                                                   *
 * IBM licenses the Software to you on an "AS IS" basis, without     *
 * warranty of any kind.  IBM HEREBY EXPRESSLY DISCLAIMS ALL         *
 * WARRANTIES OR CONDITIONS, EITHER EXPRESS OR IMPLIED, INCLUDING,   *
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OR CONDITIONS OF       *
 * MERCHANTABILITY, NON INFRINGEMENT AND FITNESS FOR A PARTICULAR    *
 * PURPOSE.  You are solely responsible for determining the          *
 * appropriateness of using this Software and assume all risks       *
 * associated with the use of this Software, including but not       *
 * limited to the risks of program errors, damage to or loss of      *
 * data, programs or equipment, and unavailability or interruption   *
 * of operations.                                                    *
 *                                                                   *
 * IBM WILL NOT BE LIABLE FOR ANY DIRECT DAMAGES OR FOR ANY SPECIAL, *
 * INCIDENTAL, OR  INDIRECT DAMAGES OR FOR ANY ECONOMIC              *
 * CONSEQUENTIAL DAMAGES (INCLUDING LOST PROFITS OR SAVINGS), EVEN   *
 * IF IBM HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.  IBM  *
 * will not be liable for the loss of, or damage to, your records or *
 * data, or any damages claimed by you based on a third party claim. *
 *                                                                   *
 * IBM wishes to obtain your feedback to assist in improving the     *
 * Software.  You grant IBM a world-wide, royalty-free right to use, *
 * copy, distribute, sublicense and prepare derivative works based   *
 * upon any feedback, including materials, error corrections,        *
 * Software Derivatives, enhancements, suggestions and the like that *
 * you provide to IBM relating to the Software (this does not        *
 * include products for which you charge a royalty and distribute to *
 * IBM under other terms and conditions).                            *
 *                                                                   *
 * You agree to distribute the Software and any Software Derivatives *
 * under a license agreement that: 1) is sufficient to notify all    *
 * licensees of the Software and Software Derivatives that IBM       *
 * assumes no liability for any claim that may arise regarding the   *
 * Software or Software Derivatives, and 2) that disclaims all       *
 * warranties, both express and implied, from IBM regarding the      *
 * Software and Software Derivatives.  (If you include this          *
 * Agreement with any distribution of the Software or Software       *
 * Derivatives you will have met this requirement.)  You agree that  *
 * you will not delete any copyright notices in the Software.        *
 *                                                                   *
 * This Agreement is the exclusive statement of your rights in the   *
 * Software as provided by IBM.   Except for the rights granted to   *
 * you in the second paragraph above, You are not granted any other  *
 * patent rights, including but not limited to the right to make     *
 * combinations of the Software with products that infringe IBM      *
 * patents. You agree to comply with all applicable laws and         *
 * regulations, including all export and import laws and regulation. *
 * This Agreement is governed by the laws of the State of New York.  *
 * This Agreement supersedes all other communications,               *
 * understandings or agreements we may have had prior to this        *
 * Agreement.                                                        *
 * ***************************************************************** */

#include <memory.h>
#include "ibmcylinkcsp.hpp"
#include "digest.hpp"

#include "exception.hpp"
#include "cssmerr.h"
#include "misc.hpp"
#include "shadigest.hpp"

/**********************************************************
 *
 *	Digest class
 *
 **********************************************************/
Digest::Digest(CSSM_CC_HANDLE ctxHandle, uint32 algId)
:Context(ctxHandle, CSSM_ALGCLASS_DIGEST, algId)
 {
	switch (algId) {
		case CSSM_ALGID_SHA1 :
			engine = new SHADigestEngine();
			break;
		default:
			throw InputException(CSSM_CSP_UNKNOWN_ALGORITHM);
			break;
	}
}

Digest::~Digest() {
	delete engine;
}

void Digest::digest(const CSSM_DATA_PTR dataBuff, uint32 dataBuffCount, CSSM_DATA_PTR digest) {
	if (dataBuffCount == 1) {
		if ((dataBuff == NULL) || (dataBuff->Data == NULL) || (dataBuff->Length == 0))
			throw InputException(CSSM_CSP_INPUT_DATA_ERROR);

		if ((digest == NULL) || (digest->Data == NULL))
			throw InputException(CSSM_CSP_NO_OUTBUF);

		if (digest->Length < getDigestLength())
			throw InputException(CSSM_CSP_NOT_ENOUGH_BUFFER);

		CssmData dataIn	 = CssmData(dataBuff);
		CssmData dataOut = CssmData(digest);
		engine->digest(dataIn, dataOut);
		if (dataOut.getLength() != 0) {
			memcpy(digest->Data, dataOut.getData(), dataOut.getLength());
		}
		else {
			digest->Data = NULL;
		}
		digest->Length = dataOut.getLength();
	}
	else {
		digestInit();
		digestUpdate(dataBuff, dataBuffCount);
		digestFinal(digest);
	}
}

void Digest::digest(CssmData *dataBuff, uint32 dataBuffCount, CssmData &digest) {
	if (dataBuffCount == 1) {
		if ((dataBuff == NULL) || (dataBuff->getLength() == 0))
			throw InputException(CSSM_CSP_INPUT_DATA_ERROR);

		if (digest.getLength() < getDigestLength())
			throw InputException(CSSM_CSP_ERR_OUTBUF_LENGTH);

		engine->digest(*dataBuff, digest);
	}
	else {
		digestInit();
		digestUpdate(dataBuff, dataBuffCount);
		digestFinal(digest);
	}
}

void Digest::digestInit() {
	engine->digestInit();
}

void Digest::digestUpdate(const CSSM_DATA_PTR dataBuff, uint32 dataBuffCount) {
	uint32	i;

	if (dataBuff == NULL)
		throw InputException(CSSM_CSP_INPUT_DATA_ERROR);

	for (i = 0; i < dataBuffCount; i ++) {
		if ((dataBuff[i].Data == NULL) || (dataBuff[i].Length == 0)) {
			throw InputException(CSSM_CSP_INPUT_DATA_ERROR);
			break;
		}
		CssmData dataIn = CssmData(&dataBuff[i]);
		engine->digestUpdate(dataIn);
	}
}

void Digest::digestUpdate(CssmData *dataBuff, uint32 dataBuffCount) {
	uint32	i;

	if (dataBuff == NULL)
		throw InputException(CSSM_CSP_INPUT_DATA_ERROR);

	for (i = 0; i < dataBuffCount; i ++) {
		if (dataBuff[i].getLength() == 0) {
			throw InputException(CSSM_CSP_INPUT_DATA_ERROR);
			break;
		}
		engine->digestUpdate(dataBuff[i]);
	}
}

void Digest::digestFinal(CSSM_DATA_PTR digest) {
	if ((digest == NULL) || (digest->Data == NULL))
		throw InputException(CSSM_CSP_INPUT_DATA_ERROR);

	if (digest->Length < getDigestLength())
		throw InputException(CSSM_CSP_ERR_OUTBUF_LENGTH);

	CssmData dataOut = CssmData(digest);
	engine->digestFinal(dataOut);
	memcpy(digest->Data, dataOut.getData(), getDigestLength());
	digest->Length = dataOut.getLength();	
}

void Digest::digestFinal(CssmData &digest) {
	if (digest.getLength() < getDigestLength())
		throw InputException(CSSM_CSP_ERR_OUTBUF_LENGTH);

	engine->digestFinal(digest);
}

uint32 Digest::getDigestLength() {
	return (engine->getDigestLength());
}

/**********************************************************
 *
 *	DigestEngine class
 *
 **********************************************************/
DigestEngine::DigestEngine(uint32 algId) {
	this->algId = algId;
}

DigestEngine::~DigestEngine() {
}

void DigestEngine::digest(CssmData &dataBuff, CssmData &digest) {
	//this method was added to Digest and DigestEngine, because there are some
	//algorithms that provide a one shot digest function, which might be more efficient
	//This is a virtual method, so if the engine has a one shot operation, it will overwrite
	//this method; if not, it won't :-)
	digestInit();
	digestUpdate(dataBuff);
	digestFinal(digest);
}