﻿using System;
using System.Threading;
using System.Threading.Tasks;
using Helpers;

namespace TreeSample {
    class TreeSample {
        static void Main(string[] args) {
            Console.WriteLine("starting TreeSample...");
            Tree t = Tree.CreateSomeTree(9, 1);

            TestRunner.Runtest( () => WalkTreeSequencial(t) );
            //TestRunner.Runtest(() => WalkTreeThreads(t)); // OutOfMemoryException bei grossen Trees
            TestRunner.Runtest(() => StartWalkTreeTasks(t));
            TestRunner.Runtest(() => WalkTreeParallelInvoke(t));
            TestRunner.Runtest(() => StartWalkTreeChilds(t));
            
            TestRunner.Runtest<int>( () => SumTreeSequencial(t) );
            TestRunner.Runtest<int>( () => SumTreeFutures(t) );

            Console.WriteLine("done. Press any key.");
            Console.ReadKey();
        }


        ////////////////////////////////////////////////////////////////////////////////
        //
        // sequenzielle Version
        //
        static void WalkTreeSequencial(Tree tree) {
            if (tree == null) return;

            WalkTreeSequencial(tree.left);
            WalkTreeSequencial(tree.right);

            tree.ProcessItem();
        }


        ////////////////////////////////////////////////////////////////////////////////
        //
        // Threads - don't do this!
        // OutOfMemoryException bei grossen Trees
        //
        static void WalkTreeThreads(Tree tree) {
            if (tree == null) return;

            Thread left = new Thread((o) => WalkTreeThreads(tree.left));
            left.Start();
            Thread right = new Thread((o) => WalkTreeThreads(tree.right));
            right.Start();

            left.Join();
            right.Join();

            tree.ProcessItem();
        }


        ////////////////////////////////////////////////////////////////////////////////
        //
        // Tasks
        //
        static void StartWalkTreeTasks(Tree tree) {
            Task task = Task.Factory.StartNew(() => WalkTreeTasks(tree));
            task.Wait();
        }

        static void WalkTreeTasks(Tree tree) {
            if (tree == null) return;
            Task left = new Task(() => WalkTreeTasks(tree.left));
            left.Start();
            Task right = new Task(() => WalkTreeTasks(tree.right));
            right.Start();

            Task.WaitAll(left, right);

            tree.ProcessItem();
        }


        ////////////////////////////////////////////////////////////////////////////////
        //
        // ParallelInvoke
        //
        static void WalkTreeParallelInvoke(Tree tree) {
            if (tree == null) return;

            Parallel.Invoke(
                  () => WalkTreeParallelInvoke(tree.left),
                  () => WalkTreeParallelInvoke(tree.right),
                  () => tree.ProcessItem()
              );
        }


        ////////////////////////////////////////////////////////////////////////////////
        //
        // Kind-Tasks
        //
        static void StartWalkTreeChilds(Tree tree) {


            Task task = Task.Factory.StartNew(() => WalkTreeChilds(tree));
            task.Wait();

            // all childs have finished, too
        }

        static void WalkTreeChilds(Tree tree) {
            if (tree == null) return;

            Task left = Task.Factory.StartNew(() => WalkTreeChilds(tree.left), TaskCreationOptions.AttachedToParent);
            Task right = Task.Factory.StartNew(() => WalkTreeChilds(tree.right), TaskCreationOptions.AttachedToParent);

            // hier kein TaskWait()!

            tree.ProcessItem();
        }


        ////////////////////////////////////////////////////////////////////////////////
        //
        // Aufsummieren sequenziell
        //
        static int SumTreeSequencial(Tree tree)
        {
            if (tree == null) return 0;

            int left = SumTreeSequencial(tree.left);
            int right = SumTreeSequencial(tree.right);

            return tree.ProcessItem() + left + right;
        }


        ////////////////////////////////////////////////////////////////////////////////
        //
        // Aufsummieren mit Futures
        //
        static int SumTreeFutures(Tree tree)
        {
            if (tree == null) return 0;

            Task<int> left = Task<int>.Factory.StartNew(() => SumTreeFutures(tree.left));
            Task<int> right = Task<int>.Factory.StartNew(() => SumTreeFutures(tree.right));

            return tree.ProcessItem() + right.Result + left.Result;
        }

    }
}
