﻿using System;
using System.Threading.Tasks;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Threading;

namespace Pipeline {

    ////////////////////////////////////////////////////////////////////////////////
    //
    // SimpleOrderingPipeline implementiert einfache lineare Pipeline.
    // Ein- und Ausgabetyp aller Stufen ist gleich.
    // SimpleOrdering Pipeline stellt die Reihenfolge der Eingaben in der letzten Stage wieder her!
    //
    class SimpleOrderingPipeline<TChunk> {
        private ConcurrentBag<Task> rootTasks;      // sammelt alle Tasks, die von hier gestartet werden;
                                                    // nötig für Wait(); Subtasks müssen mit 
                                                    // AttachedToParent erzeugt werden.

        private OrderingStage<TChunk> rootStage;    // Start-Stufe
        public List<TChunk> results;               // Ergebnis-Sammler

        private OrderingStage<TChunk> _last;        // letzte hinzugefuegte Stufe
        private long _counter = 0;                  // Sequenzzähler

        // quick and dirty: wird von den Stages direkt zugegriffen
        public long outCounter = 1;                // naechst erartete Ausgabe
        public Dictionary<long, TChunk> tempResults;
                                                    // temporaere Ablage für Ergebnisse

        public SimpleOrderingPipeline(OrderingStage<TChunk> root) {
            rootStage = root;
            _last = rootStage;
            rootTasks = new ConcurrentBag<Task>();
            results = new List<TChunk>();
            tempResults = new Dictionary<long, TChunk>();
        }

        public void Process(TChunk chunk) {
            long cnt = Interlocked.Increment(ref _counter);
            Task t;
            t = Task.Factory.StartNew(
                // counter wird hochgezaehlt, schicherheitshalber interlocked
                // falls jemand auf die Idee kommt, die Pipeline von verschiedenen
                // Threads zu fuettern
                () => rootStage.Process(chunk, cnt, this)
            );
            rootTasks.Add(t);
        }

        public void AddStage(OrderingStage<TChunk> stage) {
            _last.AddNext(stage);
            _last = stage;
        }

        public void Wait() {
            Task.WaitAll(rootTasks.ToArray());
        }

        public List<TChunk> GetResults() {
            return results;
        }

    }

    ////////////////////////////////////////////////////////////////////////////////
    //
    // Pipeline-Stufe von SimpleOrderingPipeline.
    // Jede Stufe erzeugt eine Task für die nachfolgende Stufe.
    //
    class OrderingStage<TChunk> {

        private Func<TChunk, TChunk> stageFunc;     // Funktion, die in der Stufe ausgeführt wird
        private OrderingStage<TChunk> nextStage;    // nachfolgende Stufe

        public OrderingStage(Func<TChunk, TChunk> func) {
            stageFunc = func;
        }

        public void AddNext(OrderingStage<TChunk> next) {
            nextStage = next;
        }

        public void Process(TChunk chunk, long counter, SimpleOrderingPipeline<TChunk> pipeline) {
            TChunk result = stageFunc(chunk);
            if (nextStage != null) {
                nextStage.Process(result, counter, pipeline);
            } else {
                // letze Stufe erzeugt ein Ergebnis
                lock (pipeline) {
                    bool resultFoundInTemp = true;
                    // solange gueltige Ergebnisse da sind, ablegen
                    while (resultFoundInTemp) {
                        resultFoundInTemp = false;
                        if (counter == pipeline.outCounter) {
                            // zum Ergebnis, wenn es das naechste
                            pipeline.results.Add(result);
                            pipeline.outCounter++;
                        } else {
                            // sonst merken
                            pipeline.tempResults.Add(counter, result);
                        }
                        // wenn in temporaerer Liste, nimm das
                        while (pipeline.tempResults.ContainsKey(pipeline.outCounter)) {
                            pipeline.results.Add(pipeline.tempResults[pipeline.outCounter]);
                            pipeline.tempResults.Remove(pipeline.outCounter);
                            pipeline.outCounter++;
                            resultFoundInTemp = true;
                        }
                    }
                } 
            }
        }
    }
}
