﻿using System;
using System.Linq;
using Helpers;
using System.Threading.Tasks;
using System.Collections.Concurrent;

namespace ParallelProgramming_CSharp {
    class DotProduct {

        static int ITERATIONS = 100000;       // wie oft soll das Skalarprodukt berechnet werden
        static int SIZE = 256;                // Vektorgroeße
        //static int SIZE = 50000;             // Vektorgroeße fuer Performancetests (Beispiel)

        ////////////////////////////////////////////////////////////////////////////////
        //
        // main
        //
        static void Main(string[] args) {
            double[] a = new double[SIZE];
            double[] b = new double[SIZE];

            Console.WriteLine("starting DotProduct sample...");

            // Vektoren initialisieren
            for (int i = 0; i < SIZE; i++) {
                a[i] = i * 0.5;
                b[i] = i * 2.0;
            }

            TestRunner.Runtest<double>(() => DotProductSequential(a, b));
            TestRunner.Runtest<double>(() => DotProductParallelForUnlocked(a, b));
            TestRunner.Runtest<double>(() => DotProductParallelForLocked(a, b));
            TestRunner.Runtest<double>(() => DotProductParallelAggregate0(a, b));
            TestRunner.Runtest<double>(() => DotProductParallelAggregate1(a, b));
            TestRunner.Runtest<double>(() => DotProductParallelAggregate2(a, b));
            TestRunner.Runtest<double>(() => DotProductParallelAggregatePartitioner(a, b));
            TestRunner.Runtest<double>(() => DotProductParallelAggregateLINQ(a, b));

            Console.WriteLine();
            Console.WriteLine("done. Press any key.");
            Console.ReadKey();
        }


        ////////////////////////////////////////////////////////////////////////////////
        //
        // sequenzielle Version des Skarprodukts
        //
        static double DotProductSequential(double[] a, double[] b) {
            double sum = 0;
            for (int j = 0; j < ITERATIONS; j++) {
                sum = 0;
                for (int i = 0; i < SIZE; i++) {
                    sum += a[i] * b[i];
                }
            }
            return sum;
        }

        ////////////////////////////////////////////////////////////////////////////////
        //
        // paralleles Skalarprodukt mit Race Condition
        //
        static double DotProductParallelForUnlocked(double[] a, double[] b) {
            double sum = 0;
            for (int j = 0; j < ITERATIONS; j++) {
                sum = 0;
                Parallel.For(0, SIZE, delegate(int i) {
                    sum += a[i] * b[i];     // Race Condition!
                });
            }
            return sum;
        }

        ////////////////////////////////////////////////////////////////////////////////
        //
        // korrekte, aber langsame, parallele Version
        //
        static double DotProductParallelForLocked(double[] a, double[] b) {
            double sum = 0;
            object monitor = new object();
            for (int j = 0; j < ITERATIONS; j++) {
                sum = 0;
                Parallel.For(0, SIZE, /*options,*/ delegate(int i) {
                    lock (monitor) sum += a[i] * b[i];
                });
            }
            return sum;
        }

        ////////////////////////////////////////////////////////////////////////////////
        //
        // parallele Aggregation mit Parallel.For
        //
        static double DotProductParallelAggregate0(double[] a, double[] b) {
            double sum = 0;
            object monitor = new object();
            for (int j = 0; j < ITERATIONS; j++) {
                sum = 0;
                Parallel.For(
                    0,                      // from (inclusiv)
                    SIZE,                   // to (excludiv)
                    () => { return 0.0; },  // Initialisierung threadlokaler Daten (definiert Typ von ThreadLocalState)
                    (int i, ParallelLoopState state, double threadLocalState) =>  // body...
                    {
                        return threadLocalState + a[i] * b[i];
                    },
                    (double partialSum) =>           // clean up thread local data
                    { lock (monitor) sum += partialSum; }
                );

            }
            return sum;
        }

        ////////////////////////////////////////////////////////////////////////////////
        //
        // parallele Aggregation mit Parallel.For (explizite Typangabe)
        //
        static double DotProductParallelAggregate1(double[] a, double[] b) {
            double sum = 0;
            object monitor = new object();
            for (int j = 0; j < ITERATIONS; j++) {
                sum = 0;
                Parallel.For<double>(
                    0,                    // from (inklusiv)
                    SIZE,                 // to (exklusiv)
                    () => 0,              // Initialisierung threadlokaler Daten
                    (i, parallelLoopState, threadLocalSum) => {  // body...
                        return threadLocalSum + a[i] * b[i];
                    },
                    partialSum => {
                        lock (monitor) sum += partialSum;
                    }
                );
            }
            return sum;
        }

        ////////////////////////////////////////////////////////////////////////////////
        //
        // generische Variante
        //
        static double ParallelAggregate(int fromInclusive, int toExclusive, Func<int, double> aggregator) {
            double result = 0.0;
            object monitor = new object();
            Parallel.For(fromInclusive, toExclusive, () => 0.0, (i, state, threadLocalState) => {
                return threadLocalState + aggregator(i);
            },
                (partialSum) => { lock (monitor) result += partialSum; });
            return result;
        }

        static double DotProductParallelAggregate2(double[] a, double[] b) {
            double sum = 0;
            for (int j = 0; j < ITERATIONS; j++) {
                sum = 0;
                sum += ParallelAggregate(0, SIZE, delegate(int x) { return a[x] * b[x]; });
            }
            return sum;
        }



        ////////////////////////////////////////////////////////////////////////////////
        //
        // parallele Aggregation mit Partitioner
        //
        static double DotProductParallelAggregatePartitioner(double[] a, double[] b) {
            double sum = 0;
            object monitor = new object();
            for (int j = 0; j < ITERATIONS; j++) {
                sum = 0;
                Parallel.ForEach<Tuple<int,int>, double>(
                    Partitioner.Create(0, SIZE), 
                    () => 0, 
                    (range, parallelLoopState, threadLocalSum) => { // body...
                        double rangeSum = 0;
                        for (int i = range.Item1; i < range.Item2; i++) {
                            rangeSum += a[i] * b[i];
                        }
                        return threadLocalSum + rangeSum;
                    },
                    partialSum => {
                        lock (monitor) sum += partialSum;
                    }
                );
            }
            return sum;
        }

        ////////////////////////////////////////////////////////////////////////////////
        //
        // parallele Aggregation mit PLINQ
        //
        static double DotProductParallelAggregateLINQ(double[] a, double[] b) {
            double sum = 0;
            for (int j = 0; j < ITERATIONS; j++) {
                sum = 0;
                sum += (from i in ParallelEnumerable.Range(0, SIZE)
                        select i).Aggregate(
                            0.0,  // seed (richtigen Typ verwenden!)    
                            delegate(double partialSum, int x)  // Akkumulatorfunktion
                            { return partialSum + a[x] * b[x]; }
                            );
            }
            return sum;
        }
    }
}
