
#include "StdAfx.h"
#include <math.h>

#include "SpaceShip.h"
#include "Saucer.h"
#include "GlobalSettings.h"
#include "PositionCalculator.h"
#include "GameEngine.h"

#include "GameStatus.h"


const double GameStatus::CalculationRange = 3.0;
//const double GameStatus::CalculationRange = 24.0;


GameStatus::GameStatus(void)
: mSpaceShipIsPresent(false), mNasteroids(0), mNshots(0)
{
	mSpaceShip = new SpaceShip();
	mSaucer = new Saucer();
}


GameStatus::~GameStatus(void)
{
	delete mSpaceShip;
	delete mSaucer;

	clearShotList();
	clearAsteroidList();
}


void GameStatus::clearAsteroidList(void)
{
	for (list<Asteroid*>::iterator i = mAsteroidList.begin(); i != mAsteroidList.end(); i++)
	{
		delete *i;
	}
	
	mAsteroidList.clear();
}

list<Asteroid*>* GameStatus::getAsteroidList(void)
{
	return &mAsteroidList;
}

Asteroid* GameStatus::getAsteroidById(const int& id)
{
	for (list<Asteroid*>::iterator i = mAsteroidList.begin(); i != mAsteroidList.end(); i++)
	{
		if ((*i)->getId() == id) { return *i; }
	}

	return 0;
}

MovingEntity* GameStatus::getObjectById(const int& id)
{
	if (id == MovingEntity::SaucerId)
	{
		return mSaucer;
	}
	else
	{
		return getAsteroidById(id);
	}
}


void GameStatus::addTarget(int id)
{
	mNextTargetIds.push(id);
}


void GameStatus::calculateAsteroids(const int& frameRate, const int& latenz)
{
	if (mNasteroids== 0)
	{	// Nichts zu tun
		clearAsteroidList();
		return;
	}

	for (int i = 0; i < mNasteroids; i++)
	{	// Zu Begin alle Asteroiden auf neu setzen
		mAsteroids[i].setCreationState(MovingEntity::NewEntity);
	}

	//////////////////////////////////////////////////////////////////////////
	// Ab hier versuchen wir alle Asteroiden in unserer Liste den Asteroiden
	// die gerade auf dem Screen sind zuzuordnen
	Vector2D predictedPos, prevPos;
	Asteroid* foundAsteroid = 0;

	// Wir beginnen mit denen die schon zugeordnet sind. Bei diesen wurde die Geschwindigkeit schon gesetzt und
	// so kann die nchste Position berechnet werden
	for (list<Asteroid*>::iterator iter = mAsteroidList.begin(); iter != mAsteroidList.end(); iter++)
	{
		if ((*iter)->getCreationState() != MovingEntity::AssignedEntity)
		{	// Nur die schon zugeordneten Asteroiden betrachten
			continue;
		}

		// Die nchste Position berechnen und mit allen Asteroiden abgleichen
		// TODO !!!: die vorhergesagte Position muss unter umstnden auch umgebrochen werden.
		// Auch bei Schen bercksichtigen
		foundAsteroid = 0;
		predictedPos = (*iter)->getPosition() + (*iter)->getVelocity() * static_cast<double>(frameRate);

		for (int i = 0; i < mNasteroids; i++)
		{	// Versuchen diesen Asteroiden bei denen auf dem Schim zu finden
			if (((*iter)->getScaleFactor() != mAsteroids[i].getScaleFactor()) ||
				((*iter)->getType() != mAsteroids[i].getType()) ||
				(mAsteroids[i].getCreationState() != MovingEntity::NewEntity))
			{	// Typ oder Gre unterschiedlich oder Asteroid schon zugeordnet
				continue;
			}

			if (PositionCalculator::isPositionInWorldRange(predictedPos, mAsteroids[i].getPosition(), CalculationRange))
			{	// Asteroid gefunden
				foundAsteroid = mAsteroids + i;
				mAsteroids[i].setCreationState(MovingEntity::AssignedEntity);
				break;
			}
		}

		if (foundAsteroid != 0)
		{	// Asteroid gefunden -> Position setzen und Geschwindigkeit berechnen lassen
			prevPos = (*iter)->getPosition();
			(*iter)->setPosition(foundAsteroid->getPosition());
			(*iter)->calculateVelocityFromPrevPosition(prevPos, frameRate);
		}
		else
		{	// Asteroid nicht gefunen. Mglicherweise zerstrt
			(*iter)->setCreationState(MovingEntity::DeadEntity);
		}
	}

	// Im nchsten Schritt werden alle Asteroiden in der Liste untersucht, die den Status new haben
	// fr diese wird ein Asteroid im Array gesucht, der dem Typ entspricht und am nchsten ist.
	double distanceToClosestAsteroidSq = GlobalSettings::Instance()->maxDistanceWorldSq;
	Vector2D vecToAsteroid;

	for (list<Asteroid*>::iterator iter = mAsteroidList.begin(); iter != mAsteroidList.end(); iter++)
	{
		if ((*iter)->getCreationState() != MovingEntity::NewEntity)
		{	// Nur die neuen Asteroiden betrachten
			continue;
		}

		foundAsteroid = 0;
		distanceToClosestAsteroidSq = GlobalSettings::Instance()->maxDistanceWorldSq;

		for (int i = 0; i < mNasteroids; i++)
		{	// Versuchen diesen Asteroiden bei denen auf dem Schim zu finden
			if (((*iter)->getScaleFactor() != mAsteroids[i].getScaleFactor()) ||
				((*iter)->getType() != mAsteroids[i].getType()) ||
				(mAsteroids[i].getCreationState() != MovingEntity::NewEntity))
			{	// Typ oder Gre unterschiedlich oder Asteroid schon zugeordnet
				continue;
			}

			// Vektor zwischen den zwei Asteroiden berechnen
			vecToAsteroid = mAsteroids[i].getPosition() - (*iter)->getPosition();

			// Wenn dieser Asteroid der nchste ist dann wird es wohl der gleiche sein
			if (vecToAsteroid.getLengthSq() < distanceToClosestAsteroidSq)
			{
				distanceToClosestAsteroidSq = vecToAsteroid.getLengthSq();
				foundAsteroid = mAsteroids + i;
			}		
		}

		if (foundAsteroid != 0)
		{
			foundAsteroid->setCreationState(MovingEntity::AssignedEntity);
			(*iter)->setCreationState(MovingEntity::AssignedEntity);

			prevPos = (*iter)->getPosition();
			(*iter)->setPosition(foundAsteroid->getPosition());
			(*iter)->calculateVelocityFromPrevPosition(prevPos, frameRate);

		}
		else
		{	// Hm, mglicherweise ist er direkt nach dem entstehen wieder zerstrt worden
			(*iter)->setCreationState(MovingEntity::DeadEntity);
		}
	}

	//////////////////////////////////////////////////////////////////////////
	// Jetzt die alte Liste bereinigen
	list<Asteroid*> tempAsteroidList;

	for (list<Asteroid*>::iterator iter = mAsteroidList.begin(); iter != mAsteroidList.end(); iter++)
	{	
		if ((*iter)->getCreationState() == MovingEntity::DeadEntity) { delete *iter; }
		else { tempAsteroidList.push_back(*iter); }
	}

	mAsteroidList.clear();

	for (list<Asteroid*>::iterator iter = tempAsteroidList.begin(); iter != tempAsteroidList.end(); iter++)
	{	
		mAsteroidList.push_back(*iter);
	}

	/////////////////////////////////////////////////////////////////
	// Alle bis hier nicht zugeordneten Asteroiden sind neu erzeugt worden
	Asteroid* newAsteroid = 0;

	for (int i = 0; i < mNasteroids; i++)
	{
		if (mAsteroids[i].getCreationState() != MovingEntity::NewEntity)
		{
			continue;
		}

		// Fr diesen Asteroiden ein neues Objekt erzeugen und in die Liste aufnehmen
		// Es wird absichtlich nicht der =operator verwendet, da dieser auch die Geschwindigkeiten
		// setzen wrde
		newAsteroid = new Asteroid();
		newAsteroid->setPosition(mAsteroids[i].getPosition());
		newAsteroid->setScaleFactor(mAsteroids[i].getScaleFactor());
		newAsteroid->setType(mAsteroids[i].getType());

		mAsteroidList.push_back(newAsteroid);
	}
}

void GameStatus::clearShotList(void)
{
	for (list<Shot*>::iterator i = mShotList.begin(); i != mShotList.end(); i++)
	{
		delete *i;
	}
	
	mShotList.clear();
}

std::list<Shot*>* GameStatus::getShotList(void)
{
	return &mShotList;
}

void GameStatus::calculateShots(const int& frameRate)
{
	// in mShots stehen die Daten aller derzeit auf dem Screen befindlichen Sche. Diese
	// mssen jetzt unseren wirklich Schu-Objekten in der Schuliste zugeordnet werden.
	if (mNshots == 0)
	{	// Nichts zu tun
		clearShotList();
		return;
	}

	for (int i = 0; i < mNshots; i++)
	{	// Zu Begin alle Sche auf newShot setzen
		mShots[i].setCreationState(MovingEntity::NewEntity);
	}

	//////////////////////////////////////////////////////////////////////////
	// Jetzt alle schon vorhanden Sche versuchen in mShots zu finden und zu aktualisiern
	Vector2D predictedPos, prevPos;
	Shot* foundShot = 0;
	for (list<Shot*>::iterator iter = mShotList.begin(); iter != mShotList.end(); iter++)
	{	
		foundShot = 0;
		predictedPos = (*iter)->getPosition() + (*iter)->getVelocity() * static_cast<double>(frameRate);

		for (int i = 0; i < mNshots; i++)
		{
			if (PositionCalculator::isPositionInWorldRange(predictedPos, mShots[i].getPosition(), CalculationRange))
			{	// Schu gefunden
				foundShot = mShots + i;
				mShots[i].setCreationState(MovingEntity::AssignedEntity);
				break;
			}
		}

		if (foundShot != 0)
		{	// Schu gefunden -> Position setzen und Geschwindigkeit berechnen lassen
			prevPos = (*iter)->getPosition();
			(*iter)->setCreationState(MovingEntity::AssignedEntity);
			(*iter)->setPosition(foundShot->getPosition());
			(*iter)->calculateVelocityFromPrevPosition(prevPos, frameRate);
		}
		else
		{	// Schu nicht gefunen. Mglicherweise alt
			(*iter)->setCreationState(MovingEntity::DeadEntity);
		}
	}

	//////////////////////////////////////////////////////////////////////////
	// Jetzt die Liste bereinigen
	list<Shot*> newShotList;

	for (list<Shot*>::iterator iter = mShotList.begin(); iter != mShotList.end(); iter++)
	{	
		if ((*iter)->getCreationState() == MovingEntity::DeadEntity) 
		{ 
			MovingEntity* target = getAsteroidById((*iter)->getTargetId());
			if (target != 0)
			{
				target->setAlreadyUnderFire(false);
				if (GlobalSettings::Instance()->writeLog)
				{
					char msg[255] = { 0 };
					sprintf_s(msg, sizeof(msg), "GameStatus::calculateShots(): Schu(%d) hat Ziel %d nicht getroffen\n", (*iter)->getId(), (*iter)->getTargetId());
					GameEngine::Instance()->log(msg);
				}
			}
			else
			{
				if (GlobalSettings::Instance()->writeLog)
				{
					char msg[255] = { 0 };
					sprintf_s(msg, sizeof(msg), "GameStatus::calculateShots(): Schu(%d) hat Ziel %d getroffen\n", (*iter)->getId(), (*iter)->getTargetId());
					GameEngine::Instance()->log(msg);
				}
			}

			delete *iter; 
		}
		else { newShotList.push_back(*iter); }
	}

	mShotList.clear();

	for (list<Shot*>::iterator iter = newShotList.begin(); iter != newShotList.end(); iter++)
	{	
		mShotList.push_back(*iter);
	}

	//////////////////////////////////////////////////////////////////////////
	// Alle Sche die in der Liste nicht gefunden wurden sind wohl neue Schsse 
	// und werden jetzt erzeugt
	Shot* newShot = 0;
	double shotSpeed = GlobalSettings::Instance()->shotSpeedSaucer;
	Vector2D toSpaceShip;
	Vector2D toSaucer;

	for (int i = 0; i < mNshots; i++)
	{
		if (mShots[i].getCreationState() != MovingEntity::NewEntity)
		{
			continue;
		}

		// Dieser Schu konnte keinem vorhanden Schu zugeordnet werden. Also erzeugen wir 
		// einen neuen. Es wird absichtlich nur die Position von mShot[i] kopiert. Alle
		// anderen Werte sollen hier berechnet werden
		shotSpeed = GlobalSettings::Instance()->shotSpeedSaucer; // Erstmal die Geschwindigkeit vom Ufoschu annehmen
		newShot = new Shot();
		if (!mNextTargetIds.empty())
		{
			newShot->setTargetId(mNextTargetIds.front());
			mNextTargetIds.pop();
		}
		newShot->setPosition(mShots[i].getPosition());
		newShot->setCreationState(MovingEntity::AssignedEntity);

		// Jenachdem, an wem der Schu nher dran ist, ordnen wir ihn dem Ufo oder dem Raumschiff zu
		toSpaceShip = newShot->getPosition() - mSpaceShip->getPosition();
		toSaucer = newShot->getPosition() - mSaucer->getPosition();

		if ((!mSaucer->getIsPresent()) || (toSpaceShip.getLengthSq() < toSaucer.getLengthSq()))
		{	// Raumschiff 
			newShot->setSender(Shot::SpaceShip);

			// Geschwindigkeit berechnen
			shotSpeed = GlobalSettings::Instance()->shotSpeedSpaceShip;
			toSpaceShip.normalize();
			newShot->setVelocity(toSpaceShip *  shotSpeed, 1);
//			newShot->setVelocity(mSpaceShip->getHeading() *  shotSpeed, 1);
		}
		else
		{
			newShot->setSender(Shot::Saucer);

			// Geschwindigkeit berechnen
			toSaucer.normalize();
			newShot->setVelocity(toSaucer *  shotSpeed, 1);
		}

		mShotList.push_back(newShot);
	}
}

GameStatus& GameStatus::operator=(const GameStatus& rhs)
{
    if (&rhs == this) { return *this; }

	mSpaceShipIsPresent = rhs.mSpaceShipIsPresent;
	mNasteroids = rhs.mNasteroids;
	mNshots = rhs.mNshots;

	// Alle Asteroiden kopieren
	for (int i = 0; i < mNasteroids; i++)
	{
		*(mAsteroids + i) = *(rhs.mAsteroids + i);
	}

	// Alle Schsse kopieren
	for (int i = 0; i < mNshots; i++)
	{
		*(mShots + i) = *(rhs.mShots+ i);
	}

	// Schiff und Ufo kopieren
	*mSpaceShip = *rhs.mSpaceShip;
	*mSaucer = *rhs.mSaucer;

    return *this; 
}

void GameStatus::clear(void)
{
	mNasteroids = 0;
	mNshots = 0;

	mSpaceShipIsPresent = false;
	mSaucer->setIsPresent(false);

	// !!! Vorsicht: Die Schuliste darf nicht gelscht werden !!!!
	// clear wird nach jedem Empfang eines Packets aufgerufen. Sonst wrde die Zuordnung 
	// verloren gehen
}

bool GameStatus::getSpaceShipIsPresent(void) const
{
	return mSpaceShipIsPresent;
}

void GameStatus::setSpaceShipIsPresent(bool isPresent)
{
	mSpaceShipIsPresent = isPresent;
}

SpaceShip* GameStatus::getSpaceShip(void)
{
	return mSpaceShip;
}

Saucer* GameStatus::getSaucer(void)
{
	return mSaucer;
}

int GameStatus::getAsteroidsCount(void) const
{
	return mNasteroids;
}

void GameStatus::setAsteroidsCount(int asteroidsCount)
{
	mNasteroids = asteroidsCount;
}

Asteroid* GameStatus::getAsteroids()
{
	return mAsteroids;
}

int GameStatus::getShotsCount(void) const
{
	return mNshots;
}

void GameStatus::setShotsCount(int shotsCount)
{
	mNshots = shotsCount;
}

Shot* GameStatus::getShots()
{
	return mShots;
}


void GameStatus::getAllMovingEntities(std::list<MovingEntity*>* entities)
{
	for (int i = 0; i < mNasteroids; i++)
	{
		entities->push_back((mAsteroids + i));
	}
	for (int i = 0; i < mNshots; i++)
	{
		entities->push_back((mShots+ i));
	}
}


void GameStatus::writeStatusToLog() 
{
	mSpaceShip->writeStatusToLog();

	if(mSaucer->getIsPresent())
	{
		mSaucer->writeStatusToLog();
	}

	for (list<Asteroid*>::iterator i = mAsteroidList.begin(); i != mAsteroidList.end(); i++)
	{
		(*i)->writeStatusToLog();
	}

	for (list<Shot*>::iterator i = mShotList.begin(); i != mShotList.end(); i++)
	{
		(*i)->writeStatusToLog();
	}
}

