// player.cpp: Beispielspieler fr Asteroids
// Harald Bgeholz / c't
#define _CRT_SECURE_NO_WARNINGS 1

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <math.h>

#include "turn.h"


#if defined(WINDOWS)
#include <winsock2.h>
#else
// 2 Includes fr socket()
#include <sys/types.h>
#include <sys/socket.h>
// 2 Includes fr inet_addr()
#include <netinet/in.h>
#include <arpa/inet.h>
// 2 Includes fr fcntl()
#include <unistd.h>
#include <fcntl.h>
// fr memset()
#define INVALID_SOCKET -1
#define WSAGetLastError() errno
#endif

#include "player.h"
#include "turn.h"

double normalizeDX(double dx) {
		while (dx < -512) dx += 1024;
		while (dx > 511) dx -= 1024;
		return dx;
	}
double normalizeDY(double dy) {
		while (dy < -384) dy += 768;
		while (dy > 383) dy -= 768;
		return dy;
	}



void Player::OriginalStrategy(int t, const GameStatus& gameToUse, 
							  KeysPacket& keys)
{
		fprintf(stdout, "OriginalStrategy\n");

		keys.clear();   // alle Tasten loslassen
		const MovingObject *nearest = 0;
		double min_dist = 1e6;
		double min_dx = 0;
		double min_dy = 0;
		if (gameToUse.ship_present)
		{
			for (int i=0; i<gameToUse.nasteroids; ++i)
			{   // nchstgelegenen Asteroiden suchen
				Vector d;
				double dist = gameToUse.asteroids[i].squareDist(gameToUse.ship, &d);
				dist -= gameToUse.asteroids[i].rsquare();
				if (dist < min_dist)
				{
					nearest = gameToUse.asteroids + i;
					min_dist = dist;
					min_dx = d.x;
					min_dy = d.y;
				}
			}
			if (gameToUse.saucer_present)
			{
				Vector d = (gameToUse.saucer.r - gameToUse.ship.r).asteroids_norm();
				int dist = int(d.square() - gameToUse.saucer.rsquare());
				if (dist < min_dist)
				{
					nearest = &gameToUse.saucer;
					min_dist = dist;
					min_dx = d.x;
					min_dy = d.y;
				}
			}

			// Schiff in Richtung auf das nchstgelegene Objekt drehen
			// mathematisch wird hier das Kreuzprodukt aus den Vektoren 
			// ship_dx/y/0 und min_dx/y/0 berechnet
			if (gameToUse.ship.dx * min_dy - gameToUse.ship.dy * min_dx > 0)
				keys.left(true);
			else
				keys.right(true);

			if (min_dist < 27*27)  // Flucht, wenn Kollision unausweichlich
				keys.hyperspace(true);

			if (min_dist > 400*400) // beschleunigen, wenn nichts in der Nhe
				keys.thrust(true);

			if (t % 2 == 0)  // Feuerknopf drcken, so schnell es geht
				keys.fire(true);
		}
}

void Player::BetterStrategy(int t, const GameStatus& gameToUse, 
							  KeysPacket& keys, bool haveSpeeds)
{
		keys.clear();   // alle Tasten loslassen
		const MovingObject *nearest = 0;
		double min_dist = 1e6;
		double min_dx = 0;
		double min_dy = 0;
		if (gameToUse.ship_present)
		{
			fprintf(stdout, "BetterStrategy\n");
			for (int i=0; i<gameToUse.nasteroids; ++i)
			{   // nchstgelegenen Asteroiden suchen
				Vector d;
				double dist = gameToUse.asteroids[i].squareDist(gameToUse.ship, &d);
				dist -= gameToUse.asteroids[i].rsquare();
				if (dist < min_dist)
				{
					nearest = gameToUse.asteroids + i;
					min_dist = dist;
					min_dx = d.x;
					min_dy = d.y;
				}
			}

			for (int i=0; i<gameToUse.nshots; ++i)
			{   // nchstgelegenen Schuss suchen
				Vector d;
				double dist = gameToUse.shots[i].squareDist(gameToUse.ship, &d);
				dist -= gameToUse.shots[i].rsquare();
				if (dist < min_dist && dist < 27*27) // Schsse erst im letzten Moment bercksichtigen
				{
					bool ack = false;
					if (haveSpeeds) {
						Vector vrel = gameToUse.shots[i].v - gameToUse.ship.v;
						Vector vrel_n = vrel.normalized();
						Vector pos_n = gameToUse.shots[i].r.normalized();
						double incoming = pos_n * vrel_n; // cos
						if (incoming < -0.99) ack = true;
					}
					if (ack) {
						nearest = gameToUse.shots + i;
						min_dist = dist;
						// min_dx = -dx; // umdrehen
						// min_dy = -dy;
					}
				}
			}

			if (gameToUse.saucer_present)
			{
				Vector d = (gameToUse.saucer.r - gameToUse.ship.r).asteroids_norm();
				double dist = d.square() - (int)gameToUse.saucer.rsquare();
				if (dist < min_dist)
				{
					nearest = &gameToUse.saucer;
					min_dist = dist;
					min_dx = d.x;
					min_dy = d.y;
				}
			}

			// Schiff in Richtung auf das nchstgelegene Objekt drehen
			// mathematisch wird hier das Kreuzprodukt aus den Vektoren 
			// ship_dx/y/0 und min_dx/y/0 berechnet
			if (gameToUse.ship.dx * min_dy - gameToUse.ship.dy * min_dx > 0)
				keys.left(true);
			else
				keys.right(true);

			if (min_dist < 27*27) { // Flucht, wenn Kollision unausweichlich
				keys.hyperspace(true);
				if (0 != dynamic_cast<const Shot*>(nearest))
					fprintf(stdout, "HYPERSPACING to avoid SHOT\n");
			}

			if (min_dist > 400*400 && gameToUse.nasteroids < 5 && gameToUse.nasteroids > 0) // beschleunigen, wenn nichts in der Nhe
				keys.thrust(true);

			if (t % 2 == 0)
				keys.fire(true);
		}
}




void Player::RunSimple(void)
{
	FramePacket frame, framePast;
	KeysPacket K[3];
	KeysPacket& keys = K[0];
	GameStatus game, gamePast;
	bool havePast = false;
	char prevframe = 0;
	int t = 0;
	int t_lastshot = -2;
	int t_lasthyper = -2;
	int t_lastthrust = -2;
	int t_last_flagged_collision = -1;
	int latency;
	bool dont_shoot;
	int angle_byte = 0;
	int last_angle_byte = 0;
	int lrfilter = 0;

	init_degrees();

	for (;;)
	{
		++t;         // Zeit
		++keys.ping; // jedes gesendete Pckchen erhlt eine individuelle Nummer zur Latenzmessung
		SendPacket(keys);
		if (1 == t) {SendName(); fprintf(stderr, "Wenn's hngt, dann weil's die Online-Version ist...\n");}
		ReceivePacket(frame);
		dont_shoot = false;


		if (frame.frameno != ++prevframe || frame.ping != keys.ping)
		{
			fprintf(stdout, "%d:\n", t);
			printf("Latenz %d. %d Frames verloren.\n", keys.ping - frame.ping, frame.frameno - prevframe);
			prevframe = frame.frameno;
			latency = keys.ping - frame.ping;
		} else latency = 0;

		InterpretScreen(frame, game, t);


		keys.clear();
		if (game.ship_present) {
			bool flagged_collision = false;
			int time_of_first_collision = LOOKAHEAD+1;
			int maxNumCollisions = 0, numCollisions = 0;
			const MovingObject *bangPresent = 0;
			const MovingObject *bangPresentTmp = 0;
			const MovingObject *bangFuture = 0;
			static GameStatus future[LOOKAHEAD];
			Shot MyShotInit, MyShot;
			int my_shot_hits = 0;

			fprintf(stdout, "%d:\n", t);

			angle_byte = synchronize_angle_byte(angle_byte, game.ship.dx, game.ship.dy);
			//if (K[1].getKeys() && K[1].KEY_LEFT) angle_byte = (angle_byte + 1) % 256;
			//else if (K[1].getKeys() && K[1].KEY_RIGHT) angle_byte = (angle_byte + 256 - 1) % 256;
			//if (K[2].getKeys() && K[2].KEY_LEFT) angle_byte = (angle_byte + 1) % 256;
			//else if (K[2].getKeys() && K[2].KEY_RIGHT) angle_byte = (angle_byte + 256 - 1) % 256;
			double angle = degree_from_angle_byte(angle_byte);
			double angle2 = 180.0 * atan2(double(game.ship.dy), double(game.ship.dx)) / 3.1415;
			if (angle2 < 0.0) angle2 += 360.0;
			fprintf(stdout, "ANGLE dx:dy:%f w:%f\n", angle2, angle);
		

			bool present_fits_past  = false;
			if (game.ship_present && havePast) {
			    present_fits_past = game.CalculateSpeed(gamePast);
				if (present_fits_past) {
					//InitializeShotFromShip(angle_byte, game.ship, MyShotInit);
					future[0] = game;
					double time_step = 1.0;
					int adapt = 1 + latency;
					char tag[256];

					//if(t>=1000 && t<1000 + LOOKAHEAD) {
					//		sprintf(tag, "Real State");
					//		game.Show(tag);
					//}

					for(int f=1; f<LOOKAHEAD; f++) {
						if (f>0) future[0].CalculateProjection(future[f], double(t) + double(f));
						//if (1000==t) {
						//	sprintf(tag, "Projection %d", f);
						//	future[f].Show(tag);
						//} 
						if (adapt == f) InitializeShotFromShip(angle_byte, future[f].ship, MyShotInit);
						MyShotInit.CalculateProjection(MyShot, double(t)+double(f));
						if (f>=adapt && f<70+adapt && future[f].shotHits(MyShot, 1.0)) my_shot_hits++;
						bangPresentTmp = future[f].shipCollidesFromReference(game, numCollisions);
						maxNumCollisions = max( maxNumCollisions, numCollisions);
						if (bangPresentTmp && !flagged_collision) {
							bangPresent = bangPresentTmp;
							bangFuture = future[f].shipCollides(numCollisions);
							fprintf(stdout,"%d: collision in %f frames\n", t, f*time_step );
							flagged_collision = true;
							time_of_first_collision = f;
						}
						// if (flagged_collision) break;
					}
				} else {
					fprintf(stdout, "No speeds, etc.\n");
				}
			}

			if (!flagged_collision && 0 == game.nasteroids) {
				if (game.saucer_present) {
					AttackSaucer(t, angle_byte, game, keys, present_fits_past, latency);
					keys.fire( (t_lastshot < t - 2) );
				} else {
					GotoCenter(game, keys, present_fits_past);
				}
			} else if (bangPresent && time_of_first_collision >= 1) {
				if (dynamic_cast<const Shot*>(bangPresent) == 0) {
					fprintf(stdout, "Attack bangPresent!\n");
					AttackObjectWithoutThrust3(angle_byte, *bangPresent, game, keys, present_fits_past, latency, false);
					keys.fire( (t_lastshot < t - 1) && (my_shot_hits > 0) );
				} else { // Shot
					EscapeFromObject(*bangPresent, game, keys, present_fits_past);
				}
			} else if (game.saucer_present && game.ship.squareDist(game.saucer) < 200*200 ) {
				AttackSaucer(t, angle_byte, game, keys, present_fits_past, latency);
				keys.fire( (t_lastshot < t - 2) );
			} else {

				if (present_fits_past) {
					AttackCheapestObject(angle_byte, game, keys, present_fits_past, latency);
					keys.fire( (t - t_lastshot) > 1
								&& ( (keys.getKeys() & keys.KEY_FIRE)
						             || (my_shot_hits > 0) 
								   )
							   );
				} else {
					//BetterStrategy(t, game, keys, present_fits_past);
					//keys.fire( (t - t_lastshot) > 1 );
					AttackNearestAsteroid(angle_byte, game, keys, false, latency);
				}
			} 
			if (time_of_first_collision <= 1+latency) keys.hyperspace(true);


			if (!present_fits_past) {
				gamePast = game;
				framePast = frame;
			} else {
				; // ltest mgliche Vergangenheit behalten (schlecht frs UFO...)
			}
			havePast = true;
			if (keys.getKeys() & keys.KEY_FIRE) t_lastshot = t;
			if (keys.getKeys() & keys.KEY_HYPERSPACE) {
				t_lasthyper = t;
				havePast = false;
			}
			if (keys.getKeys() & keys.KEY_THRUST) {
				t_lastthrust = t;
				// havePast = false; // es dauert etwas, bevor die Beschleunigung einsetzt, also passt die Vergangenheit noch
			}
			if (flagged_collision)
				t_last_flagged_collision = t;

#if 1
			if (keys.getKeys() & keys.KEY_FIRE) fprintf(stdout, " FIRE");
			if (keys.getKeys() & keys.KEY_HYPERSPACE ) fprintf(stdout, " HYPERSPACE");
			if (keys.getKeys() & keys.KEY_LEFT) fprintf(stdout, " LEFT");
			if (keys.getKeys() & keys.KEY_RIGHT) fprintf(stdout, " RIGHT");
			if (keys.getKeys() & keys.KEY_THRUST) 
				fprintf(stdout, " THRUST");
			if (keys.getKeys() != '@') fprintf(stdout, "\n");
#endif
			fflush(stdout);
			fflush(stderr);
		} // if ship_present



		last_angle_byte = angle_byte;

		if (keys.getKeys() & keys.KEY_LEFT) {
			angle_byte++;
		} else if (keys.getKeys() & keys.KEY_RIGHT) {
			angle_byte--;
		}
		angle_byte = (256 + angle_byte) % 256;
		K[2] = K[1];
		K[1] = K[0];
	}  // end for
} // end of Player::RunSimple


void Player::RunMeasureShots(void)
{
	FramePacket frame, framePast;
	KeysPacket keys;
	GameStatus game, gamePast;
	bool havePast = false;
	char prevframe = 0;
	int t = 0;
	int latency;

	int count_turns = 0;
	int count_after_shot = 0;

	for (;;)
	{
		++t;         // Zeit
		++keys.ping; // jedes gesendete Pckchen erhlt eine individuelle Nummer zur Latenzmessung
		SendPacket(keys);
		ReceivePacket(frame);

		if (frame.frameno != ++prevframe || frame.ping != keys.ping)
		{
			printf("Latenz %d. %d Frames verloren.\n", keys.ping - frame.ping, frame.frameno - prevframe);
			prevframe = frame.frameno;
			latency = keys.ping - frame.ping;
		} else latency = 0;

		InterpretScreen(frame, game, t);

		int time = 1;
		if (havePast) time = frame.frameno - framePast.frameno;
		bool present_fits_past = false;
		if (havePast)
			present_fits_past = game.CalculateSpeed(gamePast);

		if (game.nshots) {
			fprintf(stdout, "Frame:%d NShots:%d Saucer:%d Ship:%d haveSpeeds:%d\n", 
				t, game.nshots, (int)game.saucer_present, (int)game.ship_present, 
				(int)present_fits_past
				);
			if (game.ship_present) fprintf(stdout, "\tShip: x:%f y:%f |v|:%f\n", 
				game.ship.r.x, game.ship.r.y, 
				present_fits_past ? sqrt(game.ship.v.square()) : -1.0 
				);
			for(int i=0; i<game.nshots; i++) {
				Shot& s = game.shots[i];
				fprintf(stdout, "%f %f %d %d %d  Our1:%d Our2:%d", 
					s.r.x, s.r.y, s.gScale, s.pOpcode, s.lOpcode, 
					(int)game.OurShot(i, present_fits_past, false),
					(int)game.OurShot(i, present_fits_past, true)
					);
				if (present_fits_past) {
					fprintf(stdout, " vx:%f vy:%f |v|:%f", 
						s.v.x, s.v.y, sqrt(s.v.square()) 
						);
				}
				fprintf(stdout, "\n");
			} // for i 
		}

		keys.clear();
		if (game.ship_present) {
			keys.fire(6000 == t);
		}
		// keys.thrust(true);

		//if (game.ship_present) {
		//	if (count_turns < 17) {
		//		keys.left(true);
		//		count_turns++;
		//	} else if (0 == count_after_shot) {
		//		keys.fire(true);
		//		count_after_shot++;
		//	} else if (count_after_shot > 200) {
		//		exit(0);
		//	}
		//}

		if (keys.getKeys() & keys.KEY_FIRE) fprintf(stdout, " FIRE");
		if (keys.getKeys() & keys.KEY_HYPERSPACE ) fprintf(stdout, " HYPERSPACE");
		if (keys.getKeys() & keys.KEY_LEFT) fprintf(stdout, " LEFT");
		if (keys.getKeys() & keys.KEY_RIGHT) fprintf(stdout, " RIGHT");
		if (keys.getKeys() & keys.KEY_THRUST) fprintf(stdout, " THRUST");
		if (keys.getKeys() != '@') fprintf(stdout, "\n");

		gamePast = game;
		framePast = frame;
		havePast = true;
	} // end for
} // end of Player::RunMeasureShots










void Player::InterpretScreen(FramePacket &packet, GameStatus& game, double t)
{
	unsigned short *vector_ram = (unsigned short *)packet.vectorram;
	int gvx, gvy, gvs; // Werte vom letzten LABS
	int last_pOpcode; // Opcode == Skalierung der letzten Ruhezeit vor Schu.
	int dx, dy, sf, vx, vy, vz, vs;
	int v1x = 0;
	int v1y = 0;
	int shipdetect = 0;

	game.clear();
	if ((unsigned char)packet.vectorram[1] != 0xe0 && (unsigned char)packet.vectorram[1] != 0xe2)
		return; // sollte nicht vorkommen; erster Befehl ist immer ein JMPL

	int pc = 1;
	int lastop = -1;
	int op = vector_ram[pc] >> 12;
	for (;;)
	{
		lastop = op;
		op = vector_ram[pc] >> 12;
		switch (op)
		{
		case 0xa: // LABS
			gvy = vy = vector_ram[pc] & 0x3ff;
			gvx = vx = vector_ram[pc+1] & 0x3ff;
			gvs = vs = vector_ram[pc+1] >> 12;
			break;
		case 0xb: // HALT
			return;
		case 0xc: // JSRL
			switch (vector_ram[pc] & 0xfff)
			{
			case 0x8f3:
				game.asteroids[game.nasteroids++].set(vx, vy, 1, vs, t);
				break;
			case 0x8ff:
				game.asteroids[game.nasteroids++].set(vx, vy, 2, vs, t);
				break;
			case 0x90d:
				game.asteroids[game.nasteroids++].set(vx, vy, 3, vs, t);
				break;
			case 0x91a:
				game.asteroids[game.nasteroids++].set(vx, vy, 4, vs, t);
				break;
			case 0x929:
				game.saucer_present = true;
				game.saucer.r.x = vx;
				game.saucer.r.y = vy;
				game.saucer.v.x = 0;
				game.saucer.v.y = 0;
				game.saucer.size = vs;
				game.saucer.t = t;
				break;
			}  
			break;
		case 0xd: // RTSL
			return;
		case 0xe: // JMPL
			/*
			pc = vector_ram[pc] & 0xfff;
			break;
			*/
			return;
		case 0xf: // SVEC
			/*
			dy = vector_ram[pc] & 0x300;
			if ((vector_ram[pc] & 0x400) != 0)
				dy = -dy;
			dx = (vector_ram[pc] & 3) << 8;
			if ((vector_ram[pc] & 4) != 0)
				dx = -dx;
			sf = (((vector_ram[pc] & 8) >> 2) | ((vector_ram[pc] & 0x800) >> 11)) + 2;
			vz = (vector_ram[pc] & 0xf0) >> 4;
			*/
			break;
		default: // Immer Vector?
			dy = vector_ram[pc] & 0x3ff;
			if ((vector_ram[pc] & 0x400) != 0)
				dy = -dy;
			dx = vector_ram[pc+1] & 0x3ff;
			if ((vector_ram[pc+1] & 0x400) != 0)
				dx = -dx;
			sf = op;
			vz = vector_ram[pc+1] >> 12;
			if (dx == 0 && dy == 0 && vz == 0) {
				// Ruhezeit
				last_pOpcode = op;
			}

			if (dx == 0 && dy == 0 && vz == 15) {
				// Schuss
				Shot *shot = &game.shots[game.nshots++];
				shot->set(vx, vy);
				shot->gScale = gvs;
				shot->pOpcode = last_pOpcode;
				shot->lOpcode = op;
				shot->t = t;
			}
			if (op == 6 && vz == 12 && dx != 0 && dy != 0)
			{
				switch (shipdetect)
				{
				case 0:
					v1x = dx;
					v1y = dy;
					++shipdetect;
					break;
				case 1:
					game.ship_present = true;
					game.ship.r.x = vx;
					game.ship.r.y = vy;
					game.ship.v.x = 0;
					game.ship.v.y = 0;
					game.ship.dx = v1x - dx;
					game.ship.dy = v1y - dy;
					game.ship.t = t;
					++shipdetect;
					break;
				}
			}
			else if (shipdetect == 1)
				shipdetect = 0;

			break;
		}
		if (op <= 0xa)
			++pc;
		if (op != 0xe) // JMPL
			++pc;
	}   

}



void Asteroid::set(int x, int y, int type, int sf, double t)
{
	r.x = x;
	r.y = y;
	v.x = v.y = 0;
	this->type = type;
	this->sf = sf;
	this->t=t;
}


void GameStatus::clear(void)
{
	ship_present = false;
	saucer_present = false;
	nasteroids = 0;
	nshots = 0;
}

KeysPacket::KeysPacket(void)
{
	signature[0] = 'c';
	signature[1] = 't';
	signature[2] = 'm';
	signature[3] = 'a';
	signature[4] = 'm';
	signature[5] = 'e';
	keys = '@';
	ping = 0;
}

void KeysPacket::clear(void)
{
	keys = '@';
}

void KeysPacket::hyperspace(bool b)
{
	if (b)
		keys |= KEY_HYPERSPACE;
	else
		keys &= ~KEY_HYPERSPACE;
}

void KeysPacket::fire(bool b)
{
	if (b)
		keys |= KEY_FIRE;
	else
		keys &= ~KEY_FIRE;
}

void KeysPacket::thrust(bool b)
{
	if (b)
		keys |= KEY_THRUST;
	else
		keys &= ~KEY_THRUST;
}

void KeysPacket::left(bool b)
{
	if (b)
	{
		keys |= KEY_LEFT;
		right(false);
	}
	else
		keys &= ~KEY_LEFT;
}

void KeysPacket::right(bool b)
{
	if (b)
	{
		keys |= KEY_RIGHT;
		left(false);
	}
	else
		keys &= ~KEY_RIGHT;
}

void Player::ReceivePacket(FramePacket &packet)
{
	sockaddr_in sender;
	int sender_size = sizeof sender;
	fd_set readfds, writefds, exceptfds;

	do
	{
		FD_ZERO(&readfds);
		FD_ZERO(&writefds);
		FD_ZERO(&exceptfds);
		FD_SET(sd, &readfds);
		FD_SET(sd, &exceptfds);
		select(sd+1, &readfds, &writefds, &exceptfds, NULL);
		int bytes_received = recv(sd, (char *)&packet, sizeof packet, 0);
		if (bytes_received != sizeof packet)
		{
			int err = WSAGetLastError();
			fprintf(stdout, "Fehler %d bei recvfrom().\n", err);
			exit(1);
		}
		FD_ZERO(&readfds);
		FD_ZERO(&writefds);
		FD_ZERO(&exceptfds);
		FD_SET(sd, &readfds);
		timeval zero;
		zero.tv_sec = zero.tv_usec = 0;
		select(sd+1, &readfds, &writefds, &exceptfds, &zero);
	} while(FD_ISSET(sd, &readfds));
}

void Player::SendPacket(KeysPacket &packet)
{
	sockaddr_in server;
	memset(&server, 0, sizeof server);
	server.sin_family = AF_INET;
	server.sin_port = htons(1979);
	server.sin_addr.s_addr = server_ip;
	if (sizeof packet != sendto(sd, (char *)&packet, sizeof packet, 0, (sockaddr*)&server, sizeof server))
	{
#if defined(WINDOWS)
		int err = WSAGetLastError();
		if (err != WSAEWOULDBLOCK)
		{
			fprintf(stdout, "Fehler %d bei sendto().\n", err);
			exit(1);
		}
#else
		if (errno != EAGAIN)
		{
			perror("Fehler bei sendto()");
			exit(1);
		}
#endif
	}
}

void Player::SendName()
{
	char packet[38] = {0,};
	strcpy(packet, "ctnamepalemale");
	//strcpy(packet, "ctnamepoerl haribo");
	sockaddr_in server;
	memset(&server, 0, sizeof server);
	server.sin_family = AF_INET;
	server.sin_port = htons(1979);
	server.sin_addr.s_addr = server_ip;
	if (sizeof packet != sendto(sd, (char *)&packet, sizeof packet, 0, (sockaddr*)&server, sizeof server))
	{
#if defined(WINDOWS)
		int err = WSAGetLastError();
		if (err != WSAEWOULDBLOCK)
		{
			fprintf(stdout, "Fehler %d bei sendto().\n", err);
			exit(1);
		}
#else
		if (errno != EAGAIN)
		{
			perror("Fehler bei sendto()");
			exit(1);
		}
#endif
	}
}




#if 0
bool Player::CalculateSpeeds
(const GameStatus& gamePast, GameStatus& game)
 {
	static int ncalls=0;
	static int nvalid=0;
	const int pastF = -1;
	const int presentF = 2;

	ncalls++;

	// Falls Vergangenheit und Gegenwart zu unterschiedlich sind,
	// versuchen wir erst gar keine Projektion in die Zukunft.
	if (gamePast.saucer_present != game.saucer_present) return false;
	if (!gamePast.ship_present || !game.ship_present) return false;
	if (gamePast.nasteroids != game.nasteroids) return false;
	for(int i=0; i<game.nasteroids; i++)
		if (game.asteroids[i].type != gamePast.asteroids[i].type
			|| game.asteroids[i].sf != gamePast.asteroids[i].sf)
			return false;
	if (gamePast.nshots != game.nshots) return false;

	nvalid++;

	// printf("Player::Forcast %10d / %-10d\n", nvalid, ncalls);


	for(int i=0; i<game.nasteroids; i++) {
		// calculate incoming flag
		game.asteroids[i].incoming = 0.0;

		double dx = normalizeDX(game.ship_x - game.asteroids[i].x); 
		double dy = normalizeDY(game.ship_y - game.asteroids[i].y); 
		double len = sqrt( dx*dx + dy*dy );
		if (len > 0) {dx /= len; dy /= len;}

		double vx = normalizeDX(game.asteroids[i].x - game.ship_x) 
			- normalizeDX(gamePast.asteroids[i].x - gamePast.ship_x);
		double vy = normalizeDY(game.asteroids[i].y - game.ship_y) 
			- normalizeDY(gamePast.asteroids[i].y - gamePast.ship_y);
		game.asteroids[i].vx = vx;
		game.asteroids[i].vy = vy;
		
		game.asteroids[i].incoming_speed = dx * vx + dy * vy; // cos(phi) * ||v||

		len = sqrt( vx*vx + vy*vy );
		if (len > 0) game.asteroids[i].incoming = game.asteroids[i].incoming_speed / len; // cos(phi)
		else game.asteroids[i].incoming = -1.0;
	}

	for(int i=0; i<game.nshots; i++) {
		// calculate incoming flag
		game.shots[i].incoming = 0.0;

		double dx = normalizeDX(game.ship_x - game.shots[i].x); 
		double dy = normalizeDY(game.ship_y - game.shots[i].y); 
		double len = sqrt( dx*dx + dy*dy );
		if (len > 0) {dx /= len; dy /= len;}

		double vx = normalizeDX(game.shots[i].x - game.ship_x) 
			- normalizeDX(gamePast.shots[i].x - gamePast.ship_x);
		double vy = normalizeDY(game.shots[i].y - game.ship_y) 
			- normalizeDY(gamePast.shots[i].y - gamePast.ship_y);
		game.shots[i].vx = vx;
		game.shots[i].vy = vy;

		game.shots[i].incoming_speed = dx * vx + dy * vy; // cos(phi) * ||v||

		len = sqrt( vx*vx + vy*vy );
		if (len > 0) game.shots[i].incoming = game.shots[i].incoming_speed / len; // cos(phi)
		else game.shots[i].incoming = -1.0;
	}	


	return true;
}


bool Player::Forecast
(const GameStatus& gamePast, const GameStatus& game, 
 GameStatus& gameFuture)
{
	static int ncalls=0;
	static int nvalid=0;
	const int pastF = -1;
	const int presentF = 2;

	ncalls++;

	// Falls Vergangenheit und Gegenwart zu unterschiedlich sind,
	// versuchen wir erst gar keine Projektion in die Zukunft.
	if (gamePast.saucer_present != game.saucer_present) return false;
	if (!gamePast.ship_present || !game.ship_present) return false;
	if (gamePast.nasteroids != game.nasteroids) return false;
	for(int i=0; i<game.nasteroids; i++)
		if (game.asteroids[i].type != gamePast.asteroids[i].type
			|| game.asteroids[i].sf != gamePast.asteroids[i].sf)
			return false;
	if (gamePast.nshots != game.nshots) return false;

	nvalid++;

	// printf("Player::Forcast %10d / %-10d\n", nvalid, ncalls);

	gameFuture = game;

	gameFuture.ship_x   = presentF * game.ship_x + pastF * gamePast.ship_x;
	gameFuture.ship_y   = presentF * game.ship_y + pastF * gamePast.ship_y;

	gameFuture.saucer_x = presentF * game.saucer_x + pastF * gamePast.saucer_x;
	gameFuture.saucer_y = presentF * game.saucer_y + pastF * gamePast.saucer_y;

	for(int i=0; i<gameFuture.nasteroids; i++) {
		gameFuture.asteroids[i].x = 
			presentF * game.asteroids[i].x
			+ pastF * gamePast.asteroids[i].x;
		gameFuture.asteroids[i].y = 
			presentF * game.asteroids[i].y
			+ pastF * gamePast.asteroids[i].y;
		// calculate incoming flag
		gameFuture.asteroids[i].incoming = 0.0;
		double vx = normalizeDX(game.asteroids[i].x - gamePast.asteroids[i].x);
		double vy = normalizeDY(game.asteroids[i].y - gamePast.asteroids[i].y);
		double len = sqrt( vx*vx + vy*vy );
		if (len <= 0) {gameFuture.asteroids[i].incoming = 0.0;continue;}
		vx /= len; vy /= len;
		double dx = normalizeDX(game.ship_x - game.asteroids[i].x); 
		double dy = normalizeDY(game.ship_y - game.asteroids[i].y); 
		len = sqrt( dx*dx + dy*dy );
		if (len <= 0) {gameFuture.asteroids[i].incoming = 1.0;continue;}
		dx /= len; dy /= len;
		gameFuture.asteroids[i].incoming = dx * vx + dy * vy; // cos(phi)
	}

	for(int i=0; i<gameFuture.nshots; i++) {
		gameFuture.shots[i].x = 
			presentF * game.shots[i].x
			+ pastF * gamePast.shots[i].x;
		gameFuture.shots[i].y = 
			presentF * game.shots[i].y
			+ pastF * gamePast.shots[i].y;
	}
	
	return true;
}



void Player::OriginalStrategyPlusShots
	(int t, const GameStatus& gameToUse, 
	 KeysPacket& keys)
{
		keys.clear();   // alle Tasten loslassen
		int min_dist_shots = 0x7fffffff;
		int min_dx_shots = 0;
		int min_dy_shots = 0;

		int min_dist = 0x7fffffff;
		int min_dx = 0;
		int min_dy = 0;
		if (gameToUse.ship_present)
		{
			for (int i=0; i<gameToUse.nshots; i++) {
				int dx = normalizeDX(gameToUse.shots[i].x 
						             - gameToUse.ship_x);
				int dy = normalizeDY(gameToUse.shots[i].y 
					                 - gameToUse.ship_y);
				int dist = dx*dx+dy*dy;  // Quadrat des Abstands zu diesem Asteroiden
				if (dist < min_dist_shots)
				{
					min_dist_shots = dist;
					min_dx_shots = dx;
					min_dy_shots = dy;
				}
			}

			for (int i=0; i<gameToUse.nasteroids; ++i)
			{   // nchstgelegenen Asteroiden suchen
				int dx = normalizeDX(gameToUse.asteroids[i].x - gameToUse.ship_x);
				int dy = normalizeDY(gameToUse.asteroids[i].y - gameToUse.ship_y);
				int dist = dx*dx+dy*dy;  // Quadrat des Abstands zu diesem Asteroiden
				switch (gameToUse.asteroids[i].sf)
				{	// Abstand um den ungefhren Radius des Asteroiden korrigieren
					case 0:  // groer Asteroid
						dist -= 40*40;
						break;
					case 15: // mittlerer Asteroid
						dist -= 20*20;
						break;
					case 14: // kleiner Asteroid
						dist -= 8*8;
						break;
				}
				if (dist < min_dist)
				{
					min_dist = dist;
					min_dx = dx;
					min_dy = dy;
				}
			}
			if (gameToUse.saucer_present)
			{
				int dx = normalizeDX(gameToUse.saucer_x - gameToUse.ship_x);
				int dy = normalizeDY(gameToUse.saucer_y - gameToUse.ship_y);
				int dist = dx*dx+dy*dy;
				switch (gameToUse.saucer_size)
				{	// Abstand um den ungefhren Radius des UFOs korrigieren
				case 15: // groes UFO
					dist -= 20*12;
					break;
				case 14: // kleines UFO
					dist -= 10*6;
					break;
				}
				if (dist < min_dist)
				{
					min_dist = dist;
					min_dx = dx;
					min_dy = dy;
				}
			}

			// Schiff in Richtung auf das nchstgelegene Objekt drehen
			// mathematisch wird hier das Kreuzprodukt aus den Vektoren 
			// ship_dx/y/0 und min_dx/y/0 berechnet
			if (gameToUse.ship_dx * min_dy - gameToUse.ship_dy * min_dx > 0)
				keys.left(true);
			else
				keys.right(true);

			if (min_dist < 27*27) // Flucht, wenn Kollision unausweichlich
				keys.hyperspace(true);
			else if (min_dist_shots < 27*27) {
				keys.hyperspace(true);
				printf("%s: Hyperspace for shot\n", __FUNCTION__);
			} else if (min_dist_shots < 54*54) {
				keys.thrust(true);
				printf("%s: Thrust for shot\n", __FUNCTION__);
			}

			if (min_dist > 400*400) // beschleunigen, wenn nichts in der Nhe
				keys.thrust(true);
			if (t % 2 == 0)  // Feuerknopf drcken, so schnell es geht
				keys.fire(true);
		}
}

void Player::OriginalStrategyPlusShotsPlusIncoming
	(int t, const GameStatus& gameToUse, 
	 KeysPacket& keys)
{
		keys.clear();   // alle Tasten loslassen
		int min_dist_shots = 0x7fffffff;
		int min_dx_shots = 0;
		int min_dy_shots = 0;

		int min_dist = 0x7fffffff;
		int min_dx = 0;
		int min_dy = 0;

		int danger_dist = 0x7fffffff;
		int danger_dx = 0;
		int danger_dy = 0;

		if (gameToUse.ship_present)
		{
			if (gameToUse.saucer_present)
			for (int i=0; i<gameToUse.nshots; i++) 
			if (gameToUse.shots[i].incoming > 0.95)
			{
				int dx = normalizeDX(gameToUse.shots[i].x 
						             - gameToUse.ship_x);
				int dy = normalizeDY(gameToUse.shots[i].y 
					                 - gameToUse.ship_y);
				int dist = dx*dx+dy*dy;  // Quadrat des Abstands zu diesem Schuss
				if (dist < min_dist_shots)
				{
					min_dist_shots = dist;
					min_dx_shots = dx;
					min_dy_shots = dy;
				}
			}

			for (int i=0; i<gameToUse.nasteroids; ++i)
			{   // nchstgelegenen Asteroiden suchen
				int dx = normalizeDX(gameToUse.asteroids[i].x - gameToUse.ship_x);
				int dy = normalizeDY(gameToUse.asteroids[i].y - gameToUse.ship_y);
				int dist = dx*dx+dy*dy;  // Quadrat des Abstands zu diesem Asteroiden
				switch (gameToUse.asteroids[i].sf)
				{	// Abstand um den ungefhren Radius des Asteroiden korrigieren
					case 0:  // groer Asteroid
						dist -= 40*40;
						break;
					case 15: // mittlerer Asteroid
						dist -= 20*20;
						break;
					case 14: // kleiner Asteroid
						dist -= 8*8;
						break;
				}


				if (dist < min_dist)
				{
					min_dist = dist;
					min_dx = dx;
					min_dy = dy;
				}
				// 32 steps are needed make a half turn (180 degrees)
				int weighted_dist = dist 
					- 32 * int(gameToUse.asteroids[i].incoming_speed * gameToUse.asteroids[i].incoming_speed);
				if (weighted_dist < danger_dist)
				{
					danger_dist = weighted_dist;
					danger_dx = dx;
					danger_dy = dy;
				}


			}
			if (gameToUse.saucer_present)
			{
				int dx = normalizeDX(gameToUse.saucer_x - gameToUse.ship_x);
				int dy = normalizeDY(gameToUse.saucer_y - gameToUse.ship_y);
				int dist = dx*dx+dy*dy;
				switch (gameToUse.saucer_size)
				{	// Abstand um den ungefhren Radius des UFOs korrigieren
				case 15: // groes UFO
					dist -= 20*12;
					break;
				case 14: // kleines UFO
					dist -= 10*6;
					break;
				}
				if (dist < min_dist)
				{
					min_dist = dist;
					danger_dx = min_dx = dx; // shoot the ship, if that near!
					danger_dy = min_dy = dy;
				}
			}

			// Schiff in Richtung auf das nchstgelegene Objekt drehen
			// mathematisch wird hier das Kreuzprodukt aus den Vektoren 
			// ship_dx/y/0 und min_dx/y/0 berechnet
			if (gameToUse.ship_dx * danger_dy - gameToUse.ship_dy * danger_dx > 0)
				keys.left(true);
			else
				keys.right(true);

			if (min_dist < 27*27) // Flucht, wenn Kollision unausweichlich
				keys.hyperspace(true);
			else if (min_dist_shots < 27*27) { // Flucht, wenn Abschu unausweichlich (das geht nur gut fr vorhergesagte Schsse)
				keys.hyperspace(true);
				printf("%s: Hyperspace for shot\n", __FUNCTION__);
			}

			if (min_dist > 400*400) // beschleunigen, wenn nichts in der Nhe
				keys.thrust(true);
			if (t % 2 == 0)  // Feuerknopf drcken, so schnell es geht
				keys.fire(true);
		}
}


void Player::OriginalStrategyPlusShotsPlusIncomingPlusCarefullThrottle
	(int t, const GameStatus& gamePast, const GameStatus& gameToUse, 
	 KeysPacket& keys)
{
		keys.clear();   // alle Tasten loslassen
		int min_dist_shots = 0x7fffffff;
		int min_dx_shots = 0;
		int min_dy_shots = 0;

		int min_dist = 0x7fffffff;
		int min_dx = 0;
		int min_dy = 0;

		int danger_dist = 0x7fffffff;
		int danger_dx = 0;
		int danger_dy = 0;

		if (gameToUse.ship_present)
		{ 

			for (int i=0; i<gameToUse.nasteroids; ++i)
			{   // nchstgelegenen Asteroiden suchen
				int dx = normalizeDX(gameToUse.asteroids[i].x - gameToUse.ship_x);
				int dy = normalizeDY(gameToUse.asteroids[i].y - gameToUse.ship_y);
				int dist = dx*dx+dy*dy;  // Quadrat des Abstands zu diesem Asteroiden
				switch (gameToUse.asteroids[i].sf)
				{	// Abstand um den ungefhren Radius des Asteroiden korrigieren
					case 0:  // groer Asteroid
						dist -= 40*40;
						break;
					case 15: // mittlerer Asteroid
						dist -= 20*20;
						break;
					case 14: // kleiner Asteroid
						dist -= 8*8;
						break;
				}


				if (dist < min_dist)
				{
					min_dist = dist;
					min_dx = dx;
					min_dy = dy;
				}
				if (dist < danger_dist) 
				{
					danger_dist = dist;
					danger_dx = dx;
					danger_dy = dy;
				}
			}
			if (gameToUse.saucer_present)
			{
				int dx = normalizeDX(gameToUse.saucer_x - gameToUse.ship_x);
				int dy = normalizeDY(gameToUse.saucer_y - gameToUse.ship_y);
				int dist = dx*dx+dy*dy;
				switch (gameToUse.saucer_size)
				{	// Abstand um den ungefhren Radius des UFOs korrigieren
				case 15: // groes UFO
					dist -= 20*12;
					break;
				case 14: // kleines UFO
					dist -= 10*6;
					break;
				}
				if (dist < min_dist)
				{
					min_dist = dist;
					danger_dx = min_dx = dx; // shoot the ship, if that near!
					danger_dy = min_dy = dy;
				}
			}

			if (gameToUse.saucer_present)
			for (int i=0; i<gameToUse.nshots; i++) 
			if (acos(gameToUse.shots[i].incoming) < 5 * DEG) {
				int dx = normalizeDX(gameToUse.shots[i].x 
						             - gameToUse.ship_x);
				int dy = normalizeDY(gameToUse.shots[i].y 
					                 - gameToUse.ship_y);
				int dist = dx*dx+dy*dy;  // Quadrat des Abstands zu diesem Schuss
				if (dist < min_dist_shots)
				{
					min_dist_shots = dist;
					min_dx_shots = dx;
					min_dy_shots = dy;
				}
			}


			// Schiff in Richtung auf das nchstgelegene Objekt drehen
			// mathematisch wird hier das Kreuzprodukt aus den Vektoren 
			// ship_dx/y/0 und min_dx/y/0 berechnet
			if (gameToUse.ship_dx * danger_dy - gameToUse.ship_dy * danger_dx > 0)
				keys.left(true);
			else
				keys.right(true);

			if (min_dist < 27*27) // Flucht, wenn Kollision unausweichlich
				keys.hyperspace(true);
			else if (min_dist_shots < 0x7fffffff) { // Flucht, wenn Abschu droht
			    if (gameToUse.ship_dx * -min_dy_shots - gameToUse.ship_dy * -min_dx_shots > 0) {
					keys.right(false);keys.left(true);
				} else {
					keys.left(false);keys.right(true);
				}
				keys.thrust(true);
			}

			if (min_dist > 400*400 && gameToUse.nasteroids < 3 
				&& gamePast.ship_x == gameToUse.ship_x
				&& gamePast.ship_y == gameToUse.ship_y) // beschleunigen, wenn nichts in der Nhe
				keys.thrust(true);
			if (t % 2 == 0)  // Feuerknopf drcken, so schnell es geht
				keys.fire(true);
		}
}



void Player::ForecastStrategy
(int t, const GameStatus& gamePast, GameStatus& gameToUse, KeysPacket& keys)
{
	const int NUM_STEPS = 16; // Wir versuchen NUM_STEPS-1 Schritte in die Zukunft zu schauen.
	GameStatus game[NUM_STEPS];
	KeysPacket key[NUM_STEPS];
	int f;
	
	keys.clear();
	if (!gameToUse.ship_present) return;

	CalculateSpeeds(gamePast, gameToUse);
	OriginalStrategyPlusShotsPlusIncomingPlusCarefullThrottle(t, gamePast, gameToUse, keys);
	if (keys.getKeys() & keys.KEY_HYPERSPACE)
		return;

	KeysPacket originalKeys = keys;

	for(f=0; f<NUM_STEPS; f++) {
		game[f] = gameToUse;
		key[f] = keys;
	}
	game[0] = gamePast;

	int collision_f = -1;

	for(f=1; f<NUM_STEPS; f++) {
		if (!CalculateSpeeds(game[f-1], game[f]))
			break;
		// Advance
		game[f].ship_x += game[f].ship_x - game[f-1].ship_x;
		game[f].ship_y += game[f].ship_y - game[f-1].ship_y;
		if (game[f].saucer_present) {
			game[f].saucer_x += game[f].saucer_x - game[f-1].saucer_x;
			game[f].saucer_y += game[f].saucer_y - game[f-1].saucer_y;
		}
		for(int i=0; i<game[f].nasteroids; i++) {
			game[f].asteroids[i].x += (int)game[f].asteroids[i].vx;
			game[f].asteroids[i].y += (int)game[f].asteroids[i].vy;
		}
		for(int i=0; i<game[f].nshots; i++) {
			game[f].shots[i].x += (int)game[f].shots[i].vx;
			game[f].shots[i].y += (int)game[f].shots[i].vy;
		}

		OriginalStrategyPlusShotsPlusIncomingPlusCarefullThrottle(t, game[f-1], game[f], key[f]);
		if (key[f].getKeys() & keys.KEY_HYPERSPACE) {
			break;
		}
	}
	collision_f = f;
	if (collision_f < NUM_STEPS) 
		printf("t:%d collision_f:%d/%d\n", t, collision_f, NUM_STEPS);
	if (collision_f < NUM_STEPS) 
		keys = key[collision_f];
	else keys = originalKeys; // Benutze keys aus der OriginalStrategie
}



#endif