package core;

import java.awt.Color;
import java.util.ArrayList;
import java.util.Vector;

import util.Statistics;
import util.Vector2D;
import core.asteroid.AsteroidObject;
import core.asteroid.Decoration;
import core.asteroid.Ship;
import core.asteroid.Shot;
import core.asteroid.Ufo;
import core.opCodes.OpLocation;

public class GameInfo {

	private int gameTime;
	private int frameDif;
	private int latency;
	
	private int asteroidCnt = 0;
	private int curShotCnt = 0;
	
	private int shotAvailableTime = 0;
	
	private Vector<AsteroidObject> objects;
	private Ship ship;
	private Ufo ufo;
	private Vector<Shot> shots;
	private Vector<AsteroidObject> collisionObjs;
	private Vector<Decoration> decoObjs;
	private Vector<AsteroidObject> destroyableObjs;
	
	
	public GameInfo() {
		collisionObjs = new Vector<AsteroidObject>();
		destroyableObjs = new Vector<AsteroidObject>();
		decoObjs = new Vector<Decoration>();
		shots = new Vector<Shot>();
		objects = new Vector<AsteroidObject>();
	}
	
	
	public void newFrame(ArrayList<AsteroidObject> frameObjs, int gameMode, int frameDif, int latency) {
		this.frameDif = frameDif;
		this.latency = latency;
		
		if(gameMode == ScreenInterpreter.GAME_MODE_GAME) {
			gameTime+=frameDif;
		} else {
			gameTime = 0;
		}
		Statistics.putFloat(Statistics.KEY_TIME, gameTime);
		
		// Objekte verfolgen
		matchObjects(frameObjs);
		// Objekte kategorisieren
		categorizeObjects();
		// Objekte auf Kollisionskurs mit Schiff ermitteln
		if(isShipPresent()) detectCollidingObjects();
		
		/*if(gameTime % 10 == 0) {
			recalcCollisionMap();
		}*/
	}


	private void matchObjects(ArrayList<AsteroidObject> frameObjs) {
		ArrayList<AsteroidObject> deadObjs = new ArrayList<AsteroidObject>();
		
		for(int i=0; i<objects.size(); i++) {
			boolean found = false;
			AsteroidObject obj = objects.get(i);
			
			for(int j=0; j<frameObjs.size(); j++) {
				AsteroidObject fObj = frameObjs.get(j);
				if(obj.isMatching(fObj, frameDif)) {
					obj.updateLocation(fObj, frameDif);
					frameObjs.remove(j);
					found = true;
					break;
				}
			}
			
			// Nicht mehr gefundene Objekte entfernen
			if(!found) {
				deadObjs.add(objects.remove(i));
				i--;
			}
		}
		
		// Neue Objekte hinzufgen
		for(int i=0; i<frameObjs.size(); i++) {
			objects.add(frameObjs.get(i));
		}
		
		analyzeHits(deadObjs);
	}


	private void categorizeObjects() {
		asteroidCnt = 0;
		curShotCnt = 0;
		ufo = null;
		shots.clear();
		decoObjs.clear();
		destroyableObjs.clear();
		boolean shipFound = false;
		int minShotHitTime = 1000;
		
		for(int i=0; i<objects.size(); i++) {
			AsteroidObject o = objects.get(i);
			if(o instanceof Ship) {
				ship = (Ship) o;
				shipFound = true;
			} else if(o instanceof Shot) {
				Shot s = (Shot) o;
				shots.add(s);
				if(s.getShotType() == Shot.SHOT_UNKNOWN) {
					float dist = 100;
					if(ship != null) {
						// Abstand Schiff - Schuss
						dist = ship.calculateDistance(s);
					}
					if(dist < 50.0f) {
						s.setShotType(Shot.SHOT_SHIP);
						float shotCnt = Statistics.getFloat(Statistics.KEY_SHOTS);
						Statistics.putFloat(Statistics.KEY_SHOTS, shotCnt+1);
					} else {
						s.setShotType(Shot.SHOT_UFO);
					}
				}
				if(s.getShotType() == Shot.SHOT_SHIP) {
					curShotCnt++;
					if(s.getHitTime() < minShotHitTime) {
						minShotHitTime = s.getHitTime();
					}
				}
				if(s.getAge() >= AsteroidObject.MAX_RESOLUTION_AGE && !s.isCollisionCalculated()) {
					detectShotCollision(s);
					s.setCollisionCalculated(true);
				}
			} else if(o instanceof Decoration) {
				decoObjs.add((Decoration) o);
			} else if(o instanceof Ufo) {
				ufo = (Ufo) o;
				destroyableObjs.add(o);
			} else {
				asteroidCnt++;
				destroyableObjs.add(o);
			}
		}
		
		if(curShotCnt == 4) {
			// Alle vier Schuss Slots belegt
			shotAvailableTime = minShotHitTime;
		} else {
			shotAvailableTime = 0;
		}
		
		if(!shipFound) {
			ship = null;
		}
		
		Statistics.putFloat(Statistics.KEY_ALIVE_SHOTS, curShotCnt);
		Statistics.putFloat(Statistics.KEY_ASTEROIDS, asteroidCnt);
	}

	
	private void detectCollidingObjects() {
		collisionObjs.clear();
		
		for(int i=0; i<destroyableObjs.size(); i++) {
			AsteroidObject o = destroyableObjs.get(i);
			int colTime = o.getCollisionTime();
			if(o.getNextColCalc() <= 0 || colTime < 20) {
				colTime = calcColTime(ship, o);
				o.setCollisionTime(colTime);
			}
			if(colTime >= 0) {
				o.setColor(Color.RED);
				collisionObjs.add(o);
			} else {
				o.setColor(o.getDefaultColor());
			}
		}
		
		for(int i=0; i<shots.size(); i++) {
			Shot o = shots.get(i); 
			if(o.getShotType() != Shot.SHOT_SHIP) {
				int colTime = o.getCollisionTime();
				if(o.getNextColCalc() <= 0 || colTime < 20) {
					colTime = calcColTime(ship, o);
					o.setCollisionTime(colTime);
				}
				if(colTime >= 0) {
					o.setColor(Color.RED);
					collisionObjs.add(o);
				} else {
					o.setColor(o.getDefaultColor());
				}
			}
		}
	}
	
	
	private void analyzeHits(ArrayList<AsteroidObject> deadObjs) {
		// Treffer Statistik
		for(int i=0; i<deadObjs.size(); i++) {
			AsteroidObject o = deadObjs.get(i);
			if(o instanceof Shot) {
				Shot s = (Shot)o;
				if(s.getShotType() == Shot.SHOT_SHIP) {
					for(int j=0; j<deadObjs.size(); j++) {
						AsteroidObject p = deadObjs.get(j);
						if(p.isDestroyable() && s.calculateDistance(p) <= p.getHitRadius()+8) {
							float hitCnt = Statistics.getFloat(Statistics.KEY_KILLS);
							float shotCnt = Statistics.getFloat(Statistics.KEY_SHOTS);
							Statistics.putFloat(Statistics.KEY_KILLS, hitCnt+1);
							Statistics.putFloat(Statistics.KEY_HIT_RATE, 100.0f * hitCnt / shotCnt);
							break;
						}
					}
				}
			}
		}
	}
	
	
	private void detectShotCollision(Shot s) {
		int minTime = 10000;
		AsteroidObject colObj = null;
		
		for(int i=0; i<destroyableObjs.size(); i++) {
			AsteroidObject o = destroyableObjs.get(i);
			int colTime = calcColTime(s, o);
			if(colTime >= 0 && colTime < minTime) {
				minTime = colTime;
				colObj = o;
			}
		}
		
		if(colObj != null) {
			colObj.setHitTime(minTime);
			colObj.setHitPreliminary(false);
			s.setHitTime(minTime);
		}
	}
	
	
	private int calcColTime(AsteroidObject o1, AsteroidObject o2) {
		OpLocation loc = o1.getLocation();
		Vector2D p1 = new Vector2D(loc.x, loc.y);
		Vector2D v1 = o1.getVelocity();
		loc = o2.getLocation();
		Vector2D p2 = new Vector2D(loc.x, loc.y);
		Vector2D v2 = o2.getVelocity();
		
		float colSpeed = v1.getLength() + v2.getLength();
		Vector2D distance = new Vector2D();
		int hitDist = o1.getHitRadius() + o2.getHitRadius();

		int colTime = -1;
		int time = 0;
		while(time < 100) {
			distance.x = p1.x - p2.x;
			distance.y = p1.y - p2.y;
			float dist = distance.getLength()-hitDist;
			if(dist <= 0) {
				colTime = time;
				break;
			}
			
			int step = 1;
			if(dist > 10) step = (int)(dist * 0.75f / colSpeed);
			if(step > 10) step = 10;
			if(step <= 0) step = 1;
			time += step;
			
			p1.x += v1.x * step;
			p1.y += v1.y * step;
			
			p2.x += v2.x * step;
			p2.y += v2.y * step;
			
			if(p1.x >= 1024) p1.x -= 1024;
			if(p1.x < 0) p1.x += 1024;
			if(p1.y >= 768) p1.y -= 768;
			if(p1.y < 0) p1.y += 768;
			
			if(p2.x >= 1024) p2.x -= 1024;
			if(p2.x < 0) p2.x += 1024;
			if(p2.y >= 768) p2.y -= 768;
			if(p2.y < 0) p2.y += 768;
		}
		
		return colTime;
	}


	public Vector<AsteroidObject> getAllObjects() {
		return objects;
	}


	public Vector<Decoration> getDecoObjects() {
		return decoObjs;
	}


	public Vector<AsteroidObject> getDestroyableObjects() {
		return destroyableObjs;
	}
	
	
	public Vector<Shot> getShots() {
		return shots;
	}
	
	
	public Vector<AsteroidObject> getCollisionObjects() {
		return collisionObjs;
	}
	
	
	public boolean isObjectOnCol(int id) {
		boolean col = false;
		for(int i=0; i<collisionObjs.size(); i++) {
			AsteroidObject o = collisionObjs.get(i);
			if(o.getObjectId() == id) {
				col = true;
				break;
			}
		}
		return col;
	}
	
	
	public boolean isShipPresent() {
		if(ship != null) {
			return true;
		} else {
			return false;
		}
	}
	
	
	public Ship getShip() {
		return ship;
	}
	
	
	public boolean isUfoPresent() {
		if(ufo != null) {
			return true;
		} else {
			return false;
		}
	}
	
	
	public Ufo getUfo() {
		return ufo;
	}
	
	
	public Shot getNearestShipShot() {
		float dist = 10000;
		Shot shot = null;
		
		if(isShipPresent()) {
			for(int i=0; i<shots.size(); i++) {
				Shot s = shots.get(i);
				if(s.getShotType() == Shot.SHOT_SHIP) {
					float d = ship.calculateDistance(s);
					if(d < dist) {
						dist = d;
						shot = s;
					}
				}
			}
		}
		
		return shot;
	}


	public int getGameTime() {
		return gameTime;
	}


	public int getPacketIncrement() {
		return frameDif;
	}


	public int getLostPacketCount() {
		return frameDif-1;
	}


	public int getLatency() {
		return latency;
	}
	
	
	public int getFrameDif() {
		return frameDif;
	}
	
	
	public int getAsteroidCount() {
		return asteroidCnt;
	}
	
	
	public int getCurrentShotCount() {
		return curShotCnt;
	}
	
	
	public int getNextShotAvailableTime() {
		return shotAvailableTime;
	}
}
