/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.gui;

import java.util.Timer;
import java.util.TimerTask;

import javax.swing.JLabel;
import javax.swing.SwingUtilities;

import de.hfaber.asteroids.client.GameStatusEvent;
import de.hfaber.asteroids.client.IGameStatusListener;
import de.hfaber.asteroids.game.state.GameStatus;

/**
 * @author Henning Faber
 */
public class TelemetryViewer implements IGameStatusListener {

    /**
     * Serial version UID for serialization API.
     */
    private static final long serialVersionUID = -2138676227124502541L;
    
    /**
     * The number of frames per second when the mame server is running
     * with option -speed 1.0 
     */
    private static final int FPS_AT_SPEED_ONE = 60;

    /**
     * Number of seconds in a minute.
     */
    private static final int SECONDS_PER_MINUTE = 60;

    private final TelemetryFrame m_frame;
    
    private int m_frameRate = 0;
    
    /**
     * Timer task for measuring the frame rate.
     */
    private TimerTask m_frameRateTimerTask = new TimerTask() {

        /* (non-Javadoc)
         * @see java.util.TimerTask#run()
         */
        @Override
        public void run() {
            synchronized (TelemetryViewer.this) {
                final int frameRate = m_frameRate;
                m_frameRate = 0;
                SwingUtilities.invokeLater(new Runnable() {

                    /* (non-Javadoc)
                     * @see java.lang.Runnable#run()
                     */
                    public void run() {
                        // set frame rate
                        m_frame.getFrameRateLbl().setText(
                                String.valueOf(frameRate));
                        
                        // set free memory
                        Runtime runtime = Runtime.getRuntime();
                        int freeMem = Math.round(runtime.freeMemory() 
                                / (1024 * 1024));
                        m_frame.getFreeMemLbl().setText(String.valueOf(freeMem) 
                                + "M");
                    }
                });
            }
        }
    };

    /**
     * 
     */
    public TelemetryViewer() {
        super();
        m_frame = new TelemetryFrame();
        m_frame.setVisible(true);
        initFrameRateTimer();
    }

    /**
     * 
     */
    private void initFrameRateTimer() {
        // schedule timer
        Timer frameRateTimer = new Timer("FrameRateTimer", true);
        frameRateTimer.schedule(m_frameRateTimerTask, 0, 1000);
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.client.IGameStatusListener#gameStatusUpdated(de.hfaber.asteroids.asteroids.client.GameStatusEvent)
     */
    public final void gameStatusUpdated(GameStatusEvent e) {
        // new frame arrived, count it for frame rate measurement
        synchronized (this) {
            m_frameRate++;
        }
        
        final GameStatus gameStatus = e.getGameStatus();
        SwingUtilities.invokeLater(new Runnable() {

            /* (non-Javadoc)
             * @see java.lang.Runnable#run()
             */
            public void run() {
                updateTelemetry(gameStatus);
            }
        });
    }
    
    private void updateTelemetry(GameStatus gs) {
        // score
        JLabel scoreLbl = m_frame.getScoreLbl();
        scoreLbl.setText(String.valueOf(gs.getPlayerOneScore()));
        
        // score per 5 minutes
        JLabel scorePerFiveMinLbl = m_frame.getScorePerFiveMinLbl();
        if ((gs.getFrameNo() > 0) && (gs.isGameRunning())) {
            int scorePerFiveMin = gs.getPlayerOneScore()
                    * (5 * 60 * 60) / gs.getFrameNo(); 
            scorePerFiveMinLbl.setText(String.valueOf(scorePerFiveMin));
        }

        // lifes
        JLabel lifesLbl = m_frame.getLifesLbl();
        lifesLbl.setText(String.valueOf(gs.getLifeCount()));
        
        // round
        JLabel roundLbl = m_frame.getRoundLbl();
        roundLbl.setText(String.valueOf(gs.getRound()));
        
        // game duration
        JLabel gameDurationLbl = m_frame.getGameDurationLbl();
        int gameDuration = gs.getFrameNo();
        gameDurationLbl.setText(String.format("%d:%02d.%02d min",
                gameDuration / (SECONDS_PER_MINUTE * FPS_AT_SPEED_ONE),
                gameDuration / FPS_AT_SPEED_ONE % SECONDS_PER_MINUTE, 
                (gameDuration % FPS_AT_SPEED_ONE) * 100 / FPS_AT_SPEED_ONE));
        
        // round duration
        RoundDurationTableModel model = m_frame.getRoundDurationModel();
        if (gs.isGameStarting()) {
            model.reset();
        } else {
            model.setDuration(gs.getRound(), gs.getRoundDuration());
        }
    }
}
