/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.game.objects;

import de.hfaber.asteroids.game.field.Screen;


/**
 * A bullet. In addition to the standard attributes of a game object, 
 * a bullet also has...</p>
 * 
 * <ul>
 *  <li>a source. The source determines, if the bullet was fired from the
 *   ship or from a saucer.
 *  <li>a time to life (ttl). The ttl indicates the number frames before
 *   the bullet will disappear. The bullet may of course even disappear 
 *   earlier, if it hits something.
 * </ul> 

 * @author Henning Faber
 */
public class Bullet extends GameObject {

    /**
     * The maximum square distance that a bullet is expected to move
     * from one frame to another.
     */
    private static final int MAX_BULLET_FRAME_DIST = 16 * 16
            * Screen.INTERNAL_PRECISION * Screen.INTERNAL_PRECISION;

    /**
     * The square radius of a bullet measured in sub pixels.   
     */
    public static final int BULLET_SQUARE_RADIUS = 1 * 1
            * Screen.INTERNAL_PRECISION * Screen.INTERNAL_PRECISION;
    
    // bullet sources
    public static final int UNKNOWN = 0;
    public static final int FRIENDLY = 1;
    public static final int HOSTILE = 2;
    
    /**
     * The number of frames a bullet stays in the game before it
     * expires. 
     */
    public static final int BULLET_LIFE_TIME = 70;

    /**
     * The source of the bullet, either the own ship (FRIENDLY), 
     * a saucer (HOSTILE) or unkown.
     */
    private int m_source;
    
    /**
     * The remaining life time of the bullet, measured in frames. 
     */
    private int m_ttl;
    
    /**
     * Creates a bullet at the given location.
     * 
     * @param x the x coordinate of the bullet
     * @param y the y coordinate of the bullet
     */
    public Bullet(int x, int y) {
        super(x, y);
        m_source = UNKNOWN;
        m_ttl = BULLET_LIFE_TIME;
    }

    /**
     * Creates a bullet that represents what the given bullet is expected to 
     * be like in the given number of frames.
     * 
     * @param toProject the bullet object to project
     * @param frameCount the number of frames to project
     */
    public Bullet(Bullet toProject, int frameCount) {
        super(toProject, frameCount);
        m_source = toProject.m_source;
        m_ttl = toProject.m_ttl - frameCount;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.game.objects.GameObject#getPrecision()
     */
    @Override
    protected int getPrecision() {
        return 8;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.game.objects.GameObject#getMaxFrameDist()
     */
    @Override
    public final int getMaxFrameDist() {
        return MAX_BULLET_FRAME_DIST;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.game.objects.GameObject#getRadius()
     */
    @Override
    public int getSquareRadius() {
        return BULLET_SQUARE_RADIUS;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.game.objects.GameObject#trackCustomProperties(de.hfaber.asteroids.game.objects.GameObject, int)
     */
    @Override
    protected final void trackCustomProperties(GameObject prev, int frameGap) {
        // inherit the bullet's source
        Bullet prevBullet = (Bullet)prev;
        m_source = prevBullet.getSource();
        
        // inherit the time to live value and decrease it by the frame gap 
        m_ttl = prevBullet.getTtl() - frameGap;
    }

    /**
     * @return the source
     */
    public final int getSource() {
        return m_source;
    }

    /**
     * @param source the source to set
     */
    public final void setSource(Ship ship, Saucer saucer) {
        if (m_source == UNKNOWN) {
            if (saucer == null) {
                m_source = FRIENDLY;
            } else if (ship == null) {
                m_source = HOSTILE;
            } else {
                int shipDist = getLocation().distance(ship.getLocation());
                int saucerDist = getLocation().distance(saucer.getLocation());
                if (shipDist < saucerDist) {
                    m_source = FRIENDLY;
                } else {
                    m_source = HOSTILE;
                }
            }
        }
    }

    /**
     * @return the remaining time to live
     */
    public final int getTtl() {
        return m_ttl;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.game.objects.GameObject#toString()
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(getClass().getSimpleName());
        sb.append("[id=");
        sb.append(getId());
        sb.append(" x=");
        sb.append(getRelativeLocation().getX());
        sb.append(" y=");
        sb.append(getRelativeLocation().getY());
        sb.append(" source=");
        switch (m_source) {
            case FRIENDLY:
                sb.append("FRIENDLY");
                break;
            case HOSTILE:
                sb.append("HOSTILE");
                break;
            case UNKNOWN:
                sb.append("UNKNOWN");
                break;
            default:
                sb.append(m_source);
                break;
        }
        sb.append(" ttl=");
        sb.append(m_ttl);
        sb.append("]");
        return sb.toString();
    }
}
