/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.game.field;

/**
 * A point on the playing field. A point has an <em>x</em> and a <em>y</em>
 * co-ordinate. 
 * 
 * @author Henning Faber
 */
public class Point {

    /**
     * The x co-ordinate of the point.
     */
    private final int m_x;

    /**
     * The y co-ordinate of the point.
     */
    private final int m_y;
    
    /**
     * Creates a point with the given co-ordinates.
     * 
     * @param x the x co-ordinate
     * @param y the y co-ordinate
     */
    public Point(int x, int y) {
        this(x, y, false);
    }

    /**
     * Creates a point with the co-ordinates from the given point.
     * 
     * @param p the other point, from which the co-ordinates are taken
     */
    public Point(Point p) {
        this(p.m_x, p.m_y, false);
    }

    /**
     * Creates a point with the given co-ordinates and normalizes the
     * co-ordinates, if desired.
     * 
     * @param x the x co-ordinate
     * @param y the y co-ordinate
     * @param normalize if <code>true</code>, the co-ordinates are normalized,
     *  if <code>false</code>, the co-ordinates are used as given
     */
    public Point(int x, int y, boolean normalize) {
        super();
        if (normalize) {
            m_x = normalize(x, Screen.WIDTH);
            m_y = normalize(y, Screen.HEIGHT);
        } else {
            m_x = x;
            m_y = y;
        }
    }

    /**
     * Normalizes this point.
     * 
     * @return a new point that is the normalization of this point
     */
    public final Point normalize() {
        int normalizedX = normalize(m_x, Screen.WIDTH);
        int normalizedY = normalize(m_y, Screen.HEIGHT);
        Point normalizedPoint = new Point(normalizedX, normalizedY);
        return normalizedPoint;
    }
    
    /**
     * Returns the square of the distance between this point and the 
     * given point.
     * 
     * @param p the other point
     * @return the distance
     */
    public final int distance(Point p) {
        int dx = m_x - p.m_x;
        int dy = m_y - p.m_y;
        dx = normalize(dx, Screen.WIDTH);
        dy = normalize(dy, Screen.HEIGHT);
        return dx * dx + dy * dy;
    }
    
    /**
     * Adds the given point to this point and returns the result
     * as a new point.
     * 
     * @param p the point to add
     * @return a new point that contains the added result
     */
    public final Point add(Point p) {
        Point result = new Point(m_x + p.m_x, m_y + p.m_y);
        return result;
    }
    
    /**
     * Multipies this point with the given scalar value and returns
     * the result as a new point.
     * 
     * @param scalar the scalar value
     * @return a new point with the multiplication result
     */
    public final Point multiply(int scalar) {
        Point result = new Point(m_x * scalar, m_y * scalar);
        return result;
    }
    
    /**
     * Calculates the difference between this point and the given point,
     * normalizes it and and returns it as a new point. 
     * 
     * @param p the point, to which the delta should be calculated
     * @return a new point that contains the delta
     */
    public final Point delta(Point p) {
        int dx = p.m_x - m_x;
        int dy = p.m_y - m_y;
        dx = normalize(dx, Screen.WIDTH);
        dy = normalize(dy, Screen.HEIGHT);
        Point delta = new Point(dx, dy);
        return delta;
    }

    /**
     * Calculates the size of the area that is spanned by this point
     * and the given point.
     * 
     * @param p the other point
     * @return the size of the surface area
     */
    public final int surfaceArea(Point p) {
        return m_x * p.m_y - m_y * p.m_x;
    }
    
    /**
     * Calculates the angle between the vector determined by
     * this point and the vector determined by the given point.
     * 
     * @param p the other point
     * @return the angle in radian units
     */
    public final double angle(Point p) {
        double cosbeta = (m_x * p.m_x + m_y * p.m_y)
                / (length() * p.length());
        double beta = Math.acos(cosbeta);
        return beta;
    }
    
    /**
     * Calculates the length of the vector that is determined by this
     * point.
     * 
     * @return the length
     */
    public final double length() {
        return Math.sqrt(m_x * m_x + m_y * m_y);
    }
    
    /**
     * Normalizes a value along a given maximum. 
     * 
     * @param value the value to normalize
     * @param maxValue the maximum
     * @return the normalized value
     */
    public static int normalize(int value, int maxValue) {
        int normalizedValue = value;
        while (normalizedValue < -1 * (maxValue >> 1)) {
            normalizedValue += maxValue;
        }
        while (normalizedValue > (maxValue >> 1) - 1) {
            normalizedValue -= maxValue;
        }
        return normalizedValue;
    }

    /**
     * Normalizes a value along a given maximum. 
     * 
     * @param value the value to normalize
     * @param maxValue the maximum
     * @return the normalized value
     */
    public static double normalize(double value, int maxValue) {
        double normalizedValue = value;
        while (normalizedValue < -1 * (maxValue >> 1)) {
            normalizedValue += maxValue;
        }
        while (normalizedValue > (maxValue >> 1) - 1) {
            normalizedValue -= maxValue;
        }
        return normalizedValue;
    }

    /**
     * @return the x
     */
    public final int getX() {
        return m_x;
    }

    /**
     * @return the y
     */
    public final int getY() {
        return m_y;
    }

    /* (non-Javadoc)
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof Point) {
            final Point other = (Point)obj;
            return (m_x == other.m_x) && (m_y == other.m_y);
        } else {
            return false;
        }
    }

    /* (non-Javadoc)
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        int result = Screen.WIDTH * m_y + m_x;
        return result;
    }

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(getClass().getSimpleName());
        sb.append("[x=");
        sb.append(getX());
        sb.append(" y=");
        sb.append(getY());
        sb.append("]");
        return sb.toString();
    }
}
