/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.bot.sittingduck;

import java.util.List;

import org.apache.log4j.Logger;

import de.hfaber.asteroids.bot.IAsteroidsBot;
import de.hfaber.asteroids.client.Commands;
import de.hfaber.asteroids.client.GameStatusEvent;
import de.hfaber.asteroids.game.field.Point;
import de.hfaber.asteroids.game.objects.GameObject;
import de.hfaber.asteroids.game.objects.Ship;
import de.hfaber.asteroids.game.state.GameStatus;

/**
 * @author Henning Faber
 */
public class SittingDuck implements IAsteroidsBot {

    /**
     * The LOGGER instance.
     */
    private static final Logger LOGGER = Logger.getLogger(SittingDuck.class);

    /**
     * The move calculator.
     */
    private MoveListManager m_moveListManager;
    
    /**
     * The last game status received from the mame server.
     */
    private GameStatus m_gameStatus;
    
    /**
     * Creates the bot.
     */
    public SittingDuck() {
        super();
        
        // create the move list manager
        m_moveListManager = new MoveListManager();
        Thread calculatorThread = new Thread(m_moveListManager, 
                "MoveListManager");
        calculatorThread.setDaemon(true);
        calculatorThread.start();
        
        // set high priority for communication thread
        Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.client.IGameStatusListener#gameStatusUpdated(de.hfaber.asteroids.asteroids.client.GameStatusEvent)
     */
    public final void gameStatusUpdated(GameStatusEvent e) {
        m_moveListManager.gameStatusUpdated(e);
        m_gameStatus = e.getGameStatus();
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.bot.IAsteroidsBot#compute()
     */
    public final Commands compute() {

        // create new commands object
        Commands c = new Commands();
        
        // do not send any commands, if ship is not present
        Ship ship = m_gameStatus.getShip();
        if (ship != null) {
            
            // get available moves
            List<Move> moves = m_moveListManager.getMoves(
                    m_gameStatus.getFrameNo(), ship.getShotAngle());
            
            // get the best move
            Move move = null;
            if (moves.size() > 0) {
                move = moves.get(0);
            }
            
            // execute move, if shot frame number is reached
            if ((move != null)
                    && (m_gameStatus.getFrameNo() == move.getShotFrameNo())) {
                    
                if (LOGGER.isInfoEnabled() 
                        && (ship.getShotAngle() != move.getAngle())) {
                    StringBuilder sb = new StringBuilder();
                    sb.append("Bullet angle missed! [desiredAngle=");
                    sb.append(move.getAngle());
                    sb.append(" actualAngle=");
                    sb.append(ship.getShotAngle());
                    sb.append("]");
                    LOGGER.info(sb.toString());
                }
                
                c.pressFire(true);
                m_moveListManager.trackShot(move, m_gameStatus.getFrameNo());
                
                // find the next move; skip all moves, that fire at
                // the target that was just fired at and that require
                // to fire in the direct next move, because that is not
                // possible without releasing the fire button first
                int nextMoveIdx = 1;
                while ((nextMoveIdx < moves.size())
                        && ((moves.get(nextMoveIdx).getTargetId() == move.getTargetId())
                                || (moves.get(nextMoveIdx).getShotFrameNo()
                                        - m_gameStatus.getFrameNo() <= 1))) {
                    nextMoveIdx++;
                }

                // get the next move
                if (nextMoveIdx < moves.size()) {
                    move = moves.get(nextMoveIdx);
                }
            }
            
            // rotate ship towards shot direction of next move
            if (move != null) {
                int rotationSteps = ship.getRequiredSteps(move.getAngle());
                
                if (rotationSteps > 0) {
                    c.pressLeft(true);
                } else if (rotationSteps < 0) {
                    c.pressRight(true);
                }
            }
            
            // press hyper space, if collision is inevitable
            considerHyperSpaceEscape(c);
        }

        return c;
    }
    
    /**
     * Checks the last received game status for inevitable collisions and
     * presses the hyperspace button on the given command object, if necessary.
     * 
     * @param commands the command object, where the hyperspace button 
     *  may be pressed
     */
    private void considerHyperSpaceEscape(Commands commands) {
    
        // create list of objects to consider
        List<GameObject> objectList = m_gameStatus.getObstacleList();
        
        // check all objects on the list for inevitable collisions
        Ship ship = m_gameStatus.getShip();
        for (GameObject o : objectList) {
            Point p = o.relativeProject(1);
            int dist = p.getX() * p.getX() + p.getY() * p.getY();
            if (dist < o.getSquareRadius() + ship.getSquareRadius()) {
                commands.pressHyperspace(true);
                break;
            }   
        }
    }
}