/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.Parser;

import de.hfaber.asteroids.bot.IAsteroidsBot;
import de.hfaber.asteroids.bot.sittingduck.SittingDuck;
import de.hfaber.asteroids.client.MameClient;
import de.hfaber.asteroids.client.ServerBusyException;
import de.hfaber.asteroids.gui.TelemetryViewer;

/**
 * @author Henning Faber
 */
public class Main {

    /**
     * The default server IP, which is used, if no server is given
     * on the command line.   
     */
    private static final String DEFAULT_SERVER_ADDRESS = "127.0.0.1";

    /**
     * The default port where the mame server is listening. 
     */
    private static final int DEFAULT_SERVER_PORT = 1979;

    // options
    private static final String GUI_OPTION = "g";
    private static final String LATENCY_OPTION = "l";
    private static final String HELP_OPTION = "h";

    /**
     * The main method, starts the bot.
     * 
     * @param args program parameters
     */
    public static void main(String[] args) {

        // output program signature
        System.out.println("Sitting Duck Asteroids Bot");
        System.out.println("Copyright (C) 2008 Henning Faber");

        // parse command line
        Options options = createOptions();
        Parser parser = new GnuParser();
        try {
            CommandLine line = parser.parse(options, args);
            
            if (line.hasOption(HELP_OPTION)) {
                
                // show help screen
                HelpFormatter formatter = new HelpFormatter();
                formatter.printHelp("SittingDuck [options] <address[:port]>", 
                        options);
            } else {
                try {
                    // start the bot
                    startBot(line);
                } catch (IOException e) {
                    System.err.println("Network error: " + e.getMessage());
                    System.exit(1);
                } catch (ServerBusyException e) {
                    System.err.println("Server is busy: " + e.getMessage());
                    System.exit(1);
                }
            }
        } catch (ParseException e) {
            System.err.println("Error parsing command line: " + e.getMessage());
            System.exit(1);
        }
    }

    /**
     * Creates an options object with the known command line options.
     * 
     * @return the known command line options
     */
    private static Options createOptions() {
        // create options object
        Options options = new Options();
    
        // add gui option
        Option o = new Option(GUI_OPTION, "gui", false, 
                "start with graphical user interface");
        options.addOption(o);
    
        // add latency option
        o = new Option(LATENCY_OPTION, "latency", true, 
                "set latency correction");
        o.setArgName("frames");
        o.setType(Number.class);
        options.addOption(o);
        
        // add help option
        o = new Option(HELP_OPTION, "help", false, "show this message");
        options.addOption(o);
        
        return options;
    }

    /**
     * Prepares and starts the bot according to the given command
     * line parameters. 
     * 
     * @param line the parsed command line
     * @throws IOException
     * @throws ServerBusyException
     */
    private static void startBot(CommandLine line) throws IOException,
            ServerBusyException {
        // get latency correction
        int latencyCorrection = 0;
        Object value = line.getOptionObject(LATENCY_OPTION);
        if (value instanceof Long) {
            Long longValue = (Long)value;
            latencyCorrection = longValue.intValue();
            System.out.println("Latency correction activated: " 
                    + latencyCorrection + " frame" 
                    + (latencyCorrection != 1 ? "s" : ""));
        }
        
        // get server address
        String server = DEFAULT_SERVER_ADDRESS;
        String[] argList = line.getArgs();
        if (argList.length > 0) {
            server = argList[argList.length - 1];
        }
        InetSocketAddress serverAddr = getServerAddress(server);
        System.out.println("Connecting to " + serverAddr.getHostName() + ":"
                + serverAddr.getPort());
        
        // create the bot and the mame client
        IAsteroidsBot bot = new SittingDuck();
        MameClient client = new MameClient(serverAddr, bot, latencyCorrection);
        
        // create the gui interface, if applicable
        if (line.hasOption(GUI_OPTION)) {
            TelemetryViewer viewer = new TelemetryViewer();
            client.addGameStatusListener(viewer);
        }
        
        // run the client
        client.run();
    }
    
    /**
     * Determines the internet socket address for the given server.
     * 
     * @param server the server name and optionally port, for which the
     *  internet socket address should be determined
     * @return the internet socket address
     * @throws IOException if the given server string cannot be converted
     *  into a valid server address 
     */
    private static InetSocketAddress getServerAddress(String server) throws 
            IOException {
        // split server address into host and port
        String[] addressParts = server.split(":");

        try {
            // get server host and port
            InetAddress serverIp = InetAddress.getByName(addressParts[0]);
            int serverPort = DEFAULT_SERVER_PORT;
            if (addressParts.length > 1) {
                serverPort = Integer.parseInt(addressParts[1]);
            }

            // create channel and connect it
            InetSocketAddress socketAddr = new InetSocketAddress(serverIp, 
                    serverPort);
            
            // return the channel
            return socketAddr;
        } catch (UnknownHostException e) {
            throw new IOException("Invalid server host: " + addressParts[0], e);
        } catch (NumberFormatException e) {
            throw new IOException("Invalid server port: " + addressParts[1], e);
        }
    }
}
