// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   (c) 2008  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.dkn.asteroids;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import de.caff.asteroid.Asteroid;
import de.caff.asteroid.Bullet;
import de.caff.asteroid.FrameInfo;
import de.caff.asteroid.FramePreparer;
import de.caff.asteroid.SpaceShip;
import de.caff.asteroid.Ufo;
import de.caff.util.Pair;

/**
 * <p>
 * Ermittlung der Geschwindigkeiten der verschiedenen Objekte, die Ursprungsimplementierung stammt
 * von Rammi, ich habe sie jedoch an einigen Stellen ergnzt.
 * </p>
 * <p>
 * HINWEIS: Die Domain de.dkn gehrt nicht mir, aber ich benutze das Krzel dkn schon sehr lange und
 * meine, bzw. unsere Domain de.familie-damken ist nicht Java-fhig und damit ist de.dkn aus meiner
 * Sicht genauso geeignet wie jede abweichende Schreibweise von de.familie-damken.
 * </p>
 * <p>
 * (c) 2008, by Uwe Damken ... basierend auf einem Framework von Rammi (rammi@caff.de)
 * </p>
 */
public class DknVelocityPreparer implements FramePreparer {

	/** Um wieviel Punkte darf sich X oder Y verschieben ohne den Origin neu zu setzen? */
	private static final int ORIGIN_RESET_MAX_XY_CHANGE = 80;

	/** Um wieviel Radiant darf der VelocityAngle abweichen ohne den Origin neu zu setzen? */
	private static final double ORIGIN_RESET_UFO_ANGLE_CHANGE = 10 * Math.PI / 180;

	/** Um wieviel Radiant darf der VelocityAngle bei der Zuordnung von Bullets abweichen? */
	private static final double NO_PAIR_BULLET_ANGLE_CHANGE = 170 * Math.PI / 180;

	/** Maximum squared distance we assume an asteroid may move between frames. */
	private static final int MAX_SQUARED_ASTEROID_VELOCITY = 6 * 6;

	/** Maximum squared distance we assume a bullet may move between frames. */
	protected static final int MAX_SQUARED_BULLET_VELOCITY = 16 * 16;

	/** Die nchste freie Id fr Asteroiden und UFOs (Bullets siehe DknDecisionPreparer) */
	private int nextId = 0;

	/**
	 * Helper class to presort asteroids to allow easier compare between frames.
	 */
	protected static class AsteroidSelector {

		/**
		 * Helper class: key for mapping asteroids.
		 */
		private static class Key {
			/** The asteroid's type. */
			private final int type;
			/** The asteroids size. */
			private final int size;

			/**
			 * Constrictor.
			 * 
			 * @param ast
			 *            asteroid
			 */
			private Key(Asteroid ast) {
				this.type = ast.getType();
				this.size = ast.getSize();
			}

			public boolean equals(Object o) {
				if (this == o) {
					return true;
				}
				if (o == null || getClass() != o.getClass()) {
					return false;
				}

				Key key = (Key) o;

				if (size != key.size) {
					return false;
				}
				if (type != key.type) {
					return false;
				}

				return true;
			}

			public int hashCode() {
				int result;
				result = type;
				result = 31 * result + size;
				return result;
			}
		}

		/** Presorted mapping of asteroids. */
		private Map<Key, Collection<Asteroid>> sorted = new HashMap<Key, Collection<Asteroid>>();

		/**
		 * Constructor.
		 * 
		 * @param asteroids
		 *            asteroids to be presorted
		 */
		public AsteroidSelector(Collection<Asteroid> asteroids) {
			for (Asteroid ast : asteroids) {
				Key key = new Key(ast);
				Collection<Asteroid> list = sorted.get(key);
				if (list == null) {
					list = new LinkedList<Asteroid>();
					sorted.put(key, list);
				}
				list.add(ast);
			}
		}

		/**
		 * Get the best matching asteroid assuming the keys collected here are from the previous
		 * frame while the given asteroid is from the current frame.
		 * 
		 * @param asteroid
		 *            current frame astreroid
		 * @return best match or <code>null</code>
		 */
		public Asteroid getBestMatch(Asteroid asteroid, List<Integer> usedIds) {
			Key key = new Key(asteroid);
			Collection<Asteroid> list = sorted.get(key);
			SortedMap<Double, Asteroid> result = new TreeMap<Double, Asteroid>();
			if (list != null) {
				double futureX = asteroid.getX() + asteroid.getVelocityX();
				double futureY = asteroid.getY() + asteroid.getVelocityY();
				for (Asteroid a : list) {
					// Doppelte IDs drfen nicht vorkommen ... darum werden benutzte rausgenommen
					if (a.getIdentity() == null || !usedIds.contains(a.getIdentity())) {
						double dist2 = a.getSquaredDistance(futureX, futureY);
						if (dist2 < MAX_SQUARED_ASTEROID_VELOCITY) {
							result.put(dist2, a);
						}
					}
				}
			}
			// Wenn es mehr oder weniger als einen Best Match gibt, ignoriere ich es lieber, ehe es
			// zu "Richtungswechseln" kommt, die *ich* sonst nicht erkennen kann. Allerdings sollen
			// frischegebackene Zwillinge schon rausgereicht werden, sie sind einander so nah, dass
			// es egal ist, welchen es trifft. Immer noch besser als zwei nicht "beschleunigte", nie
			// angeschossene Asteroiden als Prchen.
			int oldAsteroids = 0;
			for (Asteroid a : result.values()) {
				if (a.getIdentity() != null) {
					oldAsteroids++;
				}
			}
			return result.isEmpty() || oldAsteroids > 1 ? null : result.values().iterator().next();
		}
	}

	/**
	 * Prepare the ufo. Called if there are at least two frames. Overwrite to change the default
	 * behavior
	 */
	protected void prepareUfo(LinkedList<FrameInfo> frameInfos, FrameInfo prevFrame,
			FrameInfo currFrame) {
		Ufo curr = currFrame.getUfo();
		if (curr != null) {
			Ufo prev = prevFrame.getUfo();
			if (prev != null) {
				curr.inheret(prev); // originX/originY bernehmen
				curr.setLifetime(prev.getLifetime() + currFrame.getLifetimeIncrement());
				curr.setVelocityFromDelta(prev); // Nehmen wir mal an, wir htten keinen Origin
				if (curr.getIdentity() == null) { // Dann ist prev neu gewesen!
					curr.setIdentity(nextId++);
					curr.setOriginX(prev.getX());
					curr.setOriginY(prev.getY());
					// Delta- ist gleich Origin-Geschwindigkeit => lassen wie es ist
				} else {
					double ovx = (curr.getX() - curr.getOriginX()) / (double) curr.getLifetime();
					double ovy = (curr.getY() - curr.getOriginY()) / (double) curr.getLifetime();
					double vacc = Math.abs(curr.getVelocityAngle() - Math.PI); // Wechsel von 0 auf
					double vacp = Math.abs(prev.getVelocityAngle() - Math.PI); // 355 Grad ist ok!
					if (Math.abs(curr.getX() - prev.getX()) > ORIGIN_RESET_MAX_XY_CHANGE
							|| Math.abs(curr.getY() - prev.getY()) > ORIGIN_RESET_MAX_XY_CHANGE
							|| Math.abs(vacc - vacp) > ORIGIN_RESET_UFO_ANGLE_CHANGE) {
						// Riesiger Sprung durch Wechsel der Spielfeldseite oder Richtungswechsel,
						// wie UFOs es manchmal tun => Origin neu setzen!
						curr.setOriginX(curr.getX());
						curr.setOriginY(curr.getY());
						curr.setLifetime(0);
						// Delta-Geschwindigkeit lassen, es ist die einzige, die wir haben
					} else {
						curr.setVelocity(ovx, ovy);
					}
				}
			}
		}
	}

	/**
	 * Prepare the space ship. Called if there are at least two frames. Overwrite to change the
	 * default behavior
	 * 
	 * @param frameInfos
	 *            all frame infos
	 * @param prevFrame
	 *            the second to last frame
	 * @param currFrame
	 *            the last frame
	 */
	protected void prepareSpaceShip(LinkedList<FrameInfo> frameInfos, FrameInfo prevFrame,
			FrameInfo currFrame) {
		SpaceShip ship = currFrame.getSpaceShip();
		if (ship != null) {
			ship.setVelocityFromDelta(prevFrame.getSpaceShip());
		}
	}

	/**
	 * Prepare the asteroids. Called if there are at least two frames. Overwrite to change the
	 * default behavior
	 * 
	 * @param frameInfos
	 *            all frame infos
	 * @param prevFrame
	 *            the second to last frame
	 * @param currFrame
	 *            the last frame
	 */
	protected void prepareAsteroids(LinkedList<FrameInfo> frameInfos, FrameInfo prevFrame,
			FrameInfo currFrame) {
		List<Integer> usedIds = new ArrayList<Integer>();
		AsteroidSelector selector = new AsteroidSelector(prevFrame.getAsteroids());
		for (Asteroid curr : currFrame.getAsteroids()) {
			Asteroid prev = selector.getBestMatch(curr, usedIds);
			// log("ast=" + curr + ", candidate=" + prev);
			if (prev != null) {
				curr.inheret(prev); // originX/originY bernehmen
				curr.setLifetime(prev.getLifetime() + currFrame.getLifetimeIncrement());
				curr.setVelocityFromDelta(prev); // Nehmen wir mal an, wir htten keinen Origin
				if (curr.getIdentity() == null) { // Dann ist prev neu gewesen!
					curr.setIdentity(nextId++);
					if (Math.abs(curr.getX() - prev.getX()) > ORIGIN_RESET_MAX_XY_CHANGE
							|| Math.abs(curr.getY() - prev.getY()) > ORIGIN_RESET_MAX_XY_CHANGE) {
						curr.setOriginX(curr.getX());
						curr.setOriginY(curr.getY());
						curr.setLifetime(0);
					} else {
						curr.setOriginX(prev.getX());
						curr.setOriginY(prev.getY());
						// Delta-Geschwindigkeit lassen, es ist die einzige, die wir haben
					}
				} else {
					double ovx = (curr.getX() - curr.getOriginX()) / (double) curr.getLifetime();
					double ovy = (curr.getY() - curr.getOriginY()) / (double) curr.getLifetime();
					curr.setVelocity(ovx, ovy);
					if (Math.abs(curr.getX() - prev.getX()) > ORIGIN_RESET_MAX_XY_CHANGE
							|| Math.abs(curr.getY() - prev.getY()) > ORIGIN_RESET_MAX_XY_CHANGE) {
						// Riesiger Sprung (Bildschirmseitenwechsel) => Origin neu setzen!
						curr.setOriginX(curr.getX());
						curr.setOriginY(curr.getY());
						curr.setLifetime(0);
						curr.setVelocityFromDelta(prev);
					}
				}
				usedIds.add(curr.getIdentity());
			}
		}
	}

	/**
	 * Prepare the bullets. Called if there are at least two frames. Overwrite to change the default
	 * behavior
	 * 
	 * @param frameInfos
	 *            all frame infos
	 * @param prevFrame
	 *            the second to last frame
	 * @param currFrame
	 *            the last frame
	 */
	protected void prepareBullets(LinkedList<FrameInfo> frameInfos, FrameInfo prevFrame,
			FrameInfo currFrame) {
		for (Bullet oldBullet : prevFrame.getBullets()) {
			double futureX = oldBullet.getX() + oldBullet.getVelocityX();
			double futureY = oldBullet.getY() + oldBullet.getVelocityY();

			SortedMap<Double, Pair<Bullet>> result = new TreeMap<Double, Pair<Bullet>>();
			for (Bullet bullet : currFrame.getBullets()) {
				double dist2 = bullet.getSquaredDistance(futureX, futureY);
				if (dist2 < MAX_SQUARED_BULLET_VELOCITY) {
					// Bullets als Paare aufnehmen, wenn sie nah beieinander sind, aber nur wenn der
					// alte und neue Geschwindigkeitsvektor nicht gleichzeitig eine gegenstzliche
					// X- und Y-Richtung haben. Nur gegenstzlich X- oder Y-Richtung ist z.B. mglich,
					// wenn die Geschwindigkeit mal ber, mal unter die 0-Grad-Linie zeigt.
					bullet.setVelocityFromDelta(oldBullet); // testweise mal berechnen ...
					boolean oppositeVelocityX = Math.signum(Math.round(bullet.getVelocityX())) == -Math
							.signum(Math.round(oldBullet.getVelocityX()));
					boolean oppositeVelocityY = Math.signum(Math.round(bullet.getVelocityY())) == -Math
							.signum(Math.round(oldBullet.getVelocityY()));
					bullet.setVelocity(0.0, 0.0); // ... dann aber wieder zurcksetzen
					if (!oppositeVelocityX || !oppositeVelocityY) {
						result.put(dist2, new Pair<Bullet>(oldBullet, bullet));
					}
				}
			}
			LinkedList<Pair<Bullet>> pairs = new LinkedList<Pair<Bullet>>(result.values());
			while (!pairs.isEmpty()) {
				Pair<Bullet> pair = pairs.remove(0);
				pair.second.setVelocityFromDelta(pair.first);
				pair.second
						.setLifetime(pair.first.getLifetime() + currFrame.getLifetimeIncrement());
				pair.second.setIdentity(pair.first.getIdentity()); // bernehmen, wenn vorhanden
				for (ListIterator<Pair<Bullet>> it = pairs.listIterator(); it.hasNext();) {
					Pair<Bullet> p = it.next();
					if (p.first.equals(pair.first) || p.second.equals(pair.second)) {
						it.remove();
					}
				}
			}
		}
	}

	/**
	 * Prepare frame info if there is only a single frame available.
	 * 
	 * @param frame
	 *            frame info
	 */
	protected void prepareSingleFrame(FrameInfo frame) {
		// set bullet lifetimes to 1 to indicate lifetimes are calculated
		for (Bullet bullet : frame.getBullets()) {
			bullet.setLifetime(1);
		}
	}

	/**
	 * Prepare the frame(s).
	 * 
	 * @param frameInfos
	 *            the collected frame infos
	 */
	public void prepareFrames(LinkedList<FrameInfo> frameInfos) {
		if (frameInfos.size() >= 2) {
			FrameInfo currFrame = frameInfos.getLast();
			FrameInfo prevFrame = frameInfos.get(frameInfos.size() - 2);
			int idMinusLastId = currFrame.getId() - prevFrame.getId();
			if (idMinusLastId == 0) { // Analyse einer gz-Datei (enthlt keine ID-Bytes)
				currFrame.setLifetimeIncrement(1); // Defaults annehmen, auch wenn er falsch ist
			} else if (idMinusLastId < 0) {
				currFrame.setLifetimeIncrement(idMinusLastId + 256); // Byte-Diff normalisieren
			} else {
				currFrame.setLifetimeIncrement(idMinusLastId);
			}

			prepareUfo(frameInfos, prevFrame, currFrame);

			prepareSpaceShip(frameInfos, prevFrame, currFrame);

			prepareAsteroids(frameInfos, prevFrame, currFrame);

			prepareBullets(frameInfos, prevFrame, currFrame);

		} else if (!frameInfos.isEmpty()) {
			prepareSingleFrame(frameInfos.getLast());
		}
	}
}
