package de.dkn.asteroids;

import java.awt.Point;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import de.caff.asteroid.FrameInfo;
import de.caff.asteroid.FramePreparer;
import de.caff.asteroid.SpaceShip;
import de.caff.util.Tools;

/**
 * <p>
 * Diese Klasse ermittelt tatschliche zu erwartende Schussrichtung auf Basis der letzten Bewegungen
 * und einer Tabelle von Winkeln und Schussrichtungen, die in Rammis Framework bereits enthalten
 * war.
 * </p>
 * <p>
 * HINWEIS: Die Domain de.dkn gehrt nicht mir, aber ich benutze das Krzel dkn schon sehr lange und
 * meine, bzw. unsere Domain de.familie-damken ist nicht Java-fhig und damit ist de.dkn aus meiner
 * Sicht genauso geeignet wie jede abweichende Schreibweise von de.familie-damken.
 * </p>
 * <p>
 * (c) 2008, by Uwe Damken ... basierend auf einem Framework von Rammi (rammi@caff.de)
 * </p>
 */
public class DknShootingDirPreparer implements FramePreparer {

	/** Anzahl der fr die Synchronisation FrameInfos => eine ShipDirection weniger wird genutzt */
	private static final int MAX_EXTRACTS_FOR_SYNCH = 6;

	/** Die fr die Synchronisation verwendeten ShipDirections => bei Turns + 1 Extract */
	private int usedShipDirs = 0;

	/**
	 * Prepare the frame(s).
	 */
	@SuppressWarnings("unchecked")
	public void prepareFrames(LinkedList<FrameInfo> frameInfos) {
		// Bis zu MAX_FRAMES_FOR_SYNCH FrameInfos mit Schiff fr die Synchronisation sammeln
		// int possibleSynchQuality = 0;
		List<FrameInfoExtract> extracts = new ArrayList<FrameInfoExtract>();
		for (int i = frameInfos.size(); i > frameInfos.size() - MAX_EXTRACTS_FOR_SYNCH && i > 0; i--) {
			FrameInfoExtract extract = new FrameInfoExtract(frameInfos.get(i - 1));
			if (extract.isWithShip()) {
				extracts.add(extract);
				// possibleSynchQuality += extract.getTurn();
				// log("extract " + i + "=" + extract);
			} else {
				break;
			}
		}
		// Die neue ShootingDirection ist die vorherige ShootingDirection, allerding unter
		// Bercksichtigung des Turns im vorvorherigen Extract
		if (extracts.size() > 2) {
			// log("old shooting direction: " + extracts.get(0).getShootingDirection());
			extracts.get(0).setShootingDirection(
					normalize(extracts.get(1).getShootingDirection() + extracts.get(2).getTurn()));
			// log("calculated shooting direction: " + extracts.get(0).getShootingDirection());
			frameInfos.getLast().setProbableShootingDir(
					normalize(extracts.get(0).getShootingDirection() + extracts.get(1).getTurn()));
		}
		// Wenn es ein Extract gibt, ist das aktuelle vorne und es ist auch ein Schiff drin!
		if (extracts.size() > 0) {
			FrameInfoExtract currExtract = extracts.get(0);
			// Die Synchronisation wird immer vollstndig berprft => Tabelle mit mglichen
			// Positionen je Anzahl der verwendeten ShipDirections aufbauen
			List<Integer>[] allMatches = (List<Integer>[]) new ArrayList[MAX_EXTRACTS_FOR_SYNCH - 1];
			for (int i = 0; i < allMatches.length; i++) {
				allMatches[i] = new ArrayList<Integer>();
			}
			// An jedem Eintrag der Tabelle alle gefundenen Extracts berpfen
			for (int i = 0; i < FrameInfo.SHOOTING_DIRECTIONS.length; i++) {
				int entry = i; // Nchstes Extract an diesem Entry berprfen
				// Alle Extracts bis auf den letzten prfen, der dient nur der
				// Richtungsermittlung
				for (int j = 0; j < extracts.size(); j++) {
					// Fr alle Extracts auer dem letzten von mehreren Extracts muss die
					// ShipDirection mit der aus der Tabelle bereinstimmen (beim letzten ist
					// unklar, in welche Richtung der Eintrag in der Tabelle zu suchen wre)
					if (j == extracts.size() - 1 && extracts.size() > 1) {
						// Hier ist nichts zu tun
					} else if (FrameInfo.SHOOTING_DIRECTIONS[entry].getShipDirection().equals(
							extracts.get(j).getShipDirection())) {
						allMatches[j].add(i); // j+1 Extracts ergeben i als gltige Position
						if (j < extracts.size() - 2) {
							// Nchstes Extract an dem Entry berprfen, der sich aus der
							// Drehrichtung des vorvorherigen FrameInfos ergibt; links => +1 =>
							// jetzt -1 in der Tabelle
							entry = normalize(entry - extracts.get(j + 2).getTurn());
						}
					} else {
						break; // Nur innere Schleife beenden, Mehrdeutigkeiten suchen
					}
				}
			}
			// Die Liste auswhlen, die mit den meisten Extracts zustande gekommen ist. Mindestens
			// fr 1 Extract muss es einen Eintrag geben, weil der aktuelle Extract ein Schiff hat.
			int matchesPos = -1;
			for (int i = allMatches.length - 1; i >= 0 && matchesPos < 0; i--) {
				if (allMatches[i].size() > 0) {
					matchesPos = i;
				}
			}
			// Gewhlte Liste der mglichen Positionen zur Synchronisations-/Besttigung nutzen
			if (matchesPos < 0) {
				frameInfos.getLast().getDecisionData().synchInfo = (new StringBuffer()).append(
						"Could not synchronize: ").append(extracts.size()).append(" extracts=")
						.append(extracts).toString();
			} else {
				usedShipDirs = matchesPos + 1;
				List<Integer> matches = allMatches[matchesPos];
				if (matches.contains(currExtract.getShootingDirection())) {
					frameInfos.getLast().getDecisionData().synchInfo = (new StringBuffer()).append(
							"Confirmed synchronization: ").append(
							currExtract.getShootingDirection()).append(", usedShipDirs=").append(
							usedShipDirs).append(", with ").append(extracts.size()).append(
							" extracts=").append(extracts).toString();
				} else {
					int synchDirection = matches.get(0);
					extracts.get(0).setShootingDirection(synchDirection);
					frameInfos.getLast().getDecisionData().synchInfo = (new StringBuffer()).append(
							"Resynchronized to: ").append(synchDirection).append(", usedShipDirs=")
							.append(usedShipDirs).append(", with ").append(extracts.size()).append(
									" extracts=").append(extracts).toString();
				}
			}
			// Jetzt die errechnete, die neu synchronisierte oder die alte ShootingDirection auch
			// wieder ins FrameInfo schreiben
			frameInfos.getLast().setBothShootingDirectionsLowLevel(
					unsignedToByte(currExtract.getShootingDirection()));
		}
	}

	/**
	 * Normalisiert den Directions-Tabellen-Index
	 */
	public static int normalize(int index) {
		return (index + 256) % 256;
	}

	private Byte unsignedToByte(int index) {
		return new Byte(Integer.toString((index < 128) ? index : index - 256));
	}

	/**
	 * Klasse zum Halten der fr den DknShootingPreparer notwendigen Werte aus dem FrameInfo.
	 */
	public static class FrameInfoExtract {

		private boolean withShip;

		private Point shipDirection;

		private int shootingDirection;

		private int turn;

		public FrameInfoExtract(FrameInfo frame) {
			SpaceShip ship = frame.getSpaceShip();
			withShip = ship != null;
			if (withShip) {
				shipDirection = ship.getDirection();
				shootingDirection = Tools.byteToUnsigned(frame.getShootingDirectionLowLevel());
				int nextShootingDirection = Tools.byteToUnsigned(frame
						.getNextShootingDirectionLowLevel());
				if (nextShootingDirection == 0 && shootingDirection == 255) {
					turn = -1;
				} else {
					turn = nextShootingDirection - shootingDirection;
				}
			}
		}

		public boolean isWithShip() {
			return withShip;
		}

		public Point getShipDirection() {
			return shipDirection;
		}

		public int getTurn() {
			return turn;
		}

		public String toString() {
			StringBuffer sb = new StringBuffer();
			sb.append("FrameInfoExtract[").append("withShip=").append(withShip).append(
					",shipDirection=").append(shipDirection).append(",shootingDirection=").append(
					shootingDirection).append(",turn=").append(turn).append("]");
			return sb.toString();
		}

		public int getShootingDirection() {
			return shootingDirection;
		}

		public void setShootingDirection(int shootingDirection) {
			this.shootingDirection = shootingDirection;
		}

	}

}
