// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   (c) 2008  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.asteroid;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;
import java.awt.geom.Point2D;
import java.util.Collection;

import de.caff.util.Tools;

/**
 * A moving game object has a position and a velocity and it can be drawn.
 * 
 * This is part of a solution for a competition <a
 * href="http://www.heise.de/ct/creativ/08/02/details/">by the German computer magazine c't</a>
 */
public abstract class MovingGameObject extends GameObject {
	/** Score of objects which can't be hit. */
	public static final int NO_SCORE = 0;
	/** TEST: calculate maximum velocities? */
	protected static final boolean WATCH_VELOCITIES = false;

	// if you are interested which y coordinates are used change the "if (false)" line in contructor
	// to "if (true)"
	private static int minY = Integer.MAX_VALUE;
	private static int maxY = Integer.MIN_VALUE;

	/** Size of head of velocity arrow. */
	private static final int ARROW_HEAD_SIZE = 8;

	/** The arrow head for velocity drawings. */
	private static final GeneralPath ARROW_HEAD = new GeneralPath();
	static {
		ARROW_HEAD.moveTo(0, 0);
		ARROW_HEAD.lineTo(-ARROW_HEAD_SIZE, ARROW_HEAD_SIZE);
		ARROW_HEAD.lineTo(-ARROW_HEAD_SIZE, -ARROW_HEAD_SIZE);
		ARROW_HEAD.closePath();
	}

	/** The x coordinate of the velocity vector. */
	private double vx;
	/** The y coordinate of the velocity vector. */
	private double vy;
	/** Lifetime of this object (set externally). */
	private int lifetime;
	/** The identity of this object (set externally). */
	private Integer identity;

	private int originX;

	private int originY;

	private int squaredDistanceToShip;

	private int hitShipInFrames;

	private int lastHitShipInFrames;

	private int dangerousDetected;

	private boolean dangerous;

	/**
	 * Constructor.
	 * 
	 * @param x
	 *            x coordinate
	 * @param y
	 *            y coordinate
	 */
	protected MovingGameObject(int x, int y) {
		super(x, y);
		if (false) {
			if (y < minY) {
				minY = y;
				System.out.println("new min/max: " + minY + "," + maxY);
			}
			if (y > maxY) {
				maxY = y;
				System.out.println("new min/max: " + minY + "," + maxY);
			}
		}
	}

	/**
	 * Get the center of the object.
	 * 
	 * @return center point
	 */
	public Point getCenter() {
		return new Point(x, y);
	}

	/**
	 * Get the size of the object.
	 * 
	 * The size returned by this method is half the length of a square which contains the object, so
	 * the object's bounding box is between (x - size, y - size) and (x + size, y + size).
	 * 
	 * @return object size
	 */
	public abstract int getSize();

	/**
	 * Draw the object.
	 * 
	 * @param g
	 *            graphics context
	 */
	public void draw(Graphics2D g) {
		g.setColor(Color.white);
		int size = getSize();
		g.drawOval(x - size, y - size, 2 * size, 2 * size);

		drawVelocityVector(g, Color.red);
	}

	/**
	 * Get the squared size of this object.
	 * 
	 * @return squared size
	 */
	public int getSquaredSize() {
		int size = getSize();
		return size * size;
	}

	/**
	 * Set the velocity.
	 * 
	 * The velocity is the step between frames.
	 * 
	 * It is not calculated internally but has to be set from outside. For an example see
	 * {@link de.caff.asteroid.SimpleVelocityPreparer#prepareFrames(java.util.LinkedList)}.
	 * 
	 * @param x
	 *            x coordinate of velocity
	 * @param y
	 *            y coordinate of velocity
	 */
	public void setVelocity(double x, double y) {
		vx = x;
		vy = y;
	}

	/**
	 * Set the velocity.
	 * 
	 * The velocity is the step between frames.
	 * 
	 * @param v
	 *            velocity vector
	 */
	public void setVelocity(Point2D v) {
		setVelocity(v.getX(), v.getY());
	}

	/**
	 * Set the velocity assuming that the given object is at the same place as this object in the
	 * last frame.
	 * 
	 * @param obj
	 *            comparision object (<code>null</code> allowed, but then nothing happens)
	 */
	public void setVelocityFromDelta(MovingGameObject obj) {
		if (obj != null) {
			setVelocity(obj.getDelta(this));
		}
	}

	/**
	 * Get the velocity in x.
	 * 
	 * @return x component of velocity vector
	 */
	public double getVelocityX() {
		return vx;
	}

	/**
	 * Get the velocity in y.
	 * 
	 * @return y component of velocity vector
	 */
	public double getVelocityY() {
		return vy;
	}

	/**
	 * Get the velocity vector.
	 * 
	 * @return velocity vector (movement between frames)
	 */
	public Point2D getVelocity() {
		return new Point2D.Double(vx, vy);
	}

	/**
	 * Get the velocity angle.
	 * 
	 * The angle is measured counterclockwise, with <code>0</code> pointing to the right
	 * 
	 * @return velocity in radians, or -Math.PI if ship has no velocity
	 */
	public double getVelocityAngle() {
		return vx != 0 || vy != 0 ? Math.atan2(vy, vx) : -Math.PI;
	}

	/**
	 * Has this object a velocity.
	 * 
	 * @return velocity
	 */
	public boolean hasKnownVelocity() {
		return vx != 0 || vy != 0;
	}

	/**
	 * Draw a vector displaying the current velocity.
	 * 
	 * @param g
	 *            graphics context
	 * @param color
	 *            color to use
	 */
	protected void drawVelocityVector(Graphics2D g, Color color) {
		if (vx != 0 || vy != 0) {
			int scale = 16;
			g.setColor(color);
			g.drawLine(x, y, x + (int) (scale * vx), y + (int) (scale * vy));
			double angle = Math.atan2(vy, vx);
			AffineTransform at = AffineTransform.getTranslateInstance(x + scale * vx, y + scale
					* vy);
			at.concatenate(AffineTransform.getRotateInstance(angle));
			g.fill(at.createTransformedShape(ARROW_HEAD));
		}
	}

	/**
	 * Get the bounding box of this rectangle.
	 * 
	 * @return the bounding box
	 */
	public Rectangle getBounds() {
		Point center = getCenter();
		int size = getSize();
		return new Rectangle(center.x - size, center.y - size, 2 * size, 2 * size);
	}

	/**
	 * Get the position in the next frame assuming that the velocity is constant.
	 * 
	 * @return position in next frame
	 */
	public Point getNextLocation() {
		return new Point(getX() + (int) getVelocityX(), getY() + (int) getVelocityY());
	}

	/**
	 * Get the position in a coming (or gone) frame assuming that the velocity is constant.
	 * 
	 * @param nrFrames
	 *            number of frames to skip
	 * @return the position
	 */
	public Point getPredictedLocation(int nrFrames) {
		return new Point(getX() + (int) (nrFrames * getVelocityX()), getY()
				+ (int) (nrFrames * getVelocityY()));
	}

	/**
	 * Get the position in a coming (or gone) frame assuming that the velocity is constant.
	 * 
	 * @param nrFrames
	 *            number of frames to skip
	 * @return the position
	 */
	public Point2D getPredictedLocation(double nrFrames) {
		return new Point2D.Double(getX() + (nrFrames * getVelocityX()), getY()
				+ (nrFrames * getVelocityY()));
	}

	/**
	 * Get the properties of this object.
	 * 
	 * @return collection of properties
	 */
	@SuppressWarnings("unchecked")
	@Override
	public Collection<Property> getProperties() {
		Collection<Property> props = super.getProperties();
		props.add(new Property<Integer>("ID", getIdentity()));
		props.add(new Property<Integer>("Size", getSize()));
		props.add(new Property<Rectangle>("Bounds", getBounds()));
		props.add(new Property<Point2D>("Velocity", getVelocity()));
		props.add(new Property<Double>("Velocity Angle ()", 180 * getVelocityAngle() / Math.PI));
		props.add(new Property<Point2D>("Origin", new Point(originX, originY)));
		props.add(new Property<Integer>("Lifetime", getLifetime()));
		props.add(new Property<Integer>("Hit ship in frames", hitShipInFrames));
		props.add(new Property<Integer>("dangerousDetected", dangerousDetected));
		return props;
	}

	/**
	 * Get the score which is added if this object is hit.
	 * 
	 * @return score or {@link #NO_SCORE}
	 */
	public int getScore() {
		return NO_SCORE;
	}

	/**
	 * Get the lifetime of this buller.
	 * 
	 * Because it is not always easy to connect two bullets between frames the returned lifetime is
	 * not 100% correct. Also it is not calculated internally but has to be set from outside. For an
	 * example see {@link SimpleVelocityPreparer#prepareFrames(java.util.LinkedList)}.
	 * 
	 * @return lifetime (number of frames this bullet was displayed)
	 */
	public int getLifetime() {
		return lifetime;
	}

	/**
	 * Set the lifetime.
	 * 
	 * @param lifetime
	 *            life time (number of frames 'this' bullet exists)
	 */
	public void setLifetime(int lifetime) {
		this.lifetime = lifetime;
	}

	/**
	 * Get the identity of this object. The identity has to be set from the outside.
	 * 
	 * @return identity or <code>null</code> if not set
	 */
	public Integer getIdentity() {
		return identity;
	}

	/**
	 * Set the identity of this object.
	 * 
	 * @param identity
	 *            identity
	 */
	public void setIdentity(Integer identity) {
		this.identity = identity;
	}

	/**
	 * Inheret properties from another game object. This basic implementation sets the identity to
	 * the same value as the one of the other object.
	 */
	public void inheret(MovingGameObject obj) {
		setIdentity(obj.getIdentity());
		originX = obj.getOriginX();
		originY = obj.getOriginY();
		dangerous = obj.isDangerous();
		dangerousDetected = obj.getDangerousDetected();
	}

	/**
	 * Will the two game objects hit in the given number of frames?
	 */
	public boolean isHitting(MovingGameObject other, int frames) {
		Point delta = getTorusDelta(getPredictedLocation(frames), other
				.getPredictedLocation(frames));
		return delta.x * delta.x + delta.y * delta.y < Tools.square(getSize() + other.getSize());
	}

	/**
	 * Liefert die Anzahl der Frames, in denen sich dieses Objekt mit dem bergebenen Objekt treffen
	 * wird. Geprft wird von fromFrames bis toFrames. Wenn sich die Objekte in der Zeit nicht
	 * treffen, wird 0 geliefert. Es muss fromFrames > 0 und toFrames >= fromFrames sein.
	 */
	public int hitInFrames(MovingGameObject other, int fromFrames, int toFrames) {
		for (int f = fromFrames; f < toFrames + 1; f++) {
			if (isHitting(other, f)) {
				return f;
			}
		}
		return 0;
	}

	/**
	 * Liefert die Anzahl der Frames, in denen sich dieses Objekt mit dem bergebenen Objekt treffen
	 * wird. Geprft wird von fromFrames bis toFrames.
	 */
	public int hitInFrames(MovingGameObject other, int toFrames) {
		return hitInFrames(other, 1, toFrames);
	}

	/**
	 * Berechnet, wann das Objekt auf das Schiff treffen wird (bis maximal in toFrames) und die
	 * quadrierte Entfernung zum Schiff.
	 */
	public void setShipRelatedValues(SpaceShip ship, int toFrames) {
		squaredDistanceToShip = getSquaredDistance(ship);
		hitShipInFrames = hitInFrames(ship, toFrames);
		// So sollte die Gefhrlichkeit gegen Pendeln und gegen Sturheit gewappnet sein
		if (hitShipInFrames > 0) {
			lastHitShipInFrames = hitShipInFrames;
			dangerousDetected = 10;
			dangerous = true;
		} else if (dangerous) {
			if (--dangerousDetected <= 0) {
				lastHitShipInFrames = hitShipInFrames;
				dangerous = false;
			}
		}
	}

	/**
	 * Gefhrlichkeit von auen setzen.
	 */
	public void setDangerous(boolean dangerous) {
		this.dangerous = dangerous;
		this.dangerousDetected = 0;
	}

	/**
	 * Ab hier nur noch generierte Getter und Setter.
	 */
	public int getOriginX() {
		return originX;
	}

	public void setOriginX(int originX) {
		this.originX = originX;
	}

	public int getOriginY() {
		return originY;
	}

	public void setOriginY(int originY) {
		this.originY = originY;
	}

	public Point getOrigin() {
		return new Point(originX, originY);
	}

	public int getSquaredDistanceToShip() {
		return squaredDistanceToShip;
	}

	public boolean isDangerous() {
		return dangerous;
	}

	public int getHitShipInFrames() {
		return hitShipInFrames;
	}

	public int getDangerousDetected() {
		return dangerousDetected;
	}

	public int getLastHitShipInFrames() {
		return lastHitShipInFrames;
	}

}
