// ============================================================================
// File:               $File$
//
// Project:            Asteroids game solver.
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   (c) 2008  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.asteroid;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;
import java.security.Identity;
import java.util.Collection;

/**
 * Asteroid object in automated player of Asteroids game.
 * 
 * This class is part of a solution for a <a
 * href="http://www.heise.de/ct/creativ/08/02/details/">competition by the German computer magazine
 * c't</a>.
 */
public class Asteroid extends MovingGameObject {
	/** Color of asteroid. */
	// 0-Alive, nicht gefhrlich
	// 1-Death, nicht gefhrlich
	// 2-Alive, gefhrlich
	// 3-Death, gefhrlich
	private static final Color[] ASTEROID_COLOR = new Color[] { new Color(0x80, 0x80, 0x80),
			new Color(0x30, 0x30, 0x30), new Color(0xC0, 0x00, 0x00), new Color(0x60, 0x00, 0x60) };
	/** The scores by scale. */
	private static final int[] SCORES = new int[] { 0, 0, 100, 0, 50, 0, 0, 0, 20 };
	/** Scaling for shapes when drawing big asteroid. */
	private static int NORMAL_SCALE = 8;
	/** Scaling of this asteroid. */
	private int scale;
	/** Type of this asteroid. There are 4 different types numbered from 0 to 3. */
	private int type;
	/** Index of the asteroid in datagram. */
	private final int index;
	/** Nur fr die Anzeige die Information, in welcher Farbe er dargestellt werden soll */
	private int colorCode;

	private static final GeneralPath PATH0 = new GeneralPath();
	private static final GeneralPath PATH1 = new GeneralPath();
	private static final GeneralPath PATH2 = new GeneralPath();
	private static final GeneralPath PATH3 = new GeneralPath();
	static {
		PATH0.moveTo(1, 0);
		PATH0.lineTo(4, 1);
		PATH0.lineTo(4, 2);
		PATH0.lineTo(1, 4);
		PATH0.lineTo(-2, 4);
		PATH0.lineTo(-1, 3);
		PATH0.lineTo(-4, 3);
		PATH0.lineTo(-4, -1);
		PATH0.lineTo(-2, -4);
		PATH0.lineTo(1, -3);
		PATH0.lineTo(2, -4);
		PATH0.lineTo(4, -2);
		PATH0.closePath();

		PATH1.moveTo(3, 0);
		PATH1.lineTo(4, 2);
		PATH1.lineTo(2, 4);
		PATH1.lineTo(0, 2);
		PATH1.lineTo(-2, 4);
		PATH1.lineTo(-4, 2);
		PATH1.lineTo(-4, -2);
		PATH1.lineTo(-2, -4);
		PATH1.lineTo(1, -4);
		PATH1.lineTo(4, -2);
		PATH1.closePath();

		PATH2.moveTo(4, 1);
		PATH2.lineTo(2, 4);
		PATH2.lineTo(-1, 4);
		PATH2.lineTo(-4, 1);
		PATH2.lineTo(-2, 0);
		PATH2.lineTo(-4, -1);
		PATH2.lineTo(-2, -4);
		PATH2.lineTo(0, -1);
		PATH2.lineTo(0, -4);
		PATH2.lineTo(2, -4);
		PATH2.lineTo(4, -1);
		PATH2.closePath();

		PATH3.moveTo(2, 1);
		PATH3.lineTo(4, 2);
		PATH3.lineTo(2, 4);
		PATH3.lineTo(0, 3);
		PATH3.lineTo(-2, 4);
		PATH3.lineTo(-4, 2);
		PATH3.lineTo(-3, 0);
		PATH3.lineTo(-4, -2);
		PATH3.lineTo(-2, -4);
		PATH3.lineTo(-1, -3);
		PATH3.lineTo(2, -4);
		PATH3.lineTo(4, -1);
		PATH3.closePath();
	}
	/** The asteroid shapes sorted by type. */
	private static final Shape[] SHAPES = new Shape[] { PATH1, PATH3, PATH2, PATH0, };

	/**
	 * Constructor.
	 * 
	 * @param index
	 *            index of this asteroid in the incoming datagram
	 * @param x
	 *            position (x coordinate)
	 * @param y
	 *            position (y coordinate)
	 * @param size
	 *            size (as given by game, so 0 is big and 14 is small)
	 * @param type
	 *            type of asteroid (0, 1, 2, or 3)
	 */
	public Asteroid(int index, int x, int y, int size, int type) {
		super(x, y);
		switch (size) {
		case 0:
			this.scale = NORMAL_SCALE;
			break;

		case 15:
			this.scale = NORMAL_SCALE / 2;
			break;

		case 14:
			this.scale = NORMAL_SCALE / 4;
			break;
		}
		assert (type >= 0 && type < SHAPES.length);
		this.type = type;
		this.index = index;
	}

	/**
	 * Get the size of the object.
	 * 
	 * The size returned by this method is half the length of a square which contains the object, so
	 * the object's bounding box is between (x - size, y - size) and (x + size, y + size).
	 * 
	 * @return object size
	 */
	public int getSize() {
		return 4 * scale;
	}

	/**
	 * Get the type of this asteroid.
	 * 
	 * @return anumber between 0 and 3
	 */
	public int getType() {
		return type;
	}

	/**
	 * Get the index of this asteroid in the datagram.
	 * 
	 * @return index in datagram
	 */
	public int getIndex() {
		return index;
	}

	/**
	 * Draw the object.
	 * 
	 * @param g
	 *            graphics context
	 */
	@Override
	public void draw(Graphics2D g) {
		g.setColor(ASTEROID_COLOR[colorCode]);
		AffineTransform trafo = new AffineTransform(scale, 0, 0, scale, x, y);
		g.draw(trafo.createTransformedShape(SHAPES[type]));

		drawVelocityVector(g, Color.red);
	}

	/**
	 * Returns a string representation of the object.
	 * 
	 * @return a string representation of the object.
	 */
	@Override
	public String toString() {
		return String.format("Asteroid(%s)@(%d,%d)*%d?%d", (getIdentity() == null) ? ""
				: getIdentity().toString(), x, y, 4 * scale, type);
	}

	/**
	 * Get the properties of this object.
	 * 
	 * @return collection of properties
	 */
	@Override
	public Collection<Property> getProperties() {
		Collection<Property> props = super.getProperties();
		props.add(new Property<Integer>("Asteroid Type", getType()));
		props.add(new Property<Integer>("Scale Factor", scale));
		return props;
	}

	/**
	 * Get the type of game object.
	 * 
	 * @return game object type
	 */
	public String getObjectType() {
		return "Asteroid";
	}

	/**
	 * Get the score which is added if this object is hit.
	 * 
	 * @return score or {@link #NO_SCORE}
	 */
	@Override
	public int getScore() {
		return SCORES[scale];
	}

	public void setColor(int color) {
		this.colorCode = color;
	}

}
