package de.wens02;

import java.awt.Point;

abstract public class MovingObject extends VectorObject
{
	protected static int MOVINGOBJECT_ID = 1;

	protected int			id;

	/** Bewegungsinkremente (falls mindestens zweimal gesehen) */
	protected double	dx,dy;

	/** Bei x0,y0 wurde das Objekt das erste Mal erfasst */
	protected int			x0,y0;

	/** Eine virtuelle, hochgerechnete Koordinate ohne Bercksichtigung des TORUS */
	protected int			x_virtual,y_virtual;

	/** Gre des Objekts in Pixel bei gedachter Modellierung als Kreis */
	protected int			pixelSize										= 0;

	/** Quadrat von pixelSize, zur Vereinfachung von Kollisionsprfungen, Vermeidung von sqrt */
	protected int			squaredPixelSize						= 0;

	/** So viele Frames existiert das Objekt schon */
	protected int			frames;

	/** Projekt auf x/y innerhalb projected_frames, zur Kontrolle der Vorhersagegenauigkeit */
	protected int			projected_frames;
	protected int			projected_x;
	protected int			projected_y;
	
	/** Objekt ist auf Kollisionskurs mit dem Schiff */
	protected boolean	collision				= false;
	protected int			collisionFrame	= 0;
	
	/** Abstand zum Schiff */
	protected double	distance	= 0.0;

	protected Point		tmpPoint	= new Point(0,0);
	
	public MovingObject( int x, int y, int s, int type )
	{
		super( x, y, s, type );

		id = MOVINGOBJECT_ID++;

		x0 = x_virtual = x;
		y0 = y_virtual = y;

		frames = 1;
	}
	
	/** Geschwindigkeit dieses Objekts */
	public double getSpeed()
	{
		return Math.sqrt( dx*dx + dy*dy );
	}
	
	/** Winkel der Bewegungsrichtung aus -180..180 Grad */
	public double getAlpha()
	{
		return Math.toDegrees( Math.atan2( dx, dy ) );
	}

	/** Position in #frameHorizon Frames in p zurckgeben */
	public void getPosition( Point p, int frameHorizon )
	{
		p.x = x + (int)( frameHorizon * dx );
		p.y = y + (int)( frameHorizon * dy );

		normalize( p );
	}

	/** Punkt auerhalb des Spielfeldes mit Torusumbrchen wieder in das Spielfeld normalisieren */
	public static void normalize( Point p )
	{
		while ( p.x < 0 ) p.x += AsteroidApp.EXTENT_X;
		while ( p.y < 0 ) p.y += AsteroidApp.EXTENT_Y;

		while ( p.x > AsteroidApp.MAX_X ) p.x -= AsteroidApp.EXTENT_X;
		while ( p.y > AsteroidApp.MAX_Y ) p.y -= AsteroidApp.EXTENT_Y;
	}

	/** Normale Distanz zum Punkt p */
	public int getSquaredNormalDistance( Point p, int frameHorizon )
	{
		getPosition( tmpPoint, frameHorizon );

		final int dx = p.x - tmpPoint.x;
		final int dy = p.y - tmpPoint.y;

		return dx * dx + dy * dy;
	}
	
	public static int NormalDistSq( Point p1, Point p2 )
	{
		final int dx = p1.x - p2.x;
		final int dy = p1.y - p2.y;
		
		return dx * dx + dy * dy;
	}
	
	public static int NormalDistSq( Point p, int x, int y )
	{
		final int dx = p.x - x;
		final int dy = p.y - y;

		return dx * dx + dy * dy;
	}
	
	public int getSquaredNormalDistance( Shot shot, int frameHorizon )
	{
		getPosition( tmpPoint, frameHorizon );
		shot.getPosition( shot.tmpPoint, frameHorizon );

		final int dx = tmpPoint.x - shot.tmpPoint.x;
		final int dy = tmpPoint.y - shot.tmpPoint.y;

		return dx * dx + dy * dy;
	}

	/** Abstand zum Punkt vx,vy */
	public double getTorusDistance( int vx, int vy )
	{
		int odx = x - vx;
		int ody = y - vy;

		while ( odx >  AsteroidApp.EXTENT_X / 2 ) odx -= AsteroidApp.EXTENT_X;
		while ( odx < -AsteroidApp.EXTENT_X / 2 ) odx += AsteroidApp.EXTENT_X;
		while ( ody >  AsteroidApp.EXTENT_Y / 2 ) ody -= AsteroidApp.EXTENT_Y;
		while ( ody < -AsteroidApp.EXTENT_Y / 2 ) ody += AsteroidApp.EXTENT_Y;

		return Math.sqrt( odx * odx + ody * ody );
	}
	
	/** Quadrat des Abstands zu Punkt p in #N Frames */
	public double getSquaredTorusDistance( Point p, int frameHorizon )
	{
		double odx = ( x - p.x ) + frameHorizon * dx;
		double ody = ( y - p.y ) + frameHorizon * dy;

		while ( odx >  AsteroidApp.EXTENT_X / 2 ) odx -= AsteroidApp.EXTENT_X;
		while ( odx < -AsteroidApp.EXTENT_X / 2 ) odx += AsteroidApp.EXTENT_X;
		while ( ody >  AsteroidApp.EXTENT_Y / 2 ) ody -= AsteroidApp.EXTENT_Y;
		while ( ody < -AsteroidApp.EXTENT_Y / 2 ) ody += AsteroidApp.EXTENT_Y;

		return odx * odx + ody * ody;
	}
	
	/** Abstand zu Punkt p in #N Frames */
	public double getTorusDistance( Point p, int frameHorizon )
	{
		return Math.sqrt( getSquaredTorusDistance( p, frameHorizon ) );
	}
	
	/** Quadrat des Abstands zu einem Objekt in #N Frames */
	public double getSquaredTorusDistance( MovingObject obj, int frameHorizon )
	{
		double odx = ( x - obj.x ) + frameHorizon * ( dx - obj.dx );
		double ody = ( y - obj.y ) + frameHorizon * ( dy - obj.dy );

		while ( odx >  AsteroidApp.EXTENT_X / 2 ) odx -= AsteroidApp.EXTENT_X;
		while ( odx < -AsteroidApp.EXTENT_X / 2 ) odx += AsteroidApp.EXTENT_X;
		while ( ody >  AsteroidApp.EXTENT_Y / 2 ) ody -= AsteroidApp.EXTENT_Y;
		while ( ody < -AsteroidApp.EXTENT_Y / 2 ) ody += AsteroidApp.EXTENT_Y;

		return odx * odx + ody * ody;
	}

	/** Abstand zu einem Objekt in #N Frames */
	public double getTorusDistance( MovingObject obj, int frameHorizon )
	{
		return Math.sqrt( getSquaredTorusDistance( obj, frameHorizon ) );
	}
	
	/** Distanz zu Objekt obj */
	public double getTorusDistance( MovingObject obj )
	{
		return getTorusDistance( obj, 0 );
	}
	
	/** Wird ein Schuss an Position p in #n Frames treffen? p muss normalisiert sein! */
	public boolean shotWillHit( Point p, int frameHorizon )
	{
		return getSquaredNormalDistance( p, frameHorizon ) < squaredPixelSize / 4;
	}
	
	/** Wird der Schuss shot in #n Frames treffen? */
	public boolean shotWillHit( Shot shot, int frameHorizon )
	{
		return getSquaredNormalDistance( shot, frameHorizon ) < squaredPixelSize / 4;
	}

	public void update( Point p )
	{
		int odx = p.x - x;
		int ody = p.y - y;

		while ( odx >  AsteroidApp.EXTENT_X / 2 ) odx -= AsteroidApp.EXTENT_X;
		while ( odx < -AsteroidApp.EXTENT_X / 2 ) odx += AsteroidApp.EXTENT_X;
		while ( ody >  AsteroidApp.EXTENT_Y / 2 ) ody -= AsteroidApp.EXTENT_Y;
		while ( ody < -AsteroidApp.EXTENT_Y / 2 ) ody += AsteroidApp.EXTENT_Y;

		x_virtual += odx;
		y_virtual += ody;

		if ( frames <= 8 ) {
			// Nach 8-maliger Distanzermittlung steht das exakte dx,dy fest.
			// Multipliziert mit 8 ergibt dx,dy nun die "Subraumpixel"
			dx = (double)( x_virtual - x0 ) / frames;
			dy = (double)( y_virtual - y0 ) / frames;
		}
		x = p.x;
		y = p.y;

		if ( ( 8 == frames ) && Debug.subraumpixel ) {
			Log.debug( String.format(
					"Frame#%5d: MovingObject ID#%5d: dx,dy=%5f,%5f",
					AsteroidApp.frame_cnt, id, 8 * dx, 8 * dy ) );
		}
		if ( Debug.asteroid_projection ) {
			if ( 8 == frames ) {
				// Falls das Objekt schon #n Frames lebt, #m Frames im voraus x,y berechnen und grafisch darstellen
				projected_frames	= 100;
				projected_x				= (int)( x + projected_frames * dx );
				projected_y				= (int)( y + projected_frames * dy );
	
				while ( projected_x < 0 ) projected_x += AsteroidApp.EXTENT_X;
				while ( projected_y < 0 ) projected_y += AsteroidApp.EXTENT_Y;

				while ( projected_x > AsteroidApp.MAX_X ) projected_x -= AsteroidApp.EXTENT_X;
				while ( projected_y > AsteroidApp.MAX_Y ) projected_y -= AsteroidApp.EXTENT_Y;
			}
			else if ( projected_frames > 0 ) {
				if ( --projected_frames == 0 && this instanceof Asteroid ) {
					double dist = getTorusDistance( projected_x, projected_y );
		
					if ( dist > 2.0 ) {
						Log.error( String.format(
								"Frame#%5d: PROJEKTIONSFEHLER: MovingObject ID#%5d[%4d,%4d], P=[%4d,%4d], frames=%4d, dist=%5f",
								AsteroidApp.frame_cnt, id, x, y, projected_x, projected_y, frames, dist ) );
					}
				}
			}
		}
		// TODO: Bei Frame Loss hier unbedingt ggf. zwei oder mehr addieren.
		frames++;
	}
}
