// This file is part of "Omniroid", an Asteroids bot written for the 2008 c't anniversary contest
// Omniroid was written by Vladimir "CyberShadow" Panteleev <thecybershadow@gmail.com>
// This file is written in the D Programming Language ( http://digitalmars.com/d/ )

/// The master bot logic.
module omnibot;

/*

Strategy:

- precalculate all possible asteroid/UFO positions for 72 frames onwards
  - both if we can shoot this frame and if we can't (split these in threads)
  - trim pool to a sane number (64) if it grows too large

- fire if
  - we can shoot
  - shooting now will destroy a target
  - shooting now does not put us in immediate danger (die in 20 frames or less of doing nothing)

- enter hyperspace if 
  - death imminent (in 20 frames or less)
  - entering now will give us at least 20 frames after we respawn
- if we enter hyperspace, stop thinking and return

- select a target
  - trace bullets from: 
    - firing from current position 
    - turning and firing next frame
  - select object that is closest to the bullet at any point in time
- turn towards target (or in one direction if there are none)

- acceleration:
  - calculate trajectory paths for 50 frames:
    - hold down thrust
    - press thrust only for the next frame
    - do not thrust
  - pick the one in which we die later; if we don't die in any, accelerate

*/

import std.thread;
import asteroids;
import syncbot;
import network;
import utils;
import std.stdio;
debug import std.file, std.string;

enum SizeIndex
{
	Bullet,
	Ship,
	MAX
}

struct Obstacle
{
	short x, y;
	ubyte[SizeIndex.MAX] size; // 0 == vs. bullet, 1 == vs. ship
	ubyte index;

	uint toHash()
	{
		return x | (y<<12) | (size[SizeIndex.Bullet]<<24);
	}

	int opCmp(Obstacle* s)
	{
		return (cast(long)toHash()) - (cast(long)s.toHash());
	}
}

struct Coord
{
	short x, y;

	uint toHash()
	{
		return x | (y<<16);
	}

	int opCmp(Coord* s)
	{
		return (cast(long)toHash()) - (cast(long)s.toHash());
	}
}

enum { MAPPER_MAX_GAMES = 64 }
enum { GAME_WIDTH = 1024*8, GAME_HEIGHT = 768*8 }

class Mapper : Thread
{
	bool[Obstacle][72] obstacles;
	FastGame[] pool;
	KeysData input;

	this(FastGame[] pool, KeysData input)
	{
		this.pool = pool;
		this.input = input;
		start();
	}
	
	override int run()
	{
		if (pool.length > MAPPER_MAX_GAMES)
			pool.length = MAPPER_MAX_GAMES;

		for (int f=0; f<72; f++)
		{
			// map objects
			foreach (ref game; pool)
			{
				foreach (i,t; game.ObjType[0..OBJECT_LAST_UFO_BULLET+1])
					if (t>0 && i!=OBJECT_SHIP)
					{
						Obstacle o;
						o.x = (GAME_WIDTH +game.ObjX[i]+game.ObjSpeedX[i])%GAME_WIDTH ;
						o.y = (GAME_HEIGHT+game.ObjY[i]+game.ObjSpeedY[i])%GAME_HEIGHT;
						
						// size vs. bullet
						if (i >= OBJECT_BULLETS)
							o.size[SizeIndex.Bullet] = 0;
						else
							o.size[SizeIndex.Bullet] = t&1 ? 42 : t&2 ? 72 : 132;
						
						// size vs. ship
						if (i < MAX_ASTEROIDS)
							o.size[SizeIndex.Ship] = (t&1 ? 42 : t&2 ? 72 : 132) + 28;
						else
						if (i == OBJECT_UFO)
							o.size[SizeIndex.Ship] = 42 + (t>1 ? 37 : 19);
						else // bullet
							o.size[SizeIndex.Ship] = 42;

						o.index = i;

						if (!(o in obstacles[f]))
							obstacles[f][o] = true;
					}
				game.ObjType[OBJECT_SHIP] = -96;
				SyncBot.stepOneUsing(input, game);
			}
		}
		return 0;
	}

	debug bool debugData;

	bool collides(int f, Coord c, SizeIndex sizeIndex)
	{
		debug if (debugData) writefln("Testing collision with %04X, %04X", c.x, c.y);
		if (c.x==0 && c.y==0) 
			return false;

		foreach(o,b;obstacles[f]) 
		{
			Coord objX, objY;
			// bullets are always x; ship is x vs asteroids, otherwise y
			// see Game.ProcessCollisions for more info
			if (sizeIndex == SizeIndex.Bullet || o.index < MAX_ASTEROIDS)   
			{
				objX = c;
				objY = Coord(o.x, o.y);
			}
			else
			{
				objX = Coord(o.x, o.y);
				objY = c;
			}
			if (inRange(objY.x-objX.x, -0x200, 0x1F0) &&
				inRange(objY.y-objX.y, -0x200, 0x1F0))
			{
				ubyte size = o.size[sizeIndex];
				debug if (debugData) writef("Size=%3d xy=%04X,%04X index=%2d", size, o.x, o.y, o.index);
				if (size == 0)
					continue;
				ubyte dx = (abs(objY.x-objX.x) - (objY.x<objX.x)) / 2;
				ubyte dy = (abs(objY.y-objX.y) - (objY.y<objX.y)) / 2;
				debug if (debugData) writefln(" dx=%3d  dy=%3d    (size >= dx) is %d    size >= dy is %d    size+size/2 > dx+dy is %d", dx, dy, size >= dx, size >= dy, size+size/2 > dx+dy);
				if (size >= dx && size >= dy && size+size/2 > dx+dy)
				{
					debug if (debugData) writefln(" Collision at %04X,%04X in frame +%d      ", objX.x, objX.y, f);
					return true;
				}
			}
		}
		return false;
	}

	debug void dump(string prefix, Coord[] trajectory)
	{
		foreach (f,ref frame;obstacles)
		{
			int w=64, h=32;
			string[] screen = new string[h];
			string[] descs;
			foreach(ref s;screen)
			{
				s = new char[w];
				s[] = ' ';
			}
			if (trajectory)
				screen[(0x17FF-trajectory[f].y)*h/0x1800][trajectory[f].x*w/0x2000] = 'A';
			foreach (o;frame.keys.sort)
			{
				char c = '?';
				if (o.size[0] > 100)
					c = '#';
				else
				if (o.size[0] > 66)
					c = '*';
				else
				if (o.size[0] > 50)
					c = '+';
				else
					c = '.';
				screen[(0x17FF-o.y)*h/0x1800][o.x*w/0x2000] = c;
				descs ~= format("%3d/%3d @ %04X,%04X", o.size[0], o.size[1], o.x, o.y);
			}
			write(format("%s%02d.txt", prefix, f), join(screen~descs, \r\n));
		}
	}
}

class OmniBot : SyncBot
{
	this(NetworkClient network)
	{
		super(network);
	}

	override KeysData getInput()
	{
		KeysData result;

		if (pool[0].ObjType[OBJECT_SHIP] == -0x60)
			writefln("* [%04X] We died :(", pool[0].FrameCount);
		if (pool[0].ObjType[OBJECT_SHIP] != 1) // ship is not visible
			return result; 

		bool canFire = (pool[0].FireHistory >> 7) == 0;
		if (canFire)
		{
			canFire = false;
			foreach (t;pool[0].ObjType[OBJECT_SHIP_BULLETS..MAX_OBJECTS])
				if (t == 0)  // free bullet slot
				{
					canFire = true;
					break;
				}
		}

		Mapper mapDidntFire, mapFired;
		KeysData empty, fire;
		fire.fire = true;
		mapDidntFire = new Mapper(getFastPool(), empty);
		if (canFire)
			mapFired = new Mapper(getFastPool(), fire);
		
		static void mapObject(ref FastGame game, int f, int index, Coord[] results, KeysData input)
		{
			for (int i=0; i<f; i++)
				results[i] = Coord(0,0); // clear first part
			for (; f<72; f++)
			{
				results[f] = Coord(game.ObjX[index], game.ObjY[index]);
				stepOneUsing(input, game);
			}
		}

		Coord[72] fireStraight, fireLeft, fireRight;      // bullet trajectories
		Coord[72] thrustNever, thrustOnce, thrustAlways;  // ship trajectories
		FastGame game;

		KeysData turnLeft;  turnLeft .left  = true;
		KeysData turnRight; turnRight.right = true;
		KeysData thrust;    thrust.thrust   = true;
		
		getFastGame(game); game.ClearObstacles();                                     mapObject(game,  0, OBJECT_LAST_SHIP_BULLET, fireStraight[], fire);
		getFastGame(game); game.ClearObstacles();                                     mapObject(game,  0, OBJECT_SHIP            , thrustNever [], empty);

		mapDidntFire.wait();
		if (canFire)
			mapFired.wait();
		
		int scan(Mapper map, Coord[] trajectory, SizeIndex sizeIndex, int* minDist = null)
		{
			if (minDist) *minDist = int.max;
			for (int f=1; f<trajectory.length; f++)
			{
				if (minDist)
					foreach (o,b;map.obstacles[f])
						if (o.size[sizeIndex]>0)
						{
							int dist = (abs(trajectory[f].x-o.x) + abs(trajectory[f].y-o.y));
							//dist += (10 * dist) * trajectory.length / f;
							//dist /= f;
							if (*minDist > dist)
								*minDist = dist;
						}
				if (map.collides(f, trajectory[f], sizeIndex))
					return f;
			}
			return -1;
		}

		int fireStraightDist, fireLeftDist, fireRightDist, fireFarLeftDist, fireFarRightDist;
		//writefln("Testing firing straight ahead.");
		int fireStraightResult = scan(mapDidntFire, fireStraight[], SizeIndex.Bullet, &fireStraightDist);
		
		if (canFire)
		{
			if (fireStraightResult > 0)
			{
				//writefln("Testing backfire.");
				int fireStraightLife = scan(mapFired, thrustNever [0..20], SizeIndex.Ship);
				if (fireStraightLife == -1) // we don't collide within 20 frames
				{
					result.fire = true;
					//writefln(" Firing!");
				}
				else
					writefln("* [%04X] Didn't fire because a fragment asteroid would kill us in %d frames.", pool[0].FrameCount, fireStraightLife);
			}
		}

		Mapper map = result.fire ? mapFired : mapDidntFire;
		
		//writefln("Testing our expected life span.");
		int idleLife = scan(map, thrustNever [0..21], SizeIndex.Ship);
		if (idleLife > 0) // we collide within 20 frames
		{
			writefln("* [%04X] Trying to teleport...", pool[0].FrameCount);
			bool entrySafe = true;
			FastGame[] myPool = getFastPool();
			KeysData enterHyperspace;
			enterHyperspace.hyperspace = true;
			stepUsing(enterHyperspace, myPool);
			bool[Coord] possibleExits;
			foreach (ref poolGame; myPool)
			{
				if (poolGame.SafeJump == 0x80)
				{
					entrySafe = false;
					writefln("  Failed (jump unsafe)");
					break;
				}
				assert (poolGame.SafeJump == 1);
				possibleExits[Coord(poolGame.ObjX[OBJECT_SHIP], poolGame.ObjY[OBJECT_SHIP])] = true;
			}
			if (entrySafe)
			hyperspaceExitLoop:
				foreach (exit,b;possibleExits)
				{
					for (int f=48; f<70; f++)
						if (map.collides(f, exit, SizeIndex.Ship))
						{
							writefln("  Failed (re-entry collision at %04X,%04X)", exit.x, exit.y);
							entrySafe = false;
							break hyperspaceExitLoop;
						}
				}
			if (entrySafe)
			{
				result.hyperspace = true;
				return result;
			}
		}
		debug
		{
			static bool dumped;
			if (!dumped)
			{
				getFastGame(game);
				stepOneUsingN(empty, game, 10);
				if (game.Lives < pool[0].Lives)
				{
					map.dump("map/", thrustNever);
					map.debugData = true;
					scan(map, thrustNever [0..21], SizeIndex.Ship);
					dumped = true;
					map.debugData = false;
					//assert(0);
				}
			}
		}


		if (result.fire)
		{
			//writefln("Re-testing firing straight ahead.");
			scan(map, fireStraight[], SizeIndex.Bullet, &fireStraightDist);
		}
		int angle;
		do 
		{
			angle++;
			getFastGame(game); game.ClearObstacles(); stepOneUsingN(turnLeft , game, angle); mapObject(game, angle, OBJECT_LAST_SHIP_BULLET, fireLeft    [], fire);
			getFastGame(game); game.ClearObstacles(); stepOneUsingN(turnRight, game, angle); mapObject(game, angle, OBJECT_LAST_SHIP_BULLET, fireRight   [], fire);
			//writefln("Testing fire to the left.");
			scan(map, fireLeft    [], SizeIndex.Bullet, &fireLeftDist);
			//writefln("Testing fire to the right.");
			scan(map, fireRight   [], SizeIndex.Bullet, &fireRightDist);
			if (fireLeftDist == int.max)
				break;
		} while (angle<42 && min(fireLeftDist, fireRightDist)>1000);
		//debug if (angle != 1) writefln("* No targets ahead; turning (a target became reachable after %d turns)", angle);
		//writefln("Distances: Left=%d, Straight=%d, Right=%d", fireLeftDist, fireStraightDist, fireRightDist);
		int minDist = fireStraightDist;
		if (minDist > fireLeftDist)
			minDist = fireLeftDist;
		if (minDist > fireRightDist)
			minDist = fireRightDist;
		//debug writefln("* MinDist: %d", minDist);

		if (minDist == int.max) // no targets
		{
			result.left = true;
			//writefln(" No targets, turning left.");
		}
		else
		if (minDist == fireStraightDist)
		{
			//writefln(" Standing still.");
		}
		else
		if (minDist == fireLeftDist || minDist == fireFarRightDist)
		{
			result.left  = true;
			//writefln(" Turning left.");
		}
		else
		{
			result.right = true;
			//writefln(" Turning right.");
		}

		KeysData resultThrust = result;
		resultThrust.thrust = true;

	  //getFastGame(game); game.ClearObstacles(); stepOneUsing (result      , game); mapObject(game,  1, OBJECT_SHIP            , thrustNever [], empty);
		getFastGame(game); game.ClearObstacles(); stepOneUsing (resultThrust, game); mapObject(game,  1, OBJECT_SHIP            , thrustOnce  [], empty);
		getFastGame(game); game.ClearObstacles(); stepOneUsing (resultThrust, game); mapObject(game,  1, OBJECT_SHIP            , thrustAlways[], thrust);

		int thrustNeverLife  = scan(map, thrustNever [], SizeIndex.Ship); if (thrustNeverLife ==-1) thrustNeverLife  = int.max-1;
		int thrustOnceLife   = scan(map, thrustOnce  [], SizeIndex.Ship); if (thrustOnceLife  ==-1) thrustOnceLife   = int.max-1;
		int thrustAlwaysLife = scan(map, thrustAlways[], SizeIndex.Ship); if (thrustAlwaysLife==-1) thrustAlwaysLife = int.max;
		
		int thrustLeftLife=-1, thrustRightLife=-1;
		
		int thrustMaxLife = max(thrustNeverLife, max(thrustOnceLife, thrustAlwaysLife));
		if (thrustMaxLife < int.max-1)
		{
			writefln("* [%04X] Expecting crash in %d...", pool[0].FrameCount, thrustMaxLife);
			
			Coord[72] thrustLeft, thrustRight;
			
			KeysData resultThrustLeft  = result; resultThrustLeft .thrust = true; resultThrustLeft .left  = true; resultThrustLeft .right = false;
			KeysData resultThrustRight = result; resultThrustRight.thrust = true; resultThrustRight.right = true; resultThrustRight.left  = false;
			KeysData keyThrustLeft ; keyThrustLeft .thrust = true; resultThrustLeft .left  = true; 
			KeysData keyThrustRight; keyThrustRight.thrust = true; resultThrustRight.right = true; 

			getFastGame(game); game.ClearObstacles(); stepOneUsing (resultThrustLeft , game); mapObject(game,  1, OBJECT_SHIP, thrustLeft [], keyThrustLeft );
			getFastGame(game); game.ClearObstacles(); stepOneUsing (resultThrustRight, game); mapObject(game,  1, OBJECT_SHIP, thrustRight[], keyThrustRight);

			thrustLeftLife  = scan(map, thrustLeft [], SizeIndex.Ship); if (thrustLeftLife ==-1) thrustLeftLife  = int.max-1;
			thrustRightLife = scan(map, thrustRight[], SizeIndex.Ship); if (thrustRightLife==-1) thrustRightLife = int.max-1;
			
			int thrustMaxLife2 = max(thrustLeftLife, thrustRightLife);
			if (thrustMaxLife2 > thrustMaxLife)
			{
				thrustMaxLife = thrustMaxLife2;
				writefln("  Averted - now it's %d", thrustMaxLife);
			}
		}
		
		if (thrustMaxLife == thrustOnceLife || thrustMaxLife == thrustAlwaysLife)
			result.thrust = true;
		else
		if (thrustMaxLife == thrustNeverLife)
			{}
		else
		if (thrustMaxLife == thrustLeftLife)
		{
			result.thrust = true;
			result.left = true;
			result.right = false;
		}
		else
		if (thrustMaxLife == thrustRightLife)
		{
			result.thrust = true;
			result.right = true;
			result.left = false;
		}
		else
		assert(0);

		return result;
	}
}

NetworkClient client;

void main(string[] args)
{
	if (client is null)
		client = new UdpNetworkClient(args.length>1?args[1]:"127.0.0.1");
	auto bot = new OmniBot(client);
	bot.run();
}
