// This file is part of "Omniroid", an Asteroids bot written for the 2008 c't anniversary contest
// Omniroid was written by Vladimir "CyberShadow" Panteleev <thecybershadow@gmail.com>
// This file is written in the D Programming Language ( http://digitalmars.com/d/ )

/// This module contains a D implementation of the simplified game engine.
module asteroids;

import utils;
import crc32, std.string; // hashing, memory comparison
debug import std.stdio;

enum
{
	OBJECT_LAST_ASTEROID = 26,
	OBJECT_SHIP = 27,
	OBJECT_UFO = 28,
	OBJECT_UFO_BULLETS = 29,
	OBJECT_BULLETS = OBJECT_UFO_BULLETS,
	OBJECT_LAST_UFO_BULLET = 30,
	OBJECT_SHIP_BULLETS = 31,
	OBJECT_LAST_SHIP_BULLET = 34,
	OBJECT_LAST = 34,

	MAX_ASTEROIDS = OBJECT_LAST_ASTEROID+1,
	MAX_OBJECTS = OBJECT_LAST+1
}

/// Contains the state of an Asteroids game and the code to work with it.
struct CustomGame(bool VIDEO)
{
	ubyte GlobalScaleFactor; // 0
	ushort VGRAM_Cursor; // 2
	ubyte TranscriptionScale; // 17
	ubyte PlayerCount; // 1C
	int Score; // 52 - stored as seen on SCREEN (but without wrap-around)
	ubyte StartingLives; // 56
	ubyte Lives; // 57
	ubyte SafeJump; // 59
	ubyte PlayerPrompt_Timeout; // 5A
	ushort FrameCount; // 5C
	ubyte RandomSeed1, RandomSeed2; // 5F
	ubyte[2] ObjAngle; // 61
	ubyte FireHistory; // 63
	ubyte ShipSpeedXLow, ShipSpeedYLow; // 64
	short[7] ShipExplosionX, ShipExplosionY; // 7E
	byte[MAX_OBJECTS] ObjType; // 200
	byte[MAX_OBJECTS] ObjSpeedX, ObjSpeedY;
	short[MAX_OBJECTS] ObjX, ObjY;
	ubyte StartingAsteroidCount; // 2F5
	ubyte AsteroidCount; // 2F6
	ubyte UFO_Timeout; // 2F7
	ubyte UFO_Level; // 2F8
	ubyte Idle_Timer; // 2F9
	ubyte Respawn_Timeout; // 2FA
	ubyte Asteroid_Spawn_Timeout; // 2FB
	ubyte UFO_AsteroidThreshhold; // 2FD

	bool EndOfGameState; // marker that denotes the end of variables that represent the game's state
	
	bool Input_HyperSpace; // 2003
	bool Input_Fire; // 2004
	bool Input_Start; // 2403
	bool Input_Thrust; // 2405
	bool Input_Clockwise; // 2406
	bool Input_CounterClockwise; // 2407

	// interop stuff:
	bool Option_UnknownFrame; // prevents UFOs from turning until frame is thus determined
	bool Option_UnknownUFOY; // not used here, but used externally

	ushort[0x400] VectorRAM;


	/// Initialize the game variables for a new game.
	void Initialize()
	{
		// Clear variables
		//*this = *this.init;
		(cast(ubyte[])(this[0..1]))[] = 0;
		
		/* 7CF3 locInitialization */
		VectorRAM[0] = 0xE201; // JMPL $201
		VectorRAM[1] = 0xB000; // HALT

		VectorRAM[0x200] = 0xE201; // JMPL $201 - not in original code, but it aids analysis

		ClearVars();
		PlaceAsteroids();
		FlipVideoPage();
	}

	/// Run one cycle (frame) of the game's logic (the border is at the avgdvg "go" signal, 6822)
	void Step()
	{
		FrameCount++;
		VGRAM_Cursor = (VectorRAM[0]&0x0200)? 0x0002 : 0x0402;
		if (ProcessGameStart())
		{
			// starting game.. clear vars etc.
			// jump to 6803
			ClearVars();
			PlaceAsteroids(); // clear them actually, as Asteroid_Spawn_Timeout will be != 0
			FlipVideoPage();
			return;
		}
		if (PlayerPrompt_Timeout==0)
		{
			ProcessPlayerFire();
			ProcessHyperspaceEntry();
			ProcessInput();
			ProcessUFO();
		}
		ProcessObjects();
		ProcessCollisions();
		DrawInterface();
		WriteLABS4(0x7F, 0x7F);
		//writefln("%04X", RandomSeed);
		Random();
		WriteHALT();
		if (Asteroid_Spawn_Timeout)
			Asteroid_Spawn_Timeout--;
		else
			if (AsteroidCount==0)
			// loop around here
				PlaceAsteroids();

		FlipVideoPage();
	}
	
	/// 681A
	void FlipVideoPage()
	{
		VectorRAM[0] ^= 0x0200; // flip video page
	}

	/// 6885
	bool ProcessGameStart()
	{
		if (PlayerCount)
			if (PlayerPrompt_Timeout) // "Player N" prompt
			{
				PlayerPrompt_Timeout--;
				PrintPlayerN();
				return false;
			}
			else // game is running
			{
				/// 6960 locGameRunning
				if (Lives==0)
					throw new Exception("We died");
				if (ObjType[OBJECT_SHIP]==0 && Respawn_Timeout==0x80)
				{
					Respawn_Timeout = 0x10;
					ResetUFO();
				}
				return false;
			}
		else // no players
			if (Input_Start)
			{
				/// 68DE
				PlayerCount = 1;
				CenterShip();
				Respawn_Timeout = 1;
				UFO_Timeout = UFO_Level = 0x92;
				Asteroid_Spawn_Timeout = 0x7F; 
				UFO_AsteroidThreshhold = 5;
				PlayerPrompt_Timeout = 0x80;
				Lives = StartingLives;
				return true;
			}
			else
			{
				/// 693B locNotStarting
				// flashing "Press Start" text code omitted
				// leds code omitted
				return false;
			}
	}


	/// 69E2
	void PrintPlayerN()
	{
	}
	
	/// 69F0
	void ProcessCollisions()
	{
		//       x     /          y
		// our bullets / asteroids, ship, ufo
		// ufo bullets / asteroids, ship
		// ufo         / asteroids, ship
		// ship        / asteroids

		//       x     /     y     /     size
		// our bullets / asteroids / 42/72/132
		// our bullets / ship      / 42
		// our bullets / ufo       / 42/72
		// ufo bullets / asteroids / 42/72/132
		// ufo bullets / ship      / 42
		// ufo         / asteroids / 42/72/132 + 19/37
		// ufo         / ship      / 42 + 19/37
		// ship        / asteroids / 42/72/132 + 28

		for (byte x=OBJECT_LAST;x>=OBJECT_SHIP;x--)
			if (ObjType[x]>0)  // occupied and not exploding
			{
				/// 69FD locProcessObject
				ubyte start = 
					x<OBJECT_SHIP_BULLETS ?
						x==OBJECT_SHIP ?
							OBJECT_LAST_ASTEROID
						: 
							OBJECT_SHIP
					: OBJECT_UFO;
				for (byte y=start;y>=0;y--)
					if (ObjType[y]>0)  // occupied and not exploding
					{
						/// 6A0A locCheckAsteroid
						/+debug if ((x==OBJECT_SHIP && y==19) && (FrameCount==0x1442||FrameCount==0x1443))
						{
							writefln(" x==%d y==%d  ObjType[y]=%02X ObjXY[x]==%04X,%04X  ObjXY[y]==%04X,%04X", x, y, ObjType[y], ObjX[x], ObjY[x], ObjX[y], ObjY[y]);
							writefln("Test 1: %d", inRange(ObjX[y]-ObjX[x], -0x200, 0x1FF));
							writefln("Test 2: %d", inRange(ObjY[y]-ObjY[x], -0x200, 0x1FF));
							//assert(0);
						}// +/
						//XObj-YObj<=0x200 && YObj-XObj<0x200
						//(abs(YObj-XObj) - (YObj<XObj)) / 2
						if (inRange(ObjX[y]-ObjX[x], -0x200, 0x1FF) &&
							inRange(ObjY[y]-ObjY[x], -0x200, 0x1FF))
						{
							ubyte dx = (abs(ObjX[y]-ObjX[x]) - (ObjX[y]<ObjX[x])) / 2;
							ubyte dy = (abs(ObjY[y]-ObjY[x]) - (ObjY[y]<ObjY[x])) / 2;
							ubyte size = ObjType[y]&1 ? 42 : ObjType[y]&2 ? 72 : 132;
							if (x == OBJECT_SHIP)
								size += 28;
							if (x == OBJECT_UFO)
								size += ObjType[OBJECT_UFO]>1 ? 37 : 19;
							/+debug if ((x==OBJECT_SHIP && y==19) && (FrameCount==0x1442||FrameCount==0x1443))
							{
								writefln("dx=%d dy=%d size=%d", dx, dy, size);
								writefln("size >= dx = ", size >= dx);
								writefln("size >= dy = ", size >= dy);
								writefln("size+size/2 >= dx+dy = ", size+size/2 >= dx+dy);
							}// +/
							if (size >= dx && size >= dy && size+size/2 > dx+dy)
							{
								Collide(x, y);
								break;
							}
						}
					}
			}
	}

	/// 6A9D
	void CopyAsteroid(ubyte to, ubyte from)
	{
		ObjType[to] = (ObjType[from]&0b00000111) | (Random()&0b00011000);
		ObjX[to] = ObjX[from];
		ObjY[to] = ObjY[from];
		ObjSpeedX[to] = ObjSpeedX[from];
		ObjSpeedY[to] = ObjSpeedY[from];
	}

	/// 6AD7
	void TranscribeVideoSubroutine(ref ushort address, ubyte xor1, ubyte xor2)
	{
		for (ubyte y=0;;y++)
		{
			ushort w = readROMWord(address); address+=2;
			w ^= xor2<<8;
			if (w > 0xF000) // SVEC
				WriteWord((w ^ xor1) + TranscriptionScale);
			else
			if (w > 0xA000) // not VCTR
				return;
			else
			{
				WriteWord(w);
				w = readROMWord(address); address+=2;
				WriteWord((w ^ (xor1<<8)) + (TranscriptionScale<<8));
			}
		}
	}

	/// 6B0F
	void Collide(ubyte x, ubyte y)    // x = SHIP/UFO/BULLETS, y = ASTEROID/SHIP/UFO
	{
		/+debug if ((x==29 && y==25) && FrameCount==0xC245)
		{
			writefln("Collide(%d, %d)", x, y);
		}// +/
		if (x==OBJECT_UFO && y==OBJECT_SHIP) // swap
		{
			x = OBJECT_SHIP;
			y = OBJECT_UFO;
		}
	
		// what are we destroying this with?
		if (x==OBJECT_SHIP) // our own ship - die
		{
			Respawn_Timeout = 0x81;
			Lives--;
			//if (FrameCount==0x1443) writefln("We died from colliding with %d at frame %04X", y, FrameCount);
		}
		if (x<=OBJECT_UFO) // a ship - explode it
		{
			/// 6B29 locDestroyShip
			ObjType[x] = -0x60;
			ObjSpeedX[x] = ObjSpeedY[x] = 0;
		}
		else // a bullet - erase it
		{
			/// 6B3C locNotShip
			ObjType[x] = 0;
		}
	
		// what are we destroying?
		if (y<OBJECT_SHIP) // an asteroid - break it
		{
			/// 6B47 locAsteroid
			BreakAsteroid(x, y);
		}
		else
		if (y==OBJECT_SHIP) // our ship - die
		{
			/// 6B66 locShip
			Respawn_Timeout = 0x81;
			Lives--;
		}
		else // an UFO
		{
			/// 6B73 locUpdateScore
			UFO_Timeout = UFO_Level;
			if (PlayerCount)
			{
				AddToScore(ObjType[OBJECT_UFO]&1 ? 1_000 : 200);
			}
		}
		/// 6B4A locDestroyObject
		ObjType[y] = -0x60;
		ObjSpeedX[y] = 0;
		ObjSpeedY[y] = 0;
	}

	/// 6B93
	void ProcessUFO()
	{
		if (FrameCount&3) return;
		// every 4 frames
		if (ObjType[OBJECT_UFO]<0) return; // exploding
		if (ObjType[OBJECT_UFO]==0)
		{
			/// 6BA4 locNoUFO
			if (PlayerCount && (ObjType[OBJECT_SHIP]<=0)) // the player's ship is exploding/respawning
				return;
			if (Idle_Timer)
				Idle_Timer--;
			UFO_Timeout--;
			if (UFO_Timeout>0)
				return;
			UFO_Timeout = 18;
			if (Idle_Timer==0 || (AsteroidCount>0 && AsteroidCount<UFO_AsteroidThreshhold))
			{
				if (UFO_Level>=38 || UFO_Level<6) // underflow emulation
					UFO_Level -= 6;
				ObjX[OBJECT_UFO] = 0;
				ubyte y = Random();
				ObjY[OBJECT_UFO] = ((ObjY[OBJECT_UFO]&0xFF)>>3) | ((y<<5)&0xFF) | (((y>>3)>=24?(y>>3)&23:(y>>3))<<8);
				if (RandomSeed2 & 0b01000000)
				{
					ObjSpeedX[OBJECT_UFO] = 16;
				}
				else
				{
					ObjX[OBJECT_UFO] = 0x1FFF;
					ObjSpeedX[OBJECT_UFO] = -16;
				}
				if (UFO_Level >= 0x80)     // UFO_Level starts at $92, so first 3 UFOs are always big
					ObjType[OBJECT_UFO] = 2;
				else
				if ((Score%100_000) >= 30_000)
					ObjType[OBJECT_UFO] = 1;
				else
				if (UFO_Level/2 >= Random())
					ObjType[OBJECT_UFO] = 2;
				else
					ObjType[OBJECT_UFO] = 1;
			}
		}
		else
		{
			/// 6C34 locUFOPresent
			if (!Option_UnknownFrame && (FrameCount & 0x7F) == 0)
			{
				static const byte[] NewUFOYSpeeds = [-16, 0, 0, 16];
				ObjSpeedY[OBJECT_UFO] = NewUFOYSpeeds[Random()&3];
			}
			if (PlayerCount && Respawn_Timeout) return; // player is respawning
			UFO_Timeout--;
			if (UFO_Timeout) return; // timeout still ticking
			UFO_Timeout = 10;
			if (ObjType[OBJECT_UFO] & 1)
			{
				/// 6C65 locSmallUFO
				ObjAngle[1] = CalcAngle(((ObjX[OBJECT_SHIP] - ObjX[OBJECT_UFO]) >> 6) - (ObjSpeedX[OBJECT_UFO] >> 1), ((ObjY[OBJECT_SHIP] - ObjY[OBJECT_UFO]) >> 6) - (ObjSpeedY[OBJECT_UFO] >> 1));
				ubyte mode = (Score%100_000) >= 35_000;
				ubyte error = mode ? 0b10000111 : 0b10001111;
				/+debug if (FrameCount == 0x6808)
				{
					writefln("Angle = %02X; error = %02X", ObjAngle[1], error);
					assert(0);
				}// +/
				ubyte x = Random() & error;
				if (x&0x80) x |= ~error;
				ObjAngle[1] += x + mode;
					
			}
			else  // big UFO
				ObjAngle[1] = Random();
			Fire(1, OBJECT_UFO_BULLETS, OBJECT_LAST_UFO_BULLET);
		}
	}

	/// 6CD7
	void ProcessPlayerFire()
	{
		if (PlayerCount==0) return;
		FireHistory = (FireHistory>>1) | (Input_Fire<<7);
		if (FireHistory >> 6 != 0b10) return; // on key press only..
		if (Respawn_Timeout) return;
		Fire(0, OBJECT_SHIP_BULLETS, OBJECT_LAST_SHIP_BULLET);
	}

	/// 6CF2
	void Fire(ubyte parentIndex, ubyte firstBullet, ubyte lastBullet)
	{
		for (ubyte y=lastBullet;y>=firstBullet;y--)
			if (ObjType[y]==0)
			{
				ObjType[y] = 18;

				byte bx = Cos(ObjAngle[parentIndex]) >> 1;
				byte sx = bx + ObjSpeedX[OBJECT_SHIP + parentIndex];
				if (sx> 111) sx =  111;
				if (sx<-111) sx = -111;
				ObjSpeedX[y] = sx;
					
				byte by = Sin(ObjAngle[parentIndex]) >> 1;
				byte sy = by + ObjSpeedY[OBJECT_SHIP + parentIndex];
				if (sy> 111) sy =  111;
				if (sy<-111) sy = -111;
				ObjSpeedY[y] = sy;

				ObjX[y] = ObjX[OBJECT_SHIP + parentIndex] + bx + (bx>>1);
				ObjY[y] = ObjY[OBJECT_SHIP + parentIndex] + by + (by>>1);

				return;
			}
	}

	/// 6E74
	void ProcessHyperspaceEntry()
	{
		if (PlayerCount==0) return;
		if (ObjType[OBJECT_SHIP]<0) return;
		if (Respawn_Timeout) return;
		if (!Input_HyperSpace) return;
		ObjType[OBJECT_SHIP] = 0;
		ObjSpeedX[OBJECT_SHIP] = ObjSpeedY[OBJECT_SHIP] = 0;
		Respawn_Timeout = 48;
		ubyte x = Random() & 31;
		if (x>28) x=28;
		if (x< 3) x= 3;
		ObjX[OBJECT_SHIP] = (ObjX[OBJECT_SHIP]&0xFF) | (x<<8);
		for (int n=4;n>0;n--) Random();
		ubyte y = Random() & 31;
		SafeJump = 1;
		if (y>=24)
		{
			y = ((y&7)<<1)+4; // 4..18
			if (y>=AsteroidCount)
				SafeJump = 0x80;  // hyperspace exit failure
		}
		if (y>20) y = 20;
		if (y< 3) y =  3;
		ObjY[OBJECT_SHIP] = (ObjY[OBJECT_SHIP]&0xFF) | (y<<8);
	}

	/// 6ED8
	void ClearVars()
	{
		StartingAsteroidCount = 2;
		StartingLives = 3;
		ObjType[OBJECT_SHIP..MAX_OBJECTS] = 0;
		AsteroidCount = 0;
	}

	/// 6F35
	void PrintGlyph(ubyte x2)
	{
		WriteWord(readROMWord(Glyphs + x2));
	}

	/// 6F3E
	void DrawLives(ubyte lives, ubyte x)
	{
		if (lives==0) return;
		GlobalScaleFactor = 0xE0;
		WriteLABS4(x, 0xD5);
		for (ubyte l=lives;l>0;l--)
			WriteJSRL(0x54DA); // (A6D) - ship
	}

	/// 6F57
	void ProcessObjects()
	{
		for (byte x=OBJECT_LAST;x>=0;x--)
			if (ObjType[x])
				/// 6F62 locProcessObject
				if (ObjType[x]<0) // explosion
				{
					byte a = (-ObjType[x]) >> 4;
					if (x==OBJECT_SHIP)
						a = FrameCount&1;
					else
						a++;
					a += ObjType[x];
					if (a>=0)
					{
						if (x==OBJECT_SHIP)
						{
							/// 6F93 locShipDoneExploding
							CenterShip();
						}
						else
						if (x>=OBJECT_UFO)
						{
							/// 6F99 locUFODoneExploding
							UFO_Timeout = UFO_Level;
						}
						else
						{
							AsteroidCount--;
							if (AsteroidCount==0)
								Asteroid_Spawn_Timeout = 0x7F;
						}
						/// 6F8C locDeleteObject
						ObjType[x] = 0;
					}
					else
					{
						/// 6FA1 locStillExploding
						ObjType[x] = a;
						DrawObject(x, x==OBJECT_SHIP ? 0 : (ObjType[x] & 0xF0) + 0x10, ObjX[x], ObjY[x]);
					}
				}
				else
				{
					/// 6FC7 locNotExplosion
					short newX = ObjX[x] + ObjSpeedX[x];
					if ((newX & ~0x1FFF) && x==OBJECT_UFO)
					{
						ObjX[x] = (ObjX[x] & 0xFF00) | (newX & 0x00FF);  // only set the low byte
						ResetUFO();
						continue;
					}
					ObjX[x] = newX;
					ObjX[x] &= 0x1FFF;
					ObjY[x] += ObjSpeedY[x];
					if (ObjY[x]<0)
						ObjY[x] += 0x1800;
					if (ObjY[x]>=0x1800)
						ObjY[x] -= 0x1800;
					ubyte scale;
					if (ObjType[x]&1)
						scale = 0xE0;
					else
					if (ObjType[x]&2)
						scale = 0xF0;
					else
						scale = 0x00;
					DrawObject(x, scale, ObjX[x], ObjY[x]);
				}
	}

	/// 702D
	void ResetUFO()
	{
		UFO_Timeout = UFO_Level;
		ObjType[OBJECT_UFO] = 0;
		ObjSpeedX[OBJECT_UFO] = ObjSpeedY[OBJECT_UFO] = 0;
	}

	/// 703F ProcessInput
	void ProcessInput()
	{
		if (PlayerCount==0) return;
		if (ObjType[OBJECT_SHIP]<0) return;
		if (Respawn_Timeout)
		{
			Respawn_Timeout--;
			if (Respawn_Timeout) return;
			// respawning...
			if (SafeJump>=0x80)
			{
				/// 706F locCrashExit
				ObjType[OBJECT_SHIP] = -0x60;
				Lives--;
				Respawn_Timeout = 0x81;
			}
			else
			if (SafeJump!=0)
			{
				/// 7068 locSafeExit
				ObjType[OBJECT_SHIP] = 1;
			}
			else // SafeJump==0 - very safe exit
			{
				if (!CheckRespawnOverlap()) // true if overlap
				{
					if (ObjType[OBJECT_UFO]==0)
						ObjType[OBJECT_SHIP] = 1;  // spawn if no overlap and no UFO
					else
					{
						Respawn_Timeout = 2;
						return;
					}
				}
			}
			/// 7081 locExit
			SafeJump = 0;
		}
		else
		{
			/// 7086 locPlaying
			if (Input_CounterClockwise)
				ObjAngle[0] += 3;
			else
			if (Input_Clockwise)
				ObjAngle[0] -= 3;
			if (FrameCount&1) return;
			if (Input_Thrust)
			{
				ShipSpeedX = CapSpeed(ShipSpeedX + (Cos(ObjAngle[0]) << 1));
				ShipSpeedY = CapSpeed(ShipSpeedY + (Sin(ObjAngle[0]) << 1));
			}
			else
			{
				short slowDown(short x)
				{
					 return x - (x>>8)*2 - (x>=0);
				}
				
				/// 70E1 locNoThrust
				if (ShipSpeedX)
					ShipSpeedX = slowDown(ShipSpeedX);
				if (ShipSpeedY)
					ShipSpeedY = slowDown(ShipSpeedY);
			}
		}
	}

	/// 7125
	short CapSpeed(short x)
	{
		if (x >=  0x4000)
			return 0x3FFF;
		if (x <= -0x4000)
			return -0x3FFF;
		return x;
	}

	/// 7139
	bool CheckRespawnOverlap()
	{
		for (byte i=OBJECT_UFO;i>=0;i--)
			if (ObjType[i])
			{
				byte dx = (ObjX[i]>>8) - (ObjX[OBJECT_SHIP]>>8);
				byte dy = (ObjY[i]>>8) - (ObjY[OBJECT_SHIP]>>8);
				if (dx<4 && dx>=-4 && dy<4 && dy>=-4)
				{
					Respawn_Timeout++;
					return true;
				}
			}
		return false;
	}

	/// 7168
	void PlaceAsteroids()
	{
		//writefln("Placing asteroids with seed %04X", RandomSeed);
		if (Asteroid_Spawn_Timeout)
		{
			ObjType[0..MAX_ASTEROIDS] = 0;
			return;
		}
		if (ObjType[OBJECT_UFO])
			return;
		//writefln("PlaceAsteroids");
		ObjType[0..MAX_ASTEROIDS] = 0;
		ObjSpeedX[OBJECT_UFO] = ObjSpeedY[OBJECT_UFO] = 0;
		if (UFO_AsteroidThreshhold<10) UFO_AsteroidThreshhold++;
		AsteroidCount = StartingAsteroidCount = StartingAsteroidCount>=9?11:StartingAsteroidCount+2;
		ubyte asteroidsLeft = AsteroidCount; // byte_B
		ubyte currentAsteroid = OBJECT_LAST_ASTEROID; // X
		do
		{
			ObjType[currentAsteroid] = (Random() & 0b00011000) | 0b00000100;
			GenerateRelativeSpeeds(currentAsteroid, OBJECT_UFO);
			ubyte x = Random();
			if (x&1)
			{
				x = (x>>1)&31;
				if (x >= 24) x &= 23;
				ObjY[currentAsteroid] = (ObjY[currentAsteroid] & 0x00FF) | (x << 8); // set high byte only
				ObjX[currentAsteroid] = 0;
			}
			else
			{
				x = (x>>1)&31;
				ObjX[currentAsteroid] = (ObjX[currentAsteroid] & 0x00FF) | (x << 8); // set high byte only
				ObjY[currentAsteroid] = 0;
			}
			currentAsteroid--;
		} while (--asteroidsLeft);
		UFO_Timeout = 127;
	}

	/// 71E8
	void CenterShip()
	{
		ObjSpeedX[OBJECT_SHIP] = ObjSpeedY[OBJECT_SHIP] = 0; // high bytes only
		ObjX[OBJECT_SHIP] = 0x1060;
		ObjY[OBJECT_SHIP] = 0x0C60;
	}

	/// 7203
	void GenerateRelativeSpeeds(ubyte indexTo, ubyte indexFrom)
	{
		byte r = Random() & 0b10001111;
		if (r<0) r |= 0xF0; // r <= Random(-16..15)
		ObjSpeedX[indexTo] = AdjustSpeeds(ObjSpeedX[indexFrom] + r);
		Random(); Random(); Random();
		r = Random() & 0x8F;
		if (r<0) r |= 0xF0; // r <= Random(-16..15)
		ObjSpeedY[indexTo] = AdjustSpeeds(ObjSpeedY[indexFrom] + r);
	}

	/// 7233
	byte AdjustSpeeds(byte x)
	{
		if (x<0)
		{
			if (x<-31) x = -31;
			if (x>- 6) x = - 6;
		}
		else
		{
			if (x<  6) x =   6;
			if (x> 31) x =  31;
		}
		return x;
	}

	/// 724F
	void DrawInterface()
	{
		static if (VIDEO)
		{
			GlobalScaleFactor = 0x10;
			WriteJSRL(0x50A4); // (852) - copyright notice
			WriteLABS4(0x19, 0xDB);
			WriteDelay(0x70);
			/+bool f = true;
			PrintNumber((Score%100_000)/10, 2, 0, f); // N, digits/2, scale, prefix with spaces
			PrintScaledDigit(0, f);+/
			FastPrintScore(); // faster
		
			DrawLives(Lives, 0x28);
			GlobalScaleFactor = 0;
			WriteLABS4(0x78, 0xDB);
			WriteDelay(0x50);
			/+f = true;
			PrintNumber(0, 2, 0, f); // print "0" for highscore
			PrintDigit(0, f);+/
			WriteWildcard(); WriteWildcard(); WriteWildcard(); WriteWildcard(); WriteWord(0xCADD); // high score - four wildcards and a zero
			GlobalScaleFactor = 0x10;
			WriteLABS4(0xC0, 0xDB);
			WriteDelay(0x50);
		}
	}

	/// 72FE
	void DrawObject(ubyte i, ubyte scale, short x, short y)
	{
		static if (VIDEO)
		{
			GlobalScaleFactor = scale;
			x = x >> 3;
			y = (y+0x400) >> 3;
			WriteLABS(x, y);
			ubyte a = 0x70 - GlobalScaleFactor;
			while (a >= 0xA0)
			{
				WriteDelay(0x90);
				a -= 0x10;
			}
			WriteDelay(a);
			if (ObjType[i]<0) // explosion?
			{
				if (i == OBJECT_SHIP)
					DrawShipExplosion();
				else
					WriteWord(readROMWord(Explosions + ((ObjType[i] & 0b00001100) >> 1)));
			}
			else
			{
				/// 735B locNotExplosion
				if (i == OBJECT_SHIP)
					DrawShip();
				else
				if (i == OBJECT_UFO)
					WriteWord(readROMWord(UFO));
				else
				if (i >= OBJECT_BULLETS)
				{
					WriteVCTR(0x70, 0xF0);
					if ((FrameCount&3) == 0)
						ObjType[i]--;
				}
				else // asteroid
				{
					WriteWord(readROMWord(Asteroids + ((ObjType[i] & 0b00011000) >> 2)));
				}
			}
		}
		else
		{
			if (i >= OBJECT_BULLETS && (FrameCount&3)==0)
				ObjType[i]--;
		}
	}

	/// 7397
	void AddToScore(ushort points)
	{
		if (Score/10_000 != (Score+points)/10_000)
			Lives++;
		Score += points;
		//Score %= 100_000;
	}

	/// 745A
	byte FindFreeAsteroidSlot()
	{
		return FindFreeAsteroidSlotFrom(OBJECT_LAST_ASTEROID);
	}

	///745C
	byte FindFreeAsteroidSlotFrom(byte index)
	{
		do 
		{
			if (ObjType[index]==0)
				return index;
		} while (--index>=0);
		return -1;
	}

	/// 7465
	void DrawShipExplosion()
	{
		if (ObjType[OBJECT_SHIP] < -94)
			for (ubyte i=0; i<=5; i++)
			{
				// set the high bytes
				ShipExplosionX[i] = (ShipExplosionX[i] & 0x00FF) | ((((readROMByte(ExplosionPieces + i*2    ) >> 4) + 0xF8) ^ 0xF8) << 8);
				ShipExplosionY[i] = (ShipExplosionY[i] & 0x00FF) | ((((readROMByte(ExplosionPieces + i*2 + 1) >> 4) + 0xF8) ^ 0xF8) << 8);
			}
		for (byte pieceNr = ((cast(ubyte)(ObjType[OBJECT_SHIP]) ^ 0xFF) & 0x70) >> 3; pieceNr>=0; pieceNr-=2)
		{
			ShipExplosionX[pieceNr/2] += cast(byte)readROMByte(ExplosionPieces + pieceNr    );
			ShipExplosionY[pieceNr/2] += cast(byte)readROMByte(ExplosionPieces + pieceNr + 1);
			auto savedPosition = VGRAM_Cursor;
			//writefln("DrawShipExplosion: ShipType=%d, pieceNr=%d, ShipExplosionX=%d, ShipExplosionY=%d", ObjType[OBJECT_SHIP], pieceNr, ShipExplosionX[pieceNr/2], ShipExplosionY[pieceNr/2]);
			WriteShipSegmentPosition(ShipExplosionX[pieceNr/2] >> 8, ShipExplosionY[pieceNr/2] >> 8);
			auto vector = readROMWord(ShipExplosionVectors + pieceNr);
			WriteWord( vector);
			WriteWord((vector & 0xFF0F) ^ 0x0404);
			WriteWord(VectorRAM[savedPosition/2  ] ^ 0x0400);
			WriteWord(VectorRAM[savedPosition/2+1] ^ 0x0400);
		}
		//writefln();
	}

	/// 750B
	void DrawShip()
	{
		TranscriptionScale = 0;
		ubyte xor1=0, xor2 = 0;
		ubyte a = ObjAngle[0];
		if (a >= 0x80)
		{
			xor2 = 4;
			a = -a;
		}
		if (a & 0xC0)
		{
			xor1 = 4;
			a = 0x80 - a;
		}
		a = (a >> 1)&0xFE;
		ushort address = readROMWord(ShipAngles + a);
		TranscribeVideoSubroutine(address, xor1, xor2);
		if (Input_Thrust && (FrameCount&4))
		{
			TranscribeVideoSubroutine(address, xor1, xor2);
		}
	}

	/// 75EC
	void BreakAsteroid(ubyte parent, ubyte asteroid)
	{
		Idle_Timer = 0x50;
		ubyte newSize = (ObjType[asteroid] & 0b00000111)>>1;
		ubyte newType = newSize;
		if (newSize)  // break?
			newType |= ObjType[asteroid] & 0b01111000; // copy some bits
		ObjType[asteroid] = newType;
		if (PlayerCount && (parent==OBJECT_SHIP || parent>=OBJECT_SHIP_BULLETS))
		{
			static const ushort[3] AsteroidScores = [100, 50, 20];
			AddToScore(AsteroidScores[newSize]);
		}
		if (ObjType[asteroid]==0) return;
		
		byte slot = FindFreeAsteroidSlot();
		if (slot<0) return;
		AsteroidCount++;
		CopyAsteroid(slot, asteroid);
		GenerateRelativeSpeeds(slot, asteroid);
		ObjX[slot] ^= (ObjSpeedX[slot]&0x1F) << 1;
		
		slot = FindFreeAsteroidSlot();
		if (slot<0) return;
		AsteroidCount++;
		CopyAsteroid(slot, asteroid);
		GenerateRelativeSpeeds(slot, asteroid);
		ObjY[slot] ^= (ObjSpeedY[slot]&0x1F) << 1;
	}

	/// 76F0
	ubyte CalcAngle(byte x, byte y)
	{
		//debug if (FrameCount == 0x6808) writefln("CalcAngle(%02X, %02X)", x, y);
		if (y>=0)
			return  CalcAngle2(x,  y);
		else
			return -CalcAngle2(x, -y);
	}

	/// 76FC
	ubyte CalcAngle2(byte x, byte y)
	{
		//debug if (FrameCount == 0x6808) writefln("CalcAngle2(%02X, %02X)", x, y);
		if (x>=0)
			return   CalcAngle3( x, y);
		else
			return -(CalcAngle3(-x, y) ^ 0x80);
	}

	/// 770E
	ubyte CalcAngle3(byte x, byte y)
	{
		//debug if (FrameCount == 0x6808) writefln("CalcAngle3(%02X, %02X)", x, y);
		if (x==y)
			return 0x20;
		else
		if (y<x)
			return  CalcAngle4(x, y, false);
		else
			return -CalcAngle4(y, x, true ) + 0x40;
	}

	/// 7728
	ubyte CalcAngle4(byte x, byte y, bool c)
	{
		//debug if (FrameCount == 0x6808) writefln("CalcAngle4(%02X, %02X, %d)", x, y, c);
		static const ubyte[16] AngleTable = [0, 2, 5, 7, 10, 12, 15, 17, 19, 21, 23, 25, 26, 28, 29, 31];
		//return AngleTable[CalcAngle5(x, y, c)];
		auto c5 = CalcAngle5(x, y, c);
		//debug if (FrameCount == 0x6808) writefln("CalcAngle5 -> %02X", c5);
		//debug if (FrameCount == 0x6808) writefln("CalcAngle4 -> %02X", AngleTable[c5]);
		return AngleTable[c5];
	}

	/// 773F
	void PrintNumber(int n, ubyte digitPairs, ubyte scale, ref bool prefixSpaces)
	{
		// convert to decimal
		ubyte[4] digits;
		for (int x=0;x<4;x++)
		{
			digits[x] = n%10;
			n/=10;
		}
		
		TranscriptionScale = scale;
		int x = digitPairs*2;
		while (x)
		{
			PrintScaled(digits[--x], prefixSpaces);
			if (x==1) prefixSpaces = false; // last digit is always printed, even if it's 0
			PrintScaled(digits[--x], prefixSpaces);
		}
	}

	/// 776C
	ubyte CalcAngle5(ubyte x, ubyte y, bool c)
	{
		//debug if (FrameCount == 0x6808) writefln("CalcAngle5(%02X, %02X, %d)", x, y, c);
		
		ubyte b = 0;
		for (ubyte i=4;i;i--)
		{
			y = (y<<1) | (b>>7);
			b = (b<<1) + c;
			if (y >= x)
			{
				c = y>=x;
				y -= x;
			}
			else
				c = 0;
		}
		b = (b<<1) + c;
		return b & 0xF;
	}

	/// 7785 
	void PrintScaled(ubyte x, ref bool prefixSpaces)
	{
		if (!prefixSpaces || x!=0)
			PrintScaledDigit(x, prefixSpaces);
		else
			Print(x, prefixSpaces);
	}

	/// 778B
	void PrintScaledDigit(ubyte x, ref bool prefixSpaces)
	{
		if (TranscriptionScale==0)
			Print(x, prefixSpaces);
		else
		{
			prefixSpaces = false;
			ushort address = ((readROMWord(Glyphs + (((x&0xF) + 1) << 1)) << 1) & 0x1FFF) | 0x4000;
			TranscribeVideoSubroutine(address, 0, 0);
		}
	}

	/// 77B5
	ubyte Random()
	{
		enum { RandomFactor = 2 }
		RandomSeed2 = (RandomSeed2 << 1) | (RandomSeed1 >> 7);
		RandomSeed1 =  RandomSeed1 << 1;
		if (RandomSeed2 & 0x80)
			RandomSeed1++;
		if (RandomSeed1 & RandomFactor)
			RandomSeed1 ^= 1;
		if ((RandomSeed1|RandomSeed2) == 0)
			RandomSeed1++;
		return RandomSeed1;
	}


	/// 77D2
	byte Cos(ubyte a)
	{
		return Sin(a + 0x40);
	}

	/// 77D5
	byte Sin(ubyte a)
	{
		if (a < 0x80)
			return SinP(a);
		else
			return -SinP(a&0x7F);
	}

	/// 77DF
	byte SinP(ubyte a)
	{
		if (a > 0x40)
			a = (a^0x7F) + 1;
		return readROMByte(SineTable + a);
	}

	/// 7BC0
	void WriteHALT()
	{
		WriteWord(0xB0B0);
	}

	/// 7BCB
	void Print(ubyte x, ref bool prefixSpaces)
	{
		if (!prefixSpaces || x!=0)
			PrintDigit(x, prefixSpaces);
		else
			PrintGlyph2(x);
	}

	/// 7BD1
	void PrintDigit(ubyte x, ref bool prefixSpaces)
	{
		prefixSpaces = false;
		PrintGlyph2((x&0xF) + 1);
	}

	/// 7BD6
	void PrintGlyph2(ubyte x)
	{
		WriteWord(readROMWord(Glyphs + (x << 1)));
	}

	/// 7BFC
	void WriteJSRL(ushort address)
	{
		WriteWord(0xC000 | ((address&0x1FFF)>>1));
	}

	/// 7C03
	void WriteLABS4(ubyte x, ubyte y)
	{
		WriteLABS(x<<2, y<<2);
	}

	/// 7C1C
	void WriteLABS(ushort x, ushort y)
	{
		WriteWord((y & 0x0FFF) | 0xA000);
		WriteWord((x & 0x0FFF) | (GlobalScaleFactor << 8));
	}

	/// 7C49
	void WriteShipSegmentPosition(short x, short y)
	{
		//writefln("WriteShipSegmentPosition(%d, %d)", x, y);
		auto b81 = x<0;
		x = abs(x);
		auto b82 = y<0;
		y = abs(y);
		/+if (x | y)
		{
			ubyte r = (x>>8) + (y>>8);
			if (r==0)
			{
				if (((x | y) & 0x80) == 0)
					
			}
		}+/
		ubyte r = (x>>8) | (y>>8);
		ubyte rx, ry;
		if (r==0)
			goto loc_7C8B;
		rx = 0;
		if (r<2)
		{
			ry = 1;
			goto loc_7C9B;
		loc_7C8B:
			ry = 2;
			rx = 9;
			ubyte r2 = x | y;
			if (r2 == 0)
				goto loc_7CAB;
			if ((r2 & 0x80) == 0)
				do
					ry++;
				while (((r2<<=1)&0x80)==0);
		loc_7C9B:
			rx = ry;
			do
			{
				x <<= 1;
				y <<= 1;
			} while (--ry);
		}
		loc_7CAB:
		ubyte b8 = (((rx - 10) ^ 0xFF) << 4) | (b82?4:0) | (b81?2:0);
		//writefln("WriteShipSegmentPosition: b8 = %d", b8);
		WriteWord(y | ((b8 & 0xF4)<<8));
		WriteWord(x | ((b8 & 0x02)<<9));
	}

	/// 7CDB
	void WriteDelay(ubyte a)
	{
		WriteVCTR(a, 0);
	}

	/// 7CE0
	void WriteVCTR(ubyte a, ubyte x)
	{
		WriteWord(a << 8);
		WriteWord(x << 8);
	}

	// short-hand properties/methods

	short ShipSpeedX()
	{
		return (ObjSpeedX[OBJECT_SHIP]<<8) | ShipSpeedXLow;
	}

	void ShipSpeedX(short x)
	{
		ObjSpeedX[OBJECT_SHIP] = x>>8;
		ShipSpeedXLow = x&0xFF;
	}

	short ShipSpeedY()
	{
		return (ObjSpeedY[OBJECT_SHIP]<<8) | ShipSpeedYLow;
	}

	void ShipSpeedY(short y)
	{
		ObjSpeedY[OBJECT_SHIP] = y>>8;
		ShipSpeedYLow = y&0xFF;
	}

	ushort RandomSeed()
	{
		return (RandomSeed2<<8) | RandomSeed1;
	}

	void RandomSeed(ushort value)
	{
		RandomSeed1 = value & 0xFF;
		RandomSeed2 = value >> 8;
	}

	void WriteWord(ushort w)
	{
		static if (VIDEO)
		{
			VectorRAM[VGRAM_Cursor/2] = w;
			VGRAM_Cursor += 2;
		}
	}

	void WriteWildcard()
	{
		WriteWord(0xEEEE);
	}

	void FastPrintScore()
	{
		uint n = Score % 100_000;
		ubyte[5] digits;
		for (int x=0;x<5;x++)
		{
			digits[x] = n%10;
			n/=10;
		}
		bool f = true;
		for (int x=4;x>=0;x--)
		{
			if (x == 1) f = false;
			auto digit = digits[x];
			if (digit==0 && f)
				WriteWord(spaceGlyph);
			else
			{
				WriteWord(digitGlyphs[digit]);
				f = false;
			}
		}
	}

	static if (VIDEO)
	{
		ushort[] ActiveVideoPage()
		{
			assert(VectorRAM[0]==0xE001 || VectorRAM[0]==0xE201);
			if (VectorRAM[0]==0xE001)
				return VectorRAM[0..0x200];
			else
				return VectorRAM[0x200..0x400];
		}

		ushort[] PassiveVideoPage()
		{
			assert(VectorRAM[0]==0xE001 || VectorRAM[0]==0xE201);
			if (VectorRAM[0]==0xE201)
				return VectorRAM[0..0x200];
			else
				return VectorRAM[0x200..0x400];
		}
	}

	// methods that query or artificially alter the state of the game - used for simulation testing/research

	/// Put the game in a playable state without any objects/asteroids
	void MakeNewGame()
	{
		Initialize();
		ObjType[] = 0;
		PlayerCount = 1;
		CenterShip();
		AsteroidCount = 1;
		Respawn_Timeout = 0;
		UFO_Timeout = UFO_Level = 0x92;
		Asteroid_Spawn_Timeout = 0; 
		UFO_AsteroidThreshhold = 0;
		PlayerPrompt_Timeout = 0;
		Lives = StartingLives;
	}

	/// Remove all obstacles, while preserving the rest of the game.
	void ClearObstacles()
	{
		ObjType[] = 0;
		ObjType[OBJECT_SHIP] = 1;
		AsteroidCount = 1; // prevent respawning
		Idle_Timer = 0xFF; // prevent UFOs for a while
		UFO_Timeout = 0xFF;
		UFO_AsteroidThreshhold = 0;
	}

	// operators

	char[] data()
	{
		return (cast(char*)this)[0..EndOfGameState.offsetof];
	}

	uint toHash()
	{
		return strcrc32(data);
	}

	int opCmp(typeof(this) s)
	{
		return std.string.cmp(this.data, s.data);
	}
}

alias CustomGame!(true) Game;
alias CustomGame!(false) FastGame;
static assert(Game.sizeof == FastGame.sizeof);

const VectorROM = import("035127.02");

const ShipExplosionVectors = 0x50E0;
const ExplosionPieces = 0x50EC;
const Explosions = 0x50F8;
const Asteroids = 0x51DE;
const UFO = 0x5250;
const ShipAngles = 0x526E;
const Glyphs = 0x56D4;
const SineTable = 0x57B9;

ushort readROMByte(ushort address)
{
	assert(address>=0x5000 && address<0x5800, "Address out of range");
	return (cast(ubyte[])VectorROM)[address-0x5000];
}

ushort readROMWord(ushort address)
{
	assert(address>=0x5000 && address<0x5800, "Address out of range");
	assert((address&1)==0); // word-aligned
	return (cast(ushort[])VectorROM)[(address-0x5000)/2];
}

// optimization pre-fetch
ushort spaceGlyph;
ushort[10] digitGlyphs;

static this()
{
	spaceGlyph = readROMWord(Glyphs);
	for (int i=0;i<10;i++)
		digitGlyphs[i] = readROMWord(Glyphs + 2*(1+i));
}
