// ============================================================================
// File:               RJMenu.java
//
// Project:            i18n 
//
// Purpose:            JMenu which can change it's Label depending on 
//                     Locale.
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2002  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================

package de.caff.i18n.swing;


import de.caff.i18n.I18n;
import de.caff.i18n.Localizable;

import javax.swing.*;
import java.awt.*;
import java.util.Locale;

/**
 *  Menu with localized label.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision$
 */
public class RJMenu
  extends    JMenu
  implements Localizable
{
  protected String resTag;         // name of the resource of this menu's name 

  /**
   *  @param  tag       resource tag for the item
   */
  public RJMenu(String tag) {
    super();
    setTag(tag);
  }

  /**
   *  @param  tag       resource tag for the item
   *  @param  tearoff   tearoff menu?
   */
  public RJMenu(String tag, boolean tearoff) {
    super("", tearoff);
    setTag(tag);
  }

  /**
   * Notifies this component that it now has a parent component.
   * When this method is invoked, the chain of parent components is
   * set up with <code>KeyboardAction</code> event listeners.
   */
  public void addNotify()
  {
    super.addNotify();
    I18n.addLocalizationChangeListener(this);
  }

  /**
   * Overrides <code>JComponent.removeNotify</code> to check if
   * this button is currently set as the default button on the
   * <code>RootPane</code>, and if so, sets the <code>RootPane</code>'s
   * default button to <code>null</code> to ensure the
   * <code>RootPane</code> doesn't hold onto an invalid button reference.
   */
  public void removeNotify()
  {
    I18n.removeLocalizationChangeListener(this);
    super.removeNotify();
  }


  /**
   *  Change the tag of this item.
   *  @param  tag       new resource tag 
   */
  public void setTag(String tag) {
    resTag = tag;
    I18nHelper.setProperties(this, tag, getLocale());
  }

  /**
   *  Change the locale.
   *  @param  l    new locale
   */
  public void setLocale(Locale l) {
    Locale oldLocale = getLocale();
    super.setLocale(l);
    if (oldLocale != null  &&  !oldLocale.equals(l)) {
      if (resTag != null) {
        I18nHelper.setProperties(this, resTag, getLocale());
      }
    }
  }


  /**
   *  Get the locale. Instead of throwing an <code>IllegalComponentStateException</code>
   *  when there's no locale we just return null.
   *  @return the Locale of this Label or null
   */
  public Locale getLocale() {
    try {
      return super.getLocale();
    } catch (IllegalComponentStateException x) {
      return null;
    }
  }

  /**
   *  Overwritten to set the locale of the JMenuItem when unset.
   *  @param  m   menu item to add
   */
  public JMenuItem add(JMenuItem m) {
    if (m instanceof Localizable) {
      Localizable l = (Localizable)m;
      if (l.getLocale() == null) {
        l.setLocale(getLocale());
      }
    }
    return super.add(m);
  }


  /**
   *  Overwritten to set the locale of the JMenuItem when unset.
   *  @param  m   menu item to add
   */
  public JMenuItem insert(JMenuItem m, int index) {
    if (m instanceof Localizable) {
      Localizable l = (Localizable)m;
      if (l.getLocale() == null) {
        l.setLocale(getLocale());
      }
    }
    return super.insert(m, index);
  }


}

