// ============================================================================
// File:               RJLabel.java
//
// Project:            I18n support.
//
// Purpose:            A JLabel with localized text.
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2002  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================

package de.caff.i18n.swing;

import de.caff.i18n.I18n;
import de.caff.i18n.Localizable;

import javax.swing.*;
import java.awt.*;
import java.util.Locale;

/**
 *  A JLabel with localized resource.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision$
 */
public class RJLabel 
  extends    JLabel 
  implements Localizable 
{
  /** Resource tag for label text. */
  protected String   textResource;  
  /** Extra text (unresourced) to add. */
  protected String   addText;     

  /**
   *  Initialize method.
   *  @param  textRes  resource tag of label text
   *  @param  add      unresourced tag to add
   */
  private void init(String textRes, String add)
  {
    if (add != null) {
      setText(I18n.getString(textRes+I18n.SUFFIX_TEXT, getLocale())+add);
    }
    else {
      setText(I18n.getString(textRes+I18n.SUFFIX_TEXT, getLocale()));
    }
    textResource = textRes;
    addText      = add;
    I18n.addLocalizationChangeListener(this);
  }

  /**
   *  Constructor.
   *  @param   textRes    resource tag of the label text
   */
  public RJLabel(String textRes) 
  {
    init(textRes, null);
  }


  /**
   *  Constructor.
   *  @param   textRes    resource tag of the label text
   *  @param   align      horizontal alignment
   */
  public RJLabel(String textRes, int align)
  {
    init(textRes, null);
    setHorizontalAlignment(align);
  }


  /**
   *  Constructor.
   *  @param   textRes    resource tag of the label text
   *  @param   add        text to add
   */
  public RJLabel(String textRes, String add) 
  {
    init(textRes, add);
  }


  /**
   *  Constructor.
   *  @param   textRes    resource tag of the label text
   *  @param   add        text to add
   *  @param   align      horizontal alignment
   */
  public RJLabel(String textRes, String add, int align)
  {
    init(textRes, add);
    setHorizontalAlignment(align);
  }


  /**
   * Notifies this component that it now has a parent component.
   * When this method is invoked, the chain of parent components is
   * set up with <code>KeyboardAction</code> event listeners.
   */
  public void addNotify()
  {
    super.addNotify();
    I18n.addLocalizationChangeListener(this);
  }

  /**
   * Overrides <code>JComponent.removeNotify</code> to check if
   * this button is currently set as the default button on the
   * <code>RootPane</code>, and if so, sets the <code>RootPane</code>'s
   * default button to <code>null</code> to ensure the
   * <code>RootPane</code> doesn't hold onto an invalid button reference.
   */
  public void removeNotify()
  {
    I18n.removeLocalizationChangeListener(this);
    super.removeNotify();
  }


  /**
   *  Change the locale.
   *  @param  l   new locale
   */
  public void setLocale(Locale l) 
  {
    Locale oldLocale = getLocale();
    super.setLocale(l);
    if (oldLocale != null  &&  !oldLocale.equals(l)) {
      if (textResource != null) {
        if (addText == null) {
          setText(I18n.getString(textResource+I18n.SUFFIX_TEXT, l));
        }
        else {
          setText(I18n.getString(textResource+I18n.SUFFIX_TEXT, l)+addText);
        }
        invalidate();
      }
    }
  }

  /**
   *  Get the locale. Instead of throwing 
   *  an <code>IllegalComponentStateException</code>
   *  when there's no locale we just return null.
   *  @return the Locale of this Label or null
   */
  public Locale getLocale() {
    try {
      return super.getLocale();
    } catch (IllegalComponentStateException x) {
      return null;
    }
  }
}


