#ifndef MOVINGOBJECT_H
#define MOVINGOBJECT_H

#include <cmath>
#include <string>
#include "Point.h"
#include "../FrameStatus/LinkedClass.h"

const int MAX_SHOT_AGE = 40;
const int MAX_SHOT_CNT =  1;


///  Collision check moves asteroid ahead by this many frames.
const int COL_MOVE_ASTEROID = 1;

/// Fictive shot: Ship is moved ahead by this many frames.
const int FICTIVE_SHOT_MOVE_SHIP = 1;

/// Fictive shot: Shot is moved back by this many frames.
const int FICTIVE_SHOT_MOVE_BACK = 0;


class FrameStatus;
class Ship;

/// \brief Base class for any objects that can move across the game screen.

/**
 *  MovingObject is the base class for anything that
 *  may move across the screen. Such entities have
 *  basic properties like location and velocity.
 */

class MovingObject : public LinkedInstance<MovingObject>
{
public:

    /// Maximum number of shots to be fired on a single target.
    static const int MAX_AIM = 2;

    int lastShot;
    int shotAtRound[MAX_AIM];

    int x;  ///< x position of the object, in pixel
    int y;  ///< y position of the object, in pixel
    double u;  ///< u velocity, in pixel/second (pixel/60 frames)
    double v;  ///< v velocity, in pixel/second (pixel/60 frames)
    double speed;

    FrameStatus* frame;  ///< frame the object is located in

    virtual ~MovingObject() {}

    virtual std::string className() = 0;


    MovingObject(FrameStatus* frame, int x, int y)
    : frame(frame), x(x), y(y)
    {
        clear();
    }


    MovingObject()
    : frame(0)
    {
        clear();
    }


    int underfire[4];


    void clear()
    {
        for (int i=0;i<4;i++)
            underfire[i] = 1000;
    }

    virtual void listInsertAfter(MovingObject* o) {
        LinkedInstance<MovingObject>::listInsertAfter(o);
        if (listNext)
            for (int i=0;i<4;i++)
                listNext->underfire[i] = underfire[i]+1;
    }



    void setShotRound(int ttl = MAX_SHOT_AGE) 
    {
        for (int i=0;i<4;i++)
            if (underfire[i]>MAX_SHOT_AGE) {
                underfire[i] = MAX_SHOT_AGE-ttl;
                return;
            }
    }

    int getShotRound(int i = 0) {
        int j = lastShot-i;
        while (j<0) j+=MAX_AIM;
        return shotAtRound[j];
    }


    virtual double diameter() {return 0.4;}


    /**
     *  If enough information is available from previous instances,
     *  velocity is deduced accordingly. Otherwise, velocity is
     * set to zero.
     */
    virtual void updateVelocity() {

        u = 0;
        v = 0;

        /// First, the number of available previous object data
        /// is counted, up to a maximum number.
        int nprev = 0;
        MovingObject* prev = this;
        while (nprev<20 && prev->listPrev) {
            prev = prev->listPrev;
            nprev++;
        }

        /// If any previous data is found, this is used to
        /// compute velocity, in pixel per second (assuming
        /// 60 frames per second).
        if (nprev) {
            u = getx()-prev->getx();
            v = gety()-prev->gety();

            // correct over-/underrun in delta
            while (u>=512) u-=1024;
            while (u<-512) u+=1024;
            while (v>=384) v-= 768;
            while (v<-384) v+= 768;

            // normalize delta to 60 frames
            u *= 60/(double)nprev;
            v *= 60/(double)nprev;
            u = u<-480?-480:u;
            u = u> 480? 480:u;
            v = v<-480?-480:v;
            v = v> 480? 480:v;

        }

        speed = sqrt(u*u+v*v);

    }


    virtual int getx() {return x;}
    virtual int gety() {return y;}

    virtual double getu() {return u;}
    virtual double getv() {return v;}

    class Point getPos_shotBy(Ship &o, bool inclTurnTime = false);
};



#endif
