// player.cpp: Beispielspieler fr Asteroids
// Harald Bgeholz / c't
// Changed by David Schulz, 30.06.2008

//TODO:

// - Bewegung des Schiffes beim Schiessen beruecksichtigen

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <math.h>

#if defined(WINDOWS)
#include <winsock2.h>
#else
// 2 Includes fr socket()
#include <sys/types.h>
#include <sys/socket.h>
// 2 Includes fr inet_addr()
#include <netinet/in.h>
#include <arpa/inet.h>
// 2 Includes fr fcntl()
#include <unistd.h>
#include <fcntl.h>
// fr memset()
#define INVALID_SOCKET -1
#define WSAGetLastError() errno
#endif

#include "player.h"

void Player::Run(void)
{
	FramePacket frame;
	KeysPacket keys;
	GameStatus game;
	GameMove game_move;
	char prevframe = 0;
	int t = 0;
	int duration = 40; //holds the time, needed to destroy an object (rotation+shot)
	int nast_last = 0; //number of asteroids on last screen
	int t_empty_screen = 0; //time when screen was first empty
	int n_thrust = 0; //how often the thrust key was pressed
	int thrust_max = 10; //how often the thrust key must be pressed

	for (;;)
	{
		++t;         // time
		++keys.ping; // every package has an individual number for calculation of latency..
		SendPacket(keys);
		ReceivePacket(frame);

		int latency=0;
		if (frame.frameno != ++prevframe || frame.ping != keys.ping)
		{
			latency = frame.frameno - prevframe;
			printf("Latenz %d. %d Frames verloren.\n", keys.ping - frame.ping, latency);
			prevframe = frame.frameno;
		}

		////GameStatus last_game = game.clone();
		InterpretScreen(frame, game);

		if ( game_move.nasteroids != 0 && game_move.nasteroids != game.nasteroids){
			if (game.nasteroids == 0)
				game_move.clear_asteroids();
			else
				game_move.reorder_asteroids(game, latency);

		}
		if ( game_move.nshots != 0 && game_move.nshots != game.nshots){
			if (game.nshots == 0)
				game_move.clear_shots();
			else
				game_move.reorder_shots(game, latency);
		}
		game_move.add (game, latency);
		
		keys.clear();   // release all keys
		int min_dist = 0x7fffffff;
		int min_factor_k_of_colliding_asteroid = 1000;
		int min_dx = 0;
		int min_dy = 0;
		int closest_asteroid = 0;
		int closest_colliding_asteroid = 0;
		int shot_taken = 0;
		int n_shots_taken = 0;
		int last_dx = 0;
		int turn_counter = 0;
		int dist_saucer = 0;
		
		if (game.ship_present)
		{
			for (int i=0; i<game.nasteroids; ++i)
			{   // find closest asteroid
				Distance dist = Distance( game.asteroids[i].x, game.asteroids[i].y, game.ship_x, game.ship_y);
				switch (game.asteroids[i].sf)
				{	// correct distance by radius of asteroid
					case 0:  // big asteroid
						dist.correct_distance(40*40);
						break;
					case 15: // middle asteroid
						dist.correct_distance(20*20);
						break;
					case 14: // small asteroid
						dist.correct_distance(8*8);
						break;
				}
				if (dist.get_distance() < min_dist)
				{
					min_dist = dist.get_distance();
					min_dx = dist.get_dx();
					min_dy = dist.get_dy();
					closest_asteroid = i;
					
					//define what close is:
					int close = 350*350; //slow asteroid
					if (game_move.asteroids[i].dx*game_move.asteroids[i].dx+game_move.asteroids[i].dy*game_move.asteroids[i].dy > 16 ) {
						close = 450*450; //fast asteroid
					}
					
					//asteroid is close to us -> check if it will collide..
					if (dist.get_distance() < close ) {
						//will this asteroid collide with us?
						Vector p = Vector(game.ship_x, game.ship_y);
						DistStraightPoint d = DistStraightPoint( p, game_move.asteroids[i]);
						if (d.get_asteroid_distance(game_move.asteroids[i]) <= 30*30 && d.get_k() > 0) {
							//collision possible
							if (d.get_k() < min_factor_k_of_colliding_asteroid) {
								min_factor_k_of_colliding_asteroid = d.get_k();
								closest_colliding_asteroid = i;
							}
						}
					}
					
				}
			}
			if (game.saucer_present)
			{
				Distance dist = Distance( game.saucer_x, game.saucer_y, game.ship_x, game.ship_y);
				switch (game.saucer_size)
				{	// correct distance to ufo by size
				case 15: // big UFO
					dist.correct_distance(20*12);
					break;
				case 14: // small UFO
					dist.correct_distance(10*6);
					break;
				}
				dist_saucer = dist.get_distance();
				if (dist_saucer < min_dist)
				{
					min_dist = dist_saucer;
					min_dx = dist.get_dx();
					min_dy = dist.get_dy();
				}
				dist_saucer = (int)sqrt(dist_saucer); //get real distance
			}
			//avoid collision with shots:
			if (game.nshots > 0)
			{
				for (int i=0; i<game.nshots; ++i)
				{
					Vector p = Vector(game.ship_x, game.ship_y);
					DistStraightPoint d = DistStraightPoint( p, game_move.shots[i]);

					if (game_move.shots[i].self_shot == false && d.get_k() < 10 && d.get_distance() < 25*25)
					{
						keys.hyperspace(true);
						break;
					}
				}
			}

			if ( (shot_taken == 0 || n_shots_taken < 5) && game_move.numvals > 8) {
			//if ( game_move.numvals > 8) {
				
				if (shot_taken == 0) {
					n_shots_taken = 0;
				}
				//shot colliding asteroids first:
				if (closest_colliding_asteroid != 0){
					closest_asteroid = closest_colliding_asteroid;
				}
				MovingObject closest_object = game_move.asteroids[closest_asteroid];
				Distance dist = Distance( game.asteroids[closest_asteroid].x, game.asteroids[closest_asteroid].y, game.ship_x, game.ship_y);
				int must_distance = 100*100; //min distance to asteroid
				if ((closest_colliding_asteroid != 0) && (closest_object.dx*closest_object.dx)+(closest_object.dy*closest_object.dy) > 16) must_distance = 200*200;
				bool shot_ufo = false;
				if ((game.saucer_present && game.nasteroids == 0)||(game.saucer_present && dist_saucer < 600 && dist.get_distance() > must_distance))
				{
					closest_object = game_move.saucer; //shot UFO, if it is closer than 600px and any asteroid is more than 100px away
					shot_ufo = true;
				}

				//Calculation for closest asteroid:
				//asteroid position duration (default 40) time steps in future:
				int astx = closest_object.x + closest_object.dx * duration;
				int asty = closest_object.y + closest_object.dy * duration;
//				if (shot_ufo) {
//					astx = closest_object.x + closest_object.dx10 * duration;
//					asty = closest_object.y + closest_object.dy10 * duration;					
//				}
			
			//vector to our position
			int rot_dx = astx - game.ship_x;
			int rot_dy = asty - game.ship_y;
			
			//find appropriate rotation vector
			float min_deviation = 1200;
			int min_rotx = 0;
			int min_roty = 0;
			int signx=1;
			int signy=1;
			if (rot_dx < 0) {signx=-1;}
			if (rot_dy < 0) {signy=-1;}
			
			for (int r1=0; r1<NROT_VECTS; ++r1)
			{
				int r2 = NROT_VECTS - r1 -1;
				int rotvx = signx*ROTATION_VECTS[r1];
				int rotvy = signy*ROTATION_VECTS[r2];
				
				//check vector
				float factor_x = (float)rotvx / rot_dx;
				float factor_y = (float)rotvy / rot_dy;
				float deviation = fabs(factor_x - factor_y);
				
				if ( deviation < min_deviation )
				{
					min_deviation = deviation;
					min_rotx = rotvx;
					min_roty = rotvy;
				}
					
			}
			
			//how long will a shot take?
//			shot: 
			int length = (int)sqrt(rot_dx*rot_dx + rot_dy*rot_dy);
			int shot_steps = 1 + ((length - 19) / 8); //delay 1, start 19, per timestep 8
			
//			rotation:
			int now_index=get_index(game.ship_dx);
			int target_index=get_index(min_rotx);
			int spin_steps=abs(now_index-target_index);
			spin_steps += spin_steps / 3; //add 1 per 3 steps
			if (t<16) spin_steps += 6-(t-10); //delay 6
						
			if (turn_counter < 2 && game.ship_dx != min_rotx)
			{
				if (last_dx == game.ship_dx){
					turn_counter++;
				}
				//rotation needed
				if (game.ship_dx * min_roty - game.ship_dy * min_rotx > 0)
					keys.left(true);
				else
					keys.right(true);

				if (t % 12 == 0) keys.fire(true);
			} else {
				//shot
				if (duration <= shot_steps+5)
				{
					turn_counter=0;
					keys.fire(true);
					shot_taken = duration;
					n_shots_taken++;
				} else if (t % 12 == 0) keys.fire(true);
			}

			//print debug information
//			printf ("Berechnung:\n");
//			printf ("-----------------------\n");
//			printf ("astnr: %i\n", closest_asteroid);
//			printf ("co (x,y): (%i,%i)\n", closest_object.x, closest_object.y);
//			printf ("co +%i (x,y): (%i,%i)\n", duration, astx, asty);
//			printf ("dist_saucer: %i\n", dist_saucer);
//			printf ("dx,dy: %i,%i\n", rot_dx, rot_dy);
//			printf ("rot (x,y): (%i,%i)\n", min_rotx, min_roty);
//			printf ("deviation: %.3f\n", min_deviation);
//			printf ("now_index: %i\n", now_index);
//			printf ("target_index: %i\n", target_index);
//			printf ("steps: sum,shot,spin: %i,%i,%i\n", shot_steps+spin_steps, shot_steps, spin_steps);
			
			duration = shot_steps+spin_steps;
			last_dx = min_rotx;

			} else {
				shot_taken--;
			
				// Schiff in Richtung auf das nchstgelegene Objekt drehen
				// mathematisch wird hier das Kreuzprodukt aus den Vektoren 
				// ship_dx/y/0 und min_dx/y/0 berechnet
				
				if (game.ship_dx * min_dy - game.ship_dy * min_dx > 0){
					if (last_dx == 2)
						turn_counter++;
					else 
						turn_counter=0;
					if (turn_counter < 3) {
						last_dx=1; //save that we pressed left
						keys.left(true);
					}
					else
						turn_counter++;
				}
				else {
					if (last_dx == 1)
						turn_counter++;
					else
						turn_counter=0;
					if (turn_counter < 3) {
						last_dx=2; //save that we pressed right
						keys.right(true);
					}
					else
						turn_counter++;
				}
				if (turn_counter >= 40 || shot_taken == 0) turn_counter=0;
				
				if (min_dist > 400*400 && t % 40) // accelarate slowly if nothing is close
					keys.thrust(true);

				if (t % 2 == 0)  // fire as often as possible
					keys.fire(true);

			}

			//no ufo and everything else out of reach -> go to center of the screen:
			if (!game.saucer_present && min_dist > 640*640) {
				keys.clear();
				
				if (n_thrust < thrust_max) {

					int center_x = 524;
					int center_y = 524;
					int center_dx = center_x - game.ship_x;
					int center_dy = center_y - game.ship_y;
					Distance dist = Distance( center_x, center_y, game.ship_x, game.ship_y);
					int scalar_product = game.ship_dx * center_dy - game.ship_dy * center_dx; 

					if (t_empty_screen == 0) {
						t_empty_screen = t;
						if (dist.get_distance() > 500*500 ) {
							thrust_max = 35;
						} else if (dist.get_distance() > 300*300 ) {
							thrust_max = 23;
						} else if (dist.get_distance() > 200*200 ) {
							thrust_max = 17;
						} else if (dist.get_distance() > 100*100 ) {
							thrust_max = 10;
						} else {
							thrust_max = 5;
						} 
					}
					
					
					if (dist.get_distance() > 50*50 ){
						
						//rotate till center is in front:
						if ( abs(scalar_product) > 120000 || (center_dx*game.ship_dx < 0) || (center_dy*game.ship_dy < 0)){
							if ( scalar_product > 0){
								keys.left(true);
							} else { 
								keys.right(true);
							}
						} else {
							keys.thrust(true);
							n_thrust++;
						}
					}
				}
				
			} else if (t_empty_screen != 0) {
				//reset some values:
				n_thrust = 0;
				t_empty_screen = 0;
			}
			
			if (min_dist < 27*27)  // collision unavoidably -> escape
				keys.hyperspace(true);

			nast_last = game.nasteroids;

//			if ( t > 100) 
//			{
//				break;
//			}
		}
	}
}

int Player::get_index(int dx)
{
	if (dx < 0) dx = -dx;
	
	for (int i=0; i<NROT_VECTS; ++i)
	{
		if ( dx == ROTATION_VECTS[i] )
			return i;
	}
	
	return -1;
}

Vector::Vector (int x, int y) {
	this->x = x;
	this->y = y;
}

void Vector::set (int x, int y) {
	this->x = x;
	this->y = y;
}

MovingAsteroid::MovingAsteroid (void)
{
	MovingObject::MovingObject();
	type = 0;
	sf = 0;
}

void MovingAsteroid::clear (void)
{
	type = 0;
	sf = 0;
	MovingObject::clear();
}

void MovingAsteroid::copy (MovingAsteroid from)
{
	type = from.type;
	sf = from.sf;
	MovingObject::copy(from);
}

void MovingAsteroid::add(Asteroid a, int latency)
{
	if ( numvals == 0 ){
		this->type = a.type;
		this->sf   = a.sf;
	}

	MovingObject::add(a.x, a.y, latency);
}

MovingShot::MovingShot (void)
{
	MovingObject::MovingObject();
	self_shot = true;
}

void MovingShot::clear (void)
{
	self_shot = false;
	MovingObject::clear();
}

void MovingShot::copy (MovingShot from)
{
	self_shot = from.self_shot;
	MovingObject::copy(from);
}

void MovingShot::add(Shot s, MovingObject saucer, bool saucer_present, int latency)
{
	if ( saucer_present && numvals == 0 ){
		Distance dist = Distance( s.x, s.y, saucer.x, saucer.y);
		if (dist.get_distance() <= 30*30) {
			self_shot = false;
		}
	}

	MovingObject::add(s.x, s.y, latency);
}

MovingObject::MovingObject (void)
{
	x = 0;
	y = 0;
	dx = 0;
	dy = 0;
	x_all = 0;
	y_all = 0;
	x10 = 0;
	y10 = 0;
	dx10 = 0;
	dy10 = 0;
	numvals = 0;
}

void MovingObject::add(int x, int y, int latency)
{
	// avoid deviation:
	if (numvals > 300) {
		this->clear();
		numvals=0;
	}
	
	if ( numvals == 0 ){
		this->x = x;
		this->y = y;
		numvals++;
		//printf ("NEW: n=%i l=%i (x,y)=(%i,%i)\n", numvals, latency, x, y);
	}
	else {
		latency++; //1 is neutral
		x_all += (x - this->x);
		y_all += (y - this->y);
// #### the following code should give us the movement of the last 10 frames - doesnt work until now.. ####
//		dx_last_10[numvals % 10] = (x - this->x);
//		dy_last_10[numvals % 10] = (y - this->y);
//		x10 += (x - this->x);
//		y10 += (y - this->y);
//		if (numvals > 10){
//			int last_index = (numvals % 10) - 1;
//			if ( last_index < 0 ) last_index = 9;
//			x10 -= dx_last_10[last_index];
//			y10 -= dy_last_10[last_index];
//			int l = latency;
//			while (l > 1) {
//				dx_last_10[last_index + l] = 0;
//				dy_last_10[last_index + l] = 0;
//				l--;
//			}
//		}
// #### END: last 10 movement ####
		float new_dx = (float)x_all / (numvals + latency - 1);
		float new_dy = (float)y_all / (numvals + latency - 1);
		dx = new_dx;
		dy = new_dy;
//		dx10 = x10 / 10;
//		dy10 = y10 / 10;
		//printf ("n= %i l=%i (x,y)=(%i,%i) (nx,ny)=(%i,%i) (dx,dy)=(%.2f,%.2f)\n", numvals, latency, this->x, this->y, x, y, dx, dy);
		//printf ("n= %i l=%i (x,y)=(%i,%i) (nx,ny)=(%i,%i) (xall=%i,yall=%i) (dx,dy)=(%.2f,%.2f) (dxo,dyo)=(%.2f,%.2f)\n", numvals, latency, this->x, this->y, x, y, x_all, y_all, dx, dy, new_dx, new_dy);

		this->x = x;
		this->y = y;
		numvals+=latency;
	}
}

void MovingObject::copy(MovingObject from)
{
	x = from.x;
	y = from.y;
	dx = from.dx;
	dy = from.dy;
	x_all = from.x_all;
	y_all = from.y_all;
	x10 = from.x10;
	y10 = from.y10;
	dx10 = from.dx10;
	dy10 = from.dy10;
	numvals = from.numvals;
}

void MovingObject::clear(void)
{
	x = 0;
	y = 0;
	dx = 0;
	dy = 0;
	x_all = 0;
	y_all = 0;
	x10 = 0;
	y10 = 0;
	dx10 = 0;
	dy10 = 0;
	numvals = 0;
}

int MovingObject::propagate_x(int interval)
{
	return (int)(x + dx * interval + 0.5);
}

int MovingObject::propagate_y(int interval)
{
	return (int)(y + dy * interval + 0.5);
}

GameMove::GameMove (void)
{
	numvals = 0;
	reordered_asteroids = false;
	reordered_shots = false;
}

void GameMove::add(GameStatus game, int latency)
{

	ship.add(game.ship_x, game.ship_y, latency);
	if (game.saucer_present)
		saucer.add(game.saucer_x, game.saucer_y, latency);
	else 
		saucer.clear();
	
	if (!reordered_asteroids){
		for (int i=0; i<game.nasteroids; ++i)
		{   // Asteroidenbewegung berrechnen
			asteroids[i].add( game.asteroids[i], latency);
		}
	}

	if (!reordered_shots){
		for (int i=0; i<game.nshots; ++i)
		{   // Schussbewegung berrechnen
			shots[i].add( game.shots[i], saucer, game.saucer_present, latency);
		}
	}

	numvals++;
	nasteroids = game.nasteroids;
	nshots = game.nshots;
	reordered_asteroids = false;
	reordered_shots = false;
}

void GameMove::reorder_asteroids(GameStatus game, int latency)
{
	MovingAsteroid new_asteroids[MAX_ASTEROIDS];
	bool found[nasteroids];
	
	for (int i=0; i<nasteroids; ++i)
	{	
		found[i] = false;
	}

	//find asteroids in old list:
	for (int i=0; i<game.nasteroids; ++i){
		//printf ("searching (%i,%i)..\n", game.asteroids[i].x, game.asteroids[i].y);
		for (int j=0; j<nasteroids; ++j){
			//suche asteroid i:
			if (found[j] || game.asteroids[i].type != asteroids[j].type || game.asteroids[i].sf != asteroids[j].sf) 
				continue;
			//printf ("comparing (%i,%i)..\n", asteroids[j].x, asteroids[j].y);
			int diff_x = abs((game.asteroids[i].x - (int)(asteroids[j].x + asteroids[j].dx * (latency+1) + 0.5)));
			int diff_y = abs((game.asteroids[i].y - (int)(asteroids[j].y + asteroids[j].dy * (latency+1) + 0.5)));
			//printf ("calc      (%i,%i),(%i,%i)..\n", (int)(asteroids[j].x + asteroids[j].dx * (latency+1) + 0.5), (int)(asteroids[j].y + asteroids[j].dy * (latency+1) + 0.5), diff_x, diff_y);
			if ( diff_x <= 1 && diff_y <= 1 ){
				//printf ("found i=%i as j=%i\n", i, j);
				new_asteroids[i].copy(asteroids[j]);
				new_asteroids[i].add( game.asteroids[i], latency);
				found[j] = true;
				break;
			}
		}
		if (new_asteroids[i].numvals == 0) {
			//asteroid not found, use i:
			new_asteroids[i].add( game.asteroids[i], latency);
			//printf ("inserted i=%i..\n", i);
		}
	}
	//copy new asteroid list:
	for (int i=0; i<game.nasteroids; ++i){
		asteroids[i].copy(new_asteroids[i]);
	}
	
	nasteroids=game.nasteroids;
	reordered_asteroids = true;
}

void GameMove::reorder_shots(GameStatus game, int latency)
{
	MovingShot new_shots[MAX_SHOTS];
	bool found[nasteroids];
	
	for (int i=0; i<nshots; ++i)
	{	
		found[i] = false;
	}

	for (int i=0; i<game.nshots; ++i)
	{   // find shots in old list
		//printf ("searching (%i,%i)..\n", game.shots[i].x, game.shots[i].y);
		for (int j=0; j<nshots; ++j){
			//suche asteroid i:
			if (found[j]) continue;
			//printf ("comparing (%i,%i)..\n", shots[j].x, shots[j].y);
			int diff_x = abs((game.shots[i].x - (int)(shots[j].x + shots[j].dx * (latency+1) + 0.5)));
			int diff_y = abs((game.shots[i].y - (int)(shots[j].y + shots[j].dy * (latency+1) + 0.5)));
			//printf ("calc      (%i,%i),(%i,%i)..\n", (int)(shots[j].x + shots[j].dx * (latency+1) + 0.5), (int)(shots[j].y + shots[j].dy * (latency+1) + 0.5), diff_x, diff_y);
			if ((diff_x <= 4 && diff_y <= 4)||(diff_x <= 4 && diff_y > 1000)||(diff_x > 1000 && diff_y <= 4)){
				//printf ("found i=%i as j=%i\n", i, j);
				new_shots[i].copy(shots[j]);
				new_shots[i].add( game.shots[i], saucer, game.saucer_present, latency);
				found[j] = true;
				break;
			}
		}
		if (new_shots[i].numvals == 0) {
			//shot not found, use i:
			new_shots[i].add( game.shots[i], saucer, game.saucer_present, latency);
			//printf ("inserted i=%i..\n", i);
		}
	}
	//copy new asteroid list:
	for (int i=0; i<game.nshots; ++i){
		shots[i].copy(new_shots[i]);
	}
	nshots=game.nshots;
	reordered_shots = true;
}

void GameMove::clear_asteroids(void)
{
	for (int i=0; i<nasteroids; ++i)
	{   // clear asteroid moves
		asteroids[i].clear();
	}	
}

void GameMove::clear_shots(void)
{
	for (int i=0; i<MAX_SHOTS; ++i)
	{   // clear shot moves
		shots[i].clear();
	}
}


void GameMove::clear(void)
{
	ship.clear();
	saucer.clear();
	
	clear_asteroids();
	clear_shots();
	
	numvals=0;
	nasteroids=0;
}

void GameMove::print(void)
{
	printf ("Bewegungen:\n");
	printf ("-----------\n");

	printf ("Werte: %i\n", numvals);
	printf ("ship: (%.2f,%.2f)\n", ship.dx, ship.dy);	
	printf ("saucer: (%.2f,%.2f)\n", saucer.dx, saucer.dy);
	
	printf ("asteroids:\n");

	for (int i=0; i<nasteroids; ++i)
	{
		printf ("ast[%i] (%.2f,%.2f)\n", i, asteroids[i].dx, asteroids[i].dy);
	}

	printf ("shots:\n");
	for (int i=0; i<nshots; ++i)
	{
		printf ("shot[%i] (%.2f,%.2f)\n", i, shots[i].dx, shots[i].dy);
	}

	printf ("\n");
}

void GameMove::print_future(GameStatus game, int interval)
{
	printf ("Bildschirm in der Zukunft:\n");
	printf ("--------------------------\n");

	printf ("Werte: %i\n", numvals);
	printf ("Frames: +%i\n", interval);
	printf ("ship: (%i,%i)\n", ship.propagate_x(interval), ship.propagate_y(interval));	
	printf ("saucer: (%i,%i)\n", saucer.propagate_x(interval), saucer.propagate_y(interval));	
	
	printf ("asteroids:\n");

	for (int i=0; i<4; ++i)
	{
		//Koordinaten sind identisch:
		printf ("ast[%i] (%i,%i)\n", i, asteroids[i].propagate_x(interval), asteroids[i].propagate_y(interval));
	}

	printf ("shots:\n");
	for (int i=0; i<2; ++i)
	{
		printf ("shot[%i] (%i,%i)\n", i, shots[i].propagate_x(interval), shots[i].propagate_y(interval));
	}

	printf ("\n");
}

/* Abstand zwischen zwei Mittelpunkten berechnen */
Distance::Distance(int x1, int y1, int x2, int y2)
{
	dx = x1 - x2;
	dy = y1 - y2;
	dist = dx*dx+dy*dy;  // Quadrat des Abstands zu diesem Asteroiden

}

void Distance::correct_distance(int length)
{
	dist -= length;
}

int Distance::get_distance(void)
{
	return this->dist;
}

int Distance::get_dx(void)
{
	return this->dx;
}

int Distance::get_dy(void)
{
	return this->dy;
}

DistStraightPoint::DistStraightPoint(Vector p, MovingObject o){

	dist = 1000000; //some big value
	k = 1000; //some big value
	
	if ( o.numvals >= 10 ){
		//calculate point q on the plane
		float k = (p.x*o.dx - o.x*o.dx + p.y*o.dy - o.y*o.dy) / (o.dx*o.dx + o.dy*o.dy);
		if ( k > 0 ) {
			this->k = (int)(k + 0.5); 
			Vector q = Vector((int)(o.x + (o.dx * k)), (int)(o.y + (o.dy * k)));
	
			//get distance between q and p
			Distance d = Distance(p.x, p.y, q.x, q.y);
			dist = d.get_distance();
			//if (dist <= 27*27) printf ("px=%i, py=%i, ax=%i, ay=%i, adx=%.2f, ady=%.2f, qx=%i, qy=%i, d=%i\n", p.x, p.y, a.x, a.y, a.dx, a.dy, q.x, q.y, dist);
		} else {
			//wrong direction..
			Distance d = Distance(p.x, p.y, o.x, o.y);
			dist = d.get_distance();			
		}
	} else {
		//not enough values -> use distance between a and p..
		Distance d = Distance(p.x, p.y, o.x, o.y);
		dist = d.get_distance();
	}
}

int DistStraightPoint::get_asteroid_distance(MovingAsteroid a)
{
	switch (a.sf)
	{	// correct distance by radius of asteroid
		case 0:  // big asteroid
			dist-=40*40;
			break;
		case 15: // middle asteroid
			dist-=20*20;
			break;
		case 14: // small asteroid
			dist-=8*8;
			break;
	}
	if (dist < 0) dist=0;
	return this->dist;
}

int DistStraightPoint::get_distance(void)
{
	return this->dist;
}

int DistStraightPoint::get_k(void)
{
	return this->k;
}

void Player::InterpretScreen(FramePacket &packet, GameStatus& game)
{
	unsigned short vector_ram[512];
	int dx, dy, sf, vx, vy, vz, vs;
	int v1x = 0;
	int v1y = 0;
	int shipdetect = 0;

	game.clear();

	/* Vektor-RAM in 16-Bit-Worte konvertieren. War in der ersten Version mal ein sportlicher
	Typecast: unsigned short *vector_ram = (unsigned short*)packet.vectorram;
	Das klappt aber nur auf Little-Endian-Systemen, daher besser portabel: */
	for (int i=0; i<512; ++i)
		vector_ram[i] = (unsigned char)packet.vectorram[2*i] | (unsigned char)packet.vectorram[2*i+1] << 8;

	if (vector_ram[0] != 0xe001 && vector_ram[0] != 0xe201)
		return; // sollte nicht vorkommen; erster Befehl ist immer ein JMPL

	int pc = 1;
	while (pc < 512)
	{
		int op = vector_ram[pc] >> 12;
		switch (op)
		{
		case 0xa: // LABS
			vy = vector_ram[pc] & 0x3ff;
			vx = vector_ram[pc+1] & 0x3ff;
			vs = vector_ram[pc+1] >> 12;
			break;
		case 0xb: // HALT
			return;
		case 0xc: // JSRL
			switch (vector_ram[pc] & 0xfff)
			{
			case 0x8f3:
				game.asteroids[game.nasteroids++].set(vx, vy, 1, vs);
				break;
			case 0x8ff:
				game.asteroids[game.nasteroids++].set(vx, vy, 2, vs);
				break;
			case 0x90d:
				game.asteroids[game.nasteroids++].set(vx, vy, 3, vs);
				break;
			case 0x91a:
				game.asteroids[game.nasteroids++].set(vx, vy, 4, vs);
				break;
			case 0x929:
				game.saucer_present = true;
				game.saucer_x = vx;
				game.saucer_y = vy;
				game.saucer_size = vs;
				break;
			}  
			break;
		case 0xd: // RTSL
			return;
		case 0xe: // JMPL
			/*
			pc = vector_ram[pc] & 0xfff;
			break;
			*/
			return;
		case 0xf: // SVEC
			/*
			dy = vector_ram[pc] & 0x300;
			if ((vector_ram[pc] & 0x400) != 0)
				dy = -dy;
			dx = (vector_ram[pc] & 3) << 8;
			if ((vector_ram[pc] & 4) != 0)
				dx = -dx;
			sf = (((vector_ram[pc] & 8) >> 2) | ((vector_ram[pc] & 0x800) >> 11)) + 2;
			vz = (vector_ram[pc] & 0xf0) >> 4;
			*/
			break;
		default:
			dy = vector_ram[pc] & 0x3ff;
			if ((vector_ram[pc] & 0x400) != 0)
				dy = -dy;
			dx = vector_ram[pc+1] & 0x3ff;
			if ((vector_ram[pc+1] & 0x400) != 0)
				dx = -dx;
			sf = op;
			vz = vector_ram[pc+1] >> 12;
			if (dx == 0 && dy == 0 && vz == 15)
				game.shots[game.nshots++].set(vx, vy);
			if (op == 6 && vz == 12 && dx != 0 && dy != 0)
			{
				switch (shipdetect)
				{
				case 0:
					v1x = dx;
					v1y = dy;
					++shipdetect;
					break;
				case 1:
					game.ship_present = true;
					game.ship_x = vx;
					game.ship_y = vy;
					game.ship_dx = v1x - dx;
					game.ship_dy = v1y - dy;
					++shipdetect;
					break;
				}
			}
			else if (shipdetect == 1)
				shipdetect = 0;

			break;
		}
		if (op <= 0xa)
			++pc;
		if (op != 0xe) // JMPL
			++pc;
	}   

}


void Asteroid::set(int x, int y, int type, int sf)
{
	this->x = x;
	this->y = y;
	this->type = type;
	this->sf = sf;
}

void Shot::set(int x, int y)
{
	this->x = x;
	this->y = y;
}

void GameStatus::clear(void)
{
	ship_present = false;
	saucer_present = false;
	nasteroids = 0;
	nshots = 0;
}

void GameStatus::print(int time, char frameno)
{
	printf ("(%i) Neuer Bildschirm:\n", time);
	printf ("-----------------------\n");
	printf ("-----------------------\n");
	printf ("frame: %i\n", frameno);  // Frame
	printf ("ship_present: %i\n", ship_present);  // Schiff sichtbar
	printf ("ship_x: %i\n", ship_x);         // Mittelpunkt des Schiffs
	printf ("ship_y: %i\n", ship_y);
//	printf ("ship (dx,dy) (ddx, ddy): (%i,%i) (%i,%i)\n", ship_dx, ship_dy, ship_dx - ship_last_dx, ship_dy - ship_last_dy);        // Blickrichtung des Schiffes
	printf ("ship_dx: %i\n", ship_dx);        // Blickrichtung des Schiffes
	printf ("ship_dy: %i\n", ship_dy);
	printf ("saucer_present: %i\n", saucer_present);// UFO sichtbar
	printf ("saucer_x: %i\n", saucer_x);       // Mittelpunkt des UFOs
	printf ("saucer_y: %i\n", saucer_y);
	printf ("saucer_size: %i\n", saucer_size);    // Groesse: 15 = gross, 14 = klein
	printf ("asteroids: %i\n", nasteroids); // Anzahl Asteroiden
	printf ("-----------------\n");

	for (int i=0; i<nasteroids; ++i)
	{   // Asteroiden kopieren
		printf ("ast[%i] (%i,%i,%i,%i)\n", i, asteroids[i].x, asteroids[i].y, asteroids[i].type, asteroids[i].sf);
	}

	printf ("shots: %i\n", nshots);     // Anzahl Schsse
	printf ("-----------------\n");
	for (int i=0; i<nshots; ++i)
	{   // Asteroiden kopieren
		printf ("shot[%i] (%i,%i)\n", i, shots[i].x, shots[i].y);
	}

	printf ("\n");

}

void GameStatus::print_diff(GameStatus game_b)
{
	printf ("Unterschiede:\n");
	printf ("-------------\n");

	if (ship_present == true)
	{
		printf ("ship: (%i,%i)\n", game_b.ship_x - ship_x, game_b.ship_y - ship_y);         // Mittelpunkt des Schiffs
	}
	
	if (saucer_present == true)
	{
		printf ("saucer: (%i,%i)\n", game_b.saucer_x - saucer_x, game_b.saucer_y - saucer_y);         // Mittelpunkt des Schiffs
	}
	
	printf ("asteroids:\n"); // Anzahl Asteroiden

	for (int i=0; i<nasteroids; ++i)
	{   // Asteroiden kopieren
		printf ("ast[%i] (%i,%i)\n", i, game_b.asteroids[i].x - asteroids[i].x, game_b.asteroids[i].y - asteroids[i].y);
	}

	printf ("shots:\n");     // Anzahl Schsse
	for (int i=0; i<nshots; ++i)
	{   // Asteroiden kopieren
		printf ("shot[%i] (%i,%i)\n", i, game_b.shots[i].x - shots[i].x, game_b.shots[i].y - shots[i].y);
	}

	printf ("\n");
}

GameStatus GameStatus::clone(void)
{
	GameStatus clone;

	clone.ship_present = ship_present;  // Schiff sichtbar
	clone.ship_x = ship_x;         // Mittelpunkt des Schiffs
	clone.ship_y = ship_y;
	clone.ship_dx = ship_dx;        // Blickrichtung des Schiffes
	clone.ship_dy = ship_dy;
	clone.saucer_present = saucer_present;// UFO sichtbar
	clone.saucer_x = saucer_x;       // Mittelpunkt des UFOs
	clone.saucer_y = saucer_y;
	clone.saucer_size = saucer_size;    // Gre: 15 = gro, 14 = klein
	clone.nasteroids = nasteroids; // Anzahl Asteroiden
	clone.nshots = nshots;     // Anzahl Schsse

	for (int i=0; i<nasteroids; ++i)
	{   // Asteroiden kopieren
		clone.asteroids[i].x = asteroids[i].x;
		clone.asteroids[i].y = asteroids[i].y;
		clone.asteroids[i].type = asteroids[i].type;
		clone.asteroids[i].sf = asteroids[i].sf;
	}
	
	for (int i=0; i<nshots; ++i)
	{   // Schuesse kopieren
		clone.shots[i].x = shots[i].x;
		clone.shots[i].y = shots[i].y;
	}
	
	return clone;
}

KeysPacket::KeysPacket(void)
{
	signature[0] = 'c';
	signature[1] = 't';
	signature[2] = 'm';
	signature[3] = 'a';
	signature[4] = 'm';
	signature[5] = 'e';
	keys = '@';
	ping = 0;
}

void KeysPacket::clear(void)
{
	keys = '@';
}

void KeysPacket::hyperspace(bool b)
{
	if (b)
		keys |= KEY_HYPERSPACE;
	else
		keys &= ~KEY_HYPERSPACE;
}

void KeysPacket::fire(bool b)
{
	if (b)
		keys |= KEY_FIRE;
	else
		keys &= ~KEY_FIRE;
}

void KeysPacket::thrust(bool b)
{
	if (b)
		keys |= KEY_THRUST;
	else
		keys &= ~KEY_THRUST;
}

void KeysPacket::left(bool b)
{
	if (b)
	{
		keys |= KEY_LEFT;
		right(false);
	}
	else
		keys &= ~KEY_LEFT;
}

void KeysPacket::right(bool b)
{
	if (b)
	{
		keys |= KEY_RIGHT;
		left(false);
	}
	else
		keys &= ~KEY_RIGHT;
}

void Player::ReceivePacket(FramePacket &packet)
{
	sockaddr_in sender;
	int sender_size = sizeof sender;
	fd_set readfds, writefds, exceptfds;

	do
	{
		FD_ZERO(&readfds);
		FD_ZERO(&writefds);
		FD_ZERO(&exceptfds);
		FD_SET(sd, &readfds);
		FD_SET(sd, &exceptfds);
		select(sd+1, &readfds, &writefds, &exceptfds, NULL);
		int bytes_received = recv(sd, (char *)&packet, sizeof packet, 0);
		if (bytes_received != sizeof packet)
		{
			int err = WSAGetLastError();
			fprintf(stderr, "Fehler %d bei recvfrom().\n", err);
			exit(1);
		}
		FD_ZERO(&readfds);
		FD_ZERO(&writefds);
		FD_ZERO(&exceptfds);
		FD_SET(sd, &readfds);
		timeval zero;
		zero.tv_sec = zero.tv_usec = 0;
		select(sd+1, &readfds, &writefds, &exceptfds, &zero);
	} while(FD_ISSET(sd, &readfds));
}

void Player::SendPacket(KeysPacket &packet)
{
	sockaddr_in server;
	memset(&server, 0, sizeof server);
	server.sin_family = AF_INET;
	server.sin_port = htons(1979);
	server.sin_addr.s_addr = server_ip;
	if (sizeof packet != sendto(sd, (char *)&packet, sizeof packet, 0, (sockaddr*)&server, sizeof server))
	{
#if defined(WINDOWS)
		int err = WSAGetLastError();
		if (err != WSAEWOULDBLOCK)
		{
			fprintf(stderr, "Fehler %d bei sendto().\n", err);
			exit(1);
		}
#else
		if (errno != EAGAIN)
		{
			perror("Fehler bei sendto()");
			exit(1);
		}
#endif
	}
}
