using System;
using System.IO;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading;

namespace PlayerLogic
{
    public class PlayerWorker
    {
        private const int KEYIDX = 6;
        private const int PINGIDX = 7;
        private const byte KEY_HYPERSPACE = 1;
        private const byte KEY_FIRE = 2;
        private const byte KEY_THRUST = 4;
        private const byte KEY_RIGHT = 8;
        private const byte KEY_LEFT = 0x10;
        private const byte KEY_START = 0x20;

        private DateTime startTime;
        private long frameCounter;
        private readonly FeedbackTarget feedback;
        private string address;
        private readonly GameStatus gameState;
        private Utility.Interval angleInterval;
        private Socket socket;
        private EndPoint localEndpoint;
        private IPEndPoint serverEndpoint;
        private IPEndPoint receiveEndpoint;

        private volatile bool serverResponding;
        public bool ServerResponding
        {
            get { return serverResponding; }
        }

        private volatile bool connected;
        public bool Connected
        {
            get { return connected; }
        }

        private volatile bool loop;
        public bool Loop
        {
            get { return loop; }
            set { loop = value; }
        }

        private volatile bool stop;
        private volatile bool busy;
        private bool nameSend;
        public enum State { NONE, SELECT, PREGAME, RUNNING, GAMEOVER, ENTERHIGHSCORE };
        private State currentState = State.NONE;

        private string textContent;

        public long ReadScore
        {
            get
            {
                if (currentState == State.RUNNING)
                {
                    int tempScore;
                    if (int.TryParse(textContent.Substring(0, 5), out tempScore))
                        return tempScore;
                }
                return gameState.score;
            }
        }

        private long frameLimit;
        public long FrameLimit
        {
            get { return frameLimit; }
            set { frameLimit = value; }
        }

        private Behaviour behaviour;
        private Behaviour newBehaviour;
        private static object behaviourLocker = new object();
	    public Behaviour PlayerBehaviour
	    {
		    get { return behaviour; }
            set { lock (behaviourLocker) { newBehaviour = value; } }
	    }

        public PlayerWorker(FeedbackTarget feedback)
        {
            startTime = DateTime.Now;
            frameCounter = 0;
            frameLimit = -1;
            this.feedback = feedback;
            gameState = new GameStatus();
            angleInterval = new Utility.Interval(253, 3);
            serverResponding = false;
            connected = false;
            loop = false;
            stop = false;
            busy = true;
            nameSend = false;
            textContent = "";
        }

        public void Connect(string address, int port)
        {
            this.address = address;
            localEndpoint = new IPEndPoint(IPAddress.Any, 0);
            IPAddress ipAddr;
            if (!IPAddress.TryParse(address, out ipAddr))
            {
                IPAddress.TryParse("127.0.0.1", out ipAddr);
            }
            serverEndpoint = new IPEndPoint(ipAddr, 1979);
            receiveEndpoint = new IPEndPoint(ipAddr, 1979);
            socket = new Socket(AddressFamily.InterNetwork, SocketType.Dgram, ProtocolType.Udp);
            socket.ReceiveBufferSize = 1200;
            socket.Blocking = true;
            socket.Bind(localEndpoint);
            connected = true;
        }

        public void Disconnect()
        {
            connected = false;
            while (currentState != State.NONE) Thread.Sleep(100);
            socket.Shutdown(SocketShutdown.Both);
            socket.Close();
        }

        public void Stop()
        {
            stop = true;
        }

        public bool Busy
        {
            get { return busy; }
        }

        public void Run()
        {
            byte[] frame = new byte[1050];
            byte[] keys = new byte[8];
            (new ASCIIEncoding()).GetBytes("ctmame").CopyTo(keys, 0);
            keys[KEYIDX] = (byte)'@';
            keys[PINGIDX] = 0;
            byte keysIn2 = 0;
            byte keysIn1 = 0;


            byte prevFrame = 0;
            int t = 0;

            while (!stop)
            {
                lock (behaviourLocker)
                {
                    if (behaviour != newBehaviour)
                    {
                        behaviour = newBehaviour;
                    }
                }

                if (!connected)
                {
                    currentState = State.NONE;
                    Thread.Sleep(1000);
                    continue;
                }
                t++;
                keys[PINGIDX]++;
                socket.SendTo(keys, 8, SocketFlags.None, serverEndpoint);
                keysIn1 = keysIn2;
                keysIn2 = keys[KEYIDX];
                keys[KEYIDX] = (byte)'@'; // release keys

                feedback.UpdateView(gameState);

                StateControl();
                if (currentState == State.SELECT && (t / 60) % 2 == 0)
                    keys[KEYIDX] |= KEY_START;
                else if (currentState != State.RUNNING && currentState != State.PREGAME && (t / 60) % 2 == 0)
                    keys[KEYIDX] |= KEY_HYPERSPACE;

                if (!ReceiveFrame(frame))
                    continue;

                int frameFactor;
                if (frame[1024] != ++prevFrame || frame[1025] != keys[PINGIDX])
                {
                    // frames lost!!!
                    keysIn1 = 0;
                    keysIn2 = 0;
                    frameFactor = frame[1024] - prevFrame + 1;
                    while (frameFactor < 0) frameFactor += 256;
                    if (frameFactor == 0)
                        frameFactor = 1;
                    prevFrame = frame[1024];
                }
                else
                {
                    frameFactor = 1;
                }
                frameCounter += frameFactor;
                gameState.frameNumber = frameCounter;
                

                InterpretFrame(frame, gameState, frameFactor);

                if (gameState.hasShip)
                {
	                if ((keysIn1 & KEY_LEFT) != 0)
	                {
	                    angleInterval.Shift(-3);
	                }
                    else if ((keysIn1 & KEY_RIGHT) != 0)
                    {
                        angleInterval.Shift(3);
                    }
                    angleInterval.Intersect(Utility.AngleIntervalFromShipRotation(gameState.shipRX, gameState.shipRY));
                    gameState.shipAngle = angleInterval.Value;
                    double w = (double)(256 - gameState.shipAngle) * Utility.PI128;
                    gameState.shipAX = (float)Math.Cos(w);
                    gameState.shipAY = (float)Math.Sin(w);

                    gameState.shipExpectedAngle = gameState.shipAngle;
                    if ((keysIn2 & KEY_LEFT) != 0)
                    {
                        gameState.shipExpectedAngle -= 3;
                    }
                    else if ((keysIn2 & KEY_RIGHT) != 0)
                    {
                        gameState.shipExpectedAngle += 3;
                    }

                    w = (double)(256 - gameState.shipExpectedAngle) * Utility.PI128;
                    gameState.shipEX = (float)Math.Cos(w);
                    gameState.shipEY = (float)Math.Sin(w);

                }

                if (behaviour != null && currentState == State.RUNNING)
                {
                	keys[KEYIDX] |= behaviour.AnalyzeFrame(gameState, frameFactor, feedback);
                }
            }
            busy = false;

            currentState = State.NONE;
            feedback.AcknowledgeExit();
        }

        private bool ReceiveFrame(byte[] frame)
        {
//            ArrayList readList = new ArrayList();
//            readList.Add(socket);
//            Socket.Select(readList, null, null, 500000);
            if (socket.Poll(500000, SelectMode.SelectRead))
            {
                EndPoint recvEndpoint = receiveEndpoint;
                try
                {
                    do 
                    {
	                    if (socket.ReceiveFrom(frame, ref recvEndpoint) < 1026)
	                    {
	                        string frameString = Encoding.UTF8.GetString(frame);
	                        if (frameString.Substring(0, 5) == "busy ")
	                        {
	                            int frames;
	                            int.TryParse(frameString.Substring(5, 5), out frames);
	                            Console.WriteLine("Server busy, waiting {0} seconds...", (frames*17/1000)+1);
	                            Thread.Sleep(17 * frames + 1000);
	                        }
	                        else if (frameString.Substring(0, 9) == "game over")
	                        {
	                            Console.WriteLine("Game over, score: {0}", gameState.score); 
	                            Stop();
	                        }
	                        else
	                        {
	                            serverResponding = false;
	                        }
	                        return false;
	                    }
	                    else
	                    {
	                        if (!nameSend)
	                        {
	                            byte[] ctname = new byte[38];
	                            for (int i = 0; i < 38; ++i)
	                                ctname[i] = 0;
	                            (new ASCIIEncoding()).GetBytes("ctnameAlpha-Gulrak").CopyTo(ctname, 0);
	                            socket.SendTo(ctname, 38, SocketFlags.None, serverEndpoint);
	                            nameSend = true;
	                        }
	                    }
                    } while (socket.Poll(0, SelectMode.SelectRead));
                }
                catch (System.Exception)
                {
                    currentState = State.NONE;
                    Thread.Sleep(1000);
                    serverResponding = false;
                    return false;
                }
                serverResponding = true;
            }
            return true;
        }

        private void StateControl()
        {
            switch(currentState)
            {
                case State.NONE:
                    gameState.score = 0;
                    frameCounter = 0;
                    if (textContent.IndexOf("STARTKN0EPFE") != -1)
                        currentState = State.SELECT;
                    else if (gameState.hasShip)
                        currentState = State.RUNNING;
                    if (textContent.IndexOf("BUCHSTABENWAHL") != -1)
                        currentState = State.ENTERHIGHSCORE;
                    break;
                case State.SELECT:
                    if (textContent.IndexOf("SPIELER 1") != -1)
                    {
                        currentState = State.PREGAME;
                        Console.WriteLine("Game starting...");
                    }
                    break;
                case State.PREGAME:
                    if (gameState.hasShip)
                    {
                        gameState.score = 0;
                        frameCounter = 0;
                        startTime = DateTime.Now;
                        currentState = State.RUNNING;
                    }
                    break;
                case State.RUNNING:
                    if ((frameLimit >0 && frameCounter >= frameLimit) || (!gameState.hasShip && textContent.IndexOf("SPIELENDE") != -1))
                    {
                        try
                        {
                            StreamWriter file = new StreamWriter("asteroid_games.log", true);
                            file.WriteLine("{0} - {1} min, {2} frames - Score: {3}", startTime.ToString("yyyyMMdd-HHmm"), DateTime.Now.Subtract(startTime).Minutes, frameCounter, gameState.score);
                            file.Close();
                        }
                        catch (System.Exception)
                        {
                            // do nothing on logfile error
                        }

                        Console.WriteLine("Game over, score: {0}", gameState.score);
                        currentState = State.GAMEOVER;
                        if (!loop)
                        {
                            Stop();
                        }
                    }
                    break;
                case State.GAMEOVER:
                    if (textContent.IndexOf("BUCHSTABENWAHL") != -1)
                    {
                        currentState = State.ENTERHIGHSCORE;
                    }
                    else if (textContent.IndexOf("STARTKN0EPFE") != -1)
                    {
                        currentState = State.SELECT;
                    }
                    break;
                case State.ENTERHIGHSCORE:
                    if (textContent.IndexOf("STARTKN0EPFE") != -1)
                    {
                        currentState = State.SELECT;
                    }
                    break;
            }
        }

        public void InterpretFrame(byte[] frame, GameStatus game, int frameFactor)
        {
            ushort[] vector_ram = new ushort[512];
            int dx = 0, dy = 0, sf = 0, vx = 0, vy = 0, vz = 0, vs = 0;
            int v1x = 0;
            int v1y = 0;
            int shipdetect = 0;

            textContent = "";
            game.Clear();

            /* Vektor-RAM in 16-Bit-Worte konvertieren. War in der ersten Version mal ein sportlicher
            Typecast: unsigned short *vector_ram = (unsigned short*)packet.vectorram;
            Das klappt aber nur auf Little-Endian-Systemen, daher besser portabel: */
            for (int i = 0; i < 512; ++i)
                vector_ram[i] = (ushort) (frame[2*i] | frame[2*i + 1] << 8);

            if (vector_ram[0] != 0xe001 && vector_ram[0] != 0xe201)
                return; // sollte nicht vorkommen; erster Befehl ist immer ein JMPL

            int pc = 1;
            while (pc < 512)
            {
                int op = vector_ram[pc] >> 12;
                switch (op)
                {
                    case 0xa: // LABS
                        vy = vector_ram[pc] & 0x3ff;
                        vx = vector_ram[pc + 1] & 0x3ff;
                        vs = vector_ram[pc + 1] >> 12;
                        break;
                    case 0xb: // HALT
                        pc = 512;
                        break;
                    case 0xc: // JSRL
                        switch (vector_ram[pc] & 0xfff)
                        {
                            case 0x8f3:
                                game.AddAsteroid(vx, vy, 1, vs, frameFactor);
                                break;
                            case 0x8ff:
                                game.AddAsteroid(vx, vy, 2, vs, frameFactor);
                                break;
                            case 0x90d:
                                game.AddAsteroid(vx, vy, 3, vs, frameFactor);
                                break;
                            case 0x91a:
                                game.AddAsteroid(vx, vy, 4, vs, frameFactor);
                                break;
                            case 0x929:
                                game.hasSaucer = true;
                                if (game.hadSaucer)
                                {
                                    game.saucer.HasDirection = true;
                                    game.saucer.DX = (vx - game.saucer.X) / frameFactor;
                                    game.saucer.DY = (vy - game.saucer.Y) / frameFactor;
                                    //game.saucerDX = game.saucerDX * 0.9f + (vx - game.saucerX) * 0.1f / frameFactor;
                                    //game.saucerDY = game.saucerDY * 0.9f + (vy - game.saucerY) * 0.1f / frameFactor;
                                }
                                else
                                {
                                    game.saucer.HasDirection = false;
                                    game.saucer.DX = 0;
                                    game.saucer.DY = 0;
                                }
                                game.saucer.X = vx;
                                game.saucer.Y = vy;
                                game.saucer.Size = vs;
                                break;

                            case 0xA78:
                                textContent += 'A';
                                break;
                            case 0xA80:
                                textContent += 'B';
                                break;
                            case 0xA8D:
                                textContent += 'C';
                                break;
                            case 0xA93:
                                textContent += 'D';
                                break;
                            case 0xA9B:
                                textContent += 'E';
                                break;
                            case 0xAA3:
                                textContent += 'F';
                                break;
                            case 0xAAA:
                                textContent += 'G';
                                break;
                            case 0xAB3:
                                textContent += 'H';
                                break;
                            case 0xABA:
                                textContent += 'I';
                                break;
                            case 0xAC1:
                                textContent += 'J';
                                break;
                            case 0xAC7:
                                textContent += 'K';
                                break;
                            case 0xACD:
                                textContent += 'L';
                                break;
                            case 0xAD2:
                                textContent += 'M';
                                break;
                            case 0xAD8:
                                textContent += 'N';
                                break;
                            case 0xADD:
                                textContent += '0';
                                break;
                            case 0xAE3:
                                textContent += 'P';
                                break;
                            case 0xAEA:
                                textContent += 'Q';
                                break;
                            case 0xAF3:
                                textContent += 'R';
                                break;
                            case 0xAFB:
                                textContent += 'S';
                                break;
                            case 0xB02:
                                textContent += 'T';
                                break;
                            case 0xB08:
                                textContent += 'U';
                                break;
                            case 0xB0E:
                                textContent += 'V';
                                break;
                            case 0xB13:
                                textContent += 'W';
                                break;
                            case 0xB1A:
                                textContent += 'X';
                                break;
                            case 0xB1F:
                                textContent += 'Y';
                                break;
                            case 0xB26:
                                textContent += 'Z';
                                break;
                            case 0xB2C:
                                textContent += ' ';
                                break;
                            case 0xB2E:
                                textContent += '1';
                                break;
                            case 0xB32:
                                textContent += '2';
                                break;
                            case 0xB3A:
                                textContent += '3';
                                break;
                            case 0xB41:
                                textContent += '4';
                                break;
                            case 0xB48:
                                textContent += '5';
                                break;
                            case 0xB4F:
                                textContent += '6';
                                break;
                            case 0xB56:
                                textContent += '7';
                                break;
                            case 0xB5B:
                                textContent += '8';
                                break;
                            case 0xB63:
                                textContent += '9';
                                break;
                        }
                        break;
                    case 0xd: // RTSL
                        return;
                    case 0xe: // JMPL
                        /*
                        pc = vector_ram[pc] & 0xfff;
                        break;
                        */
                        return;
                    case 0xf: // SVEC
                        /*
                        dy = vector_ram[pc] & 0x300;
                        if ((vector_ram[pc] & 0x400) != 0)
                            dy = -dy;
                        dx = (vector_ram[pc] & 3) << 8;
                        if ((vector_ram[pc] & 4) != 0)
                            dx = -dx;
                        sf = (((vector_ram[pc] & 8) >> 2) | ((vector_ram[pc] & 0x800) >> 11)) + 2;
                        vz = (vector_ram[pc] & 0xf0) >> 4;
                        */
                        break;
                    default:
                        dy = vector_ram[pc] & 0x3ff;
                        if ((vector_ram[pc] & 0x400) != 0)
                            dy = -dy;
                        dx = vector_ram[pc + 1] & 0x3ff;
                        if ((vector_ram[pc + 1] & 0x400) != 0)
                            dx = -dx;
                        sf = op;
                        vz = vector_ram[pc + 1] >> 12;
                        if (dx == 0 && dy == 0 && vz == 15)
                            game.AddBullet(vx, vy, frameFactor);
                        if (op == 6 && vz == 12 && dx != 0 && dy != 0)
                        {
                            switch (shipdetect)
                            {
                                case 0:
                                    v1x = dx;
                                    v1y = dy;
                                    ++shipdetect;
                                    break;
                                case 1:
                                    game.hasShip = true;
                                    game.shipDX = (vx - game.shipX) / frameFactor;
                                    game.shipDY = (vy - game.shipY) / frameFactor;
                                    game.shipX = vx;
                                    game.shipY = vy;
                                    float len = (float)Math.Sqrt((v1x - dx) * (v1x - dx) + (v1y - dy) * (v1y - dy));
                                    game.shipRX = (v1x - dx) / len;
                                    game.shipRY = (v1y - dy) / len;
                                    ++shipdetect;
                                    break;
                            }
                        }
                        else if (shipdetect == 1)
                            shipdetect = 0;

                        break;
                }
                if (op <= 0xa)
                    ++pc;
                if (op != 0xe) // JMPL
                    ++pc;
            }
            
            game.PostprocessStatus(frameFactor);

            if (currentState == State.RUNNING)
            {
                long newScore = (long)(game.score / 100000) * 100000 + ReadScore;
                game.score = game.score - newScore > 50000 ? newScore + 100000 : newScore;
            }
        }
    }
}
