package de.curdreinert.asteroids.intelligence;

import java.util.Arrays;
import java.util.Comparator;
import java.util.List;

import de.curdreinert.asteroids.base.Log;
import de.curdreinert.asteroids.base.Parameter;
import de.curdreinert.asteroids.base.Size;
import de.curdreinert.asteroids.base.Statistics;
import de.curdreinert.asteroids.geometry.Approximation;
import de.curdreinert.asteroids.geometry.Position;
import de.curdreinert.asteroids.geometry.Vector;
import de.curdreinert.asteroids.net.Command;
import de.curdreinert.asteroids.screenobject.Asteroid;
import de.curdreinert.asteroids.screenobject.Saucer;
import de.curdreinert.asteroids.screenobject.ScreenObject;
import de.curdreinert.asteroids.screenobject.Ship;
import de.curdreinert.asteroids.screenobject.Shot;

public class Analyzer {

	private static History history = new History();

	private Screen screen;

	private int framesToAsteroidHit = Parameter.getAsteroidForesight() + 1;

	private int framesToShotHit = Parameter.getShotForesight() + 1;

	private Asteroid nearestOnTrack;

	private static int lastLives;

	private String description;

	private Analysis analysis;

	private static Statistics statistics = null;

	private static boolean didShoot = false;

	public Analysis analyze() {
		if (screen.getShip() != null) {
			if (statistics == null) {
				statistics = new Statistics();
			}
			analyzeAll(analysis);
		}
		lastLives = screen.getLives();
		history.add(screen);
		setCommand();
		analysis.setDescription(description);
		return analysis;
	}

	public Analyzer(Screen screen) {
		this.analysis = new Analysis();
		this.screen = screen;
		analysis.setScreen(screen);
	}

	private void setCommand() {
		Command result = new Command();
		if (framesToAsteroidHit <= Parameter.getHyperspaceRange()
				|| framesToShotHit <= Parameter.getHyperspaceRange()) {
			result.pressHyper();
			statistics.hyperspacing();
			Log.info("Hyperspacing");
			analysis.setCommand(result);
			return;
		}

		ScreenObject target = pickTarget();
		analysis.setTarget(target);
		if (target == null) {
			analysis.setCommand(result);
			return;
		}

		Ship ship = screen.getShip();
		Log.info("Target: " + target);

		double delta = calculateDegree(ship, target);
		if (delta > Parameter.getDeviation()) {
			result.pressRight();
		} else if (delta < -Parameter.getDeviation()) {
			result.pressLeft();
		}

		if (didShoot) {
			didShoot = false;
		} else if (delta < Parameter.getShootingDegree()) {
			result.pressFire();
			didShoot = true;
		}
		analysis.setCommand(result);
	}

	private double calculateDegree(Ship ship, ScreenObject target) {
		Position shipPosition = ship.getPosition();
		int frames = Parameter.getDeviationPredictionFrames();
		Position[] targetTrajectory = target.getTrajectory(frames);
		analysis.setTargetTrajectory(targetTrajectory);

		int frame = 0;
		for (; frame < frames - 1; frame++) {
			double timeToHit = new Vector(shipPosition, targetTrajectory[frame])
					.getLength()
					/ Parameter.getShotVelocity();
			if (timeToHit <= frame) {
				break;
			}
		}
		Log.info("Deviation frames: " + frame);
		Log.info("Prospected target position: " + targetTrajectory[frame]);
		analysis.setProspectedTargetPosition(targetTrajectory[frame]);

		Vector targetVector = new Vector(shipPosition, targetTrajectory[frame]);
		analysis.setTargetVector(targetVector);

		Log.info("Target vector: " + targetVector);
		Log.info("Ship: " + ship);
		double delta = targetVector.radians()
				- ship.getViewDirection().radians();
		while (delta > Math.PI) {
			delta -= 2 * Math.PI;
		}
		while (delta < -Math.PI) {
			delta += 2 * Math.PI;
		}
		return delta;
	}

	private void analyzeAll(Analysis analysis) {
		if (lastLives > screen.getLives()) {
			Log.info("Died. " + screen.getLives() + " ship(s) left.");
			Log.info("Last analysis:");
			Log.info(lastLives);
			if (screen.getLives() == 0) {
				Log.info("Game over!");
				statistics.finish(screen.getScore());
				Log.info(statistics);
			}
		}
		description = "";
		Ship ship = screen.getShip();
		Ship lastShip = (history.last() != null && history.last().getShip() != null) ? history
				.last().getShip()
				: ship;
		ship.calculateDirection(lastShip.getPosition());

		analyseAsteroids(ship);
		analyseShots(ship);
		analyseSaucer(ship);

		Log.debug(description);
		Log.debug("Analysis finished\n");
	}

	private void analyseSaucer(Ship ship) {
		Saucer saucer = screen.getSaucer();
		if (saucer == null) {
			return;
		}
		Saucer lastSaucer = history.last() == null ? null : history.last()
				.getSaucer();
		if (lastSaucer != null) {
			saucer.calculateDirection(lastSaucer.getPosition());
		}
		description += saucer + "\n";
		double currentDistance = ship.distance(saucer);
		description += "  Current distance to ship: " + currentDistance + "\n";
	}

	private void analyseAsteroids(Ship ship) {
		Screen oldest = history.getOldestScreenWithSameAsteroids(screen);
		if (oldest == null) {
			Log.debug("Asteroids changed");
			return;
		}
		int frames = screen.getId() - oldest.getId();

		List<Asteroid> previousAsteroids = oldest.getAsteroids();
		List<Asteroid> currentAsteroids = screen.getAsteroids();
		Approximation[][] approximations = new Approximation[previousAsteroids
				.size()][];
		for (int i = 0; i < previousAsteroids.size(); i++) {
			Asteroid previousAsteroid = previousAsteroids.get(i);
			Asteroid currentAsteroid = currentAsteroids.get(i);
			currentAsteroid.calculateDirection(previousAsteroid.getPosition(),
					frames);
			approximations[i] = currentAsteroid.getApproximation(ship,
					Parameter.getAsteroidForesight());
			description += currentAsteroid + "\n";
			double currentDistance = ship.distance(currentAsteroid);
			double nextDistance = ship.nextPosition().distance(
					currentAsteroid.nextPosition());
			description += "  Current distance to ship: " + currentDistance
					+ "\n";
			description += "  Next distance to ship: " + nextDistance + "\n";
			description += "  "
					+ ((currentDistance - nextDistance > 0) ? "Closing in"
							: "Going away") + "\n";
			boolean dangerous = false;
			for (int j = 0; j < approximations[i].length; j++) {
				if (approximations[i][j].approximation() <= Parameter
						.getAsteroidCollisionRange(currentAsteroid.getSize())) {
					description += "  Will hit ship in " + j + " frames.\n";
					dangerous = true;
					if (j < framesToAsteroidHit) {
						framesToAsteroidHit = j;
						nearestOnTrack = currentAsteroid;
					}
					break;
				}
			}
			if (dangerous) {
				analysis.addDangerousAsteroid(currentAsteroid);
			} else {
				analysis.addHarmlessAsteroid(currentAsteroid);
			}			
		}
		if (framesToAsteroidHit <= Parameter.getAsteroidForesight()) {
			Log.info("Frames to asteroid hit: " + framesToAsteroidHit + " ("
					+ (screen.getId() + framesToAsteroidHit) + ")");
		}
	}

	private void analyseShots(Ship ship) {
		Screen last = history.last();
		if (last == null || !screen.sameShots(last)) {
			Log.debug("Shots changed");
			return;
		}

		List<Shot> previousShots = last.getShots();
		List<Shot> currentShots = screen.getShots();
		Approximation[][] approximations = new Approximation[previousShots
				.size()][];
		for (int i = 0; i < previousShots.size(); i++) {
			Shot previousShot = previousShots.get(i);
			Shot currentShot = currentShots.get(i);
			currentShot.calculateDirection(previousShot.getPosition());
			approximations[i] = currentShot.getApproximation(ship, Parameter
					.getShotForesight());
			description += currentShot + "\n";
			double currentDistance = ship.distance(currentShot);
			double nextDistance = ship.nextPosition().distance(
					currentShot.nextPosition());
			description += "  Current distance to ship: " + currentDistance
					+ "\n";
			description += "  Next distance to ship: " + nextDistance + "\n";
			description += "  "
					+ ((currentDistance - nextDistance > 0) ? "Closing in"
							: "Going away") + "\n";
			boolean dangerous = false;
			for (int j = 0; j < approximations[i].length; j++) {
				if (approximations[i][j].approximation() <= Parameter
						.getShotCollisionRange()) {
					dangerous = true;
					description += "  Will hit ship in " + j + " frames.\n";
					if (j < framesToShotHit) {
						framesToShotHit = j;
					}
					break;
				}
			}
			if (dangerous) {
				analysis.addDangerousShot(currentShot);
			} else {
				analysis.addHarmlessShot(currentShot);
			}
		}
		if (framesToShotHit <= Parameter.getShotForesight()) {
			Log.info("Frames to shot hit: " + framesToShotHit + " ("
					+ (screen.getId() + framesToShotHit) + ")");
		}
	}

	public String toString() {
		return description;
	}

	public ScreenObject pickTarget() {

		final Ship ship = screen.getShip();
		if (ship == null) {
			return null;
		}

		// Asteroiden, die uns sofort treffen, zuerst - die Abfrage ist
		// berflssig?
		if (framesToAsteroidHit <= Parameter.getImmediateDanger()) {
			return nearestOnTrack;
		}

		Saucer saucer = screen.getSaucer();
		if (saucer != null) {
			if (framesToAsteroidHit <= Parameter
					.getMoreDangerousThanSaucer(saucer.getSize())) {
				return nearestOnTrack;
			}
			return saucer;
		}

		if (framesToAsteroidHit <= Parameter.getFrameRange()) {
			return nearestOnTrack;
		}

		Asteroid[] asteroids = (Asteroid[]) screen.getAsteroids().toArray(
				new Asteroid[screen.getAsteroids().size()]);
		Arrays.sort(asteroids, new Comparator<Asteroid>() {

			public int compare(Asteroid a, Asteroid b) {
				if (a.getSize().equals(b.getSize())) {
					return (int) (a.distance(ship) - b.distance(ship));
				}

				return a.getSize().compareTo(b.getSize());
			}

		});

		if (asteroids.length == 0) {
			return null;
		}

		if (asteroids[0].getSize().equals(Size.SMALL)) {
			return asteroids[0];
		}

		for (int i = 0; i < asteroids.length; i++) {
			Asteroid asteroid = asteroids[i];
			double distance = ship.distance(asteroid);
			if (distance <= Parameter.getMaxRange()
					&& distance >= Parameter.getMinRange()) {
				return asteroid;
			}
		}
		return null;
	}
}
