package asteroid.printer;

import java.io.PrintStream;

import asteroid.model.AFlyable;
import asteroid.model.Asteroid;
import asteroid.model.AsteroidVector;
import asteroid.model.GameModel;
import asteroid.model.Ship;
import asteroid.model.Shot;
import asteroid.model.ShotVector;
import asteroid.model.Ufo;

public class AnalyseHit extends APrinter {
	private static final int PRINT_OBJECT = 0;
	private static final int PRINT_REMEMBER = 1;
	private static final int FIND_HITS = 2;
	private static final int FIND_ASTEROID = 3;
	private static final int USE_CHARGE = 9;
	private int mAnalyse;
	private Remember[] mLastAsts = new Remember[AsteroidVector.MAX_ASTEROIDS];
	private Remember mLastShip = new Remember();
	private Remember[] mLastShots = new Remember[ShotVector.MAX_SHOTS];
	private Remember mLastUfo = new Remember();
	private GameModel mModel;
	private Remember mTestShot;

	public AnalyseHit( GameModel model, String file) {
		super( file);
		init( model);
	}

	public AnalyseHit( GameModel model, PrintStream ps) {
		super( ps);
		init( model);
	}

	static int getDist2( int x1, int y1, int vx, int vy, int x2, int y2) {
		double v = Math.sqrt( (vx * vx) + (vy * vy));
		int dx = x2 - x1;
		int dy = y2 - y1;
		double dd = Math.abs( (dx * vy) - (dy * vx));
		if (v != 0) {
			return (int) (Math.round( 10.0 * dd / v) / 10.0);
		}
		return Integer.MAX_VALUE;
	}

	private void findAsteroid( Asteroid ast) {
		Remember lastAst = mLastAsts[ast.getIndex()];
		if (lastAst.mActive && !lastAst.mExplosion && ast.isExplosion()) {
			mOut.print( "!");
			int shotX8 = mTestShot.mX << 3;
			int shotY8 = mTestShot.mY << 3;
			int astX8 = lastAst.mX << 3;
			int astY8 = lastAst.mY << 3;
//			mPrint.print( getDist2( shotX8, shotY8, lastShot.mMoveX8, lastShot.mMoveY8, astX8, astY8));
//			mPrint.print( ',');
//			mPrint.print( getRadius( shotX8, shotY8, lastShot.mMoveX8, lastShot.mMoveY8, astX8, astY8));
			mOut.print( astX8 - shotX8);
			mOut.print( ',');
			mOut.print( astY8 - shotY8);
			int size = 0;
			switch (lastAst.mZ) {
				case 0:
					mOut.print( "G:");
					size = 320;
					break;
				case 15:
					mOut.print( "M:");
					size = 160;
					break;
				case 14:
					mOut.print( "K:");
					size = 80;
					break;
			}
			int hit = isHit( shotX8, shotY8, mTestShot.mMoveX8, mTestShot.mMoveY8, astX8, astY8, lastAst.mMoveX8, lastAst.mMoveY8, size);
			if (hit > 0) {
				mOut.print( 'x');
				mOut.print( hit);
			}
			mOut.print( '!');
			mOut.print( 'M');
			mOut.print( ':');
			mOut.print( mTestShot.mX);
			mOut.print( ',');
			mOut.print( mTestShot.mY);
			mOut.print( ',');
			mOut.print( mTestShot.mMoveX8);
			mOut.print( ',');
			mOut.print( mTestShot.mMoveY8);
			mOut.print( ',');
			mOut.print( 'A');
			mOut.print( ':');
			mOut.print( lastAst.mX);
			mOut.print( ',');
			mOut.print( lastAst.mY);
			mOut.print( ',');
			mOut.print( lastAst.mZ);
			mOut.print( ',');
			mOut.print( lastAst.mMoveX8);
			mOut.print( ',');
			mOut.print( lastAst.mMoveY8);
			mOut.print( ',');
			mOut.print( size);
			mOut.print( ' ');
		}
	}

	void findHits() {
		mAnalyse = FIND_HITS;
		mModel.forAllShots( this);
	}

	public void forAsteroid( Asteroid ast) {
		Remember lastAst = mLastAsts[ast.getIndex()];
		switch (mAnalyse) {
			case PRINT_OBJECT:
				if ((ast.isVisible()) && !lastAst.mActive) {
					mOut.print( '+');
				}
				else if (!(ast.isVisible()) && lastAst.mActive) {
					mOut.print( '-');
				}
				else if (!(ast.isVisible()) && !lastAst.mActive) {
					mOut.print( '.');
				}
				else if (ast.isExplosion() && !lastAst.mExplosion) {
					mOut.print( '?');
				}
				else if (ast.isExplosion() && lastAst.mExplosion) {
					mOut.print( 'E');
				}
				else {
					mOut.print( 'A');
				}
				break;
			case PRINT_REMEMBER:
				if (lastAst.mActive) {
					mOut.print( lastAst.mExplosion ? "E:" : "A:");
					printRemeber( lastAst);
				}
				break;
			case FIND_ASTEROID:
				findAsteroid( ast);
				break;
			case USE_CHARGE:
				forFlyable( lastAst, ast);
				lastAst.mActive = ast.isVisible();
				lastAst.mExplosion = ast.isExplosion();
				break;
		}
	}

	private void forFlyable( Remember last, AFlyable fly) {
		last.mX = fly.getX8() >> 3;
		last.mY = fly.getY8() >> 3;
		last.mZ = fly.getZ();
		last.mMoveX8 = fly.getMoveX8();
		last.mMoveY8 = fly.getMoveY8();
	}

	public void forShip( Ship ship) {
		switch (mAnalyse) {
			case PRINT_OBJECT:
				if (ship.isVisible() && !mLastShip.mActive) {
					mOut.print( '+');
				}
				else if (ship.isHidden() && mLastShip.mActive) {
					mOut.print( '-');
				}
				else if (ship.isHidden() && !mLastShip.mActive) {
					mOut.print( '.');
				}
				else {
					mOut.print( 'S');
				}
				break;
			case PRINT_REMEMBER:
				if (mLastShip.mActive) {
					mOut.print( "S:");
					printRemeber( mLastShip);
				}
				break;
			case USE_CHARGE:
				forFlyable( mLastShip, ship);
				mLastShip.mActive = ship.isVisible();
				break;
		}
	}

	public void forShot( Shot shot) {
		Remember lastShot = mLastShots[shot.getIndex()];
		switch (mAnalyse) {
			case PRINT_OBJECT:
				if ((shot.isVisible()) && !lastShot.mActive) {
					mOut.print( '+');
				}
				else if (shot.isHidden() && lastShot.mActive) {
					mOut.print( '-');
				}
				else if (shot.isHidden() && !lastShot.mActive) {
					mOut.print( '.');
				}
				else {
					mOut.print( 'M');
				}
				break;
			case PRINT_REMEMBER:
				if (lastShot.mActive) {
					mOut.print( "M:");
					printRemeber( lastShot);
				}
				break;
			case FIND_HITS:
				if (shot.isHidden() && lastShot.mActive) {
					mTestShot = lastShot;
					mOut.print( shot.getIdent());
					mOut.print( ": ");
					mAnalyse = FIND_ASTEROID;
					mModel.forAllAsteroids( this);
					mAnalyse = FIND_HITS;
				}
				break;
			case USE_CHARGE:
				forFlyable( lastShot, shot);
				lastShot.mActive = shot.isVisible();
				break;
		}
	}

	public void forUfo( Ufo ufo) {
		Remember lastUfo = mLastUfo;
		switch (mAnalyse) {
			case PRINT_OBJECT:
				if ((ufo.isVisible()) && !lastUfo.mActive) {
					mOut.print( '+');
				}
				else if (!ufo.isHidden() && lastUfo.mActive) {
					mOut.print( '-');
				}
				else if (!ufo.isHidden() && !lastUfo.mActive) {
					mOut.print( '.');
				}
				else if (ufo.isExplosion() && !lastUfo.mExplosion) {
					mOut.print( '?');
				}
				else if (ufo.isExplosion() && lastUfo.mExplosion) {
					mOut.print( 'E');
				}
				else {
					mOut.print( 'U');
				}
				break;
			case PRINT_REMEMBER:
				if (lastUfo.mActive) {
					mOut.print( lastUfo.mExplosion ? "X:" : "U:");
					printRemeber( lastUfo);
				}
				break;
			case FIND_HITS:
				break;
			case USE_CHARGE:
				forFlyable( lastUfo, ufo);
				lastUfo.mActive = ufo.isVisible();
				lastUfo.mExplosion = ufo.isExplosion();
				break;
		}
	}

	static int isHit( int x1, int y1, int vx, int vy, int x2, int y2, int ax, int ay, int size) {
		int dx = x2 - x1;
		int dy = y2 - y1;
		if ((Math.abs( dx) <= size) && (Math.abs( dy) <= size)) {
			return 1;
		}
//		int dd = (dx * vy) - (dy * vx);
		dx -= vx;
		dy -= vy;
		if ((Math.abs( dx) <= size) && (Math.abs( dy) <= size)) {
			return 2;
		}
		dx -= vx;
		dy -= vy;
		if ((Math.abs( dx) <= size) && (Math.abs( dy) <= size)) {
			return 3;
		}
		dx = x2 - x1 + ax;
		dy = y2 - y1 + ay;
		if ((Math.abs( dx) <= size) && (Math.abs( dy) <= size)) {
			return 4;
		}
		dx -= vx;
		dy -= vy;
		if ((Math.abs( dx) <= size) && (Math.abs( dy) <= size)) {
			return 5;
		}
		dx -= vx;
		dy -= vy;
		if ((Math.abs( dx) <= size) && (Math.abs( dy) <= size)) {
			return 6;
		}
		return 0;
//		int dr = (dx * vx) + (dy * vy);
//		int vSqr = (vx * vx) + (vy * vy);
//		return (dr > 0) && (dr <= vSqr) && ((dd * dd) < (size * size * vSqr));
	}

	private void init( GameModel model) {
		mModel = model;
		for (int i = 0; i < ShotVector.MAX_SHOTS; ++i) {
			mLastShots[i] = new Remember();
		}
		for (int i = 0; i < AsteroidVector.MAX_ASTEROIDS; ++i) {
			mLastAsts[i] = new Remember();
		}
	}

	void prepareNext() {
		mAnalyse = USE_CHARGE;
		mModel.forShip( this);
		mModel.forUfo( this);
		mModel.forAllAsteroids( this);
		mModel.forAllShots( this);
	}

	public void print() {
		if (mModel.getShip().isVisible()) {
			mOut.print( "Frame ");
			mOut.print( GameModel.sFrame);
			mOut.print( "   ");
//			mOut.printKey( mKey);
			printStatus();
			findHits();
//			printLast( mModel);
			prepareNext();
			mOut.println();
		}
	}

	void printLast() {
		mAnalyse = PRINT_REMEMBER;
		mModel.forShip( this);
		mModel.forUfo( this);
		mModel.forAllAsteroids( this);
		mModel.forAllShots( this);
	}

	void printRemeber( Remember rem) {
		mOut.print( rem.mX);
		mOut.print( ',');
		mOut.print( rem.mY);
		mOut.print( ';');
		mOut.print( rem.mMoveX8);
		mOut.print( ',');
		mOut.print( rem.mMoveY8);
		mOut.print( ' ');
	}

	void printStatus() {
		mAnalyse = PRINT_OBJECT;
		mModel.forShip( this);
		mOut.print( ' ');
		mModel.forUfo( this);
		mOut.print( ' ');
		mModel.forAllAsteroids( this);
		mOut.print( ' ');
		mModel.forAllShots( this);
		mOut.print( "   ");
	}

	static double getRadius( int x1, int y1, int vx, int vy, int x2, int y2) {
		double v2 = (vx * vx) + (vy * vy);
		int dx = x2 - x1;
		int dy = y2 - y1;
		double rr = (dx * vx) + (dy * vy);
		if (v2 != 0) {
			return Math.round( 100.0 * rr / v2) / 100.0;
		}
		return Integer.MAX_VALUE;
	}

	private class Remember {
		boolean mActive;
		boolean mExplosion;
		int mMoveX8, mMoveY8;
		int mX, mY, mZ;
	}
}
