package asteroid.model;

import java.nio.ByteBuffer;
import java.util.StringTokenizer;

import asteroid.Player;
import asteroid.SimpleLogging;
import asteroid.printer.Source;
import asteroid.strategy.AStrategie;
import asteroid.udp.AHandler;

public class GameModel {
	public static final int FRAME_TYPE_READY = 1;
	public static final int FRAME_TYPE_HIGH = 2;
	public static final int FRAME_TYPE_START = 3;
	public static final int FRAME_TYPE_GAME = 4;
	public static final int FRAME_TYPE_GAME_OVER = 5;
	public static final int FRAME_TYPE_END = 6;
	public static final int FRAME_TYPE_SCORE = 7;
	private static final String USER_SIGN = "GRA";
	private static final int KEY_HOLD_TIME = 10;
	public static int sFrame;
	public static int sFrameOffset;
	public static int sFrameID;
	Ship mShip = new Ship();
	Ufo mUfo = new Ufo( mShip);
	AsteroidVector mAsts = new AsteroidVector( mShip, mUfo);
	ShotVector mShots = new ShotVector( mShip, mUfo);
	Collider mCollider = new Collider( mShip, mUfo, mAsts, mShots);
	Detector mDetector = new Detector( mShip, mUfo, mAsts, mShots);
	Striker mStriker = new Striker( mShip, mUfo, mAsts, mShots);
	private Angle mAngle = new Angle( mShip);
	private boolean mAutoStart;
	private int mFrameType = FRAME_TYPE_GAME;
	private int mRelease;
	private boolean mRepeat = false;
	private Source mSource;
	private AHandler mSrc;
	private boolean mTopList = false;

	public GameModel( AHandler src) {
		mSrc = src;
//		mSource = new Source( System.out);
	}

	public void addKey( int key) {
		mCollider.detect( Angle.sForward);
		mSrc.addKey( mAngle.getKey( key | mCollider.getKey()));
	}

	public void setAutoStart() {
		mAutoStart = true;
	}

	public boolean isAutoStart() {
		return mAutoStart || mRepeat;
	}

	public AEnemy getEnemy( int index) {
		if (index == AsteroidVector.MAX_ASTEROIDS) {
			return mUfo;
		}
		return mAsts.get( index);
	}

	public int getFireDirection() {
		return mAngle.getFireDirection();
	}

	public void forAllAsteroids( ICompute com) {
		mAsts.forAll( com);
	}

	public void forAllDetectAsteroids( ICompute com) {
		mAsts.forAllDetect( com);
	}

	public void forAllDetectShots( ICompute com) {
		mShots.forAllDetect( com);
	}

	public void forAllShots( ICompute com) {
		mShots.forAll( com);
	}

	public void forShip( ICompute com) {
		mShip.doIt( com);
	}

	public void forUfo( ICompute com) {
		mUfo.doIt( com);
	}

	public int getFrameType() {
		return mFrameType;
	}

	public int getHighScore() {
		return mDetector.mHighScore;
	}

	public int getHits() {
		return mStriker.mStrikes;
	}

	public void interpret( ByteBuffer bb) {
		syncFrame( bb.get( AHandler.FRAME_POS) & 0xFF);
		if (sFrame >= 550) {
			sFrame += 0;
		}
		int lastScore = mDetector.mScore;
		mDetector.prepare();
		mDetector.interpret( bb);
		mDetector.detect( mAngle.getFireDirection());
		mFrameType = mDetector.mFrameType;
		mStriker.detect();
		if ((mStriker.mScore != lastScore) && isPlayTime()) {
			if (mStriker.mScore > 0) {
				SimpleLogging.addLog( "missing stikes " + lastScore + "!=" + mStriker.mScore);
			}
			mStriker.mScore = lastScore;
		}
		syncType();
		mAngle.synchronice( bb.get( AHandler.FRAME_PING) & 0xFF);
		if (mSource != null) {
			mSource.print( bb);
		}
	}

	public int getLevel() {
		return mDetector.mLevel;
	}

	public int getLives() {
		return mDetector.mLives;
	}

	public void move( int steps) {
		Angle.sDirection += 3 * steps;
		mAngle.initDirection();
		int time = Math.abs( steps);
		for (int i = 0; i <= AsteroidVector.MAX_ASTEROIDS; ++i) {
			getEnemy( i).move( time);
		}
	}

	public int getOtherScore() {
		return mDetector.mOtherScore;
	}

	public void parse( String text) {
		StringTokenizer st = new StringTokenizer( text);
		String status = st.nextToken(); // "Frame"
		StringTokenizer ss = new StringTokenizer( status, ",");
		sFrame = parseInt( ss.nextToken().trim());
		ss.nextToken(); // key
		Angle.sReceivePing = parseInt( ss.nextToken().trim());
		Angle.sSendPing = parseInt( ss.nextToken().trim());
		Angle.sDirection = parseInt( ss.nextToken().trim());
		Angle.sForward = parseInt( ss.nextToken().trim());
		mAngle.initDirection();
		while (st.hasMoreTokens()) {
			String detect = st.nextToken();
			StringTokenizer dd = new StringTokenizer( detect, ":,");
			String type = dd.nextToken();
			int x = parseInt( dd.nextToken().trim());
			int y = parseInt( dd.nextToken().trim());
			int z = parseInt( dd.nextToken().trim());
			int moveX = parseInt( dd.nextToken().trim());
			int moveY = parseInt( dd.nextToken().trim());
			if ("S".equals( type)) {
				mShip.set( x, y, z, moveX, moveY, AFlyable.STATE_VISIBLE);
			}
			else if ("T".equals( type)) {
				mShip.set( x, y, z, moveX, moveY, AFlyable.STATE_EXPLOSION);
			}
			else if ("J".equals( type)) {
				mShip.set( x, y, z, moveX, moveY, AFlyable.STATE_JUMP);
			}
			else if ("s".equals( type)) {
				mShip.set( x, y, z, moveX, moveY, AFlyable.STATE_HIDDEN);
			}
			else if ("U".equals( type)) {
				mUfo.set( x, y, z, moveX, moveY, AFlyable.STATE_VISIBLE);
			}
			else if ("X".equals( type)) {
				mUfo.set( x, y, z, moveX, moveY, AFlyable.STATE_EXPLOSION);
			}
			else if ("u".equals( type)) {
				mUfo.set( x, y, z, moveX, moveY, AFlyable.STATE_HIDDEN);
			}
			else if (type.startsWith( "A")) {
				int index = parseInt( type.substring( 1));
				mAsts.set( index, x, y, z, moveX, moveY, AFlyable.STATE_VISIBLE);
			}
			else if (type.startsWith( "E")) {
				int index = parseInt( type.substring( 1));
				mAsts.set( index, x, y, z, moveX, moveY, AFlyable.STATE_EXPLOSION);
			}
			else if (type.startsWith( "a")) {
				int index = parseInt( type.substring( 1));
				mAsts.set( index, x, y, z, moveX, moveY, AFlyable.STATE_HIDDEN);
			}
			else if (type.startsWith( "e")) {
				int index = parseInt( type.substring( 1));
				mAsts.set( index, x, y, z, moveX, moveY, AFlyable.STATE_BLOCKED);
			}
			else if (type.startsWith( "M")) {
				int index = parseInt( type.substring( 1));
				mShots.set( index, x, y, z, moveX, moveY, AFlyable.STATE_VISIBLE);
			}
			else if (type.startsWith( "_")) {
				int index = parseInt( type.substring( 1));
				mShots.set( index, x, y, z, moveX, moveY, AFlyable.STATE_HIDDEN);
			}
			else {
				throw new IllegalArgumentException( "missing type: " + detect);
			}
		}
	}

	private static int parseInt( String s) {
		int sign = 1;
		if (s.startsWith( "+")) {
			s = s.substring( 1);
		}
		if (s.startsWith( "-")) {
			s = s.substring( 1);
			sign = -1;
		}
		while (s.startsWith( "0")) {
			s = s.substring( 1);
		}
		if (s.length() == 0) {
			return 0;
		}
		return sign * Integer.parseInt( s);
	}

	public int getPlayFrame() {
		return mDetector.mPlayFrame;
	}

	public boolean isPlayTime() {
		return mFrameType == FRAME_TYPE_GAME;
	}

	public void setRepeat( boolean value) {
		mRepeat = value;
	}

	public int getScore() {
		return mDetector.mScore;
	}

	public String getScore( int index) {
		return mDetector.mHighScores.mValues[index];
	}

	public Ship getShip() {
		return mShip;
	}

	public boolean isShipVisible() {
		return mShip.isVisible();
	}

	public FlyInfo getShot( int index) {
		return mShots.get( index);
	}

	public int getShots() {
		return mStriker.mCreates;
	}

	private static void syncFrame( int frame) {
		int last = sFrame & 0xFF;
		if (frame < last) {
			last -= 0x100;
		}
		last -= frame;
		sFrame -= last;
	}

	private void syncType() {
		switch (mFrameType) {
			case FRAME_TYPE_READY:
			case FRAME_TYPE_HIGH:
				break;
			case FRAME_TYPE_START:
				mAsts.clearAll();
				mShots.clearAll();
				mDetector.reset();
				mStriker.reset();
				mAutoStart = false;
				break;
			case FRAME_TYPE_GAME:
			case FRAME_TYPE_GAME_OVER:
				break;
			case FRAME_TYPE_END:
				break;
			case FRAME_TYPE_SCORE:
				break;
		}
	}

	public void setTopList( boolean value) {
		mTopList = value;
	}

	public boolean isTopList() {
		return mTopList;
	}

	public Ufo getUfo() {
		return mUfo;
	}

	public void updateKey( AStrategie strategy) {
		int key = 0;
		switch (mFrameType) {
			case FRAME_TYPE_READY:
			case FRAME_TYPE_HIGH:
				if (isAutoStart()) {
					key = ICompute.KEY_START;
				}
				mRelease = KEY_HOLD_TIME;
				break;
			case FRAME_TYPE_START:
				if ((Player.sHost == Player.HOST_HEISE) && (mRelease > 0)) {
					key = ICompute.KEY_NAME;
					--mRelease;
				}
				break;
			case FRAME_TYPE_GAME:
				key = strategy.getKey();
				break;
			case FRAME_TYPE_GAME_OVER:
				key = ((sFrame & 4) != 0) ? ICompute.KEY_HYPERSPACE : 0;
				break;
			case FRAME_TYPE_END:
				mRelease = 0;
				break;
			case FRAME_TYPE_SCORE:
				key = userScore();
				break;
		}
		addKey( key);
		mDetector.setPlayTime();
	}

	private int userScore() {
		if (mRelease > 0) {
			return (mRelease-- > KEY_HOLD_TIME) ? ICompute.KEY_HYPERSPACE : 0;
		}
		String s = mDetector.mSign;
		int pos = s.length() - 1;
		int dif = s.charAt( pos) - USER_SIGN.charAt( pos);
		if (isTopList() && (dif != 0)) {
			return (dif > 0) ? ICompute.KEY_RIGHT : ICompute.KEY_LEFT;
		}
		mRelease = KEY_HOLD_TIME << 1;
		return ICompute.KEY_HYPERSPACE;
	}

	public boolean isVanished() {
		return mStriker.mVanish > 0;
	}
}
