package asteroid;

import java.io.BufferedOutputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.Writer;

import asteroid.model.ICompute;

public class PrintWriterFormat extends PrintWriter implements Runnable {
	private int mLinePos;
	private int mSavePos;

	public PrintWriterFormat( String file) {
		super( createFileWriter( file), true);
		Runtime.getRuntime().addShutdownHook( new Thread( this));
	}

	public PrintWriterFormat( OutputStream out) {
		super( out, true);
	}

	public PrintWriterFormat( Writer out) {
		super( out, true);
	}

	private static PrintStream createFileWriter( String file) {
		try {
			return new PrintStream( new BufferedOutputStream( new FileOutputStream( file)), true);
		}
		catch (IOException ex) {
			SimpleLogging.addException( ex);
		}
		return System.out;
	}

	public void fill( int size) {
		mSavePos += size;
		while (mLinePos < mSavePos) {
			write( ' ');
		}
	}

	private void movePos( int c) {
		if (c == '\n') {
			mSavePos -= mLinePos;
			mLinePos = 0;
		}
		else {
			++mLinePos;
		}
	}

	public int getPos() {
		return mLinePos;
	}

	public void printByte( int val) {
		val &= 0xFF;
		if (val < 10) {
			print( "00" + String.valueOf( val));
		}
		else if (val < 100) {
			print( "0" + String.valueOf( val));
		}
		else {
			print( String.valueOf( val));
		}
	}

	public void printKey( int key) {
		print( ((key & ICompute.KEY_HYPERSPACE) != 0) ? 'H' : '_');
		print( ((key & ICompute.KEY_FIRE) != 0) ? 'F' : '_');
		print( ((key & ICompute.KEY_THRUST) != 0) ? 'T' : '_');
		print( ((key & ICompute.KEY_RIGHT) != 0) ? 'R' : '_');
		print( ((key & ICompute.KEY_LEFT) != 0) ? 'L' : '_');
		print( ((key & ICompute.KEY_START) != 0) ? 'S' : '_');
	}

	public void println() {
		super.println();
		mLinePos = 0;
	}

	public void printPos( int val) {
		if (val < 10) {
			print( "000" + String.valueOf( val));
		}
		else if (val < 100) {
			print( "00" + String.valueOf( val));
		}
		else if (val < 1000) {
			print( "0" + String.valueOf( val));
		}
		else {
			print( String.valueOf( val));
		}
	}

	public void printSign( int val) {
		print( (val < 0) ? '-' : '+');
		printZoom( Math.abs( val));
	}

	public void printZoom( int val) {
		if (val < 10) {
			print( "0" + String.valueOf( val));
		}
		else {
			print( String.valueOf( val));
		}
	}

	public void run() {
		try {
			flush();
			close();
		}
		catch (Exception ex) {
			System.err.println( "Unerwartete Exception: " + ex.getMessage());
		}
	}

	public void savePos() {
		mSavePos = mLinePos;
	}

	public void write( int b) {
		super.write( b);
		movePos( b);
	}

	public void write( String s, int off, int len) {
		super.write( s, off, len);
		for (int i = 0; i < len; ++i) {
			movePos( s.charAt( off + i));
		}
	}

	public void write( char[] buf, int off, int len) {
		super.write( buf, off, len);
		for (int i = 0; i < len; ++i) {
			movePos( buf[off + i]);
		}
	}
}
