// ============================================================================
// File:               RJMenuBar.java
//
// Project:            i18n 
//
// Purpose:            JMenuBar which can change it's JMenus depending on 
//                     Locale.
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2002  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================

package de.caff.i18n.swing;


import de.caff.i18n.I18n;
import de.caff.i18n.Localizable;

import javax.swing.*;
import java.awt.*;
import java.util.Locale;

/**
 *  Localized Menubar. All menus and items get the same locale.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision$
 */
public class RJMenuBar 
  extends    JMenuBar 
  implements Localizable 
{
  /**
   *  @param  l       Locale for this bar
   */
  public RJMenuBar(Locale l) {
    super();
    setLocale(l);
  }

  /**
   * Notifies this component that it now has a parent component.
   * When this method is invoked, the chain of parent components is
   * set up with <code>KeyboardAction</code> event listeners.
   */
  public void addNotify()
  {
    super.addNotify();
    I18n.addLocalizationChangeListener(this);
  }

  /**
   * Overrides <code>JComponent.removeNotify</code> to check if
   * this button is currently set as the default button on the
   * <code>RootPane</code>, and if so, sets the <code>RootPane</code>'s
   * default button to <code>null</code> to ensure the
   * <code>RootPane</code> doesn't hold onto an invalid button reference.
   */
  public void removeNotify()
  {
    I18n.removeLocalizationChangeListener(this);
    super.removeNotify();
  }


  /**
   *  Change the locale.
   *  @param  l    new locale
   */
  public void setLocale(Locale l) {
    Locale oldLocale = getLocale();
    super.setLocale(l);
    if (oldLocale != null  &&  !oldLocale.equals(l)) {
      for (int i = getMenuCount()-1;   i >= 0;   i--) {
        JMenu m = getMenu(i);
        if (m instanceof Localizable) {
          ((Localizable)m).setLocale(l);
        }
      }
    }
  }


  /**
   *  Get the locale. Instead of throwing an <code>IllegalComponentStateException</code>
   *  when there's no locale we just return null.
   *  @return the Locale of this Label or null
   */
  public Locale getLocale() {
    try {
      return super.getLocale();
    } catch (IllegalComponentStateException x) {
      return null;
    }
  }

  /**
   *  Overwritten to set the locale of the Menu when unset.
   *  @param  m   menu to add
   */
  public JMenu add(JMenu m) {
    if (m instanceof Localizable) {
      Localizable l = (Localizable)m;
      if (l.getLocale() == null) {
	l.setLocale(getLocale());
      }
    }
    return super.add(m);
  }

}
