// ====================================================================
// File:		VisibleState
// 
// Project:		General Swing support.
//
// Purpose:		A state with visible components.
//
// Author:		rammi
// Date: 		31.03.2002
// Time: 		12:47:18
// --------------------------------------------------------------------
// Copyright Notice:	(c) 2002 Rammi (rammi@caff.de)
//                      This code is in the public domain.
//                      Use at own risk.
//                      No guarantees given.
//
// Latest change:       $Date$
//
// History: 		$Log$
// ====================================================================
package de.caff.gimmicks.swing;

import de.caff.i18n.I18n;
import de.caff.i18n.Localizable;
import de.caff.util.Utility;

import javax.swing.*;
import java.util.Locale;
import java.util.MissingResourceException;

/**
 *  A visible state is a state with a psooible label,
 *  different icons for different states and a possible tooltip text.
 *
 *  @author  <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision$
 */
public class VisibleState
        extends AbstractState
        implements Localizable
{
  /** Property name for label resource key. */
  private static final String LABEL_RESOURCE_KEY_PROPERTY = "LABEL_RESOURCE_KEY";
  /** Property name for tooltip resource key. */
  private static final String TOOLTIP_RESOURCE_KEY_PROPERTY = "TOOLTIP_RESOURCE_KEY";

  /** The current locale. */
  private Locale locale;

  /**
   *  Create a visible state.
   *  @param labelResourceKey resource key for labels and menu items
   *  @param tooltipResourceKey resource key for tooltips
   *  @param activeIcon icon to be displayed when state is active
   *  @param inactiveIcon icon to be displayed when state is inactive
   *  @param disabledIcon icon to be displayed when state is disabled
   *         (if <code>null</code> the Swing standard is used)
   */
  public VisibleState(String       labelResourceKey,
                      String       tooltipResourceKey,
                      Icon         activeIcon,
                      Icon         inactiveIcon,
                      Icon         disabledIcon)
  {
    init(labelResourceKey, tooltipResourceKey, activeIcon, inactiveIcon, disabledIcon);
  }

  /**
   *  Internal initialization.
   *  @param labelResourceKey resource key for labels and menu items
   *  @param tooltipResourceKey resource key for tooltips
   *  @param activeIcon icon to be displayed when state is active
   *  @param inactiveIcon icon to be displayed when state is inactive
   *  @param disabledIcon icon to be displayed when state is disabled
   *         (if <code>null</code> the Swing standard is used)
   */
  private void init(String labelResourceKey, String tooltipResourceKey, Icon activeIcon, Icon inactiveIcon,
                    Icon disabledIcon)
  {
    setProperty(LABEL_RESOURCE_KEY_PROPERTY, labelResourceKey);
    setProperty(TOOLTIP_RESOURCE_KEY_PROPERTY, tooltipResourceKey);
    setProperty(ACTIVE_ICON_PROPERTY, activeIcon);
    setProperty(INACTIVE_ICON_PROPERTY, inactiveIcon);
    setProperty(DISABLED_ICON_PROPERTY, disabledIcon);

    I18n.addLocalizationChangeListener(this);

    setLocale(I18n.getDefaultLocale());
  }

  /**
   *  Create a visible state with default icons. The inactive icon is the same as the
   *  active and the disabled icon is the standard disabled view.
   *  @param labelResourceKey resource key for labels and menu items
   *  @param tooltipResourceKey resource key for tooltips
   *  @param activeIcon icon to be displayed when state is active
   */
  public VisibleState(String       labelResourceKey,
                      String       tooltipResourceKey,
                      Icon         activeIcon)
  {
    this(labelResourceKey, tooltipResourceKey, activeIcon, activeIcon,  null);
  }

  /**
   *  Create a visible state from I18n settings.
   *  @param baseTag   I18n base tag
   */
  public VisibleState(String baseTag)
  {
    Icon normal = new ImageIcon(Utility.loadImage(I18n.getString(baseTag+ I18n.SUFFIX_ICON)));
    Icon disabled = normal;
    Icon inactive = null;
    try {
      disabled = new ImageIcon(Utility.loadImage(I18n.getString(baseTag+ I18n.SUFFIX_DISABLED_ICON)));
    } catch (MissingResourceException e) {
    }
    try {
      inactive = new ImageIcon(Utility.loadImage(I18n.getString(baseTag+ I18n.SUFFIX_INACTIVE_ICON)));
    } catch (MissingResourceException e) {
    }
    init(baseTag+I18n.SUFFIX_TEXT,
         baseTag+I18n.SUFFIX_TOOLTIP,
         normal,
         disabled,
         inactive);
  }

  /**
   *  Set the locale.
   *  @param  l   locale to set.
   */
  public void setLocale(Locale l)
  {
    locale = l;
    Object key = getValue(LABEL_RESOURCE_KEY_PROPERTY);
    if (key != null) {
      setProperty(LABEL_TEXT_PROPERTY,  I18n.getString(key.toString(), l));
    }
    key = getValue(TOOLTIP_RESOURCE_KEY_PROPERTY);
    if (key != null) {
      setProperty(TOOLTIP_TEXT_PROPERTY, I18n.getString(key.toString(), l));
    }
  }

  /**
   *  Get the locale.
   *  @return  the locale
   */
  public Locale getLocale()
  {
    return locale;
  }

  /**
   *  Finalizer. Remove this from I18n listeners.
   */
  protected void finalize()
  {
    I18n.removeLocalizationChangeListener(this);
  }
}
