// thread_comm.hpp
// ---------------
//
//  (C) Copyright Gerald Thaler 2008.
//
// Distributed under the Boost Software License, Version 1.0.
//    (See accompanying file LICENSE_1_0.txt or copy at
//          http://www.boost.org/LICENSE_1_0.txt)

#ifndef INTREPID_THREAD_COMM_HPP
#define INTREPID_THREAD_COMM_HPP

#if defined(_MSC_VER) && (_MSC_VER >= 1200)
# pragma once
#endif // defined(_MSC_VER) && (_MSC_VER >= 1200)

namespace intrepid
{
    // Handles main thread communication.
    class thread_comm
    {
    public:
        thread_comm();

        bool finish() const; // Worker threads

        uint8_t get_result(); // Main thread
        void push_frame(); // Main thread
        void push_result(uint8_t result,
                         int rating); // Worker threads
        void reset(int nof_threads); // Main thread
        void set_finish(); // Main thread
        void start_computing(); // Worker thread 0
        void interrupt(); // Worker threads
        void worker_wait(); // Worker threads (not Worker thread 0)
        void worker0_wait(); // Worker thread 0

    private:
        thread_comm(thread_comm const &); // = delete;
        thread_comm &operator=(thread_comm const &); // = delete;

        class comm_queue
        {
        public:
            comm_queue();

            void interrupt();
            void reset(int capacity);

        protected:
            comm_queue(comm_queue const &); // = delete;
            comm_queue &operator=(comm_queue const &); // = delete;

            void interruption_point();

            mutex              mutex_;
            condition_variable cond_can_pull_;
            condition_variable cond_can_push_;

            int  capacity_, count_;
            bool interrupted_;
        };

        class pull_queue : public comm_queue
        {
        public:
            void pull_all();
            void push_one();
        };

        class push_queue : public comm_queue
        {
        public:
            void pull_one();
            void push_all();
        };

        pull_queue    queue_results_;
        push_queue    queue_worker_;
        push_queue    queue_worker0_;
        pull_queue    queue_worker0_done_;
        mutex         mutex_;
        int           rating_;
        uint8_t       result_;
        volatile bool finish_, interrupted_;
    };

// class thread_comm
// public:

    inline thread_comm::thread_comm()
    {
    }

    inline bool thread_comm::finish() const
    {
        // Worker threads poll frequently if they should finish with their
        // computations.
        return finish_ || interrupted_;
    }

    inline void thread_comm::push_frame()
    {
        // Start worker 0.
        rating_ = 0;
        queue_worker0_.push_all();
    }

    inline void thread_comm::set_finish()
    {
        queue_worker0_done_.pull_all();
        finish_ = true;
    }

    inline void thread_comm::start_computing()
    {
        queue_worker0_done_.push_one();
        queue_worker_.push_all();
    }

    inline void thread_comm::worker_wait()
    {
        queue_worker_.pull_one();
    }

    inline void thread_comm::worker0_wait()
    {
        queue_worker0_.pull_one();
    }

// private:
// class thread_comm::comm_queue

    inline thread_comm::comm_queue::comm_queue()
    {
    }

    inline void thread_comm::comm_queue::reset(int capacity)
    {
        capacity_ = capacity;
        count_ = 0;
        interrupted_ = false;
    }

} // end of namespace intrepid

#endif // include guard
