// thread_comm.cpp
// ---------------
//
//  (C) Copyright Gerald Thaler 2008.
//
// Distributed under the Boost Software License, Version 1.0.
//    (See accompanying file LICENSE_1_0.txt or copy at
//          http://www.boost.org/LICENSE_1_0.txt)

#include "stdafx.hpp"

#include "thread_comm.hpp"

namespace intrepid
{
// class thread_comm
// public:

    uint8_t thread_comm::get_result()
    {
        // Wait until all workers have finished.
        queue_results_.pull_all();
        finish_ = false;
        return result_;
    }

    void thread_comm::push_result(uint8_t result,
                                  int rating)
    {
        {
            unique_lock<mutex> lock(mutex_);
            if (rating > rating_)
            {
                result_ = result;
                rating_ = rating;
            }
        }
        queue_results_.push_one();
    }

    void thread_comm::reset(int nof_threads)
    {
        result_ = 0;
        rating_ = 0;
        finish_ = false;
        interrupted_ = false;
        queue_worker_.reset(nof_threads - 1);
        queue_worker0_.reset(1);
        queue_worker0_done_.reset(1);
        queue_worker0_done_.push_one();
        queue_results_.reset(nof_threads);
        for (int n = 0; n < nof_threads; ++n)
        {
            queue_results_.push_one();
        }
    }

    void thread_comm::interrupt()
    {
        interrupted_ = true;
        queue_worker_.interrupt();
        queue_worker0_.interrupt();
        queue_worker0_done_.interrupt();
        queue_results_.interrupt();
    }

// private:
// class thread_comm::comm_queue

    void thread_comm::comm_queue::interrupt()
    {
        unique_lock<mutex> lock(mutex_);
        interrupted_ = true;
        cond_can_pull_.notify_all();
        cond_can_push_.notify_all();
    }

    void thread_comm::comm_queue::interruption_point()
    {
        // lock must be held.
        if (interrupted_)
        {
            throw thread_interrupted();
        }
    }

// class thread_comm::pull_queue

    void thread_comm::pull_queue::pull_all()
    {
        unique_lock<mutex> lock(mutex_);
        while (count_ != capacity_)
        {
            interruption_point();
            cond_can_pull_.wait(lock);
        }
        count_ = 0;
        cond_can_push_.notify_all();
    }

    void thread_comm::pull_queue::push_one()
    {
        unique_lock<mutex> lock(mutex_);
        while (count_ == capacity_)
        {
            interruption_point();
            cond_can_push_.wait(lock);
        }
        if (++ count_ == capacity_)
        {
            cond_can_pull_.notify_all();
        }
    }

// class thread_comm::push_queue

    void thread_comm::push_queue::push_all()
    {
        unique_lock<mutex> lock(mutex_);
        while (count_ != 0)
        {
            interruption_point();
            cond_can_push_.wait(lock);
        }
        count_ = capacity_;
        cond_can_pull_.notify_all();
    }

    void thread_comm::push_queue::pull_one()
    {
        unique_lock<mutex> lock(mutex_);
        while (count_ == 0)
        {
            interruption_point();
            cond_can_pull_.wait(lock);
        }
        if (-- count_ == 0)
        {
            cond_can_push_.notify_all();
        }
    }
} // end of namespace intrepid
