/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.netbeans.modules.payara.eecommon.dd.wizard;

import java.io.File;
import javax.swing.JPanel;
import org.openide.filesystems.FileUtil;
import org.openide.util.Lookup;
import org.openide.util.NbBundle;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ProjectUtils;
import org.netbeans.modules.j2ee.deployment.devmodules.api.J2eeModule;
import org.netbeans.modules.j2ee.deployment.devmodules.spi.J2eeModuleProvider;
import org.netbeans.modules.payara.spi.Utils;

/*
 *
 * @author Peter Williams
 * @author Gaurav Gupta
 */
public final class PayaraDDVisualPanel extends JPanel {
    
    private Project project;
    private String payaraDDFileName;
    private File payaraDDFile;
    private File payaraDDLocation;
    
    private static final String WEB_XML = "web.xml";

    public PayaraDDVisualPanel() {
        initComponents();
    }

    void setProject(final Project project) {
        this.project = project;

        // get list of config files for this module type
        // figure out which ones exist already
        // 
        Lookup lookup = project.getLookup();
        J2eeModuleProvider provider = lookup.lookup(J2eeModuleProvider.class);
        J2eeModule j2eeModule = provider.getJ2eeModule();
        payaraDDFileName = getConfigFileName(j2eeModule,provider.getServerInstanceID());

        // Calculate location:
        payaraDDLocation = j2eeModule.getDeploymentConfigurationFile(WEB_XML).getParentFile();
        payaraDDFile = new File(payaraDDLocation, payaraDDFileName);
        
        // initialize visual components
        textFileName.setText(payaraDDFileName); // NOI18N
        textProjectName.setText(ProjectUtils.getInformation(project).getDisplayName());

        File projectFolder = FileUtil.toFile(project.getProjectDirectory());
        textLocation.setText((payaraDDLocation != null) ? getRelativePath(payaraDDLocation, projectFolder) : null);
        // only fill 'created file' in if location is valid.
        textCreatedFile.setText((payaraDDLocation != null) ? getRelativePath(payaraDDFile, projectFolder) : null);
    }
    
    String getFileName() {
        return payaraDDFileName;
    }
    
    File getFile() {
        return payaraDDFile;
    }
    
    File getSelectedLocation() {
        return payaraDDLocation;
    }
    
    @Override
    public String getName() {
        return NbBundle.getMessage(PayaraDDVisualPanel.class, "LBL_CreatePayaraDeploymentDescriptor"); // NOI18N
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        labelFileName = new javax.swing.JLabel();
        textFileName = new javax.swing.JTextField();
        labelProjectName = new javax.swing.JLabel();
        textProjectName = new javax.swing.JTextField();
        labelLocation = new javax.swing.JLabel();
        textLocation = new javax.swing.JTextField();
        labelCreatedFile = new javax.swing.JLabel();
        textCreatedFile = new javax.swing.JTextField();
        filler1 = new javax.swing.JLabel();

        setLayout(new java.awt.GridBagLayout());

        labelFileName.setLabelFor(textFileName);
        org.openide.awt.Mnemonics.setLocalizedText(labelFileName, org.openide.util.NbBundle.getMessage(PayaraDDVisualPanel.class, "LBL_Name")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        add(labelFileName, gridBagConstraints);

        textFileName.setEditable(false);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 12, 0, 0);
        add(textFileName, gridBagConstraints);
        textFileName.getAccessibleContext().setAccessibleName(org.openide.util.NbBundle.getMessage(PayaraDDVisualPanel.class, "ASCN_Name")); // NOI18N
        textFileName.getAccessibleContext().setAccessibleDescription(org.openide.util.NbBundle.getMessage(PayaraDDVisualPanel.class, "ASCD_Name")); // NOI18N

        labelProjectName.setLabelFor(textProjectName);
        org.openide.awt.Mnemonics.setLocalizedText(labelProjectName, org.openide.util.NbBundle.getMessage(PayaraDDVisualPanel.class, "LBL_Project")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(12, 0, 0, 0);
        add(labelProjectName, gridBagConstraints);

        textProjectName.setEditable(false);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(12, 12, 0, 0);
        add(textProjectName, gridBagConstraints);
        textProjectName.getAccessibleContext().setAccessibleName(org.openide.util.NbBundle.getMessage(PayaraDDVisualPanel.class, "ASCN_Project")); // NOI18N
        textProjectName.getAccessibleContext().setAccessibleDescription(org.openide.util.NbBundle.getMessage(PayaraDDVisualPanel.class, "ASCD_Project")); // NOI18N

        labelLocation.setLabelFor(textLocation);
        org.openide.awt.Mnemonics.setLocalizedText(labelLocation, org.openide.util.NbBundle.getMessage(PayaraDDVisualPanel.class, "LBL_Location")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(12, 0, 0, 0);
        add(labelLocation, gridBagConstraints);

        textLocation.setEditable(false);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(12, 12, 0, 0);
        add(textLocation, gridBagConstraints);
        textLocation.getAccessibleContext().setAccessibleName(org.openide.util.NbBundle.getMessage(PayaraDDVisualPanel.class, "ASCN_Location")); // NOI18N
        textLocation.getAccessibleContext().setAccessibleDescription(org.openide.util.NbBundle.getMessage(PayaraDDVisualPanel.class, "ASCD_Location")); // NOI18N

        labelCreatedFile.setLabelFor(textCreatedFile);
        org.openide.awt.Mnemonics.setLocalizedText(labelCreatedFile, org.openide.util.NbBundle.getMessage(PayaraDDVisualPanel.class, "LBL_CreatedFile")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(12, 0, 11, 0);
        add(labelCreatedFile, gridBagConstraints);

        textCreatedFile.setEditable(false);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(12, 12, 11, 0);
        add(textCreatedFile, gridBagConstraints);
        textCreatedFile.getAccessibleContext().setAccessibleName(org.openide.util.NbBundle.getMessage(PayaraDDVisualPanel.class, "ASCN_CreatedFile")); // NOI18N
        textCreatedFile.getAccessibleContext().setAccessibleDescription(org.openide.util.NbBundle.getMessage(PayaraDDVisualPanel.class, "ASCD_CreatedFile")); // NOI18N

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        add(filler1, gridBagConstraints);
    }// </editor-fold>//GEN-END:initComponents
    
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JLabel filler1;
    private javax.swing.JLabel labelCreatedFile;
    private javax.swing.JLabel labelFileName;
    private javax.swing.JLabel labelLocation;
    private javax.swing.JLabel labelProjectName;
    private javax.swing.JTextField textCreatedFile;
    private javax.swing.JTextField textFileName;
    private javax.swing.JTextField textLocation;
    private javax.swing.JTextField textProjectName;
    // End of variables declaration//GEN-END:variables
    

    // TODO avoid the hard coding that is done in here

    private String getConfigFileName(J2eeModule j2eeModule,String serverInstanceID) {
        String result = null;
        Object moduleType = j2eeModule.getType();
        if(J2eeModule.Type.WAR.equals(moduleType)) {
            result = Utils.useGlassFishPrefix(serverInstanceID) ?
                    "payara-web.xml" : "glassfish-web.xml"; // NOI18N;
        } else if(J2eeModule.Type.EJB.equals(moduleType)) {
            result = Utils.useGlassFishPrefix(serverInstanceID) ?
                    "glassfish-ejb-jar.xml" : "sun-ejb-jar.xml"; // NOI18N
        } else if(J2eeModule.Type.EAR.equals(moduleType)) {
            result = Utils.useGlassFishPrefix(serverInstanceID) ?
                    "glassfish-application.xml" : "sun-application.xml"; // NOI18N
        } else if(J2eeModule.Type.CAR.equals(moduleType)) {
            result = Utils.useGlassFishPrefix(serverInstanceID) ?
                    "glassfish-application-client.xml" : "sun-application-client.xml"; // NOI18N
        }
        return result;
    }
    
    private static String getRelativePath(File file, File base) {
        String basePath = base.getAbsolutePath();
        String filePath = file.getAbsolutePath();
        
        if(filePath.startsWith(basePath)) {
            String prefix = "";
            int baseIndex = basePath.lastIndexOf(File.separatorChar);
            if(baseIndex < 0 || baseIndex >= filePath.length()) {
                baseIndex = 0;
            } else {
                prefix = "..."; // only apply prefix if we're calculating a true substring.
            }
            return prefix + filePath.substring(baseIndex);
        }
        return filePath;
    }
    
}
