TextDocs.NewDoc     k   CWindowsLeft   WindowsRight b  WindowsTop C   WindowsButtom   Color    Flat  Locked  Controls  Org F   BIER           3  i  Oberon10.Scn.Fnt     Syntax10.Scn.Fnt             Syntax10i.Scn.Fnt                     0        4        0        +        6                               _8                            )   Syntax10b.Scn.Fnt                     Q   f  (* ETH Oberon, Copyright 2001 ETH Zuerich Institut fuer Computersysteme, ETH Zentrum, CH-8092 Zuerich.
Refer to the "General ETH Oberon System Source License" contract available at: http://www.oberon.ethz.ch/ *)

(* Windows specific parts: Copyright (c) 1994 - 2000 Emil J. Zeller *)

MODULE System; (** portable, except where noted / source: Win32.System.Mod *)	(* JG/NW; MH 13.9.93, ejz   *)

(* contributions: JG 3.10.90 / NW 26.11.91 / ARD 4. 2. 93 / nm / jm / ejz *)

IMPORT SYSTEM, Kernel32, Kernel, Registry, FileDir, Files, Modules, Objects, Displays, Display, Fonts, Dates,
	Texts, Viewers, Input, Threads, Oberon, Printer, Configuration, Types, Exceptions, Strings, User32, ADVAPI32;

	CONST
		MaxString = 64; MaxArray = 8;
		SystemEditor = "TextDocs";

	TYPE
		Bytes = POINTER TO ARRAY OF CHAR;

	VAR
		W, Wt: Texts.Writer;
		pattern: ARRAY 32 OF CHAR;	(* for Directory command *)
		curpath: FileDir.FileName;	(* for Directory command *)
		detail, textDocs: BOOLEAN;
		trapper: Oberon.Task;
		trapMtx: Threads.Mutex;
		count, noDirs: LONGINT;

	PROCEDURE InitSystemEditor();
		VAR mod: Modules.Module;
	BEGIN
		textDocs := FALSE;
		IF Oberon.OpenText = NIL THEN
			mod := Modules.ThisMod(SystemEditor)
		END;
		IF Oberon.OpenText = NIL THEN
			User32.MessageBox(0, "No system editor installed.", "ETH Oberon - System.InitSystemEditor", User32.MBOk + User32.MBIconError);
			RETURN
		END;
		IF ~textDocs THEN
			mod := Kernel.GetMod(SYSTEM.VAL(LONGINT, Oberon.OpenText));
			textDocs := (mod # NIL) & (mod.name = "TextDocs")
		END
	END InitSystemEditor;

	PROCEDURE OpenText(title: ARRAY OF CHAR; T: Texts.Text; system: BOOLEAN);
		VAR W: INTEGER;
	BEGIN
		IF Oberon.OpenText = NIL THEN InitSystemEditor() END;
		IF system THEN W := Display.Width DIV 8*3 ELSE W := 400 END;
		Oberon.OpenText(title, T, W, 240)
	END OpenText;

	PROCEDURE NewText(name: ARRAY OF CHAR): Texts.Text;
		VAR T: Texts.Text;
	BEGIN
		NEW(T); Texts.Open(T, name); RETURN T
	END NewText;

	PROCEDURE OpenArgs(VAR S: Texts.Scanner; VAR end: LONGINT);
		VAR
			T: Texts.Text;
			beg, time: LONGINT;
	BEGIN
		Texts.OpenScanner(S, Oberon.Par.text, Oberon.Par.pos); Texts.Scan(S);
		IF (S.class = Texts.Char) & (S.c = "^") THEN
			Oberon.GetSelection(T, beg, end, time);
			IF time # -1 THEN Texts.OpenScanner(S, T, beg); Texts.Scan(S)
			ELSE S.class := Texts.Inval; end := -1
			END
		ELSE
			end := Oberon.Par.text.len
		END
	END OpenArgs;

(* --- Toolbox for system control *)

	PROCEDURE SetFont*;
		VAR S: Texts.Scanner; end: LONGINT; f: Fonts.Font;
	BEGIN
		OpenArgs(S, end);
		IF S.class IN {Texts.Name, Texts.String} THEN
			f := Fonts.This(S.s)
		ELSE
			f := NIL
		END;
		IF (f = NIL) & (S.lib # NIL) & (S.lib IS Fonts.Font) THEN
			f := S.lib(Fonts.Font)
		END;
		IF f # NIL THEN Oberon.SetFont(f) END
	END SetFont;

	PROCEDURE SetColor*;
		VAR S: Texts.Scanner; end: LONGINT;
	BEGIN
		OpenArgs(S, end);
		IF S.class = Texts.Int THEN
			Oberon.SetColor(SHORT(SHORT(S.i)))
		ELSE
			Oberon.SetColor(S.col)
		END
	END SetColor;

	PROCEDURE SetOffset*;
		VAR S: Texts.Scanner; end: LONGINT;
	BEGIN
		OpenArgs(S, end);
		IF S.class = Texts.Int THEN
			Oberon.SetOffset(SHORT(SHORT(S.i)))
		ELSE
			Oberon.SetOffset(S.voff)
		END
	END SetOffset;

	PROCEDURE Time*;
		VAR S: Texts.Scanner; end, t, d: LONGINT; hr, min, sec, yr, mo, day: INTEGER;
	BEGIN
		OpenArgs(S, end);
		IF S.class = Texts.Int THEN (*set date*)
			day := SHORT(S.i); Texts.Scan(S);
			mo := SHORT(S.i); Texts.Scan(S);
			yr := SHORT(S.i); Texts.Scan(S);
			hr := SHORT(S.i); Texts.Scan(S);
			min := SHORT(S.i); Texts.Scan(S);
			sec := SHORT(S.i); Texts.Scan(S);
			t := Dates.ToTime(hr, min, sec);
			d := Dates.ToDate(yr, mo, day);
			Oberon.SetClock(t, d)
		ELSE (*read date*)
			Texts.WriteString(W, "System.Time");
			Oberon.GetClock(t, d); Texts.WriteDate(W, t, d); Texts.WriteLn(W);
			Texts.Append(Oberon.Log, W.buf)
		END
	END Time;

	PROCEDURE Collect*;
	BEGIN
		Oberon.Collect()
	END Collect;

(* --- Toolbox for standard display *)

	PROCEDURE Open*;
		VAR S: Texts.Scanner; end: LONGINT;
	BEGIN
		OpenArgs(S, end);
		IF S.class IN {Texts.Name, Texts.String} THEN
			OpenText(S.s, NewText(S.s), TRUE)
		END
	END Open;

	PROCEDURE OpenLog*;
	BEGIN
		OpenText("System.Log", Oberon.Log, TRUE)
	END OpenLog;

	(** Clear the text of the current text-document *)
	PROCEDURE Clear*;
		VAR
			S: Texts.Scanner;
			par: Oberon.ParList;
			F: Display.Frame;
			L: Objects.LinkMsg;
			A: Objects.AttrMsg;
			end: LONGINT;
	BEGIN
		par := Oberon.Par; F := NIL;
		L.id := Objects.get; L.name := "Model"; L.obj := NIL;
		OpenArgs(S, end);
		IF (S.class = Texts.Char) & (S.c = "*") THEN
			F := Oberon.MarkedFrame()
		ELSIF (par.vwr # NIL) & (par.vwr.dsc # NIL) THEN
			F := par.vwr.dsc; IF F.next # NIL THEN F := F.next END; (* Control vs. Document *)
			F.handle(F, L);
			IF (L.obj # NIL) & (L.obj IS Display.Frame) THEN
				A.id := Objects.get; A.name := "Gen"; L.obj.handle(L.obj, A);
				IF A.s = "PanelDocs.NewDoc" THEN (* Desktop *)
					F := par.obj(Display.Frame);
					F := F.dlink(Display.Frame);
					F := F.next.dsc
				ELSE (* TextGadget, ... *)
					F := L.obj(Display.Frame)
				END
			END
		END;
		IF F # NIL THEN
			F.handle(F, L);
			IF (L.obj # NIL) & (L.obj IS Texts.Text) THEN
				Texts.Delete(L.obj(Texts.Text), 0, L.obj(Texts.Text).len)
			END
		END
	END Clear;

	PROCEDURE Close*;
		VAR par: Oberon.ParList; V: Viewers.Viewer;
	BEGIN par := Oberon.Par;
		IF par.frame = par.vwr.dsc THEN V := par.vwr
		ELSE V := Oberon.MarkedViewer()
		END;
		Viewers.Close(V)
	END Close;

	PROCEDURE CloseTrack*;
		VAR V: Viewers.Viewer;
	BEGIN
		V := Oberon.MarkedViewer(); Viewers.CloseTrack(V.X)
	END CloseTrack;

	PROCEDURE Recall*;
		VAR V: Viewers.Viewer; M: Display.ControlMsg;
	BEGIN
		Viewers.Recall(V);
		IF (V # NIL) & (V.state = 0) & (V.kind = Viewers.IsViewer) THEN
			Viewers.Open(V, V.X, V.Y + V.H); M.F := NIL; M.id := Display.restore; V.handle(V, M)
		END
	END Recall;

	PROCEDURE Copy*;
		VAR V, V1: Viewers.Viewer; M: Objects.CopyMsg; N: Display.ControlMsg;
	BEGIN
		M.id := Objects.shallow;
		V := Oberon.Par.vwr; V.handle(V, M); V1 := M.obj(Viewers.Viewer);
		Viewers.Open(V1, V.X, V.Y + V.H DIV 2);
		N.F := NIL; N.id := Display.restore; V1.handle(V1, N)
	END Copy;

	PROCEDURE Grow*;
		VAR par: Oberon.ParList; V, V1: Viewers.Viewer;
			M: Objects.CopyMsg; N: Display.ControlMsg;
			DW, DH: INTEGER;
	BEGIN par := Oberon.Par;
		IF par.frame = par.vwr.dsc THEN V := par.vwr
		ELSE V := Oberon.MarkedViewer()
		END;
		DW := Oberon.DisplayWidth(V.X); DH := Oberon.DisplayHeight(V.X);
		IF V.H < DH - Viewers.minH THEN Oberon.OpenTrack(V.X, V.W)
		ELSIF V.W < DW THEN Oberon.OpenTrack(Oberon.UserTrack(V.X), DW)
		END;
		IF (V.H < DH - Viewers.minH) OR (V.W < DW) THEN
			M.id := Objects.shallow;
			V.handle(V, M); V1 := M.obj(Viewers.Viewer);
			Viewers.Open(V1, V.X, DH);
			N.F := NIL; N.id := Display.restore; V1.handle(V1, N)
		END
	END Grow;

(* --- Toolbox for module management *)

	PROCEDURE Free*;
		VAR S: Texts.Scanner; i, end: LONGINT;
	BEGIN
		Texts.WriteString(W, "System.Free"); Texts.WriteLn(W);
		OpenArgs(S, end); Texts.Append(Oberon.Log, W.buf);
		WHILE (S.class = Texts.Name) & ((Texts.Pos(S)-S.len) <= end) DO
			i := 0; WHILE (S.s[i] # 0X) & (S.s[i] # ".") DO INC(i) END; S.s[i] := 0X;
			Modules.Free(S.s, FALSE);
			IF Modules.res # 0 THEN
				Texts.Write(W, " "); Texts.WriteString(W, Modules.resMsg)
			ELSE
				Texts.WriteString(W, S.s); Texts.WriteString(W, " unloaded")
			END;
			Texts.WriteLn(W); Texts.Append(Oberon.Log, W.buf);
			Texts.Scan(S);
			WHILE (S.class = Texts.Int) & ((Texts.Pos(S)-S.len) <= end) DO
				Texts.Scan(S)
			END
		END
	END Free;

	PROCEDURE ShowModules*;
		VAR T: Texts.Text; M: Modules.Module; dsize, csize, n, var, const, code: LONGINT;
	BEGIN
		T := NewText("");
(*
	Texts.WriteString(W, "MODULE"); Texts.Write(W, 9X);
	Texts.WriteString(W, "VAR"); Texts.Write(W, 9X);
	Texts.WriteString(W, "CONST"); Texts.Write(W, 9X);
	Texts.WriteString(W, "CODE"); Texts.Write(W, 9X);
	Texts.WriteString(W, "REFCNT"); Texts.Write(W, 9X);
	Texts.WriteLn(W);
*)
		M := Kernel.modules; n := 0; var := 0; const := 0; code := 0;
		WHILE M # NIL DO
			Texts.WriteString(W, M.name); Texts.Write(W, 9X);
			dsize := M.sb - SYSTEM.ADR(M.data[0]);	(* data size *)
			(*Texts.WriteInt(W, dsize, 1); Texts.Write(W, 9X);*)
			INC(var, dsize); csize := LEN(M.data)-dsize;
			(*Texts.WriteInt(W, csize, 1); Texts.Write(W, 9X);*)	(* const size *)
			INC(const, csize);
			(*Texts.WriteInt(W, LEN(M.code), 1); Texts.Write(W, 9X);*)	(* code size *)
			INC(code, LEN(M.code));
			Texts.WriteInt(W, dsize + csize + LEN(M.code), 1); Texts.Write(W, 9X);
			Texts.WriteInt(W, M.refcnt, 1); Texts.WriteLn(W);
			M := M.next; INC(n)
		END;
		IF n > 1 THEN
			Texts.WriteString(W, "TOTAL"); Texts.Write(W, 9X);
(*
		Texts.WriteInt(W, var, 1); Texts.Write(W, 9X);
		Texts.WriteInt(W, const, 1); Texts.Write(W, 9X);
		Texts.WriteInt(W, code, 1); Texts.Write(W, 9X);
*)
			Texts.WriteInt(W, var + const + code, 1); Texts.Write(W, 9X);
			Texts.WriteInt(W, n, 1); Texts.Write(W, 9X)
		END;
		Texts.WriteLn(W); Texts.Append(T, W.buf);
		OpenText("Modules | System.Free ^", T, TRUE)
	END ShowModules;

(* --- Toolbox for library management *)

	PROCEDURE *ListLibrary(L: Objects.Library);
	BEGIN
		Texts.WriteString(W, L.name); Texts.WriteLn(W); INC(count)
	END ListLibrary;

	PROCEDURE ShowLibraries*;
		VAR T: Texts.Text;
	BEGIN
		T := NewText(""); count := 0;
		Objects.Enumerate(ListLibrary);
		IF count > 1 THEN
			Texts.WriteLn(W); Texts.WriteInt(W, count, 1); Texts.WriteString(W, " public libraries")
		END;
		Texts.WriteLn(W); Texts.Append(T, W.buf);
		OpenText("Libraries", T, TRUE)
	END ShowLibraries;

	PROCEDURE FreeLibraries*;
		VAR S: Texts.Scanner; end: LONGINT;
	BEGIN
		Texts.WriteString(W, "System.FreeLibraries "); Texts.WriteLn(W);
		OpenArgs(S, end);
		WHILE (S.class = Texts.Name) & ((Texts.Pos(S)-S.len) <= end) DO
			Objects.FreeLibrary(S.s); Texts.WriteString(W, S.s); Texts.WriteLn(W);
			Texts.Scan(S)
		END;
		Texts.Append(Oberon.Log, W.buf)
	END FreeLibraries;

(* --- Toolbox of file system *)

	PROCEDURE ChangeDirectory*;	(** non-portable *)
		VAR
			S: Texts.Scanner; end: LONGINT;
			dir: FileDir.FileName; done: BOOLEAN;
	BEGIN
		OpenArgs(S, end);
		IF S.class IN {Texts.Name, Texts.String} THEN
			Texts.WriteString(W, "System.ChangeDirectory ");
			FileDir.ChangeDirectory(S.s, done);
			IF done THEN
				FileDir.GetWorkingDirectory(dir); Texts.WriteString(W, dir);
			ELSE
				Texts.WriteString(W, S.s); Texts.WriteString(W, " not found")
			END;
			Texts.WriteLn(W); Texts.Append(Oberon.Log, W.buf)
		END
	END ChangeDirectory;

	PROCEDURE CurrentDirectory*;	(** non-portable *)
		VAR dir: FileDir.FileName;
	BEGIN
		FileDir.GetWorkingDirectory(dir);
		Texts.WriteString(W, "System.CurrentDirectory "); Texts.WriteString(W, dir);
		Texts.WriteLn(W); Texts.Append(Oberon.Log, W.buf)
	END CurrentDirectory;

	PROCEDURE CreateDirectory*;	(** non-portable *)
		VAR
			S: Texts.Scanner; end: LONGINT;
			done: BOOLEAN;
	BEGIN
		OpenArgs(S, end);
		IF S.class IN {Texts.Name, Texts.String} THEN
			Texts.WriteString(W, "System.CreateDirectory "); Texts.WriteString(W, S.s);
			FileDir.CreateDirectory(S.s, done);
			IF ~done THEN Texts.WriteString(W, "failed") END;
			Texts.WriteLn(W); Texts.Append(Oberon.Log, W.buf);
		END
	END CreateDirectory;

	PROCEDURE DeleteDirectory*;	(** non-portable *)
		VAR
			S: Texts.Scanner; end: LONGINT;
			done: BOOLEAN;
	BEGIN
		OpenArgs(S, end);
		IF S.class IN {Texts.Name, Texts.String} THEN
			Texts.WriteString(W, "System.DeleteDirectory "); Texts.WriteString(W, S.s);
			FileDir.DeleteDirectory(S.s, done);
			IF ~done THEN Texts.WriteString(W, "failed") END;
			Texts.WriteLn(W); Texts.Append(Oberon.Log, W.buf);
		END
	END DeleteDirectory;

	PROCEDURE CopyFile(name: ARRAY OF CHAR; VAR S: Texts.Scanner);
		VAR res: INTEGER;
	BEGIN
		Texts.Scan(S);
		IF (S.class = Texts.Char) & (S.c = "=") THEN
			Texts.Scan(S);
			IF (S.class = Texts.Char) & (S.c = ">") THEN
				Texts.Scan(S);
				IF S.class IN {Texts.Name, Texts.String} THEN
					Texts.WriteString(W, name); Texts.WriteString(W, " => "); Texts.WriteString(W, S.s);
					Texts.WriteString(W, " copying"); Texts.Append(Oberon.Log, W.buf);
					Files.Copy(name, S.s, res);
					IF res # 0 THEN Texts.WriteString(W, " failed"); S.eot := TRUE END;
					Texts.WriteLn(W); Texts.Append(Oberon.Log, W.buf)
				END
			END
		END
	END CopyFile;

	PROCEDURE CopyFiles*;
		VAR S: Texts.Scanner; end: LONGINT;
	BEGIN
		Texts.WriteString(W, "System.CopyFiles"); Texts.WriteLn(W);
		OpenArgs(S, end); Texts.Append(Oberon.Log, W.buf);
		WHILE ~S.eot & (S.class IN {Texts.Name, Texts.String}) & ((Texts.Pos(S)-S.len) <= end) DO
			CopyFile(S.s, S); IF ~S.eot THEN Texts.Scan(S) END
		END
	END CopyFiles;

	PROCEDURE RenameFile(name: ARRAY OF CHAR; VAR S: Texts.Scanner);
		VAR res: INTEGER;
	BEGIN
		Texts.Scan(S);
		IF (S.class = Texts.Char) & (S.c = "=") THEN Texts.Scan(S);
			IF (S.class = Texts.Char) & (S.c = ">") THEN Texts.Scan(S);
				IF S.class IN {Texts.Name, Texts.String} THEN
					Texts.WriteString(W, name); Texts.WriteString(W, " => "); Texts.WriteString(W, S.s);
					Texts.WriteString(W, " renaming"); Texts.Append(Oberon.Log, W.buf);
					Files.Rename(name, S.s, res);
					IF res # 0 THEN Texts.WriteString(W, " failed"); S.eot := TRUE END;
					Texts.WriteLn(W); Texts.Append(Oberon.Log, W.buf)
				END
			END
		END
	END RenameFile;

	PROCEDURE RenameFiles*;
		VAR S: Texts.Scanner; end: LONGINT;
	BEGIN
		Texts.WriteString(W, "System.RenameFiles"); Texts.WriteLn(W);
		OpenArgs(S, end); Texts.Append(Oberon.Log, W.buf);
		WHILE ~S.eot & (S.class IN {Texts.Name, Texts.String}) & ((Texts.Pos(S)-S.len) <= end) DO
			RenameFile(S.s, S); IF ~S.eot THEN Texts.Scan(S) END
		END
	END RenameFiles;

	PROCEDURE DeleteFile(name: ARRAY OF CHAR; VAR S: Texts.Scanner);
		VAR res: INTEGER;
	BEGIN
		Texts.WriteString(W, name); Texts.WriteString(W, " deleting"); Texts.Append(Oberon.Log, W.buf);
		Files.Delete(name, res);
		IF res # 0 THEN Texts.WriteString(W, " failed"); S.eot := TRUE END;
		Texts.WriteLn(W); Texts.Append(Oberon.Log, W.buf)
	END DeleteFile;

	PROCEDURE DeleteFiles*;
		VAR S: Texts.Scanner; end: LONGINT;
	BEGIN
		Texts.WriteString(W, "System.DeleteFiles"); Texts.WriteLn(W);
		OpenArgs(S, end); Texts.Append(Oberon.Log, W.buf);
		WHILE ~S.eot & (S.class IN {Texts.Name, Texts.String}) & ((Texts.Pos(S)-S.len) <= end) DO
			DeleteFile(S.s, S); IF ~S.eot THEN Texts.Scan(S) END
		END
	END DeleteFiles;

	PROCEDURE WriteFileAttributes(attrs: SET; dir: BOOLEAN);
	BEGIN
		IF dir & (FileDir.Directory IN attrs) THEN Texts.WriteString(W, "  directory") END;
		IF FileDir.Readonly IN attrs THEN Texts.WriteString(W, "  readonly") END;
		IF FileDir.Hidden IN attrs THEN Texts.WriteString(W, "  hidden") END;
		IF FileDir.System IN attrs THEN Texts.WriteString(W, "  system") END;
		IF FileDir.Archive IN attrs THEN Texts.WriteString(W, "  archive") END
	END WriteFileAttributes;

	PROCEDURE UpdateFileAttributes(name: ARRAY OF CHAR; incl, excl: SET; VAR S: Texts.Scanner);
		VAR file: FileDir.FileName; old, new: SET;
	BEGIN
		IF FileDir.FindFile(name, file, TRUE) THEN
			Texts.WriteString(W, file);
			old := FileDir.GetAttributes(file);
			new := old + incl - excl;
			IF new # old THEN
				FileDir.SetAttributes(file, new);
				old := FileDir.GetAttributes(file)
			END;
			WriteFileAttributes(old, TRUE)
		ELSE
			Texts.WriteString(W, name); Texts.WriteString(W, " failed"); S.eot := TRUE
		END;
		Texts.WriteLn(W); Texts.Append(Oberon.Log, W.buf)
	END UpdateFileAttributes;

	PROCEDURE FileAttributes*;
		VAR S: Texts.Scanner; end, attr: LONGINT; incl, excl: SET; op: CHAR;
	BEGIN
		Texts.WriteString(W, "System.FileAttributes"); Texts.WriteLn(W);
		OpenArgs(S, end); Texts.Append(Oberon.Log, W.buf);
		incl := {}; excl := {};
		WHILE ~S.eot & (S.class = Texts.Char) & ((S.c = "+") OR (S.c = "-")) DO
			op := S.c; Texts.Scan(S);
			IF S.class = Texts.Name THEN
				IF CAP(S.s[0]) = "R" THEN
					attr := FileDir.Readonly
				ELSIF CAP(S.s[0]) = "A" THEN
					attr := FileDir.Archive
				ELSIF CAP(S.s[0]) = "H" THEN
					attr := FileDir.Hidden
				ELSIF CAP(S.s[0]) = "S" THEN
					attr := FileDir.System
				ELSE
					attr := -1
				END;
				IF attr >= 0 THEN
					IF op = "+" THEN
						INCL(incl, attr)
					ELSIF op = "-" THEN
						INCL(excl, attr)
					END
				END;
				Texts.Scan(S)
			END
		END;
		WHILE ~S.eot & (S.class IN {Texts.Name, Texts.String}) & ((Texts.Pos(S)-S.len) <= end) DO
			UpdateFileAttributes(S.s, incl, excl, S); IF ~S.eot THEN Texts.Scan(S) END
		END	
	END FileAttributes;

	PROCEDURE *Enumerator(path, name: ARRAY OF CHAR; time, date, size: LONGINT; attrs: SET);
		VAR
			A: Objects.AttrMsg;
			t, d: LONGINT;
			res: INTEGER;
	BEGIN
		IF (FileDir.Directory IN attrs) & textDocs THEN
			Texts.SetColor(W, (*Display3.blue*)3)
		END;
		IF path # curpath THEN
			Texts.WriteString(W, path); Texts.Write(W, FileDir.PathChar)
		END;
		Texts.WriteString(W, name);
		IF detail THEN
			Texts.WriteString(W, "  "); Texts.WriteDate(W, time, date);
			IF ~(FileDir.Directory IN attrs) THEN Texts.WriteString(W, "  "); Texts.WriteInt(W, size, 0) END;
			WriteFileAttributes(attrs, FALSE)
		END;
		IF FileDir.Directory IN attrs THEN
			INC(noDirs);
			IF textDocs THEN
				Oberon.Call("TextGadgets.NewControl", Oberon.Par, FALSE, res);
				IF res = 0 THEN
					A.name := "Cmd"; A.class := Objects.String; A.id := Objects.set; A.s := "System.Directory ";
					t := 0; WHILE A.s[t] # 0X DO INC(t) END;
					A.s[t] := 022X; INC(t);
					d := 0;
					WHILE (path[d] # 0X) & (t < 61) DO
						A.s[t] := path[d]; INC(t); INC(d)
					END;
					IF (A.s[t-1] # FileDir.PathChar) & (t < 61) THEN
						A.s[t] := FileDir.PathChar; INC(t)
					END;
					d := 0;
					WHILE (name[d] # 0X) & (t < 61) DO
						A.s[t] := name[d]; INC(t); INC(d)
					END;
					d := 0;
					A.s[t] := FileDir.PathChar; INC(t);
					WHILE (pattern[d] # 0X) & (t < 62) DO
						A.s[t] := pattern[d]; INC(t); INC(d)
					END;
					A.s[t] := 022X; INC(t); A.s[t] := 0X;
					Objects.NewObj.handle(Objects.NewObj, A);
					Texts.WriteObj(W, Objects.NewObj)
				END;
				Texts.SetColor(W, (*Display3.textC*)Display.FG)
			ELSE
				Texts.WriteString(W, "   <DIR>")
			END
		ELSE
			INC(count)
		END;
		Texts.WriteLn(W)
	END Enumerator;

	PROCEDURE Directory*;
		VAR
			T: Texts.Text; R: Texts.Reader;
			path, dirpath: FileDir.FileName;
			Title: ARRAY 64 OF CHAR;
			beg, end, time, pos, i, j: LONGINT;
			ch: CHAR; quoted, tdocs: BOOLEAN;
	BEGIN
		Texts.OpenReader(R, Oberon.Par.text, Oberon.Par.pos); Texts.Read(R, ch);
		WHILE ~R.eot & (ch <= " ") & (ch # 0DX) DO Texts.Read(R, ch) END;
		IF (ch = "^") OR (ch = 0DX) THEN
			Oberon.GetSelection(T, beg, end, time);
			IF time # -1 THEN
				Texts.OpenReader(R, T, beg); Texts.Read(R, ch);
				WHILE ~R.eot & (ch <= " ") DO Texts.Read(R, ch) END
			END
		END;
		i := 0; j := 0; pos := 0;
		IF ch = 022X THEN Texts.Read(R, ch); quoted := TRUE ELSE quoted := FALSE END;
		WHILE ~R.eot & ((~quoted & (ch > " ")) OR (quoted & (ch # 022X))) & (ch # Oberon.OptionChar) DO
			path[i] := ch; pattern[j] := ch; INC(j);
			IF ch = FileDir.PathChar THEN pos := i; j := 0 END;
			INC(i); Texts.Read(R, ch)
		END;
		pattern[j] := 0X;
		IF pattern = "" THEN pattern := "*" END;
		IF pos = 0 THEN (* no path *)
			path[0] := 0X
		ELSIF (pos = 0) OR (pos = 2) & (path[1] = ":") THEN (* keep trailing \ *)
			path[pos+1] := 0X
		ELSE (* cut last \ *)
			path[pos] := 0X
		END;
		IF ch = Oberon.OptionChar THEN
			Texts.Read(R, ch); detail := CAP(ch) = "D"
		ELSE
			detail := FALSE
		END;
		T := NewText(""); i := 0;
		Title := "System.Directory "; (* 17 chars *)
		WHILE pattern[i] # 0X DO Title[i+17] := pattern[i]; INC(i) END;
		Title[i+17] := 0X;
		FileDir.GetWorkingDirectory(curpath);
		COPY(path, dirpath);
		IF path # "" THEN
			Texts.WriteString(W, "Directory of "); Texts.WriteString(W, path);
			Texts.WriteLn(W); Texts.WriteLn(W); Texts.Append(T, W.buf)
		END;
		count := 0; noDirs := 0;
		tdocs := textDocs; textDocs := path # "";
		FileDir.EnumerateFiles(dirpath, pattern, detail, Enumerator);
		textDocs := tdocs;
		Texts.WriteLn(W); Texts.WriteInt(W, count, 0); Texts.WriteString(W, " Files");
		Texts.Write(W, 09X); Texts.WriteInt(W, noDirs, 0); Texts.WriteString(W, " Directories");
		Texts.WriteLn(W); Texts.Append(T, W.buf);
		OpenText(Title, T, TRUE)
	END Directory;

(* --- Toolbox for system inspection *)

PROCEDURE WriteK(VAR W: Texts.Writer; k: LONGINT);
VAR suffix: CHAR;
BEGIN
	IF k < 10*1024 THEN suffix := "K"
	ELSIF k < 10*1024*1024 THEN suffix := "M"; k := k DIV 1024
	ELSE suffix := "G"; k := k DIV (1024*1024)
	END;
	Texts.WriteInt(W, k, 1); Texts.Write(W, suffix); Texts.Write(W, "b")
END WriteK;

PROCEDURE Watch*;
VAR free, total, largest: LONGINT;
BEGIN
	Texts.WriteString(W, "System.Watch"); Texts.WriteLn(W);
	free := (Kernel.Available()+512) DIV 1024;
	total := (Kernel.Available()+Kernel.Used()+512) DIV 1024;
	largest := (Kernel.LargestAvailable()+512) DIV 1024;
	Texts.Write(W, 9X); Texts.WriteString(W, "Heap has ");
	WriteK(W, free); Texts.WriteString(W, " of ");
	WriteK(W, total); Texts.WriteString(W, " free (");
	WriteK(W, largest); Texts.WriteString(W, " contiguous)"); Texts.WriteLn(W);
	Texts.Append(Oberon.Log, W.buf)
END Watch;

PROCEDURE GetNum(refs: Bytes; VAR i, num: LONGINT);
VAR n, s: LONGINT; x: CHAR;
BEGIN
	s := 0; n := 0; x := refs[i]; INC(i);
	WHILE ORD(x) >= 128 DO
		INC(n, ASH(ORD(x) - 128, s)); INC(s, 7); x := refs[i]; INC(i)
	END;
	num := n + ASH(ORD(x) MOD 64 - ORD(x) DIV 64 * 64, s)
END GetNum;

(*
	Reference = {OldRef | ProcRef} .
	OldRef = 0F8X offset/n name/s {Variable} .
	ProcRef = 0F9X offset/n nofPars/n RetType procLev/1 slFlag/1 name/s {Variable} .
	RetType = 0X | Var | ArrayType | Record .
	ArrayType = 12X | 14X | 15X .	(* static array, dynamic array, open array *)
	Record = 16X .
	Variable = VarMode (Var | ArrayVar | RecordVar ) offset/n name/s .
	VarMode = 1X | 3X .	(* direct, indirect *)
	Var = 1X .. 0FX .	(* byte, boolean, char, shortint, integer, longint, real, longreal, set, ptr, proc, string *)
	ArrayVar = (81X .. 8EX) dim/n .	(* byte, boolean, char, shortint, integer, longint, real, longreal, set, ptr, proc *)
	RecordVar = (16X | 1DX) tdadr/n .	(* record, recordpointer *)
*)

(* FindProc - Find a procedure in the reference block. Return index of name, or -1 if not found. *)

PROCEDURE FindProc(refs: Bytes; ofs: LONGINT): LONGINT;
VAR i, m, t, proc: LONGINT; ch: CHAR;
BEGIN
	proc := -1; i := 0; m := LEN(refs^);
	ch := refs[i]; INC(i);
	WHILE (i < m) & ((ch = 0F8X) OR (ch = 0F9X)) DO	(* proc *)
		GetNum(refs, i, t);	(* pofs *)
		IF t > ofs THEN	(* previous procedure was the one *)
			ch := 0X	(* stop search *)
		ELSE	(* ~found *)
			IF ch = 0F9X THEN
				GetNum(refs, i, t);	(* nofPars *)
				INC(i, 3)	(* RetType, procLev, slFlag *)
			END;
			proc := i;	(* remember this position, just before the name *)
			REPEAT ch := refs[i]; INC(i) UNTIL ch = 0X;	(* pname *)
			IF i < m THEN
				ch := refs[i]; INC(i);	(* 1X | 3X | 0F8X | 0F9X *)
				WHILE (i < m) & (ch >= 1X) & (ch <= 3X) DO	(* var *)
					ch := refs[i]; INC(i);	(* type *)
					IF (ch >= 81X) OR (ch = 16X) OR (ch = 1DX) THEN
						GetNum(refs, i, t)	(* dim/tdadr *)
					END;
					GetNum(refs, i, t);	(* vofs *)
					REPEAT ch := refs[i]; INC(i) UNTIL ch = 0X;	(* vname *)
					IF i < m THEN ch := refs[i]; INC(i) END	(* 1X | 3X | 0F8X | 0F9X *)
				END
			END
		END
	END;
	IF (proc = -1) & (i # 0) THEN proc := i END;	(* first procedure *)
	RETURN proc
END FindProc;

(*
(*
	refs = { 0F8X pofs pname { mode type [ dim ] vofs vname } } 	(* use refblk length for termination *)
	pofs = num .	(* procedure offset *)
	pname = string .	(* procedure name *)
	mode = 1X | 3X .	(* 1X = direct, 3X = indirect *)
	type = 1X .. 0FX | 81X..8EX .	(* byte, boolean, char, shortint, integer, longint, real, longreal, set, ptr, proc, string *)
	vofs = num .	(* variable offset *)
	vname = string .	(* variable name *)
*)

(* FindProc - Find a procedure in the reference block. Return index of name, or -1 if not found. *)

PROCEDURE FindProc(refs: Bytes; ofs: LONGINT): LONGINT;
VAR i, m, t, proc: LONGINT; ch: CHAR;
BEGIN
	proc := -1; i := 0; m := LEN(refs^);
	ch := refs[i]; INC(i);
	WHILE (i < m) & (ch = 0F8X) DO	(* proc *)
		GetNum(refs, i, t);	(* pofs *)
		IF t > ofs THEN	(* previous procedure was the one *)
			ch := 0X	(* stop search *)
		ELSE	(* ~found *)
			proc := i;	(* remember this position, just before the name *)
			REPEAT ch := refs[i]; INC(i) UNTIL ch = 0X;	(* pname *)
			IF i < m THEN
				ch := refs[i]; INC(i);	(* 1X | 3X | 0F8X *)
				WHILE (i < m) & (ch >= 1X) & (ch <= 3X) DO	(* var *)
					ch := refs[i]; INC(i);	(* type *)
					IF ch >= 81X THEN GetNum(refs, i, t) END;	(* dim *)
					GetNum(refs, i, t);	(* vofs *)
					REPEAT ch := refs[i]; INC(i) UNTIL ch = 0X;	(* vname *)
					IF i < m THEN ch := refs[i]; INC(i) END	(* 1X | 3X | 0F8X *)
				END
			END
		END
	END;
	IF (proc = -1) & (i # 0) THEN proc := i END;	(* first procedure *)
	RETURN proc
END FindProc;
*)

PROCEDURE WriteProc(VAR W: Texts.Writer; mod: Modules.Module; pc, fp: LONGINT; VAR refs: Bytes; VAR refpos, base: LONGINT);
VAR ch: CHAR;
BEGIN
	refpos := -1;
	IF mod = NIL THEN
		Texts.WriteString(W, "Unknown EIP ="); Texts.WriteHex(W, pc); Texts.Write(W, "H");
		IF fp # -1 THEN
			Texts.WriteString(W, " EBP ="); Texts.WriteHex(W, fp); Texts.Write(W, "H")
		END
	ELSE
		Texts.WriteString(W, mod.name);
		DEC(pc, SYSTEM.ADR(mod.code[0]));
		refs := SYSTEM.VAL(Bytes, mod.refs);
		IF (refs # NIL) & (LEN(mod.refs) # 0) THEN
			refpos := FindProc(refs, pc);
			IF refpos # -1 THEN
				Texts.Write(W, ".");
				ch := refs[refpos]; INC(refpos);
				IF ch = "$" THEN base := mod.sb ELSE base := fp END;	(* for variables *)
				WHILE ch # 0X DO Texts.Write(W, ch); ch := refs[refpos]; INC(refpos) END
			END
		END;
		Texts.WriteString(W, "  PC = "); Texts.WriteInt(W, pc, 1)
	END
END WriteProc;

PROCEDURE GetTypeName(ptr: LONGINT; VAR name: ARRAY OF CHAR);
	VAR tag, i, n: LONGINT; type: Types.Type;
BEGIN
	IF ptr > 1024*1024 THEN
		SYSTEM.GET(ptr-4, tag);
		IF tag > 1024*1024 THEN
			SYSTEM.GET(tag-4, tag);
			IF tag > 1024*1024 THEN
				type := SYSTEM.VAL(Types.Type, tag);
				IF type.module # NIL THEN
					n := LEN(name);
					Kernel32.ReadProcessMemory(Kernel32.GetCurrentProcess(), SYSTEM.ADR(type.module.name[0]), name, n, i);
					DEC(n); name[n] := 0X; i := 0;
					WHILE (i < n) & (name[i] > " ") DO
						INC(i)
					END;
					name[i] := "."; name[i+1] := 0X
				END;
				Strings.Append(name, type.name); RETURN
			END
		END
	END;
	COPY("", name)
END GetTypeName;

PROCEDURE WriteType(VAR W: Texts.Writer; dynamic, static: LONGINT);
	VAR dname, sname: ARRAY 64 OF CHAR;
BEGIN
	GetTypeName(dynamic, dname);
	IF dname # "" THEN
		Texts.Write(W, " "); Texts.WriteString(W, dname)
	END;
	GetTypeName(static, sname);
	IF (sname # dname) & (sname # "") THEN
		Texts.WriteString(W, " ("); Texts.WriteString(W, sname); Texts.Write(W, ")")
	END
END WriteType;

PROCEDURE Variables(VAR W: Texts.Writer; mod: Modules.Module; refs: Bytes; i, base: LONGINT);
VAR
	mode, ch: CHAR; m, adr, type, n, lval, size, tmp1, tmp2, tdadr0, tdadr1: LONGINT; etc: BOOLEAN;
	sval: SHORTINT; ival: INTEGER; tmp: Bytes; set: SET; rval: REAL; lrval: LONGREAL;
BEGIN
	m := LEN(refs^); mode := refs[i]; INC(i);
	WHILE (i < m) & (mode >= 1X) & (mode <= 3X) DO	(* var *)
		tdadr0 := 0; tdadr1 := 0;
		type := ORD(refs[i]); INC(i); etc := FALSE;
		IF type > 80H THEN
			IF type = 83H THEN type := 15 ELSE DEC(type, 80H) END;
			GetNum(refs, i, n)
		ELSIF (type = 16H) OR (type = 1DH) THEN
			GetNum(refs, i, tdadr0); n := 1;
			IF (tdadr0 >= 0) & (tdadr0 <= 1024*1024) THEN
				SYSTEM.GET(mod.sb+tdadr0, tdadr0)
			END
		ELSE
			IF type = 15 THEN n := MaxString (* best guess *) ELSE n := 1 END
		END;
		GetNum(refs, i, adr);
		Texts.Write(W, 9X); ch := refs[i]; INC(i);
		WHILE ch # 0X DO Texts.Write(W, ch); ch := refs[i]; INC(i) END;
		Texts.WriteString(W, " = ");
		INC(adr, base);
		IF n = 0 THEN	(* open array *)
			SYSTEM.GET(adr+4, n)	(* real LEN from stack *)
		END;
		IF type = 15 THEN
			IF n > MaxString THEN etc := TRUE; n := MaxString END
		ELSE
			IF n > MaxArray THEN etc := TRUE; n := MaxArray END
		END;
		IF mode # 1X THEN SYSTEM.GET(adr, adr) END;	(* indirect *)
		IF (adr >= -4) & (adr < 4096) THEN
			Texts.WriteString(W, "NIL reference ("); Texts.WriteHex(W, adr); Texts.WriteString(W, "H )")
		ELSE
			IF type = 15 THEN
				Texts.Write(W, 22X);
				LOOP
					IF n = 0 THEN EXIT END;
					SYSTEM.GET(adr, ch); INC(adr);
					IF (ch < " ") OR (ch > "~") THEN EXIT END;
					Texts.Write(W, ch); DEC(n)
				END;
				Texts.Write(W, 22X); etc := (ch # 0X)
			ELSE
				CASE type OF
					1..4: size := 1
					|5: size := 2
					|6..7,9,13,14,29: size := 4
					|8, 16: size := 8
					|22: size := 0; ASSERT(n <= 1)
				ELSE
					Texts.WriteString(W, "bad type "); Texts.WriteInt(W, type, 1); n := 0
				END;
				WHILE n > 0 DO
					CASE type OF
						1,3:	(* BYTE, CHAR *)
							SYSTEM.GET(adr, ch);
							IF (ch > " ") & (ch <= "~") THEN Texts.Write(W, ch)
							ELSE Texts.WriteHex(W, ORD(ch)); Texts.Write(W, "X")
							END
						|2:	(* BOOLEAN *)
							SYSTEM.GET(adr, ch);
							IF ch = 0X THEN Texts.WriteString(W, "FALSE")
							ELSIF ch = 1X THEN Texts.WriteString(W, "TRUE")
							ELSE Texts.WriteInt(W, ORD(ch), 1)
							END
						|4:	(* SHORTINT *)
							SYSTEM.GET(adr, sval); Texts.WriteInt(W, sval, 1)
						|5:	(* INTEGER *)
							SYSTEM.GET(adr, ival); Texts.WriteInt(W, ival, 1)
						|6:	(* LONGINT *)
							SYSTEM.GET(adr, lval); Texts.WriteInt(W, lval, 1)
						|7:	(* REAL *)
							SYSTEM.GET(adr, rval); Texts.WriteReal(W, rval, 15)
						|8:	(* LONGREAL *)
							SYSTEM.GET(adr, lrval); Texts.WriteLongReal(W, lrval, 24)
						|9:	(* SET *)
							SYSTEM.GET(adr, set); Texts.WriteSet(W, set)
						|13, 29:	(* POINTER *)
							SYSTEM.GET(adr, tdadr1); Texts.WriteHex(W, tdadr1); Texts.Write(W, "H");
							IF type = 13 THEN tdadr1 := 0 END
						|22:	(* RECORD *)
							tdadr1 := tdadr0; Texts.WriteHex(W, tdadr1); Texts.Write(W, "H")
						|14:	(* PROC *)
							SYSTEM.GET(adr, lval);
							IF lval = 0 THEN Texts.WriteString(W, "NIL")
							ELSE WriteProc(W, Kernel.GetMod(lval), lval, -1, tmp, tmp1, tmp2)
							END
						|16:	(* HUGEINT *)
							SYSTEM.GET(adr+4, lval); Texts.WriteHex(W, lval);
							SYSTEM.GET(adr, lval); Texts.WriteHex(W, lval)
					END;
					IF type IN {13, 22, 29} THEN
						WriteType(W, tdadr1, tdadr0)						
					END;
					DEC(n); INC(adr, size);
					IF n > 0 THEN Texts.WriteString(W, ", ") END
				END
			END
		END;
		IF etc THEN Texts.WriteString(W, " ...") END;
		Texts.WriteLn(W);
		IF i < m THEN mode := refs[i]; INC(i) END
	END
END Variables;

PROCEDURE OutState (VAR name: ARRAY OF CHAR; t: Texts.Text);
VAR mod: Modules.Module; refpos, i: LONGINT; refs: Bytes; ch: CHAR;
BEGIN
	i := 0; WHILE (name[i] # 0X) & (name[i] # ".") DO INC(i) END;
	name[i] := 0X;
	Texts.WriteString(W, name); mod := Kernel.modules;
	WHILE (mod # NIL) & (mod.name # name) DO mod := mod.next END;
	IF mod # NIL THEN
		Texts.WriteString(W, "  SB ="); Texts.WriteHex(W, mod.sb); Texts.Write(W, "H"); Texts.WriteLn(W);
		refs := SYSTEM.VAL(Bytes, mod.refs);
		IF (refs # NIL) & (LEN(mod.refs) # 0) THEN
			refpos := FindProc(refs, 0);	(* assume module body is at PC = 0 *)
			IF refpos # -1 THEN
				REPEAT ch := refs[refpos]; INC(refpos) UNTIL ch = 0X;
				Variables(W, mod, refs, refpos, mod.sb)
			END
		END
	ELSE
		Texts.WriteString(W, " not loaded"); Texts.WriteLn(W)
	END;
	Texts.Append(t, W.buf)
END OutState;

PROCEDURE State*;
	VAR T: Texts.Text; S: Texts.Scanner; end: LONGINT;
BEGIN
	OpenArgs(S, end);
	IF S.class = Texts.Name THEN
		T := NewText(""); OutState(S.s, T);
		OpenText("State", T, TRUE)
	END
END State;

PROCEDURE ShowCommands*;
VAR M: Modules.Module; beg, end, time: LONGINT; T: Texts.Text; S: Texts.Scanner; i: INTEGER;
BEGIN
	Texts.OpenScanner(S, Oberon.Par.text, Oberon.Par.pos); Texts.Scan(S);
	IF (S.class = Texts.Char) & (S.c = "^") THEN
		Oberon.GetSelection(T, beg, end, time);
		IF time # -1 THEN Texts.OpenScanner(S, T, beg); Texts.Scan(S) END
	END;
	IF S.class = Texts.Name THEN
		i := 0; WHILE (S.s[i] # 0X) & (S.s[i] # ".") DO INC(i) END; S.s[i] := 0X;
		M := Modules.ThisMod(S.s);
		IF M # NIL THEN
			T := NewText("");
			i := 0; 
			WHILE i < LEN(M.cmds) DO
				Texts.WriteString(W, S.s); Texts.Write(W, "."); 
				Texts.WriteString(W, M.cmds[i].name); 
				Texts.WriteLn(W); INC(i)
			END;
			Texts.Append(T, W.buf);
			OpenText("Commands", T, TRUE)
		ELSE
			Texts.WriteString(W, Modules.resMsg); Texts.WriteLn(W);
			Texts.Append(Oberon.Log, W.buf)
		END
	END
END ShowCommands;

PROCEDURE ShowTasks*;
VAR T: Texts.Text; n: Oberon.Task; ofs, t: LONGINT; m: Modules.Module;
BEGIN
	n := Oberon.NextTask; t := Input.Time();
	REPEAT
		ofs := SYSTEM.VAL(LONGINT, n.handle); m := Kernel.GetMod(ofs);
		Texts.WriteString(W, m.name); Texts.WriteString(W, "  PC = ");
		Texts.WriteInt(W, ofs-SYSTEM.ADR(m.code[0]), 1);
		IF n.safe THEN Texts.WriteString(W, "  safe  ")
		ELSE Texts.WriteString(W, "  unsafe  ")
		END;
		Texts.WriteInt(W, n.time, 1);
		IF n.time - t <= 0 THEN 
			Texts.WriteString(W, " ready")
		ELSE
			Texts.WriteString(W, " waiting "); Texts.WriteInt(W, (n.time-t)*1000 DIV Input.TimeUnit, 1);
			Texts.WriteString(W, "ms")
		END;
		Texts.WriteLn(W);
		n := n.next
	UNTIL n = Oberon.NextTask;
	T := NewText("");
	Texts.Append(T, W.buf);
	OpenText("Tasks", T, TRUE)
END ShowTasks;

	PROCEDURE *DisplayTrap(VAR e: Exceptions.Exception);
		VAR
			pc, fp, lastfp, refpos, base, stackBottom: LONGINT;
			mod: Modules.Module; refs: Bytes;
			desc: ARRAY 64 OF CHAR;
	BEGIN
		Threads.Lock(trapMtx);
		pc := e.exc.ExceptionAddress; fp := e.cont.Ebp;
		IF pc = Kernel32.NULL THEN (* assume call of procedure variable with value NIL *)
			SYSTEM.GET(e.cont.Esp, pc) (* get return address on top of stack *)
		END;
		Exceptions.GetDescription(e, desc);
		Texts.WriteString(Wt, "TRAP "); Texts.WriteString(Wt, desc); Texts.WriteString(Wt, " in thread ");
		IF e.t # NIL THEN
			stackBottom := e.t.stackBottom; Texts.WriteString(Wt, e.t.name)
		ELSE
			stackBottom := 1024*1024; Texts.WriteString(Wt, "unknown thread")
		END;
		Texts.WriteLn(Wt); Texts.WriteLn(Wt);
		(* stack dump *)
		mod := Kernel.GetMod(pc);
		LOOP
			WriteProc(Wt, mod, pc, fp, refs, refpos, base); Texts.WriteLn(Wt);
			IF refpos # -1 THEN Variables(Wt, mod, refs, refpos, base) END;
			lastfp := fp;
			SYSTEM.GET(fp+4, pc); SYSTEM.GET(fp, fp); (* return addr from stack *)
			IF (fp < lastfp) OR (fp >= stackBottom) THEN EXIT END; (* not called from stack *)
			mod := Kernel.GetMod(pc)
		END;
		Threads.Unlock(trapMtx);
		trapper.time := Input.Time(); Kernel32.SetEvent(Displays.eventObj.handle)
	END DisplayTrap;

	PROCEDURE *Trapper(me: Oberon.Task);
		VAR T: Texts.Text;
	BEGIN
		IF Wt.buf.len > 1 THEN
			Viewers.Close(NIL);
			Threads.Lock(trapMtx);
			T := NewText(""); Texts.Append(T, Wt.buf);
			Texts.Delete(T, 0, 1); Texts.WriteLn(Wt);
			OpenText("System.Trap", T, TRUE);
			Threads.Unlock(trapMtx)
		END;
		trapper.time := Input.Time() + MAX(LONGINT) DIV 2
	END Trapper;

	PROCEDURE Quit*;
	BEGIN
		IF Kernel.isEXE & Kernel.CanShutdown() THEN
			Kernel.Shutdown(0)
		ELSE
			HALT(99)
		END
	END Quit;

	PROCEDURE *Show(t: Threads.Thread);
		VAR prio: LONGINT;
	BEGIN
		Texts.WriteString(W, t.name);
		Threads.GetPriority(t, prio); Texts.Write(W, 09X);
		IF prio < Threads.Normal THEN
			Texts.WriteString(W, "low")
		ELSIF prio > Threads.Normal THEN
			Texts.WriteString(W, "high")
		ELSE
			Texts.WriteString(W, "normal")
		END;
		IF t.safe THEN
			Texts.Write(W, 09X); Texts.WriteString(W, "safe")
		END;
		Texts.WriteLn(W)
	END Show;

	PROCEDURE ShowThreads*;	(** non-portable *)
		VAR T: Texts.Text;
	BEGIN
		Threads.Enumerate(Show);
		T := NewText("");
		Texts.Append(T, W.buf);
		OpenText("Threads", T, TRUE)
	END ShowThreads;

(* --- Toolbox for registry manilulation *)

	PROCEDURE WriteRegistryError();
	BEGIN
		IF Registry.res = Registry.NotFound THEN Texts.WriteString(W, "(undefined section or key)")
		END
	END WriteRegistryError;

	PROCEDURE WriteTriple(section, key, value: ARRAY OF CHAR);
	BEGIN
		Texts.WriteString(W, section); Texts.Write(W, " ");
		Texts.WriteString(W, key);
		Texts.WriteString(W, " := ");
		Texts.WriteString(W, value)
	END WriteTriple;

	(** put a value into the registry
		^ | path key ":=" value *)
	PROCEDURE Set*;	(** non-portable *)
		VAR
			S: Texts.Scanner;
			section, key, value: ARRAY 64 OF CHAR;
			path: ARRAY 128 OF CHAR;
			end: LONGINT;
			line: INTEGER;
			done, quote: BOOLEAN;
		PROCEDURE ReadAssign(VAR done: BOOLEAN; colon: BOOLEAN);
		BEGIN
			done := FALSE; Texts.Scan(S);
			IF colon THEN
				IF (S.class = Texts.Char) & (S.c = ":") THEN
					Texts.Scan(S)
				ELSE
					RETURN
				END
			END;
			done := (S.class = Texts.Char) & (S.c = "=")
		END ReadAssign;
	BEGIN
		OpenArgs(S, end);
		IF (S.class = Texts.Char) & (S.c = Oberon.OptionChar) THEN
			Texts.Scan(S); quote := TRUE
		ELSE
			quote := FALSE
		END;
		IF S.class IN {Texts.Name, Texts.String} THEN
			COPY(S.s, section); Texts.Scan(S);
			IF S.class IN {Texts.Name, Texts.String} THEN
				COPY(S.s, key);
				IF key[S.len-1] = ":" THEN
					key[S.len-1] := 0X; ReadAssign(done, FALSE)
				ELSE
					ReadAssign(done, TRUE)
				END;
				IF done THEN
					line := S.line; Texts.Scan(S);
					IF (S.class IN {Texts.Name, Texts.String}) & (line = S.line) THEN
						IF quote THEN
							value := '"'; Strings.Append(value, S.s); Strings.AppendCh(value, '"')
						ELSE
							COPY(S.s, value)
						END
					ELSE value := ""
					END;
					Texts.WriteString(W, "   ");
					end := 0;
					WHILE section[end] # 0X DO
						IF section[end] = "." THEN
							section[end] := "\"
						END;
						INC(end)
					END;
					Registry.OberonPath(section, path);
					Registry.SetKeyValue(Registry.CurrentUser, path, key, value);
					IF Registry.res = Registry.Done THEN WriteTriple(path, key, value)
					ELSE Texts.WriteString(W, "System.Set failed "); WriteRegistryError()
					END;
					Texts.WriteLn(W); Texts.Append(Oberon.Log, W.buf)
				END
			END
		END
	END Set;

	PROCEDURE *KeyValueHandler(key, value: ARRAY OF CHAR);
	BEGIN
		Texts.WriteString(W, "   "); WriteTriple("", key, value); Texts.WriteLn(W)
	END KeyValueHandler;

	PROCEDURE *PathHandler(path: ARRAY OF CHAR);
	BEGIN
		Texts.WriteString(W, "    ["); Texts.WriteString(W, path); Texts.Write(W, "]"); Texts.WriteLn(W)
	END PathHandler;

	(** get a value from the registry
		^ | path key *)
	PROCEDURE Get*;	(** non-portable *)
		VAR
			S: Texts.Scanner;
			section, value: ARRAY 64 OF CHAR;
			path: ARRAY 128 OF CHAR;
			end: LONGINT;
	BEGIN
		OpenArgs(S, end);
		IF S.class IN {Texts.Name, Texts.String} THEN
			COPY(S.s, section); Texts.Scan(S);
			end := 0;
			WHILE section[end] # 0X DO
				IF section[end] = "." THEN
					section[end] := "\"
				END;
				INC(end)
			END;
			Registry.OberonPath(section, path);
			IF (S.class IN {Texts.Name, Texts.String}) & (S.line = 0) THEN
				Registry.GetKeyValue(Registry.CurrentUser, path, S.s, value);
				Texts.WriteString(W, "   ");
				IF Registry.res = Registry.Done THEN
					WriteTriple(path, S.s, value)
				ELSE
					Texts.WriteString(W, "System.Get failed ");
					WriteRegistryError()
				END;
				Texts.WriteLn(W)
			ELSE
				Texts.Write(W, "["); Texts.WriteString(W, path); Texts.Write(W, "]"); Texts.WriteLn(W);
				Registry.EnumerateKeyValue(Registry.CurrentUser, path, KeyValueHandler);
				Registry.EnumeratePath(Registry.CurrentUser, path, PathHandler)
			END;
			Texts.Append(Oberon.Log, W.buf)
		END
	END Get;

PROCEDURE DoInit;
	VAR ver: Kernel32.OSVersionInfo; len: LONGINT;
BEGIN
	Kernel32.Str(Kernel.version); Kernel32.Ln();
	Texts.WriteString(W, "ETH PlugIn Oberon for Windows");
	Texts.SetFont(W, Fonts.This("Default8.Scn.Fnt")); Texts.SetOffset(W, 6);
	Texts.WriteString(W, "TM");
	Texts.SetFont(W, Fonts.Default); Texts.SetOffset(W, 0);
	Texts.WriteString(W, " / "); Texts.WriteString(W, Kernel.version); Texts.WriteLn(W);
	Texts.Write(W, 09X); Texts.WriteString(W, " on Windows ");
	ver.dwOSVersionInfoSize := SIZE(Kernel32.OSVersionInfo);
	Kernel32.GetVersionEx(ver);
	IF ver.dwPlatformId = Kernel32.VerPlatformWin32s THEN
		Texts.WriteString(W, "3.1 with Win32s")
	ELSIF ver.dwPlatformId = Kernel32.VerPlatformWin32Windows THEN
		IF ver.dwMinorVersion = 0 THEN
			Texts.WriteString(W, "95")
		ELSIF ver.dwMinorVersion = 10 THEN
			Texts.WriteString(W, "98")
		ELSIF ver.dwMinorVersion = 90 THEN
			Texts.WriteString(W, "ME")
		ELSE
			Texts.WriteString(W, "DOS")
		END;
		ver.dwMajorVersion := ver.dwBuildNumber DIV (256*256*256);
		ver.dwMinorVersion := (ver.dwBuildNumber DIV (256*256)) MOD 256;
		ver.dwBuildNumber := ver.dwBuildNumber MOD (256*256)
	ELSIF ver.dwPlatformId = Kernel32.VerPlatformWin32NT THEN
		IF ver.dwMajorVersion = 6 THEN
			IF ver.dwMinorVersion = 1 THEN
				Texts.WriteString(W, "2008")
			ELSE
				Texts.WriteString(W, "Vista")
			END
		ELSIF ver.dwMajorVersion = 5 THEN
			IF ver.dwMinorVersion = 0 THEN
				Texts.WriteString(W, "2000")
			ELSIF ver.dwMinorVersion = 2 THEN
				Texts.WriteString(W, "2003")
			ELSE
				Texts.WriteString(W, "XP")
			END
		ELSE
			Texts.WriteString(W, "NT")
		END
	ELSE
		Texts.WriteString(W, "?")
	END;
	Texts.WriteString(W, " Version "); Texts.WriteInt(W, ver.dwMajorVersion, 0);
	Texts.Write(W, "."); Texts.WriteInt(W, ver.dwMinorVersion, 0);
	Texts.Write(W, "."); Texts.WriteInt(W, ver.dwBuildNumber, 0);
	Texts.Write(W, " "); Texts.WriteString(W, ver.szCSDVersion);
	len := LEN(ver.szCSDVersion);
	ADVAPI32.GetUserName(ver.szCSDVersion, len);
	Texts.Write(W, " "); Texts.WriteString(W, ver.szCSDVersion);
	len := LEN(ver.szCSDVersion);
	Kernel32.GetComputerName(ver.szCSDVersion, len);
	Texts.Write(W, "@"); Texts.WriteString(W, ver.szCSDVersion);
	Texts.WriteLn(W); Texts.WriteLn(W); Texts.Append(Oberon.Log, W.buf)
END DoInit;

PROCEDURE StartEXE*;
	VAR T: Texts.Text; done: BOOLEAN;
BEGIN
	IF ~Configuration.Load("Configuration") THEN
		Oberon.OpenDisplay();
		OpenLog();
		NEW(T); Texts.Open(T, "System.Tool");
		OpenText("System.Tool", T, TRUE)
	END;
	InitSystemEditor();
	IF ~textDocs & (Viewers.tracksWin # NIL) THEN
		done := Displays.PutCmd(Viewers.tracksWin, NIL, "Oberon.RefreshDisplay", 0)
	END
END StartEXE;

PROCEDURE StartDLL*;
	VAR done: BOOLEAN;
BEGIN
	done := Configuration.Load("DLLConfiguration")
END StartDLL;

BEGIN
	Exceptions.displayTrap := DisplayTrap; NEW(trapMtx); Threads.Init(trapMtx);
	Texts.OpenWriter(W); Texts.OpenWriter(Wt); Texts.WriteLn(Wt);
	NEW(trapper); trapper.handle := Trapper; trapper.safe := TRUE; trapper.time := Input.Time() + MAX(LONGINT) DIV 2;
	Oberon.Install(trapper); DoInit()
END System.
BIERϩ         ;       f      C  Arial10.Scn.Fnt 01.02.2009  17:34:59  TimeStamps.New  