/*
   filename: WCD.C

   WCD - Powerful chdir for Dos and Unix.

   See file wcd.txt

Author: Erwin Waterlander

Address :

	  Jongemastate 125
	  5655 HS Eindhoven, The Netherlands

E-mail  : waterlan@natlab.research.philips.com
 or     : waterlan@xs4all.nl
WWW     : http://www.xs4all.nl

======================================================================
= Copyright                                                          =
======================================================================
Copyright (C) 1997-1998 Erwin Waterlander

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

=======================================================================

Jason Mathews' file filelist.c was a starting point for this file.

*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dirent.h>
#ifdef UNIX
# include <unistd.h>
#endif
#include "dosdir.h"
#include "match.h"
#include "std_macro.h"
#include "structures.h"
#include "nameset.h"
#include "WcdStack.h"
#include "text.h"
#include "wcd.h"
#include "stack.h"
#include "error.h"
#include "display.h"

/* Global variables */

FILE *outfile;
char no_match =1, perfect_match = 0;
char best_match[DD_MAXDIR];
char quiet = 1;

char filemask[DD_MAXFILE+DD_MAXEXT];
int  attrib = DD_DIREC; /* set recursive search on */
unsigned dir_files;
const char *default_mask = ALL_FILES_MASK;


typedef struct TDirTag {
	char* dirname;
	struct TDirTag *next;
} TDirEntry;

typedef struct {
	TDirEntry *head, *tail;
} TDirList;


int (*filematch)( const char*, const char*, int );


#if defined (VMS)

int vms_chdir(char* dir)
{
	/* kill version number on directories */
	char* s = strchr(dir, ';');
	if (s) *s = '\0';
	return chdir(dir);
}

/*---------------------------------------------------------------------*
Name            strupr - converts a string to upper-case

Description     strupr converts lower-case letters in string str to upper-case.
		No other changes occur.
*---------------------------------------------------------------------*/
char* strupr(const char* s)
{
	register char *ps = s;
	while (*ps)
	{
		if (islower((unsigned char)*ps))
			*ps = _toupper((unsigned char)*ps);
		ps++;
	}
	return s;
}

#elif defined (MSDOS)

/* No valid MS-DOS directories start with a period (.),
 * except for .. or ., so anything with a period prefix
 * must be special.
 */
#  define SpecialDir(f) (*(f) == '.')

#else  /* ? UNIX */


/* Function: SpecialDir
 *
 * Purpose:  Test for special directories
 *
 * Returns: 1 if path = "." or ".."
 *          0 otherwise.
 */
int SpecialDir(const char *path)
{
	if (*path != '.') return 0;
	if (*(++path) == '.') path++;
	return (*path=='/' || *path=='\0');
}

#endif /* ?VMS */

int strcompare(const char *s1,const char * s2,int ignore_case)
{
	return !strcmp(s1, s2);
}

/******************************************************************
 *
 *          q_insert - insert directory name to queue
 *
 ******************************************************************/

void q_insert(TDirList *list,const char *s)
{
	TDirEntry *ptr;
	int len = strlen(s);
	if (!len) return;
	if ((ptr = (TDirEntry*) malloc(sizeof(TDirEntry))) == NULL )
	{
		perror("malloc");
		return;
	}
	if ((ptr->dirname = (char*) malloc(len+1)) == NULL )
	{
		perror("malloc");
		free(ptr);
		return;
	}
	strcpy(ptr->dirname, s);
	ptr->next = NULL;
	if (!list->head) list->head = ptr;
	else list->tail->next = ptr;
	list->tail = ptr;
}

/*******************************************************************
 *
 *         q_remove - remove directory name from queue
 *
 *******************************************************************/

int q_remove(TDirList *list,char *s)
{
	TDirEntry *ptr = list->head;
	if (!ptr) return 0;		/* queue empty? */
	strcpy(s, ptr->dirname);
	list->head = ptr->next;
	free(ptr->dirname);
	free(ptr);
	return 1;			/* okay */
}

/********************************************************************
 *
 *                    finddirs(dir)
 *
 ********************************************************************/
void finddirs(char *dir)
{
	static dd_ffblk fb;       /* file block structure */
	static char tmp[DD_MAXDIR];      /* tmp string buffer */
	int rc;                       /* error code */
	TDirList list;                /* directory queue */
#ifdef MSDOS
	char *ptr;
#endif /* ?MSDOS */


	if (dir)
	{
#ifdef MSDOS
		int len = strlen(dir);
		/* strip ending separator (if present), which DOS doesn't like. */
		if (len > 1 && dir[len-1]==DIR_END) dir[len-1] = 0;
#endif /* ?MSDOS */
		if (chdir(dir)) return; /* ?err */
	}

	rc = dd_findfirst( default_mask, &fb, attrib | DD_DIREC );
	list.head = list.tail = 0;
	dir_files = 0;

	while (rc==0)
	{
		if (attrib & DD_DIREC && DD_ISDIREC(fb.dd_mode))
		{
#ifndef VMS
			/*  Ignore directory entries starting with '.'
	   *  which includes the current and parent directories.
	   */
			if (!SpecialDir(fb.dd_name))
#endif /* ?!VMS */
				q_insert(&list, fb.dd_name);
		}

		/* if match then do something with the file */
		if (filematch(fb.dd_name, filemask, 0))
		{
			if (!dir_files++)
			{
				int len;
				getcwd(tmp, sizeof(tmp));
#ifndef VMS
				len = strlen(tmp);
				if (len==0)
					tmp[len] = '\0';
#endif /* ?!VMS */
#ifdef MSDOS
				/* Change backslashes to DIR_SEPARATOR
		   Avoid doing compares on strings
		   with backslashes. */
				while ((ptr=strchr(tmp,'\\'))) * ptr=DIR_SEPARATOR;
				if ( (ptr=strstr(tmp,"/")) != NULL)
					fprintf(outfile,"%s\n",ptr);
#else
				fprintf(outfile,"%s\n", tmp);
#endif
			}
		}

		rc = dd_findnext(&fb);
	} /* while !rc */

	/* recursively parse subdirectories (if any) */
	while (q_remove(&list, tmp))
		finddirs(tmp);

	if (dir) chdir(DIR_PARENT); /* go to parent directory */
}

/********************************************************************
 *
 * Read banfile in a structure.
 *
 ********************************************************************/

void read_banfile(char* banfilename, nameset bd)
{
	FILE *infile;
	char ban[DD_MAXDIR];
	char c;
	int i,j,error;

	/* open treedata-file */
	if  ((infile = fopen(banfilename,"r")) != NULL)
	{
		while (!feof(infile) )
		{
			j=0;
			error = 0;
			/* read a line */
			for(i=0; ((c=getc(infile)) != '\n') && (!feof(infile)) ;i++)
				if (i < (DD_MAXDIR-1))
				{
					ban[i]=c;
					j++;
				}
				else error = 1;
			ban[j]='\0';  /* end string */

			if(error)
				printf("Error: line too long ( %d > %d). Fix: increase DD_MAXDIR.\n",i,(DD_MAXDIR-1));

			if (strlen(ban) > 0 )
	            {
		        	int len = strlen(ban);
		        	/* strip ending separator (if present) */
		        	if (len > 1 && ban[len-1]==DIR_SEPARATOR) ban[len-1] = '\0';

					addTexttoNameset(textNew(ban),bd);
	            }
		} /* while (!feof(infile) ) */
		fclose(infile);
	}

}
/********************************************************************
 *
 *                    checkban(char *dir,char *banfilename)
 *
 *  Returns 0 if directory is not banned.
 *  Returns 1 if directory is banned.
 *
 ********************************************************************/

int checkban(char *dir, nameset bd)
{
	char temp[DD_MAXDIR];
	int banned = 0,i=0;

		if(bd == NULL)
			return banned;
		else
		if( bd->size == 0)
			return banned;
		else
		while ( (i < bd->size) && (banned==0))
		{
				if(strlen(dir)>=strlen(bd->array[i]))
				{
					strncpy(temp,dir,strlen(bd->array[i]));
					temp[strlen(bd->array[i])]='\0';
#ifdef MSDOS
					if (stricmp(temp,bd->array[i])== 0)
#else
					if (strcmp(temp,bd->array[i])== 0)
#endif
					{
						banned = 1;
					}
				}
		i++;
		}
	return(banned);
}
/********************************************************************
 *
 *                    check_double_match(char *dir, int perfect,
 *                                       nameset pm, nameset wm)
 *
 *  Returns 0 if directory had no match before.
 *  Returns 1 if directory is double matched.
 *
 ********************************************************************/

int check_double_match(char *dir, int perfect,
                       nameset pm, nameset wm)
{
	int i = 0,doublematch = 0;
	char *ptr;

#ifdef UNIX
	/* skip the /tmp_mnt string */
	if( (ptr=strstr(dir,getenv("HOME")))== NULL)
	ptr = dir;
#else
	ptr = dir;
#endif

	if (ptr)
    if (perfect)
    {
    	if (pm->size > 0)  /* test for double perfect match */
    	{
     		while( (doublematch==0) && (i < pm->size))
     		{
#ifdef MSDOS
        		if( stricmp(pm->array[i],ptr) == 0) doublematch = 1;
#else
        		if( strcmp(pm->array[i],ptr) == 0) doublematch = 1;
#endif
        		i++;
      		}
    	}
    }
    else
    {
    	if (wm->size > 0)  /* test for double wild match */
    	{
     		while( (doublematch==0) && (i < wm->size))
     		{
#ifdef MSDOS
        		if( stricmp(wm->array[i],ptr) == 0) doublematch = 1;
#else
        		if( strcmp(wm->array[i],ptr) == 0) doublematch = 1;
#endif
        		i++;
     		}
     	}

    }

	return(doublematch);
}
/********************************************************************
 *
 *    scanfile(char *org_dir, char *filename, int
 *             ignore_case, nameset pm, nameset wm, nameset bd)
 *
 *
 ********************************************************************/

void scanfile(char *org_dir, char *filename, int ignore_case,
              nameset pm, nameset wm, nameset bd)
{
	FILE *infile;
	char line[DD_MAXDIR];            /* database path */
	char *line_end;                  /* database directory */
	char path_str[DD_MAXDIR];        /* path name to match */
	char dirwild_str[DD_MAXDIR];     /* directory name to wild match */
	char *dir_str ;                  /* directory name to perfect match */
	char c;
	int  i,j;
	char error = 0;
	int wild = 0;
#ifdef UNIX
	char *ptr;
#endif

	/* open treedata-file */
	if  ((infile = fopen(filename,"r")) == NULL)
	{
		fprintf(stderr,"Error opening treedata-file %s\n",filename);
	}
	else
	{

		if( (dir_str = strrchr(org_dir,DIR_SEPARATOR)) != NULL)
			dir_str++;
		else dir_str = org_dir;

		strcpy(dirwild_str,dir_str);
		strcpy(path_str,"*");
		strcat(path_str,org_dir);

		if (!dd_iswild(dir_str))
		{
			strcat(dirwild_str,"*");
			strcat(path_str,"*");
			wild = 0;
		}
		else
			wild = 1;


		while (!feof(infile) )  /* parse the file */
		{
			j=0;
			error = 0;
			/* read a line */
			for(i=0; ((c=getc(infile)) != '\n') && (!feof(infile)) ;i++)
				if (i < (DD_MAXDIR-1))
				{
					line[i]=c;
					j++;
				}
				else error = 1;
			line[j]='\0';  /* end string */


			if(error)
				printf("Error: line too long ( %d > %d). Fix: increase DD_MAXDIR.\n",i,(DD_MAXDIR-1));

			if (line)
	        {
		       	int len = strlen(line);
		       	/* strip ending separator (if present) */
		       	if (len > 1 && line[len-1]==DIR_SEPARATOR) line[len-1] = '\0';
	        }

			if( (line_end = strrchr(line,DIR_SEPARATOR)) != NULL)
				line_end++;
			else
				line_end = line;

			/* test for a perfect match */

			if ( (wild == 0) && (dd_match(line_end,dir_str,ignore_case))  &&
			     (dd_match(line,path_str,ignore_case)) )
					{
						if(!quiet) printf("%s\n",line);
							if ((checkban(line,bd) == 0) &&
								(check_double_match(line,1,pm,wm)==0))
							{
#ifdef UNIX
								/* skip the /tmp_mnt string */
								if( (ptr=strstr(line,getenv("HOME")))!= NULL)
	                            strcpy(line,ptr);
#endif
								perfect_match =1;
								addTexttoNameset(textNew(line),pm);
								strcpy(best_match,line);
								no_match = 0;

							}
					}
				else
				{

					/* test for a wild match if no perfect match */

						if ( (dd_match(line_end,dirwild_str,ignore_case)) &&
						     (dd_match(line,path_str,ignore_case)) && !perfect_match)
							{

								if(!quiet) printf("%s\n",line);
	                            if((checkban(line,bd) == 0) &&
									(check_double_match(line,0,pm,wm)==0))
	                            {
#ifdef UNIX
		                        /* skip the /tmp_mnt string */
		                        if( (ptr=strstr(line,getenv("HOME")))!= NULL)
			                    strcpy(line,ptr);
#endif
								addTexttoNameset(textNew(line),wm);
								strcpy(best_match,line);
		                        no_match = 0;
	                            }
							}
				}
		}   /* while (!feof(infile) ) */
		fclose(infile);
	}
}
/********************************************************************
 *
 *                    scanaliasfile(char *org_dir, char *filename,
 *                                   nameset *pm, nameset *wm)
 *
 *
 ********************************************************************/

void scanaliasfile(char *org_dir, char *filename,
              nameset pm, nameset wm)
{
	FILE *infile;
	char line[DD_MAXDIR];
	char alias[256];
	char c;
	int i,j,error;
#ifdef UNIX
	char *ptr;
#endif

	/* open treedata-file */
	if  ((infile = fopen(filename,"r")) != NULL)
	{

		while (!feof(infile) )
		{

			if(fscanf(infile,"%s",alias)==1)
			{

	  		/* skip spaces between alias and path */
	  		while ((line[0]=getc(infile)) == ' '){};

			j=1;
			error = 0;
			/* read a line */
			for(i=1; ((c=getc(infile)) != '\n') && (!feof(infile)) ;i++)
				if (i < (DD_MAXDIR-1))
				{
					line[i]=c;
					j++;
				}
				else error = 1;
			line[j]='\0';  /* end string */

			if(error)
				printf("Error: line too long ( %d > %d). Fix: increase DD_MAXDIR.\n",i,(DD_MAXDIR-1));

			if (strlen(line) > 0 )
			/* Only a perfect match counts, case sensitive */
				if  ((strcmp(alias,org_dir)==0) && (check_double_match(line,1,pm,wm)==0))
					{
						if(!quiet) printf("%s\n",line);

				/*		if((pm->size) <= (pm->maxsize))
							if (checkban(dir,bd) == 0)
							{    */
#ifdef UNIX
								/* skip the /tmp_mnt string */
								if( (ptr=strstr(line,getenv("HOME")))!= NULL)
	                            strcpy(line,ptr);
#endif
								perfect_match =1;
								addTexttoNameset(textNew(line),pm);
								strcpy(best_match,line);
								no_match = 0;

						/*	} */
					}
			}
		}   /* while (!feof(infile) ) */
	fclose(infile);
	}
}
/********************************************************************
 *
 *                 Get int
 *
 ********************************************************************/

int wcd_get_int()
{
	int i;
	char string[256];

	gets(string);
	fflush(stdin); /* flush the input stream in case of bad input */

	i=atoi(string);

	return(i);
}

/********************************************************************
 *
 *                 exit
 *
 ********************************************************************/

int wcd_exit(nameset pm, nameset wm, nameset ef, nameset bd, WcdStack ws)
{

    /* free datastructures */
    FreeNameset(pm, 1);
	FreeNameset(wm, 1);
	FreeNameset(ef, 1);
	FreeNameset(bd, 1);
	FreeWcdStack(ws, 1);

	return(0);
}

/********************************************************************
 *
 *                 Print help
 *
 ********************************************************************/
void printhelp()
{
	printf("WCD  - Waterlander Change Directory %s\n",VERSION);
	printf("     - by Erwin Waterlander, %s.   Powerful chdir for Dos and Unix.\n\n",VERSION_DATE);
	printf("Usage: wcd [drive:][dir] [-h] [-q] [-Q] [-b] [-l] [-c] [-v] [-e] [-E <path>]\n");
	printf("       [-g] [-s] [-S <path>] [-a] [-A <path>] [-u <username>] [-f <treefile>]\n");
	printf("       [-i] [-d <drive>] [-[#]] [+[#]] [=] [-z #]\n");
	printf("  dir (partial) name of directory to change to.\n");
	printf("      Wildcards *, ? and [SET] are supported!\n");
	printf("  -u  add treefile of other User (Unix only)\n");
	printf("  -q  unQuiet operation\n");
	printf("  -Q  Quieter operation             -c  direct CD mode\n");
	printf("  -f  add extra treeFile            -l  aLias current dir\n");
	printf("  -h  show this Help                -b  Ban current path\n");
	printf("  -s  (re)Scan disk from $HOME      -v  print Version info\n");
	printf("  -S  Scan disk from <path>         -g  print software licence\n");
	printf("  -a  Add current path to treedata  -e  add current path to Extra treedata\n");
	printf("  -A  Add tree from <path>          -E  add tree from <path> to Extra treedata\n");
	printf("  -   Push dir (# times)            -i  Ignore case (Unix)\n");
	printf("  +   Pop dir (# times)             -d  set <Drive> for stack & go files (DOS)\n");
	printf("  =   Show stack                    -z  set max stack siZe\n\n");
	printf("More info in file wcd.txt\n");

}

/********************************************************************
 *
 *             empty wcd.go file
 *
 ********************************************************************/

#ifdef UNIX
void empty_wcdgo(char *go_file)
{
	if  ((outfile = fopen(go_file,"w")) == NULL)
	{
		fprintf(stderr,"Error opening file %s for write.\n", go_file);
		exit(0);
	}

	fprintf(outfile,"\n");
	fclose(outfile);

}
#endif
/********************************************************************
 *
 *                             MAIN
 *
 ********************************************************************/

int main(int argc,char** argv)
{
	int  i;
	int  flags = 0;
	int  stack_hit = 0, stack_index;
	int  add_to_stack = 1;
	char *path = 0;
	char rootdir[DD_MAXDIR],treefile[DD_MAXDIR],banfile[DD_MAXDIR],aliasfile[DD_MAXDIR];
	char stackfile[DD_MAXDIR];
	char scandir[DD_MAXDIR];
	char extratreefile[DD_MAXDIR];
	char dir[DD_MAXDIR];
	char org_dir[DD_MAXDIR];
	FILE *infile;
	char scan_disk = 0;
	char *ptr, *stackptr;
	int  quieter = 0, cd = 0, scandirectory =0, append =0 ;
	char string[256];
	int len;
	static char tmp[DD_MAXDIR];      /* tmp string buffer */
	DIR* dirp; /* GJM */
	int  stack_is_read = 0;
	WcdStack DirStack;
	nameset extra_files, banned_dirs;
    nameset perfect_list, wild_list ;  /* match lists */

#ifdef MSDOS
	int destDisk;
	int ignore_case = 1;
	char drive[DD_MAXDRIVE];
	strcpy(rootdir,ROOTDIR);
	strcpy(treefile,TREEFILE);
	strcpy(extratreefile,EXTRA_TREEFILE);
	strcpy(banfile,BANFILE);
	strcpy(aliasfile,ALIASFILE);
	strcpy(stackfile,STACKFILE);
#else
	int  j,k;
	int ignore_case = 0;
	char help1_str[DD_MAXDIR];
	char go_file[DD_MAXDIR];

	strcpy(rootdir,getenv("HOME"));
	strcat(rootdir,ROOTDIR);
	strcpy(treefile,getenv("HOME"));
	strcat(treefile,TREEFILE);
	strcpy(extratreefile,getenv("HOME"));
	strcat(extratreefile,EXTRA_TREEFILE);
	strcpy(go_file,getenv("HOME"));
	strcat(go_file,GO_FILE);
	strcpy(banfile,getenv("HOME"));
	strcat(banfile,BANFILE);
	strcpy(aliasfile,getenv("HOME"));
	strcat(aliasfile,ALIASFILE);
	strcpy(stackfile,getenv("HOME"));
	strcat(stackfile,STACKFILE);
#endif

    strcpy(scandir,rootdir);

	strcpy(dir,"");
	strcpy(org_dir,"");

    perfect_list = namesetNew();
    wild_list = namesetNew();
	extra_files = namesetNew();
	banned_dirs = namesetNew();
	DirStack = WcdStackNew(WCD_STACK_SIZE);

	read_banfile(banfile,banned_dirs);

	stackptr = NULL;


	/* ---------------------- parse the commandline ------------*/

	for (i=1; i < argc; i++)
	{
		if (*argv[i]=='-') /* is it a switch? */
			switch (argv[i][1]) {
			case '\0':
	            if (stack_is_read == 0)
	            {
		          stack_read(DirStack,stackfile);
		          stack_is_read = 1;
	            }

				stackptr = stack_push(DirStack,1);
				if (stackptr != NULL)
				{
					stack_hit = 1;
		        	stack_write(DirStack,stackfile);
				}
				break;
			case '0':
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
			case '8':
			case '9':
	            if (stack_is_read == 0)
	            {
		          stack_read(DirStack,stackfile);
		          stack_is_read = 1;
	            }
				ptr = argv[i];
				ptr++;
				stackptr = stack_push(DirStack,atoi(ptr));
				if (stackptr != NULL)
				{
					stack_hit = 1;
		        	stack_write(DirStack,stackfile);
				}
				break;
			case 'S':
				scandirectory = 1 ;
				scan_disk = 1;
				break;
			case 's':
				scan_disk = 1;
				break;
			case 'A':
				append =1;
				scandirectory = 1 ;
				scan_disk = 1;
				break;
			case 'a':
				getcwd(tmp, sizeof(tmp));
				if(tmp != NULL)
				{
					len = strlen(tmp);
					if (len==0)
						tmp[len] = '\0';

					/* open the treedata file */
					if  ((outfile = fopen(treefile,"a")) == NULL)
					{
						fprintf(stderr,"Error opening file %s for write.\n", treefile);
					}
					else
					{
						while ((ptr=strchr(tmp,'\\'))) * ptr= DIR_SEPARATOR;
						if ( (ptr=strstr(tmp,"/")) != NULL)
						{
							fprintf(outfile,"%s\n",ptr);
							printf("%s added.\n",ptr);
						}
						fclose(outfile);
					}
				}
#ifdef UNIX       /* empty wcd.go file */
				empty_wcdgo(go_file);
#endif

				return wcd_exit(perfect_list,wild_list,extra_files,banned_dirs,DirStack);
			case 'E':
				append =2;
				scandirectory = 1 ;
				scan_disk = 1;
				break;
			case 'e':
				getcwd(tmp, sizeof(tmp));
				if(tmp != NULL)
				{
					len = strlen(tmp);
					if (len==0)
						tmp[len] = '\0';

					/* open the treedata file */
					if  ((outfile = fopen(extratreefile,"a")) == NULL)
					{
						fprintf(stderr,"Error opening file %s for write.\n", extratreefile);
					}
					else
					{
						while ((ptr=strchr(tmp,'\\'))) * ptr= DIR_SEPARATOR;
				        if ( (ptr=strstr(tmp,"/")) != NULL)
				        {
						  fprintf(outfile,"%s\n",ptr);
						  printf("%s added.\n",ptr);
						}
						fclose(outfile);
					}
				}
#ifdef UNIX       /* empty wcd.go file */
				empty_wcdgo(go_file);
#endif

				return wcd_exit(perfect_list,wild_list,extra_files,banned_dirs,DirStack);
			case 'B':
			case 'b':
				getcwd(tmp, sizeof(tmp));
				if(tmp != NULL)
				{
					len = strlen(tmp);
					if (len==0)
						tmp[len] = '\0';

					/* open the treedata file */
					if  ((outfile = fopen(banfile,"a")) == NULL)
					{
						fprintf(stderr,"Error opening file %s for write.\n", banfile);
					}
					else
					{
						while ((ptr=strchr(tmp,'\\'))) * ptr= DIR_SEPARATOR;
						if ( (ptr=strstr(tmp,"/")) != NULL)
						{
							fprintf(outfile,"%s\n",ptr);
							printf("%s added to banfile %s\n",ptr,banfile);
						}
						fclose(outfile);
					}
				}
#ifdef UNIX       /* empty wcd.go file */
				empty_wcdgo(go_file);
#endif

				return wcd_exit(perfect_list,wild_list,extra_files,banned_dirs,DirStack);
			case 'L':
			case 'l':
				printf("Enter alias for current directory: ");
				gets(string);
				fflush(stdin); /* flush the input stream in case of bad input */

				if(strcmp(string,"")!=0)
				{

				getcwd(tmp, sizeof(tmp));
				if(tmp != NULL)
				{
					len = strlen(tmp);
					if (len==0)
						tmp[len] = '\0';

					/* open the treedata file */
					if  ((outfile = fopen(aliasfile,"a")) == NULL)
					{
						fprintf(stderr,"Error opening file %s for write.\n", aliasfile);
					}
					else
					{
						while ((ptr=strchr(tmp,'\\'))) * ptr= DIR_SEPARATOR;
						if ( (ptr=strstr(tmp,"/")) != NULL)
						{
							fprintf(outfile,"%s %s\n",string,ptr);
							printf("%s added to aliasfile %s\n",ptr,aliasfile);
						}
						fclose(outfile);
					}
				}
				}
#ifdef UNIX       /* empty wcd.go file */
				empty_wcdgo(go_file);
#endif

				return wcd_exit(perfect_list,wild_list,extra_files,banned_dirs,DirStack);
			case 'q':
				quiet = 0;
				break;
			case 'Q':
				quiet = 1;
				quieter = 1;
				break;
			case 'v':
			case 'V':
				printf("WCD %s, %s\n",VERSION,VERSION_DATE);
				printf("Powerful chdir for Dos and Unix.\n\n");
				printf("See also file wcd.txt\n\n");
				printf("Download the latest executables and sources from:\n");
				printf("http://www.xs4all.nl/~waterlan/\n");

#ifdef UNIX       /* empty wcd.go file */
				empty_wcdgo(go_file);
#endif

				return wcd_exit(perfect_list,wild_list,extra_files,banned_dirs,DirStack);
			case 'g':
			case 'G':
				printf("WCD %s, %s\n",VERSION,VERSION_DATE);
				printf("Powerful chdir for Dos and Unix.\n\n");
				printf("Copyright (C) 1997-1998 Erwin Waterlander\n");
				printf("Copyright (C) 1998 Ondrej Popp on C3PO\n");
				printf("Copyright (C) 1994-1996 Jason Mathews on DOSDIR\n");
				printf("Copyright (C) 1990-1992 Mark Adler, Richard B. Wales, Jean-loup Gailly,\n");
				printf("Kai Uwe Rommel and Igor Mandrichenko on recmatch()\n\n");

				printf("This program is free software; you can redistribute it and/or\n");
				printf("modify it under the terms of the GNU General Public License\n");
				printf("as published by the Free Software Foundation; either version 2\n");
				printf("of the License, or (at your option) any later version.\n\n");

				printf("This program is distributed in the hope that it will be useful,\n");
				printf("but WITHOUT ANY WARRANTY; without even the implied warranty of\n");
				printf("MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n");
				printf("GNU General Public License for more details.\n\n");

				printf("You should have received a copy of the GNU General Public License\n");
				printf("along with this program; if not, write to the Free Software\n");
				printf("Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.\n");

#ifdef UNIX       /* empty wcd.go file */
				empty_wcdgo(go_file);
#endif

				return wcd_exit(perfect_list,wild_list,extra_files,banned_dirs,DirStack);
			case 'c':
			case 'C':
				cd = 1;
				break;
			case 'f':
				break;
			case 'z':
				break;
#ifdef UNIX
			case 'i':
				ignore_case = 1;
				break;
			case 'u':
				break;
#endif
#ifdef MSDOS
			case 'd':
				break;
#endif
			default:               /* any switch except the above */
				printhelp();

#ifdef UNIX       /* empty wcd.go file */
				empty_wcdgo(go_file);
#endif

				return wcd_exit(perfect_list,wild_list,extra_files,banned_dirs,DirStack);
			}
		else
		if (*argv[i]=='+') /* Pop dir */
			switch (argv[i][1]) {
			case '\0':
	            if (stack_is_read == 0)
	            {
		          stack_read(DirStack,stackfile);
		          stack_is_read = 1;
	            }
				stackptr = stack_pop(DirStack,1);
				if (stackptr != NULL)
				{
					stack_hit = 1;
		        	stack_write(DirStack,stackfile);
				}
				break;
			case '0':
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
			case '8':
			case '9':
	            if (stack_is_read == 0)
	            {
		          stack_read(DirStack,stackfile);
		          stack_is_read = 1;
	            }
				ptr = argv[i];
				ptr++;
				stackptr = stack_pop(DirStack,atoi(ptr));
 				if (stackptr != NULL)
				{
					stack_hit = 1;
					stack_write(DirStack,stackfile);
				}
				break;
			default:
				break;
			}
		else
		if (*argv[i]=='=') /* Print stack */
		   {
	        if (stack_is_read == 0)
	        {
		      stack_read(DirStack,stackfile);
		      stack_is_read = 1;
	        }
		   	stack_index = stack_print(DirStack);
			if (stack_index >= 0)
				{
					stackptr = DirStack->dir[stack_index] ;
 					if (stackptr != NULL)
					{
						stack_hit = 1;
						stack_write(DirStack,stackfile);
					}
				}
				else
				{
#ifdef UNIX       /* empty wcd.go file */
					empty_wcdgo(go_file);
#endif
					return wcd_exit(perfect_list,wild_list,extra_files,banned_dirs,DirStack);
				}
			}
		else /* Not a switch. Must be a dir of filename. */
		{


			if (strcmp(argv[i-1],"-f") == 0 )

			{
				addTexttoNameset(textNew(argv[i]),extra_files);
			}
			else
#ifdef UNIX
				if (strcmp(argv[i-1],"-u") == 0 )
				{
					strcpy(tmp,HOMESTRING);
					strcat(tmp,argv[i]);
					strcat(tmp,TREEFILE);
					addTexttoNameset(textNew(tmp),extra_files);
				}
				else
#endif
#ifdef MSDOS
				if (strcmp(argv[i-1],"-d") == 0 )
				{
				  if (stack_is_read == 0)
					stackfile[0] = argv[i][0];
				}
				else
#endif

				if (strcmp(argv[i-1],"-S") == 0 )
				{
				    strcpy(scandir,argv[i]);
				}
				else
				if (strcmp(argv[i-1],"-z") == 0 )
				{
				 if (stack_is_read == 0)
				   DirStack->maxsize = atoi(argv[i]);
				}
				else
				if (strcmp(argv[i-1],"-A") == 0 )
				{
				    strcpy(scandir,argv[i]);
				}
				else
				if (strcmp(argv[i-1],"-E") == 0 )
				{
				    strcpy(scandir,argv[i]);
				}
				else
					{
						strcpy(dir,argv[i]);
						strcpy(org_dir,argv[i]);
#ifdef MSDOS
						/* Change backslashes to DIR_SEPARATOR if a backslash was typed
		   Avoid doing compares on strings
		   with backslashes. */
						while ((ptr=strchr(dir,'\\'))) * ptr= DIR_SEPARATOR;
						while ((ptr=strchr(org_dir,'\\'))) *ptr= DIR_SEPARATOR;
#endif

					}

		}


	} /* for */


	/*--- end parsing commandline -----------*/

	if (stack_is_read == 0)
	{
	   stack_read(DirStack,stackfile);
	   stack_is_read = 1;
	}

	flags |= WILDCARDS;
	strcpy(filemask, default_mask);

	/* set comparison function */
	filematch = (flags & WILDCARDS) ? *dd_match : *strcompare;

	if (strcmp(org_dir,"") == 0 )
	{
		strcpy(best_match,rootdir);
		no_match = 0;
	}

	/* stack hit ?*/

	if (stack_hit==1)
	{
			strcpy(best_match,stackptr);
			if (!quieter)
				printf("-> %s\n",best_match);
#ifdef MSDOS
	/* is there a drive to go to ? */

	if (strlen(best_match)>1)
	{

		strncpy(drive,best_match,2);
		drive[DD_MAXDRIVE-1] = '\0';
		if (dd_match(drive,"[a-z]:",1))
		{
			destDisk = islower(*drive) ? *drive-'a' : *drive-'A';
			if (destDisk >= 0)
			{
				setdisk(destDisk);
			}
			ptr = best_match + 2;
			strcpy(best_match,ptr);
		}
	}
#endif


#ifdef UNIX
			j = strlen(best_match);
			k = 0;
			for (i=0; i < j ; i++)
			{
				if ( (best_match[i] == '$') || (best_match[i] == '"') )
				{
					help1_str[k] = '\\';
					k++;
				}
				help1_str[k] = best_match[i];
				k++ ;
			}
			help1_str[k] = '\0' ;

			strcpy(best_match,"\"");
			strcat(best_match,help1_str);
			strcat(best_match,"\"");

			if  ((outfile = fopen(go_file,"w")) == NULL)
			{
				fprintf(stderr,"Error opening file %s for write.\n", go_file);
				return wcd_exit(perfect_list,wild_list,extra_files,banned_dirs,DirStack);  /* exit */
			}
			fprintf(outfile,"cd %s\n", best_match);
			fclose(outfile);

#else
			chdir(best_match); /* change to directory */
#endif
			stack_write(DirStack,stackfile);
			return wcd_exit(perfect_list,wild_list,extra_files,banned_dirs,DirStack);
		}


	/* Direct CD mode */

	if ((cd==1)&&(strcmp(dir,"")!=0)) /* Try open dir direct. */
	{
		if((dirp=opendir(org_dir)) != NULL) /* GJM */
		{
			closedir(dirp); /* GJM */
			strcpy(best_match,org_dir);
			if (!quieter)
				printf("-> %s\n",best_match);

			chdir(org_dir);
			getcwd(tmp, sizeof(tmp));
			if(tmp != NULL)
			{
				len = strlen(tmp);
				if (len==0)
						tmp[len] = '\0';

				while ((ptr=strchr(tmp,'\\'))) * ptr= DIR_SEPARATOR;
				if ( (ptr=strstr(tmp,"/")) != NULL)
				{
			        strcpy(best_match,tmp);
	                /* skip the /tmp_mnt string */
#ifdef UNIX
	                if( (ptr=strstr(tmp,HOMESTRING))!= NULL)
		            strcpy(best_match,ptr);
#endif
					stack_add(DirStack,best_match);
					stack_write(DirStack,stackfile);
				}

			}


#ifdef UNIX
			j = strlen(best_match);
			k = 0;
			for (i=0; i < j ; i++)
			{
				if ( (best_match[i] == '$') || (best_match[i] == '"') )
				{
					help1_str[k] = '\\';
					k++;
				}
				help1_str[k] = best_match[i];
				k++ ;
			}
			help1_str[k] = '\0' ;

			strcpy(best_match,"\"");
			strcat(best_match,help1_str);
			strcat(best_match,"\"");



			if  ((outfile = fopen(go_file,"w")) == NULL)
			{
				fprintf(stderr,"Error opening file %s for write.\n", go_file);
				return wcd_exit(perfect_list,wild_list,extra_files,banned_dirs,DirStack); /* exit */
			}
			fprintf(outfile,"cd %s\n", best_match);
			fclose(outfile);

#else
			chdir(best_match); /* change to directory */
#endif
			return wcd_exit(perfect_list,wild_list,extra_files,banned_dirs,DirStack);
		}



	}


#ifdef MSDOS
	/* is there a drive to go to ? */

	if (strlen(dir)>1)
	{

		strncpy(drive,dir,2);
		drive[DD_MAXDRIVE-1] = '\0';
		if (dd_match(drive,"[a-z]:",1))
		{
			destDisk = islower(*drive) ? *drive-'a' : *drive-'A';
			if (destDisk >= 0)
			{
				setdisk(destDisk);
			}
			ptr = dir + 2;
			if (strcmp(ptr,"") == 0)
			{
				strcpy(org_dir,"/");
				strcpy(dir,"/");
			}
			else
			{
				strcpy(org_dir,ptr);
				strcpy(dir,org_dir);
			}

		}
	}
#endif


	/* does treedata-file exist? */
	if  ((infile = fopen(treefile,"r")) == NULL)
		scan_disk = 1;
	else fclose(infile);


	if (scan_disk)     /* (re)Scan the disk? */
	{

		if (scandirectory)
		{
			if((dirp=opendir(scandir)) != NULL)
		    {
				closedir(dirp); /* GJM */
				path=scandir;
			}
			else
			{
  		 		printf("Error: Directory %s does not exist.\n",scandir);
  		 		return wcd_exit(perfect_list,wild_list,extra_files,banned_dirs,DirStack); /* exit */
  		 	}
		}
		else
		path=rootdir;

		printf("Please wait. (re)Scanning disk. Building treedata-file from %s\n",path);

#ifdef MSDOS

		/* open the output file */
		if (append == 1)
		outfile = fopen(treefile,"a");  /* append to default database */
		else
		if (append == 2)
		outfile = fopen(extratreefile,"a");  /* append to extra database */
		else
		outfile = fopen(treefile,"w");  /* create new database */

		if (!outfile) /* Try to open in a temp dir */
		{
			if  ( (ptr = getenv("TEMP")) != NULL )
			{
			strcpy(treefile,ptr);
			strcat(treefile,TREEFILE);
			outfile = fopen(treefile,"w");
			}

			if (!outfile)
			{
				if  ( (ptr = getenv("TMP")) != NULL )
					{
					strcpy(treefile,ptr);
					strcat(treefile,TREEFILE);
					outfile = fopen(treefile,"w");
					}

				if (!outfile)
				{
					if  ( (ptr = getenv("TMPDIR")) != NULL )
						{
						strcpy(treefile,ptr);
						strcat(treefile,TREEFILE);
						outfile = fopen(treefile,"w");
						}
				}
			}
		}

		if (!outfile) /* Did we succeed? */
		{
			fprintf(stderr,"Error opening treefile for write.\n");
			fprintf(stderr,"Set TEMP environment variable if this is a read-only disk.\n");
			return wcd_exit(perfect_list,wild_list,extra_files,banned_dirs,DirStack); /* exit */
		}
#else
		/* open the output file */
		if (append == 1)
		outfile = fopen(treefile,"a");  /* append to default database */
		else
		if (append == 2)
		outfile = fopen(extratreefile,"a");  /* append to extra database */
		else
		outfile = fopen(treefile,"w");  /* create new database */

		if  (!outfile)
		{
			fprintf(stderr,"Error opening file %s for write.\n", treefile);
			return wcd_exit(perfect_list,wild_list,extra_files,banned_dirs,DirStack); /* exit */
		}
#endif



		finddirs( path ); /* Build treedata-file */
		fclose(outfile);

	}


	if((strcmp(dir,"")==0) && (no_match))
	{
#ifdef UNIX       /* empty wcd.go file */
		empty_wcdgo(go_file);
#endif
		return wcd_exit(perfect_list,wild_list,extra_files,banned_dirs,DirStack);
	}




	if (strcmp(dir,"")!=0) /* Is there a directory to go to? */
	{
		scanfile(org_dir, treefile,ignore_case,perfect_list,wild_list,banned_dirs); /* scan the treedata file */


		if  ((outfile = fopen(extratreefile,"r")) != NULL)
		{
			fclose(outfile);
			scanfile(org_dir, extratreefile,ignore_case,perfect_list,wild_list,banned_dirs); /* scan the extra treedata file */
		}


		/* search extra files */

		for (i=0;i<extra_files->size;i++)
		{
			scanfile(org_dir, extra_files->array[i],ignore_case,perfect_list,wild_list,banned_dirs); /* scan the extra treedata file */
		}

		/* search alias file */

		scanaliasfile(org_dir, aliasfile, perfect_list, wild_list);
    }


	/*********** search is done ***************/


	if (perfect_list->size > 1)
	{
		i= display_list(perfect_list,1);

		if ( (i>0) && (i<=perfect_list->size))
		{
			i--;
			strcpy(best_match,perfect_list->array[i]);
		}
		else
		{
#ifdef UNIX       /* empty wcd.go file */
			empty_wcdgo(go_file);
#endif
			return wcd_exit(perfect_list,wild_list,extra_files,banned_dirs,DirStack);
		}

	}
	else if ((perfect_list->size==0)&&(wild_list->size > 1))
	{
		i= display_list(wild_list,0);

		if ( (i>0) && (i<=wild_list->size))
		{
			i--;
			strcpy(best_match,wild_list->array[i]);
		}
		else
		{
#ifdef UNIX       /* empty wcd.go file */
			empty_wcdgo(go_file);
#endif
			return wcd_exit(perfect_list,wild_list,extra_files,banned_dirs,DirStack);
		}

	}

	/* #ifdef UNIX */
	/* skip the /tmp_mnt string */


#ifdef UNIX
	if( (ptr=strstr(best_match,HOMESTRING))!= NULL)
		strcpy(best_match,ptr);
#endif


	/*******************************/


	if ((no_match)&&(cd==0)) /* No match at all */
	{

		if((dirp=opendir(org_dir)) == NULL) /* GJM */
		{
			printf("No directory found matching %s\nPerhaps you need to rescan the disk or path is banned.\n",
			    dir);
#ifdef UNIX       /* empty wcd.go file */
			empty_wcdgo(go_file);
#endif
		}
		else
		{
			closedir(dirp); /* GJM */
			no_match = 0;
			add_to_stack = 0;
			strcpy(best_match,org_dir);

			chdir(org_dir);
			getcwd(tmp, sizeof(tmp));
			if(tmp != NULL)
			{
				len = strlen(tmp);
				if (len==0)
						tmp[len] = '\0';

				while ((ptr=strchr(tmp,'\\'))) * ptr= DIR_SEPARATOR;
				if ( (ptr=strstr(tmp,"/")) != NULL)
				{
			        strcpy(best_match,tmp);
	                /* skip the /tmp_mnt string */
#ifdef UNIX
	                if( (ptr=strstr(tmp,HOMESTRING))!= NULL)
		            strcpy(best_match,ptr);
#endif
					stack_add(DirStack,best_match);
					stack_write(DirStack,stackfile);
				}

			}
		}


	} else if (no_match)
		printf("No directory found matching %s\nPerhaps you need to rescan the disk or path is banned.\n",dir);


	/*******************************/


		if (no_match==0) /* yes, a match */
		{
			if (!quieter)
				printf("-> %s\n",best_match);

			if (add_to_stack == 1)
			{
			chdir(best_match);
			getcwd(tmp, sizeof(tmp));
			if(tmp != NULL)
			{
				len = strlen(tmp);
				if (len==0)
						tmp[len] = '\0';

				while ((ptr=strchr(tmp,'\\'))) * ptr= DIR_SEPARATOR;
				if ( (ptr=strstr(tmp,"/")) != NULL)
				{
			        strcpy(best_match,tmp);
	                /* skip the /tmp_mnt string */
#ifdef UNIX
	                if( (ptr=strstr(tmp,HOMESTRING))!= NULL)
		            strcpy(best_match,ptr);
#endif
					stack_add(DirStack,best_match);
					stack_write(DirStack,stackfile);
				}

			}
			}
#ifdef UNIX
			j = strlen(best_match);
			k = 0;
			for (i=0; i < j ; i++)
			{
				if ( (best_match[i] == '$') || (best_match[i] == '"') )
				{
					help1_str[k] = '\\';
					k++;
				}
				help1_str[k] = best_match[i];
				k++ ;
			}
			help1_str[k] = '\0' ;

			strcpy(best_match,"\"");
			strcat(best_match,help1_str);
			strcat(best_match,"\"");



			if  ((outfile = fopen(go_file,"w")) == NULL)
			{
				fprintf(stderr,"Error opening file %s for write.\n", go_file);
				return wcd_exit(perfect_list,wild_list,extra_files,banned_dirs,DirStack); /* exit */
			}
			fprintf(outfile,"cd %s\n", best_match);
			fclose(outfile);

#else
			chdir(best_match); /* change to directory */
#endif
		}

	return wcd_exit(perfect_list,wild_list,extra_files,banned_dirs,DirStack);
}

