{
* linequiz
* line oriented program for AKFQuiz
* usable for blind users (braile line or speech synthesizer)
* usable as backend for other applications
*
* Copyright (c) 2005-2006 Andreas K. Foerster <akfquiz@akfoerster.de>
*
* Environment: FreePascal or GNU-Pascal 
*
* This file is part of AKFQuiz
*
* AKFQuiz is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* AKFQuiz is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
*}

{ compatiblity definition }
{$IfDef _WIN32} {$Define Win32} {$EndIf}

{$IfDef Win32}
  {$R w32/linequiz}
{$EndIf}

{$IfDef FPC}
  {$Mode Delphi}

  {$IfDef Win32}
    {$AppType Console}
  {$EndIf}
{$EndIf}


{$I-}
{$R+} { Range checking }

program linequiz(input, output, stderr);
uses uakfquiz, qsys, qmsgs;


const MaxAnswers = 35;

type 
  Tlinequiz = 
    object(Takfquiz)
      readerror : boolean;

      { only temporarily used: }
      AnsPoints : array[1..MaxAnswers] of pointsType;
      
      destructor Done;                            virtual;
      procedure resetQuiz;                        virtual;
      procedure StartQuiz;                        virtual;
      procedure setcharset(cs: string);           virtual;
      procedure processParagraph;
      procedure processComment;                   virtual;
      procedure processHint;                      virtual;
      procedure processAssessment;                virtual;
      procedure processAssessmentPercent;         virtual;
      procedure processQuestion;                  virtual; 
      procedure processMulti;                     virtual;
      procedure nextanswer;
      procedure processAnswer;                    virtual;
      procedure processMultiAnswer;               virtual;
      procedure showAnswers(showdefault: boolean);
      procedure evaluate;                         virtual;
      procedure EndQuiz;                          virtual; 
      procedure error;                            virtual;
    end;

var MaxLength: integer = 72;

var infile: mystring;
var quiz: Tlinequiz;

var display: DisplayType;

var moreLineBreaks: boolean = false;

procedure Enter;
begin
Write('ENTER> ');
if moreLineBreaks then WriteLn;
ReadLn
end;

{---------------------------------------------------------------------}
procedure Tlinequiz.resetQuiz;
begin
inherited resetQuiz;
readerror := false
end;

destructor Tlinequiz.Done;
begin
inherited Done;

if readerror then
  WriteLn(stderr, msg_error)
end;

procedure Tlinequiz.setcharset(cs: string);
begin
inherited setcharset(cs);

cs := makeUpcase(cs); { to make checkung easier }

{ Latin1 is set as default }
if (cs='IBM850') or (cs='IBM437') or
   (cs='CP850') or (cs='CP437') or
   (cs='850') or (cs='437')  
   then case display of
          OEMdisplay:  setconverter(noconversion);
          UTF8display: setconverter(OEMtoUTF8);
	  ISOdisplay:  setconverter(OEMtoLatin1)
          end;

if (cs='UTF-8') or (cs='UTF8') then
   case display of
     OEMdisplay:  setconverter(UTF8toOEM);
     UTF8display: setconverter(noconversion);
     ISOdisplay:  setconverter(UTF8toLatin1)
     end;

if (cs='ASCII') or (cs='US-ASCII') then setconverter(forceASCII)
end;

procedure Tlinequiz.StartQuiz;
begin
inherited StartQuiz;
WriteLn;
WriteLn('INFO:');
WriteLn;
WriteLn(msg_quiz + title);

if author<>'' then 
  WriteLn(msg_author, author);
if copyright<>'' then
  WriteLn('Copyright: ', copyright);
if authorURI<>'' then 
  WriteLn(msg_authorURI, authorURI);
if translator<>'' then
  WriteLn(msg_translator, translator);
if edited<>'' then
  WriteLn(msg_edited, edited);
if license<>'' then
  WriteLn(msg_license, license);
if licenseURI<>'' then
  WriteLn(msg_licenseURI, licenseURI);
WriteLn;
{$IfDef Advertisement}
  WriteLn(msg_advertisement);
  WriteLn;
{$EndIf}
Enter
end;

procedure Tlinequiz.processParagraph;
var s, rest, outs : mystring;
begin
outs := '';
s := readLine;
while s<>'' do
   begin
   if s='.'
     then begin
          WriteLn(outs);
	  outs := '';
	  WriteLn
          end 
     else begin
          rest := s;
          while rest<>'' do
	    begin
	    if outs=''
	      then outs := format(rest, MaxLength, MaxLength)
	      else outs := outs + ' ' +
                           format(rest, MaxLength-Length(outs)-1, MaxLength);
            if rest<>'' then 
	      begin
	      WriteLn(outs);
	      outs := ''
	      end
	    end
          end;
   s := readLine
   end;
WriteLn(outs)
end;

procedure Tlinequiz.processComment;
begin
WriteLn;
WriteLn('COMMENT:');
processParagraph;
WriteLn;
Enter
end;

procedure Tlinequiz.processHint;
begin
WriteLn;
WriteLn('HINT:');
processParagraph;
WriteLn;
Enter
end;

procedure Tlinequiz.processAssessment;
begin 
WriteLn;
WriteLn('ASSESSMENT:');
processParagraph;
WriteLn;
Enter
end;

procedure Tlinequiz.processAssessmentPercent;
var rest: mystring;
begin 
WriteLn;
WriteLn('ASSESSMENT:');

rest := readAssessmentPercent;
while rest<>'' do
    WriteLn(format(rest, MaxLength, MaxLength));
WriteLn;
Enter
end;

procedure Tlinequiz.processQuestion;
begin
inherited processQuestion;

WriteLn;
WriteLn('QUESTION:');
processParagraph;

processAnswer
end;

procedure Tlinequiz.processMulti;
begin
inherited processMulti;

WriteLn;
WriteLn('MULTI-QUESTION:');
processParagraph;

processMultiAnswer
end;

procedure Tlinequiz.nextanswer;
begin
inc(answerNr);
if answerNr>MaxAnswers then 
   begin 
   error; 
   answerNr := MaxAnswers { to avoid range overruns }
   end
end;

procedure Tlinequiz.showanswers(showdefault: boolean);
var 
  s, ans: mystring;
  value: pointsType;
begin
WriteLn;
WRITELN('ANSWERS:');

answerNr := 0;
readAnswer(value, s);
while s<>'' do
  begin
  ans := s;
  nextanswer;
  AnsPoints[answerNr] := value;
  Write(ValueToKey(answerNr), ') ');
  if moreLineBreaks 
     then WriteLn
     else WriteLn(format(ans, MaxLength-3, MaxLength));
  while ans<>'' do
     WriteLn(format(ans, MaxLength, MaxLength));
  readAnswer(value, s)
  end;
  
{ show default-Answer }
if showdefault and (defanswer<>'') then
  begin
  ans := defanswer;
  nextanswer;
  AnsPoints[answerNr] := 0;
  Write(ValueToKey(answerNr), ') ');
  if moreLineBreaks 
     then WriteLn
     else WriteLn(format(ans, MaxLength-3, MaxLength));
  while ans<>'' do
     WriteLn(format(ans, MaxLength, MaxLength));
  end
end;

procedure Tlinequiz.processAnswer;
var 
  ap : pointsType;
  i : integer;
  c : char;
begin
showanswers(true);

WriteLn;

repeat
  Write('#> ');
  if moreLineBreaks then WriteLn;
  ReadLn(c);
  i := KeyToValue(c);
until i<>-1;

if (i<>-1) and not quit then 
  begin
  ap := AnsPoints[i];
  inc(Points, ap);

  if not neutral then
    if ap > 0 
      then begin 
           WriteLn(msg_right); 
	   RightSignal
	   end
      else begin
           WriteLn(msg_wrong);
	   FalseSignal
	   end;
  WriteLn;
  WriteLn(msg_points, Points);
  WriteLn;
  Enter
  end
end;

procedure Tlinequiz.processMultiAnswer;
var 
  i : integer;
  ans: integer;
  Answers : mystring;
  ap, myPoints, myMax: PointsType;
begin
myPoints := 0;
myMax    := 0;

showanswers(false);

{ calculate maximum points for this question }
for i := 1 to answerNr do
  if AnsPoints[i] > 0 then inc(myMax, AnsPoints[i]);
  
WriteLn;
Write('##> ');
if moreLineBreaks then WriteLn;
ReadLn(Answers);

for i := 1 to Length(Answers) do
    begin
    ans := KeyToValue(Answers[i]);
    if ans>0 then 
       begin
       ap := AnsPoints[ans];
       AnsPoints[ans] := 0; { to use it not more than once }
       inc(myPoints, ap);
       inc(Points, ap)
       end
    end;
WriteLn;

WriteLn(msg_points, Points,
        ' (', myPoints, '/', myMax, ')');
WriteLn;
Enter
end;

procedure Tlinequiz.evaluate;
begin
WriteLn;
if not evaluated and not quit and (MaxPoints<>0) then
  begin
  WriteLn(msg_sol1, Points, msg_sol2,
          MaxPoints, msg_sol3);
  If Points > 0 
    then WriteLn(msg_sol4, getPercentage, '%.')
    else if not neutral then WriteLn(msg_sol5)
  end;

inherited evaluate;
InfoSignal;
WriteLn;
Enter
end;

procedure Tlinequiz.EndQuiz;
begin
if not evaluated and not quit 
  then evaluate
end;

procedure Tlinequiz.error;
begin
readerror := true;
quit := true
end;



{---------------------------------------------------------------------}

procedure help;
begin
WriteLn(AKFQuizName+', linequiz, version '+AKFQuizVersion);
WriteLn('(' + platform + ')');
WriteLn('Copyright (C) ', AKFQuizCopyright);
WriteLn(msg_License, msg_GPL);
{$IfDef Advertisement}
  WriteLn;
  WriteLn(msg_advertisement);
{$EndIf}
WriteLn;
WriteLn('Syntax:');
WriteLn('  linequiz [options] [file.akfquiz]');
WriteLn('  linequiz -h | --help | /?');
WriteLn;
WriteLn('Options:');
WriteLn('-l          more line-breaks');
WriteLn('-s          no sound');
WriteLn('-d <dir>    path to quizfiles');
WriteLn('-w <width>  set maximum width of lines');
WriteLn('-OEM        display has OEM (IBM850/IBM437) charset');
WriteLn('-latin1     display has Latin1 charset');
WriteLn('-UTF8       display has UTF-8 charset');
{$IfDef FPC} {$IfDef Go32v2}
WriteLn('-LFN        use long filenames (DOS only)');
{$EndIf} {$EndIf}

WriteLn;
WriteLn('QUIZPATH: ', getQuizPath);
Halt
end;

procedure setwidth(const s: string);
var error: word;
begin
val(s, MaxLength, error);
if error<>0 then
   begin
   WriteLn(stderr, 'Error: after -w must be a number');
   Halt(1)
   end
end;

procedure setmsgconv;
begin
case display of
  ISOdisplay:  setmsgconverter(UTF8toLatin1);
  OEMdisplay:  setmsgconverter(UTF8toOEM);
  UTF8display: setmsgconverter(noconversion);
  end
end;

procedure parameters;
var 
  i: integer;
  count: integer;
  p: mystring;
begin
count := ParamCount;
i := 0;
while i<count do
    begin
    inc(i);
    p := makeUpcase(ParamStr(i));
    if p='-L' then 
        begin moreLineBreaks := true; continue end;
    if p='-S' then 
        begin DisableSignals; continue end;
    if p='-D' then 
        begin inc(i); { handled in qsys } continue end;
    if p='-LFN' then
        begin setLFNsupport; continue end;
    if p='-OEM' then
        begin display := OEMdisplay; setmsgconv; continue end;
    if (p='-LATIN1') or (p='-NOOEM') then
        begin display := ISOdisplay; setmsgconv; continue end;
    if (p='-UTF8') or (p='-UTF-8') then
        begin display := UTF8display; setmsgconv; continue end;
    if p='-W' then
        begin inc(i); setwidth(ParamStr(i)); continue end;
    if (p='-H') or (p='--HELP') or (p='/?') then help;
    if p[1]='-'    { "/" might be used in a path }
       then help { unknown parameter }
       else infile := ParamStr(i); { not Upcase }
    end
end;

procedure myshowentry(const s: string);
begin
WriteLn(stripext(s))
end;

function askfile: mystring;
var 
  found: boolean;
  path, s: mystring;
begin
{$IfDef Advertisement}
  WriteLn(msg_advertisement);
  WriteLn;
{$EndIf}

WriteLn('QUIZFILES:');
WriteLn;

found := false;
path := getquizpath;
while path<>'' do
  begin
  s := getnextdir(path);
  if ListEntries(s, quizext, myshowentry)  then found := true;
  if ListEntries(s, quizext2, myshowentry) then found := true;
  end;

if not found then begin Write(msg_noquizfound); WriteLn end;

WriteLn;
QuestionSignal;
Write('$> ');
if moreLineBreaks then WriteLn;
ReadLn(s);
askfile := s
end;

var myexitcode : byte;

begin { main }
myexitcode := 0;
display := ISOdisplay; { set a default }
if checkOEM then display := OEMdisplay;
if checkUTF8 then display := UTF8display;
setmsgconv;
useSystemLanguage;
useBeepSignals;
parameters;

if infile='' then infile := askfile;
if infile='' then Halt;

if not getquizfile(infile) then
  begin
  WriteLn(stderr, msg_filenotfound);
  Halt(1)
  end;

quiz.Init(infile);

{ assume Latin1 as default charset }
{ may be changed by the "charset:" keyword }
case display of
    ISOdisplay:  quiz.setconverter(noconversion);
    OEMdisplay:  quiz.setconverter(Latin1toOEM);
    UTF8display: quiz.setconverter(Latin1toUTF8);
    end;

quiz.process;
if quiz.readerror then myexitcode := 2;
quiz.Done;

Halt(myexitcode)
end.
