{
* grquiz
* graphics oriented program for AKFQuiz
*
* Copyright (c) 2005-2006 Andreas K. Foerster <akfquiz@akfoerster.de>
*
* Environment: 
* FreePascal (using SDL4FreePascal)
* GNU-Pascal and GRX library with GPC support (no other addon needed)
*
* This file is part of AKFQuiz
*
* AKFQuiz is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* AKFQuiz is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
*}

{ compatiblity definition }
{$IfDef _WIN32} {$Define Win32} {$EndIf}

{$IfDef Win32}
  {$R w32/grquiz}
  {$UnDef Beeps} { important }
{$EndIf}

{$IfDef DPMI}
  {$UnDef SDL}
  {$Define Beeps}
{$EndIf}

{$IfDef FPC}
  {$Mode Delphi}

  {$IfDef FPC} {$IfDef Unix} {$IfNDef SDL}
     { Under FPC + Unix SVGAlib is used }
     {$Define FPCSVGALIB}
     {$Define grNoEscKey} { cannot distiguish Esc from Esc-sequences }
     {$Define Beeps}
  {$EndIf} {$EndIf} {$EndIf}

  {$IfDef Win32}
    {$AppType GUI}
    {$Define SDL}
  {$EndIf}
{$EndIf}

{$I-}
{$X+}
{$R+} { Range checking }

program grquiz(input, output, stderr);

{$IfDef FPC}
  uses
  {$IfNDef SDL} 
    graph, clgrph,
  {$Else} { SDL }
    sdlgrph,
  {$EndIf}
  sysutils, uakfquiz, qsys, qmsgs; 
{$EndIf}

{$IfDef __GPC__}
  import uakfquiz; qmsgs; qsys; graph; grx; clgrph;
  {$UnDef SDL}
{$EndIf}

{$IfDef __GPC__}
  {$L quizhg.o}
  {$If __GPC_RELEASE__ >= 20030303}
    var AKFQuizHg: array[0..$FFFF] of byte; external name 'AKFQuizHg';
  {$Else} {@@@ FIXME: inermediate versions with other syntax }
    var AKFQuizHg: asmname 'AKFQuizHg' array[0..$FFFF] of byte;
  {$EndIf} { __GPC_RELEASE__ }

  {$IfNDef NoTitleImage}
    {$L titimg.o}
    {$If __GPC_RELEASE__ >= 20030303}
      var TitleImage:  array[0..$FFFF] of byte; external name 'TitleImage';
    {$Else} {@@@ FIXME: inermediate versions with other syntax }
      var TitleImage: asmname 'TitleImage' array[0..$FFFF] of byte;
    {$EndIf} { __GPC_RELEASE__ }
  {$EndIf} { NoTitleImage }
{$EndIf} { __GPC__ }

{$IfDef FPC}
  {$I quizhg.inc}
  {$IfNDef NoTitleImage}
    {$I titimg.inc}
  {$EndIf} { NoTitleImage }
{$EndIf} { FPC }

{$I lat9.inc}
{$I icons.inc}

type keyset = set of char;

type TgrfImage = packed record
                Width    : longint;
                Height   : longint;
                reserved : longint;
                Image    : packed array[0..ScreenWidth*ScreenHeight] of word;
                end;

const TitleImageName = 'AKFoerster';

const linespace = fontheight; { you may add something }

const MaxAnswers = 35;

const AnswerKeys = [ '1'..'9', 'A'..'Z' ];

const
  Esc   = chr(27);
  Enter = chr(13);

{ Unix uses #27 + something for any function key,
  that's still so with SVGALib (not unter X11),
  so it's too difficult to filter out the Esc key as such :-( }

{$IFDEF grNoEscKey}
  const
    ExitKey = 'Q'; { capital letter! }
    ExitKeyName = ExitKey;
{$Else}
  const
    ExitKey = Esc;
    ExitKeyName = 'Esc';
{$EndIf}


type
  Tgrquiz =
    object(Takfquiz)
      readerror : boolean;
      MaxLength : word;

      { only temporarily used: }
      AnsPoints : array[1..MaxAnswers] of pointsType;

      constructor init(infile: string);
      procedure resetQuiz;                        virtual;
      procedure wait;
      procedure StartQuiz;                        virtual;
      procedure setcharset(cs: string);           virtual;
      procedure processParagraph;
      procedure processComment;                   virtual;
      procedure processHint;                      virtual;
      procedure processAssessment;                virtual;
      procedure processAssessmentPercent;         virtual;
      procedure processQuestion;                  virtual;
      procedure processMulti;                     virtual;
      function  getAnswer(keys: keyset): integer;
      procedure processAnswer;                    virtual;
      procedure processMultiAnswer;               virtual;
      function  showAnswers(showdefault: boolean): boolean;
      procedure nextanswer;
      procedure evaluate;                         virtual;
      procedure EndQuiz;                          virtual;
      procedure error;                            virtual;
    end;

var infile: mystring;
var quiz: Tgrquiz;

var TextColor, BkColor, Color: word;

var MaxX, MaxY : TscreenPos;

var 
  endless: boolean;
  fullscreen: boolean;
  justtitle: boolean;
  usemouse: boolean;

procedure buildscreen;
{$IfDef UsePalette}
  var i: word;
{$EndIf}
begin

{$IfDef UsePalette}
  for i:=0 to MaxColor do
      setRGBPalette(i, myPalette[i, RGBred],
                       myPalette[i, RGBgreen],
                       myPalette[i, RGBblue]);
{$EndIf}

drawBackground(AKFQuizHg);

TextColor := GetPixel(TextColorX, TextColorY);
BkColor   := GetPixel(BackColorX, BackColorY);

Color := TextColor;
setColors(TextColor, BkColor)
end;


procedure initializeGraphics;
begin
initializeGraphicMode(AKFQuizName+' '+AKFQuizVersion, 'AKFQuiz', fullscreen);
setExitKey(ExitKey);

buildscreen;

defineTextArea(marginLeft, marginTop, 
               GetMaxX-marginRight, GetMaxY-marginBottom, useTextArea);
MaxX := GetMaxX - marginLeft - marginRight;
MaxY := GetMaxY - marginTop - marginBottom;

setmsgconverter(UTF8toLatin1)
end;

procedure GrfWrite(const s: string);
var 
  p: integer;
  x, y: TscreenPos;
  ly, lx: byte;
begin
x := GetX;
y := GetY;

for p := 1 to Length(s) do
  begin
  for ly:=1 to fontheight do
    for lx:=0 to 7 do
      if (lat9[s[p], ly] and (1 shl (7-lx)))<>0 then 
         PutPixel(x+lx, y+ly, color);
  inc(x, 8)
  end;

{ move to end of line }
MoveTo(x, y)
end;

function GetTextWidth(const s: string): integer;
begin
GetTextWidth := Length(s)*8
end;

function waitkey(const s: string): boolean;
var c: char;
begin
if GetY=0 
  then waitkey := true
  else begin
       GrfWrite(s);
       c := GetKey;

       clearTextarea;
       waitkey := ( Upcase(c) <> ExitKey )
       end
end;

procedure GrLn;
begin
MoveTo(0, GetY + linespace);

if GetY >= MaxY - (3*linespace) then
  begin
  MoveTo(0, GetY + linespace);
  waitkey(msg_anykey(''))
  end
end;

procedure GrfWriteLn(const s: string);
begin
GrfWrite(s);
GrLn
end;

procedure GRReadLn(var s: string);
var
  c: char;
  cwidth: integer;
begin
s := '';
repeat
  c := GetKey;
  if (c >= #32) and ( c<>#127) then { normal char }
    begin
    s := s + c;
    GrfWrite(c)
    end;
  if ((c=#127) or (c=#8)) and (s<>'') then
     begin
     c := s[length(s)]; { get last char }
     s := copy(s, 1, Length(s)-1); { remove last char from s }
     cwidth := GetTextWidth(c); { automatic typecast }
     MoveTo(GetX - cwidth, GetY);
     Color := BkColor;
     GrfWrite(c);
     MoveTo(GetX - cwidth, GetY);
     Color := TextColor
     end;
  if not ((c in [#32 .. #255, #10, #13]) or (c = ExitKey)) then ErrorSignal
until (c=#13) or (c=#10) or (c=ExitKey);

if c = ExitKey then s := '' { return nothing }
end;

procedure Symbol(correct: boolean);
var 
  x, y, sx, sy: TscreenPos;
  col : word;
begin
if correct 
  then col := GetRGBColor($00, $AA, $00)   { green }
  else col := GetRGBColor($AA, $00, $00);  { red }

x := GetX + 16; { a little behind the text }
y := GetY;      { same upper limit as text-line }

for sy := 1 to 18 do
  for sx := 1 to 18 do
    if (correct and (icon_right[sy, sx] <> ' ')) or 
       (not correct and (icon_wrong[sy, sx] <> ' ')) then 
      PutPixel(x+sx, y+sy, col);
GrLn
end;

{$IfDef __GPC__}

  procedure getImageSize(protected var img; var width, height: integer);
  var maxval: integer;
  begin
  GrQueryPnmBuffer(img, width, height, maxval)
  end;

{$Else}

  procedure getImageSize(const img; var width, height: integer);
  begin
  width := TgrfImage(img).Width;
  height := TgrfImage(img).Height
  end;

{$EndIf}


{$IfNDef NoTitleImage}

  procedure showTitleImage;
  var width, height: integer;
  begin
  getImageSize(TitleImage, width, height);
  ShowImage(MaxX-width, 0, TitleImage);
 
  MoveTo(MaxX-(width div 2)-(Length(TitleImageName) * 8 div 2), height+3);
  GrfWrite(TitleImageName);
  MoveTo(0, 0);
  end;
{$EndIf} { NoTitleImage }

procedure InfoScreen;
begin
showmouse(true);

{$IfNDef NoTitleImage}
  ShowTitleImage;
{$EndIf} { NoTitleImage }

GrfWriteLn(AKFQuizName+', grquiz, version '+AKFQuizVersion);
GrfWriteLn('Copyright '+chr(169)+' '+AKFQuizCopyright+
           ' <'+AKFQuizEmail+'>');
GrLn;
GrfWriteLn(msg_contributions);
GrfWriteLn('Italiano: Martin Guy, Dansk: Tommy Jensen,');
GrfWriteLn('Font: Guylhem Aznar');
GrLn;
GrfWriteLn(msg_License + msg_GPL);
case lang of
  deutsch : 
    begin
    GrfWriteLn('Dieses Programm wird ohne Gewhrleistung geliefert,');
    GrfWriteLn('soweit dies gesetzlich zulssig ist.');
    GrfWriteLn('Sie knnen es unter den Bedingungen der');
    GrfWriteLn('GNU General Public License weitergeben.');
    GrfWriteLn('Details dazu enthlt die Datei COPYING.');
    GrLn;
    GrfWriteLn('Quiz-Dateien sind von dieser Lizenz nicht betroffen.')
    end;
  otherwise
    begin
    GrfWriteLn('This program comes with NO WARRANTY, to the extent');
    GrfWriteLn('permitted by law.');
    GrfWriteLn('You may redistribute it under the terms of the');
    GrfWriteLn('GNU General Public License;');
    GrfWriteLn('see the file named COPYING for details.');
    GrLn;
    GrfWriteLn('Quiz-files are not affected by this license.')
    end;
end;
GrLn;

{$IfDef Advertisement}
  GrfWriteLn(msg_advertisement);
{$EndIf}

if lang=deutsch 
  then GrfWriteLn(Homepage+'de')
  else GrfWriteLn(Homepage);

GrLn;
waitkey(msg_anykey(''))
end;


procedure myshowentry(const s: string);
begin GrfWriteLn(stripext(s)) end;

function askfile: mystring;
var
  found: boolean;
  path, s: mystring;
begin
showmouse(false);
{$IfDef Advertisement}
  GrfWriteLn(msg_advertisement);
{$EndIf}
GRLn;

found := false;
path := getquizpath;
while path<>'' do
  begin
  s := getnextdir(path);
  if ListEntries(s, quizext, myshowentry)  then found := true;
  if ListEntries(s, quizext2, myshowentry) then found := true;
  end;

if not found then GrfWriteLN(msg_noquizfound);

GRLn;
QuestionSignal;
GrfWrite('Quiz: ');
GRReadLn(s);
clearTextarea;
askfile := s
end;

{---------------------------------------------------------------------}
constructor Tgrquiz.init(infile: string);
begin
inherited init(infile);

setconverter(noconversion); { we're using an Lat1 font }
MaxLength := (GetMaxX-(2*40)) div GetTextWidth('m'); { largest letter }
showmouse(usemouse)
end;

procedure Tgrquiz.resetQuiz;
begin
inherited resetQuiz;
readerror := false
end;

procedure Tgrquiz.setcharset(cs: string);
begin
inherited setcharset(cs);

cs := makeUpcase(cs);

if (cs='IBM850') or (cs='IBM437') or
   (cs='CP850') or (cs='CP437') or
   (cs='850') or (cs='437')
   then setconverter(OEMtoLatin1);
   
if (cs='UTF-8') or (cs='UTF8') then setconverter(UTF8toLatin1);
if (cs='ASCII') or (cs='US-ASCII') then setconverter(forceASCII)
end;


procedure Tgrquiz.wait;
begin
if not quit then
   quit := not waitkey(msg_anykey(ExitKeyName))
end;

procedure Tgrquiz.StartQuiz;
begin
inherited StartQuiz;

clearTextarea;
GRLN;
GRLN;
GrfWriteLn(msg_quiz + title);

if author<>'' then
  GrfWriteLn(msg_author + author);
if copyright<>'' then
  GrfWriteLn('Copyright: ' + copyright);
if authorURI<>'' then
  GrfWriteLn(msg_authorURI + authorURI);
if translator<>'' then
  GrfWriteLn(msg_translator + translator);
if edited<>'' then
  GrfWriteLn(msg_edited + edited);
if license<>'' then
  GrfWriteLn(msg_license + license);
if licenseURI<>'' then
  GrfWriteLn(msg_licenseURI + licenseURI);
GRLN;
{$IfDef Advertisement}
  GrfWriteLn(msg_advertisement);
  GRLN;
{$EndIf}

wait
end;

procedure Tgrquiz.processParagraph;
var s, rest, outs : mystring;
begin
outs := '';
s := readLine;
while s<>'' do
   begin
   if s='.'
     then begin
          GrfWriteLn(outs);
	  outs := '';
	  GRLN
          end
     else begin
          rest := s;
          while rest<>'' do
	    begin
	    if outs=''
	      then outs := format(rest, MaxLength-Length(outs), MaxLength)
	      else outs := outs + ' ' +
                           format(rest, MaxLength-Length(outs), MaxLength);
            if rest<>'' then
	      begin
	      GrfWriteLn(outs);
	      outs := ''
	      end
	    end
          end;
   s := readLine
   end;
GrfWriteLn(outs)
end;

procedure Tgrquiz.processComment;
begin
clearTextarea;
processParagraph;
GrLn;
wait
end;

procedure Tgrquiz.processHint;
begin
processComment { handle like a Comment }
end;

procedure Tgrquiz.processAssessment;
begin
processComment
end;

procedure Tgrquiz.processAssessmentPercent;
var rest: mystring;
begin
rest := readAssessmentPercent;
while rest<>'' do
    GrfWriteLn(format(rest, MaxLength, MaxLength));
GrLN;
wait
end;

procedure Tgrquiz.processQuestion;
begin
clearTextarea;
inherited processQuestion;

processParagraph;
processAnswer
end;

procedure Tgrquiz.processMulti;
begin
clearTextarea;
inherited processMulti;

processParagraph;
processMultiAnswer
end;

procedure Tgrquiz.nextanswer;
begin
if answerNr>0 then answerEnds(answerNr);
inc(answerNr);

if answerNr>MaxAnswers then 
   begin 
   error; 
   answerNr := MaxAnswers { to avoid range overruns }
   end;
answerStarts(answerNr);
end;

function Tgrquiz.getAnswer(keys: keyset): integer;
var
  c : char;
  okay: boolean;
begin
repeat
  c := Upcase(GetKey);
  okay := c in keys;
  if not okay then ErrorSignal;
until okay;

if c=ExitKey then quit := true;

if c in AnswerKeys
  then begin GrfWrite(c); getAnswer := KeyToValue(c) end
  else getAnswer := -1
end;

function Tgrquiz.showAnswers(showdefault: boolean): boolean;
var
  s, ans: mystring;
  value: pointsType;
  startpos: TscreenPos;
  answerMaxY: TscreenPos;
begin
GRLN;

{ leave enough space at the bottom 
  also for "more..." and the default answer }
answerMaxY := MaxY - (9*linespace);

{ enough space for at least starting the answer section? }
if GetY>=answerMaxY then wait;

answerNr := 0;
readAnswer(value, s);
while (s<>'') and (GetY<answerMaxY) and (answerNr<=MaxAnswers-1) do
  begin
  ans := s;
  nextanswer;
  AnsPoints[answerNr] := value;
  GrfWrite(ValueToKey(answerNr) + ') ');
  startpos := GetX;
  GrfWriteLn(format(ans, MaxLength-3, MaxLength-3));
  while ans<>'' do
     begin
     MoveTo(startpos, GetY);
     GrfWriteLn(format(ans, MaxLength-3, MaxLength-3));
     end;
  if GetY<answerMaxY 
    then readAnswer(value, s)
    else 
      if nextLineEmpty 
        then s := ''
        else s := '...' { Dummy }
  end;

if s<>''
  then begin { more answers to be shown }
       nextanswer;
       AnsPoints[answerNr] := 0;
       GrfWriteLn(ValueToKey(answerNr) + ') ' + msg_more)
       end
  else if showdefault and (defanswer<>'') then { show default-Answer }
         begin
         ans := defanswer;
         nextanswer;
         AnsPoints[answerNr] := 0;
         GrfWrite(ValueToKey(answerNr) + ') ');
         startpos := GetX;
         GrfWriteLn(format(ans, MaxLength-3, MaxLength-3));
         while ans<>'' do
           begin
           MoveTo(startpos, GetY);
           GrfWriteLn(format(ans, MaxLength-3, MaxLength-3))
           end { while }
         end; { if defanswer }

{ all answers shown already? }
showAnswers := (s='')
end;

procedure Tgrquiz.processAnswer;
var
  maxKey: char;
  a : integer;
  complete: boolean;
 
  procedure handleResult;
  var ap : pointsType;
  begin
  ap := AnsPoints[a];
  inc(Points, ap);
  
  if not neutral then
    if ap > 0
      then begin GrfWrite(msg_right); Symbol(true);  RightSignal end
      else begin GrfWrite(msg_wrong); Symbol(false); FalseSignal end;

  GrLn;
  GrfWriteLn(msg_points + IntToStr(Points));
  
  if not complete then { skip other answers }
    begin
    repeat until readline='';
    complete := true
    end;
  
  wait
  end;

begin { Tgrquiz.processAnswer }
repeat
  complete := showanswers(true);

  answerEnds(answerNr);

  GRLN;

  GrfWrite('> ');

  maxKey := ValueToKey(answerNr);
  if answerNr<10 
     then a := getAnswer([ '1'..maxKey, ExitKey ])
     else a := getAnswer([ '1'..'9', 'A'..maxKey, ExitKey ]);

  GrLn;

  if not complete and (a=answerNr) { answer is "..."? }
     then clearTextarea 
     else if (a<>-1) and not quit 
             then handleResult

until complete or quit;
end;

procedure Tgrquiz.processMultiAnswer;
var a: integer;
    keys: keyset;
    maxKey: char;
    myPoints, myMax : pointsType;
    complete : boolean;
begin
myPoints := 0;
myMax := 0;

repeat
  complete := showanswers(false);
  answerEnds(answerNr);

  GRLN;
  GrfWrite('>> ');

  { max points for this one question }
  for a := 1 to answerNr do
    if AnsPoints[a] > 0 then inc(myMax, AnsPoints[a]);

  maxKey := ValueToKey(answerNr);
  if answerNr<10 
     then keys := [ '1'..maxKey, ExitKey, Enter ]
     else keys := [ '1'..'9', 'A'..maxKey, ExitKey, Enter ];

  repeat {@@@}
    a := getAnswer(keys);
    if not complete and (a=answerNr) { answer is "..."? }
       then a:=-1
       else if a<>-1 then
              begin
              inc(Points, AnsPoints[a]);   { absolute points }
              inc(myPoints, AnsPoints[a]); { points for this question }
              { remove that key from valid keys }
              keys := keys - [ ValueToKey(a) ]
              end;
  until a=-1;
  if not complete and not quit then clearTextarea
until complete or quit;

GRLN; GRLN;
GrfWriteLn(msg_points + IntToStr(Points) +
           '  (' + IntToStr(myPoints) +
	   '/' + IntToStr(myMax) + ')');
wait
end;

procedure Tgrquiz.evaluate;
begin
clearTextarea;
if not evaluated and not quit and (MaxPoints<>0) then
  begin
  GrfWriteLn(msg_sol1 + IntToStr(Points) + msg_sol2 +
            IntToStr(MaxPoints) + msg_sol3);
  If Points > 0
    then GrfWriteLn(msg_sol4 + IntToStr(getPercentage) + '%.')
    else if not neutral then GrfWriteLn(msg_sol5)
  end;
inherited evaluate;
GRLN;

InfoSignal;
wait
end;

procedure Tgrquiz.EndQuiz;
begin
if not evaluated and not quit
  then evaluate
end;

procedure Tgrquiz.error;
begin
readerror := true;
quit := true
end;


{---------------------------------------------------------------------}

procedure help;
begin
WriteLn(AKFQuizName+', grquiz, version '+AKFQuizVersion);
WriteLn('(' + platform + ')');
WriteLn('Copyright (C) ', AKFQuizCopyright);
WriteLn(msg_License, msg_GPL);
WriteLn;
WriteLn('  grquiz [options] [inputfile]');
WriteLn('  grquiz [ -h | --help | /? ]');
WriteLn;
WriteLn('Options:');
WriteLn('-f | --fullscreen  fullscreen mode (if supported)');
WriteLn('-w | --window      window mode (if supported)');
WriteLn('-s                 no sound');
WriteLn('-m | --nomouse     no mouse');
WriteLn('-d <dir>           path to quizfiles');
WriteLn('-p                 endless');
WriteLn('-t | --title       show just the title screen');
{$IfDef FPC} {$IfDef Go32v2}
WriteLn('-LFN               use long filenames (DOS only)');
{$EndIf} {$EndIf}

WriteLn;
WriteLn('QUIZPATH: ', getQuizPath);
Halt
end;

procedure ErrorFileNotFound;
begin
endGraphics;
WriteLn(stderr, msg_filenotfound);
Halt(2)
end;

procedure parameters;
var
  i: integer;
  p: mystring;
begin
i := 0;
while i<ParamCount do
  begin
  inc(i);
  p := makeUpcase(ParamStr(i));
  if p='-LFN' then
      begin setLFNsupport; continue end;
  if p='-S' then
      begin DisableSignals; continue end;
  if (p='-D') then begin inc(i); { handled in qsys } continue end;
  if (p='-P') or (p='/P') then
     begin endless := true; nobreak; continue end;
  if (p='-F') or (p='--FULLSCREEN') then
     begin fullscreen := true; continue end;
  if (p='-W') or (p='--WINDOW') then
     begin fullscreen := false; continue end;
  if (p='-M') or (p='--NOMOUSE') then
     begin usemouse := false; continue end;
  if (p='-T') or (p='--title') then
     begin justtitle := true; continue end;
  if (p='-H') or (p='--HELP') or (p='/?') then help;
  if p[1]='-'    { "/" might be used in a path }
     then help { unknown parameter }
     else infile := ParamStr(i) { not Upcase }
  end
end;


var myexitcode : byte;

begin { main }

{$IfDef FPCSVGALIB}
  { space after messages from SVGALib }
  WriteLn;
  WriteLn;
{$EndIf}

myexitcode := 0;
endless := false;
fullscreen := false;
justtitle := false;
usemouse := true;

useSystemLanguage;
parameters;

{$IfDef Beeps}
  useBeepSignals; { no beeps possible in most cases }
  {$M Beeps}
{$EndIf}

InitializeGraphics;
InfoScreen;

if justtitle then
  begin
  endGraphics;
  Halt
  end;
 
repeat
  if infile<>'' then
     if not getquizfile(infile) then ErrorFileNotFound;

  if infile='' then
    repeat
      infile := askfile;
      if not endless and (infile='') 
         then begin endGraphics; Halt end;
    until (infile<>'') and getquizfile(infile);

  quiz.Init(infile);

  quiz.process;
  { exitcode 1 reserved for graphical error! }
  if quiz.readerror then myexitcode := 2;
  clearTextarea;
  quiz.Done;

  infile := '';

  if endless then 
     begin
     parameters; { to get input files again }
     InfoScreen
     end;
until not endless;

endGraphics;
if myexitcode=2 then WriteLn(stderr, msg_error);

Halt(myexitcode)
end.
