{
* sdlgrph (unit)
* some graph functions with SDL
*
* Copyright (c) 2005 Andreas K. Foerster <akfquiz@akfoerster.de>
*
* Environment: FreePascal and SDL4FreePascal
*
* This file is part of AKFQuiz
*
* AKFQuiz is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* AKFQuiz is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
*}

unit sdlgrph;

interface
uses uakfquiz, sdl, sdl_video, sdl_events, sdl_keyboard, sdl_mouse;

const MaxAnswers = 35;

procedure initializeGraphicMode(const title, short: string; 
                                fullscreen: boolean);
procedure endGraphics;
procedure setBkColor(c: word);
procedure PutPixel(x, y: integer; color: word);
function  GetPixel(x, y: integer): word;
procedure showImage(var img);
procedure MoveTo(x, y: integer);
function  GetX: integer;
function  GetY: integer;
function  GetMaxX: integer;
function  GetMaxY: integer;

procedure DefineTextArea(x1, y1, x2, y2: integer; useTextArea: boolean);
procedure ClearTextArea;

procedure setExitKey(c: char);
function GetKey: char;

procedure showmouse(on: boolean);
procedure answerStarts(ans: word);
procedure answerEnds(ans: word);


implementation
const 
  ScreenWidth  = 640;
  ScreenHeight = 480;
  ScreenDepth  = 16;

type 
  TScrMap = array[0..ScreenHeight-1, 0..ScreenWidth-1] of word;
  PScrMap = ^TScrMap;

type 
  Tgrfimage = record
              Width    : longint;
              Height   : longint;
              reserved : longint;
              Image    : array[0..ScreenWidth*ScreenHeight] of word;
              end;

var answerposition: array[1..MaxAnswers] of record f, t: integer end;

var screen, textarea: pSDL_SURFACE;
var tx1, ty1, tx2, ty2, tw, th : integer;
var xPos, yPos : integer;
var mode : LongInt;
var ExitKey : char;
var mouseshown: boolean;

procedure resetAnswerPositions;
var i: integer;
begin
for i:=1 to MaxAnswers do
    begin
    answerposition[i].f := 0;
    answerposition[i].t := 0
    end
end;

procedure answerStarts(ans: word);
begin
answerposition[ans].f := ypos
end;

procedure answerEnds(ans: word);
begin 
answerposition[ans].t := ypos
end;

procedure showmouse(on: boolean);
begin
mouseshown := on;
if on 
  then SDL_ShowCursor(SDL_ENABLE) 
  else SDL_ShowCursor(SDL_DISABLE)
end;

procedure initializeGraphicMode(const title, short: string; 
                                fullscreen: boolean);
begin
xPos   := 0;
yPos   := 0;
tx1    := 0;
ty1    := 0;
tx2    := ScreenWidth-1;
ty2    := ScreenHeight-1;
textarea := NIL;

SDL_INIT(SDL_INIT_VIDEO);
SDL_WM_SetCaption(PChar(title), PChar(short));

mode := SDL_SWSURFACE;
if fullscreen then mode := mode or SDL_FULLSCREEN;

screen := SDL_SETVIDEOMODE(ScreenWidth, ScreenHeight, 
                           ScreenDepth, mode);
if screen=NIL then 
   begin
   WriteLn(stderr, 'grquiz error: graphic mode not supported');
   Halt(1)
   end;
  
showmouse(true);

{ enable unicode handling }
SDL_EnableUNICODE(1)
end;

procedure setExitKey(c: char);
begin
ExitKey := c
end;

procedure endGraphics;
begin
if textarea<>NIL then SDL_FREESURFACE(textarea);
SDL_FREESURFACE(screen);
SDL_QUIT
end;

procedure setBkColor(c: word);
begin
end;

procedure PutPixel(x, y: integer; color: word);
var map: PscrMap;
begin
x := x + tx1;
y := y + ty1;
if (x<=tx2) and (y<=ty2) then
  begin
  map := screen^.pixels;
  map^[y, x] := color;
  end
end;

function GetPixel(x, y: integer): word;
var map: PscrMap;
begin
map := screen^.pixels;
GetPixel := map^[y+ty1, x+tx1];
end;

procedure showImage(var img);
begin
Move(TGrfImage(img).Image, screen^.pixels^, 
      ScreenHeight*ScreenWidth*2);
SDL_UPDATERECT(screen, 0, 0, 0, 0)
end;

procedure MoveTo(x, y: integer);
begin
xPos := x;
yPos := y;
end;

function GetX: integer;
begin
GetX := xPos
end;

function  GetY: integer;
begin
GetY := yPos
end;

function GetMaxX: integer;
begin
GetMaxX := ScreenWidth
end;

function  GetMaxY: integer;
begin
GetMaxY := ScreenHeight
end;

procedure defineTextArea(x1, y1, x2, y2: integer; useTextArea: boolean);
var rect: SDL_Rect;
begin
tx1 := x1;
ty1 := y1;
tx2 := x2;
ty2 := y2;
tw  := tx2-tx1+1;
th  := ty2-ty1+1;
if textarea<>NIL then SDL_FREESURFACE(textarea);
textarea := SDL_CreateRGBSurface(SDL_SWSURFACE, tw, th,
                                 ScreenDepth,
                                 screen^.format^.rmask,
                                 screen^.format^.gmask,
                                 screen^.format^.bmask,
                                 screen^.format^.amask);
rect.x := tx1;
rect.y := ty1;
rect.w := tw;
rect.h := th;
SDL_BlitSurface(screen, @rect, textarea, NIL);
resetAnswerPositions
end;

procedure ClearTextArea;
var rect : SDL_Rect;
begin
xPos := 0;
yPos := 0;
rect.x := tx1;
rect.y := ty1;
SDL_BlitSurface(textarea, NIL, screen, @rect);
resetAnswerPositions
end;

function GetMouseAnswer(const mouse): char;
const defaultanswer = #06; { #06 = ACK: no answer - but it's "any key" }
var 
  x, y : LongInt;
  button: byte;
  i : integer;
  answer: char;
begin
x := SDL_MouseButtonEvent(mouse).x - tx1;
y := SDL_MouseButtonEvent(mouse).y - ty1;
button := SDL_MouseButtonEvent(mouse).button;
answer := defaultanswer;

{ left mouse button in textarea? }
if (button=SDL_BUTTON_LEFT) and 
   (x>=0) and (x<=tw) and (y>=0) and (y<=th) then
     begin
     i := 1;
     repeat
       if (y>=answerposition[i].f) and (y<=answerposition[i].t)
          then answer := ValueToKey(i);
       inc(i)
     until (answer<>defaultanswer) or (i>MaxAnswers);
     end;

{ right mouse button = Enter }
if button=SDL_BUTTON_RIGHT then answer := #13;

GetMouseAnswer := answer
end;

function GetKey: char;
var 
 event: SDL_Event;
 c: char;
begin
SDL_UPDATERECT(screen, tx1, ty1, tw, th);

c := #0;
repeat
  SDL_WaitEvent(@event);
  case event.eventtype of
    SDL_EventQuit: c:=ExitKey;
    SDL_MouseButtonDown: if mouseshown then c := GetMouseAnswer(event);
    SDL_KeyDown: begin
                 c := chr(SDL_KeyboardEvent(event).keysym.unicode and $FF);
		 if c=#0 then
		   Case SDL_KeyboardEvent(event).keysym.sym of
		     SDLK_KP0 : c := '0';
		     SDLK_KP1 : c := '1';
		     SDLK_KP2 : c := '2';
		     SDLK_KP3 : c := '3';
		     SDLK_KP4 : c := '4';
		     SDLK_KP5 : c := '5';
		     SDLK_KP6 : c := '6';
		     SDLK_KP7 : c := '7';
		     SDLK_KP8 : c := '8';
		     SDLK_KP9 : c := '9';
		     SDLK_KP_Enter : c := chr(13);
		     SDLK_EURO : c:=chr(164); { ISO-8559-15 }
		     end
		 end;
    end
until c<>#0;
GetKey := c
end;

end.
