{
* grquiz
* graphics oriented program for AKFQuiz
*
* Copyright (c) 2005 Andreas K. Foerster <akfquiz@akfoerster.de>
*
* Environment: 
* FreePascal (using SDL4FreePascal)
* GNU-Pascal and GRX library with GPC support (no other addon needed)
*
* This file is part of AKFQuiz
*
* AKFQuiz is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* AKFQuiz is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
*}

{ compatiblity definition }
{$IfDef _WIN32} {$Define Win32} {$EndIf}

{$IfDef Win32}
  {$R grquiz.res}
  {$Define Beeps}
{$EndIf}

{$IfDef DPMI}
  {$UnDef SDL}
  {$Define Beeps}
{$EndIf}

{$IfDef FPC}
  {$Mode Delphi}

  {$IfDef FPC} {$IfDef Unix} {$IfNDef SDL}
     { Under FPC + Unix SVGAlib is used }
     {$Define FPCSVGALIB}
     {$Define grNoEscKey} { cannot distiguish Esc from Esc-sequences }
     {$Define Beeps}
  {$EndIf} {$EndIf} {$EndIf}

  {$IfDef Win32}
    {$AppType GUI}
    {$Define SDL}
  {$EndIf}
{$EndIf}

{$I-}
{$X+}
{$R+} { Range checking }

program grquiz;

{$IfDef FPC}
  uses
  {$IfNDef SDL} 
    graph, clgrph,
  {$Else} { SDL }
    sdlgrph,
  {$EndIf}
  sysutils, uakfquiz, qsys, qmsgs; 
{$EndIf}

{$IfDef __GPC__}
  import uakfquiz; qmsgs; qsys; graph; grx; clgrph;
  {$UnDef SDL}
{$EndIf}

{$IfDef __GPC__}

  {$L quizhg.o}
  {$I hginfo.inc}

  {$If __GPC_RELEASE__ >= 20030303}
    var AKFQuizHg: array[0..$FFFF] of byte; external name 'AKFQuizHg';
  {$Else} {@@@ FIXME: inermediate versions with other syntax }
    var AKFQuizHg: asmname 'AKFQuizHg' array[0..$FFFF] of byte;
  {$EndIf}
{$EndIf}

{$IfDef FPC}
  {$I quizhg.inc}
{$EndIf}

{$I lat9.inc}

type keyset = set of char;

const linespace = fontheight; { you may add something }

const MaxAnswers = 35;

const AnswerKeys = [ '1'..'9', 'A'..'Z' ];

const
  Esc   = chr(27);
  Enter = chr(13);

{ Unix uses #27 + something for any function key,
  that's still so with SVGALib (not unter X11),
  so it's too difficult to filter out the Esc key as such :-( }

{$IFDEF grNoEscKey}
  const
    ExitKey = 'Q'; { capital letter! }
    ExitKeyName = ExitKey;
{$Else}
  const
    ExitKey = Esc;
    ExitKeyName = 'Esc';
{$EndIf}


type
  Tgrquiz =
    object(Takfquiz)
      readerror : boolean;
      MaxLength : word;

      { only temporarily used: }
      AnsPoints : array[1..MaxAnswers] of pointsType;

      constructor init(infile: string);
      procedure resetQuiz;                        virtual;
      procedure wait;
      procedure StartQuiz;                        virtual;
      procedure setcharset(c: string);            virtual;
      procedure processParagraph;
      procedure processComment;                   virtual;
      procedure processHint;                      virtual;
      procedure processAssessment;                virtual;
      procedure processAssessmentPercent;         virtual;
      procedure processQuestion;                  virtual;
      procedure processMulti;                     virtual;
      function getanswer(keys: keyset): integer;
      procedure processAnswer;                    virtual;
      procedure processMultiAnswer;               virtual;
      procedure showAnswers;
      procedure nextanswer;
      procedure evaluate;                         virtual;
      procedure EndQuiz;                          virtual;
      procedure error;                            virtual;
    end;

var infile: mystring;
var quiz: Tgrquiz;

var TextColor, BkColor, Color: Integer;

var MaxX, MaxY : SmallInt;

var 
  endless: boolean;
  fullscreen: boolean;
  usemouse: boolean;

function GetImgColor(x, y: integer): integer;
begin
{$IfDef FPC}
  With AKFQuizHG do
    GetImgColor := Image[(Width*y)+x]
{$Else}
  GetImgColor := GetPixel(x, y)
{$EndIf}
end;

procedure buildscreen;
{$IfDef UsePalette}
  var i: smallInt;
{$EndIf}
begin

{$IfDef UsePalette}
  for i:=0 to MaxColor do
      setRGBPalette(i, myPalette[i, RGBred],
                       myPalette[i, RGBgreen],
                       myPalette[i, RGBblue]);
{$EndIf}

ShowImage(AkfQuizHg);

TextColor := GetImgColor(TextColorX, TextColorY);
BkColor   := GetImgColor(BackColorX, BackColorY);

Color := TextColor;

setBkColor(BkColor); {@@@}
end;


procedure initializeGraphics;
begin
initializeGraphicMode(AKFQuizName+' '+AKFQuizVersion, 'AKFQuiz', fullscreen);
setExitKey(ExitKey);

buildscreen;

defineTextArea(marginLeft, marginTop, 
               GetMaxX-marginRight, GetMaxY-marginBottom, useTextArea);
MaxX := GetMaxX - marginLeft - marginRight;
MaxY := GetMaxY - marginTop - marginBottom;

setmsgconverter(UTF8toLatin1)
end;

procedure GrfWrite(const s: string);
var 
  p: integer;
  x, y: integer;
  ly, lx: integer;
begin
x := GetX;
y := GetY;

for p := 1 to Length(s) do
  begin
  for ly:=1 to fontheight do
    for lx:=0 to 7 do
      if (lat9[s[p], ly] and (1 shl (7-lx)))<>0 then 
         PutPixel(x+lx, y+ly, color);
  inc(x, 8)
  end;

{ move to end of line }
MoveTo(x, y);
end;

function GetTextWidth(const s: string): integer;
begin
GetTextWidth := Length(s)*8
end;

function waitkey(const s: string): boolean;
var c: char;
begin
GrfWrite(s);
c := GetKey;

clearTextarea;
waitkey := ( Upcase(c) <> ExitKey )
end;

procedure GRLN;
begin
MoveTo(0, GetY + linespace);

if GetY >= MaxY - (3*linespace) then
  begin
  MoveTo(0, GetY + linespace);
  waitkey(msg_anykey(''))
  end
end;

procedure GrfWriteLn(const s: string);
begin
GrfWrite(s);
GRLN
end;

procedure GRReadLn(var s: string);
var
  c: char;
  cwidth: integer;
begin
s := '';
repeat
  c := GetKey;
  if (c >= #32) and ( c<>#127) then { normal char }
    begin
    s := s + c;
    GrfWrite(c)
    end;
  if ((c=#127) or (c=#8)) and (s<>'') then
     begin
     c := s[length(s)]; { get last char }
     s := copy(s, 1, Length(s)-1); { remove last char from s }
     cwidth := GetTextWidth(c); { automatic typecast }
     MoveTo(GetX - cwidth, GetY);
     Color := BkColor;
     GrfWrite(c);
     MoveTo(GetX - cwidth, GetY);
     Color := TextColor;
     end;
  if not ((c in [#32 .. #255, #10, #13]) or (c = ExitKey)) then ErrorSignal;
until (c=#13) or (c=#10) or (c=ExitKey);

if c = ExitKey then s := '' { return nothing }
end;

procedure myshowentry(const s: string);
begin GrfWriteLn(stripext(s)) end;

function askfile: mystring;
var
  found: boolean;
  s: mystring;
begin
showmouse(false);
{$IfDef Advertisement}
  GrfWriteLn(msg_advertisement);
{$EndIf}
GRLn;

found := ListEntries(getquizpath, quizext, myshowentry);
if ListEntries(getquizpath, quizext2, myshowentry) then found := true;
if not found then GrfWriteLN(msg_noquizfound);

QuestionSignal;
GRLn;
GrfWrite('Quiz: ');
GRReadLn(s);
clearTextarea;
askfile := s
end;

{---------------------------------------------------------------------}
constructor Tgrquiz.init(infile: string);
begin
inherited init(infile);

setconverter(noconversion); { we're using an Lat1 font }
MaxLength := (GetMaxX-(2*40)) div GetTextWidth('m'); { largest letter }
if usemouse then showmouse(true)
end;

procedure Tgrquiz.resetQuiz;
begin
inherited resetQuiz;
readerror := false
end;

procedure Tgrquiz.setcharset(c: string);
begin
inherited setcharset(c);

c := makeUpcase(c);

if (c='IBM850') or (c='IBM437') or
   (c='CP850') or (c='CP437') or
   (c='850') or (c='437')
   then setconverter(OEMtoLatin1);
   
if (c='UTF-8') or (c='UTF8') then setconverter(UTF8toLatin1);
if (c='US-ASCII') then setconverter(noconversion)
end;


procedure Tgrquiz.wait;
begin
if not quit then
   quit := not waitkey(msg_anykey(ExitKeyName))
end;

procedure Tgrquiz.StartQuiz;
begin
inherited StartQuiz;

clearTextarea;
GRLN;
GRLN;
GrfWriteLn(msg_quiz + title);

if Author<>'' then
  GrfWriteLn(msg_author + Author);
if Translator<>'' then
  GrfWriteLn(msg_translator + Translator);
if Copyright<>'' then
  GrfWriteLn('Copyright: ' + Copyright);
GRLN;
{$IfDef Advertisement}
  GrfWriteLn(msg_advertisement);
  GRLN;
{$EndIf}

wait
end;

procedure Tgrquiz.processParagraph;
var s, rest, outs : mystring;
begin
outs := '';
s := readLine;
while s<>'' do
   begin
   if s='.'
     then begin
          GrfWriteLn(outs);
	  outs := '';
	  GRLN
          end
     else begin
          rest := s;
          while rest<>'' do
	    begin
	    if outs=''
	      then outs := format(rest, MaxLength-Length(outs), MaxLength)
	      else outs := outs + ' ' +
                           format(rest, MaxLength-Length(outs), MaxLength);
            if rest<>'' then
	      begin
	      GrfWriteLn(outs);
	      outs := ''
	      end
	    end
          end;
   s := readLine
   end;
GrfWriteLn(outs)
end;

procedure Tgrquiz.processComment;
begin
clearTextarea;
processParagraph;
GrLN;
wait
end;

procedure Tgrquiz.processHint;
begin
processComment { handle like a Comment }
end;

procedure Tgrquiz.processAssessment;
begin
processComment
end;

procedure Tgrquiz.processAssessmentPercent;
var rest: mystring;
begin
rest := readAssessmentPercent;
while rest<>'' do
    GrfWriteLn(format(rest, MaxLength, MaxLength));
GrLN;
wait
end;

procedure Tgrquiz.processQuestion;
begin
clearTextarea;
inherited processQuestion;

processParagraph;
processAnswer
end;

procedure Tgrquiz.processMulti;
begin
clearTextarea;
inherited processMulti;

processParagraph;
processMultiAnswer
end;

procedure Tgrquiz.nextanswer;
begin
if answerNr>0 then answerEnds(answerNr);
inc(answerNr);

if answerNr>MaxAnswers then 
   begin 
   error; 
   answerNr := MaxAnswers { to avoid range overruns }
   end;
answerStarts(answerNr);
end;


function Tgrquiz.getanswer(keys: keyset): integer;
var
  c : char;
  okay: boolean;
begin
repeat
  c := Upcase(GetKey);
  okay := c in keys;
  if not okay then ErrorSignal;
until okay;

if c=ExitKey then quit := true;

if c in AnswerKeys
  then begin GrfWrite(c); getanswer := KeyToValue(c) end
  else getanswer := -1
end;

procedure Tgrquiz.showAnswers;
var
  s, ans: mystring;
  value: pointsType;
  startpos: word;
begin
GRLN;

answerNr := 0;
readAnswer(value, s);
while s<>'' do
  begin
  ans := s;
  nextanswer;
  AnsPoints[answerNr] := value;
  GrfWrite(ValueToKey(answerNr) + ') ');
  startpos := GetX;
  GrfWriteLn(format(ans, MaxLength-3, MaxLength-3));
  while ans<>'' do
     begin
     MoveTo(startpos, GetY);
     GrfWriteLn(format(ans, MaxLength-3, MaxLength-3))
     end;
  readAnswer(value, s)
  end
end;

procedure Tgrquiz.processAnswer;
var
  maxKey: char;
  ap : pointsType;
  i : integer;
begin
showanswers;

{ show default-Answer }
if defanswer<>'' then
  begin
  nextanswer;
  AnsPoints[answerNr] := 0;
  GrfWriteLn(ValueToKey(answerNr) + ') ' + defanswer)
  end;

answerEnds(answerNr);

GRLN;

GrfWrite('> ');

maxKey := ValueToKey(answerNr);
if answerNr<10 
   then i := getanswer([ '1'..maxKey, ExitKey ])
   else i := getanswer([ '1'..'9', 'A'..maxKey, ExitKey ]);

GrLn;

if (i<>-1) and not quit then
  begin
  ap := AnsPoints[i];
  inc(Points, ap);

  if not neutral then
    if ap > 0
      then begin GrfWriteLn(msg_right); RightSignal end
      else begin GrfWriteLn(msg_wrong); FalseSignal end;

  GRLN;
  GrfWriteLn(msg_points + IntToStr(Points));
  wait
  end
end;

procedure Tgrquiz.processMultiAnswer;
var i: integer;
    keys: keyset;
    maxKey: char;
    myPoints, myMax : pointsType;
begin
myPoints := 0;
myMax := 0;

showanswers;
answerEnds(answerNr);

GRLN;
GrfWrite('>> ');

{ max points for this one question }
for i := 1 to answerNr do
  if AnsPoints[i] > 0 then inc(myMax, AnsPoints[i]);

maxKey := ValueToKey(answerNr);
if answerNr<10 
   then keys := [ '1'..maxKey, ExitKey, Enter ]
   else keys := [ '1'..'9', 'A'..maxKey, ExitKey, Enter ];

repeat
  i := getAnswer(keys);
  if i<>-1 then
    begin
    inc(Points, AnsPoints[i]);   { absolute points }
    inc(myPoints, AnsPoints[i]); { points for this question }
    { remove that key from valid keys }
    keys := keys - [ ValueToKey(i) ]
    end;
until i=-1;

GRLN; GRLN;
GrfWriteLn(msg_points + IntToStr(Points) +
           '  (' + IntToStr(myPoints) +
	   '/' + IntToStr(myMax) + ')');
wait
end;

procedure Tgrquiz.evaluate;
begin
clearTextarea;
if not evaluated and not quit and (MaxPoints<>0) then
  begin
  GrfWriteLn(msg_sol1 + IntToStr(Points) + msg_sol2 +
            IntToStr(MaxPoints) + msg_sol3);
  If Points > 0
    then GrfWriteLn(msg_sol4 + IntToStr(getPercentage) + '%.')
    else if not neutral then GrfWriteLn(msg_sol5)
  end;
inherited evaluate;
GRLN;

InfoSignal;
wait
end;

procedure Tgrquiz.EndQuiz;
begin
if not evaluated and not quit
  then evaluate
end;

procedure Tgrquiz.error;
begin
readerror := true;
quit := true
end;


{---------------------------------------------------------------------}

procedure help;
begin
WriteLn(AKFQuizName+', grquiz, version '+AKFQuizVersion);
WriteLn('(' + platform + ')');
WriteLn(AKFQuizCopyright);
WriteLn;
WriteLn('  grquiz [options] [inputfile]');
WriteLn('  grquiz [ -h | --help | /? ]');
WriteLn;
WriteLn('Options:');
WriteLn('-f | --fullscreen  fullscreen mode (if supported)');
WriteLn('-w | --window      window mode (if supported)');
WriteLn('-s                 no sound');
WriteLn('-m | --nomouse     no mouse');
WriteLn('-d <dir>           path to quizfiles');
WriteLn('-p                 endless');
{$IfDef FPC} {$IfDef Go32v2}
WriteLn('-LFN               use long filenames (DOS only)');
{$EndIf} {$EndIf}
Halt
end;

procedure ErrorFileNotFound;
begin
endGraphics;
WriteLn(stderr, msg_filenotfound);
Halt(2)
end;

procedure parameters;
var
  i: integer;
  p: mystring;
begin
i := 0;
while i<ParamCount do
  begin
  inc(i);
  p := makeUpcase(ParamStr(i));
  if p='-LFN' then
      begin setLFNsupport; continue end;
  if p='-S' then
      begin DisableSignals; continue end;
  if (p='-D') then begin inc(i); { handled in qsys } continue end;
  if (p='-P') or (p='/P') then
     begin endless := true; nobreak; continue end;
  if (p='-F') or (p='--FULLSCREEN') then
     begin fullscreen := true; continue end;
  if (p='-W') or (p='--WINDOW') then
     begin fullscreen := false; continue end;
  if (p='-M') or (p='--NOMOUSE') then
     begin usemouse := false; continue end;
  if (p='-H') or (p='--HELP') or (p='/?') then help;
  if p[1]='-'    { "/" might be used in a path }
     then help { unknown parameter }
     else infile := ParamStr(i); { not Upcase }
  end;

{$IfDef Floppy}
  endless := true; { Floppy is always endless }
  nobreak
{$EndIf}
end;


var myexitcode : byte;

begin { main }

{$IfDef FPCSVGALIB}
  { space after messages from SVGALib }
  WriteLn;
  WriteLn;
{$EndIf}

myexitcode := 0;
endless := false;
fullscreen := false;
usemouse := true;
useSystemLanguage;

parameters;

{$IfNDef Beeps}
  DisableSignals; { no beeps possible in most cases }
  {$M no Sound}
{$EndIf}

InitializeGraphics;

repeat
  if infile<>'' then
     begin
     infile := usequizpath(infile);
     if not getquizfile(infile) then ErrorFileNotFound
     end;

  if infile='' then
    repeat
      infile := askfile;
      if not endless and (infile='') 
         then begin endGraphics; Halt end;
      if infile<>'' then infile := usequizpath(infile);
    until (infile<>'') and getquizfile(infile);

  quiz.Init(infile);

  quiz.process;
  { exitcode 1 reserved for graphical error! }
  if quiz.readerror then myexitcode := 2;
  clearTextarea;
  quiz.Done;

  infile := '';

  if endless then parameters { to get input files again }

until not endless;

endGraphics;
if myexitcode=2 then WriteLn(stderr, msg_error);

Halt(myexitcode)
end.
