//**************************************************************************
//*                     This file is part of the                           *
//*             Ogg Vorbis encoder output DLL plug-in for                  *
//*                      Mpxplay - audio player.                           *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2007 by PDSoft (Attila Padar)                *
//*                    http://mpxplay.cjb.net                              *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//OGG encoding

#include "mpxplay.h"
#include "vorbis\os_types.h"
#include "vorbis\codec.h"
#include "vorbis\vorbisen.h"
#include "newfunc\newfunc.h"
#include "control\control.h"
#include "utf8.h"
#include <string.h>
#include <malloc.h>
#include <time.h>

#define PRG_VERSION "1.54.3"

typedef float OGG_DOUBLE_T;
typedef float OGG_FLOAT_T;
#define FLOAT_BITS (sizeof(OGG_DOUBLE_T)*8)

typedef struct ogg_main_info{
 int current_decoder_part;
 int filehand;
 ogg_sync_state   oy;
 ogg_stream_state os;
 ogg_page         og;
 ogg_packet       op;

 vorbis_info      vi;
 vorbis_comment   vc;
 vorbis_dsp_state vd;
 vorbis_block     vb;
}ogg_main_info;

static int   opt_bitrate,opt_channel_coupling,opt_const_bitrate;
static float opt_lowpass_khz,opt_vbr_quality=-9.0f;
static int   opt_utf8_disable;
static char *opt_charset;

static topt oggenc_opts[] = {
{"oggv_br"  ,ARG1|ARG_NUM,      &opt_bitrate         ,0,0},
{"oggv_enq" ,ARG1|ARG_FLO,      &opt_vbr_quality     ,0,0},
{"oggv_ccp" ,0,                 &opt_channel_coupling,1,0},
{"oggv_cbr" ,0,                 &opt_const_bitrate   ,1,0},
{"oggv_lpf" ,ARG1|ARG_FLO,      &opt_lowpass_khz     ,0,0},
{"oggv_tn8" ,0,                 &opt_utf8_disable    ,1,0},
{"oggv_tecp",ARG2|ARG_CHAR,     &opt_utf8_disable    ,0,&opt_charset},
{NULL    ,0,0,0,0}
};

#ifdef __DOS__
struct mpxplay_resource_s *mrs;
#define OGGV_PRINTF(s) mrs->pds_textdisplay_printf(s)
#else
#include <share.h>
void OGGV_PRINTF(char *s)
{
 printf("%s\n",s);
}
#endif

static void ogg_comment_set(struct mpxplay_audioout_info_s *aui,vorbis_comment *);
static int  ogg_write_header(struct ogg_main_info *omip);

static unsigned int oec_ready;

static void ogg_encoder_config(void)
{
 if(!oec_ready){
  if(opt_bitrate)
   opt_bitrate*=1000;
  else
   opt_bitrate=128000;
  oec_ready=1;
 }
}

static int ogg_encoder_open(struct mpxplay_audioout_info_s *aui)
{
 int error;
 struct ogg_main_info *omip;
 double newkhz;
 char fullname[300];

 ogg_encoder_config();

 if(!aui->pei){
  OGGV_PRINTF("oggenc error: no aui->pei!");
  return -1;
 }

 aui->card_private_data=omip=malloc(sizeof(struct ogg_main_info));
 if(omip==NULL)
  return -1;
 memset(omip,0,sizeof(struct ogg_main_info));

 vorbis_info_init(&omip->vi);

 if(opt_channel_coupling) // we must to set this before the bitrate management
  vorbis_encode_ctl(&omip->vi,OV_ECTL_COUPLE_SET,&opt_channel_coupling);

 if((opt_vbr_quality>=0.0f) && (opt_vbr_quality<=1.0f))
  error=vorbis_encode_setup_vbr(&omip->vi,aui->chan_card,aui->freq_card,opt_vbr_quality);
 else
  error=vorbis_encode_setup_managed(&omip->vi,aui->chan_card,aui->freq_card,-1,opt_bitrate,-1);

 if(error!=0){
  OGGV_PRINTF("oggenc error: invalid freq (-of) or bitrate (-oggv_br) or -oggv_ccp required");
  return error;
 }

 if(!opt_const_bitrate)
  vorbis_encode_ctl(&omip->vi,OV_ECTL_RATEMANAGE2_SET,NULL);

 if(opt_lowpass_khz){
  newkhz=(double)opt_lowpass_khz;
  vorbis_encode_ctl(&omip->vi,OV_ECTL_LOWPASS_SET,&newkhz);
 }

 error=vorbis_encode_setup_init(&omip->vi);
 if(error!=0){
  OGGV_PRINTF("oggenc error : cannot initialize encoder! (invalid settings?)");
  return error;
 }

 if(aui->card_controlbits&AUINFOS_CARDCNTRLBIT_AUTOTAGGING)
  ogg_comment_set(aui,&omip->vc);
 vorbis_analysis_init(&omip->vd,&omip->vi);
 vorbis_block_init(&omip->vd,&omip->vb);

 pds_getfilename_noext_from_fullname(fullname,aui->pei->filename);
 pds_strcat(fullname,".OGG");

 if(pds_stricmp(fullname,aui->pei->filename)!=0) // input and output filename must be different
#ifdef __DOS__
  omip->filehand=mrs->pds_open_create(fullname,O_RDWR|O_BINARY);
#else
  omip->filehand=sopen(fullname,O_RDWR|O_BINARY|O_CREAT|O_TRUNC,SH_DENYWR,S_IREAD|S_IWRITE);
#endif
 if(!omip->filehand){
  OGGV_PRINTF("oggenc error: cannot open output file!");
  return -1;
 }

 error=ogg_write_header(omip);
 if(error!=0){
  OGGV_PRINTF("oggenc error : cannot write Ogg header! (disk error?)");
  return error;
 }
 aui->card_wave_id=0x0003; // 32-bit float pcm needed
 aui->bits_card=1;         // scale: -1.0 - +1.0
 return error;
}

static void acv_ogg_add_tag(vorbis_comment *vc,char *tagtype,char *taginfo)
{
 if(!opt_utf8_disable){
  char *utf8=alloca(500);
  utf8_encode(utf8,taginfo,500);
  vorbis_comment_add_tag(vc,tagtype,utf8);
 }else
  vorbis_comment_add_tag(vc,tagtype,taginfo);
}

static void ogg_comment_set(struct mpxplay_audioout_info_s *aui,vorbis_comment *vc)
{
 struct playlist_entry_info *pei=aui->pei;
 char strtmp[128];

 if(!opt_utf8_disable)
  utf8_init(opt_charset);

 vorbis_comment_init(vc);

 if(pei->id3info[I3I_ARTIST])  acv_ogg_add_tag(vc,"ARTIST", pei->id3info[I3I_ARTIST]);
 if(pei->id3info[I3I_TITLE])   acv_ogg_add_tag(vc,"TITLE",  pei->id3info[I3I_TITLE]);
 if(pei->id3info[I3I_ALBUM])   acv_ogg_add_tag(vc,"ALBUM",  pei->id3info[I3I_ALBUM]);
 if(pei->id3info[I3I_YEAR])    acv_ogg_add_tag(vc,"DATE",   pei->id3info[I3I_YEAR]);
 if(pei->id3info[I3I_GENRE])   acv_ogg_add_tag(vc,"GENRE",  pei->id3info[I3I_GENRE]);
 if(pei->id3info[I3I_COMMENT]) acv_ogg_add_tag(vc,"COMMENT",pei->id3info[I3I_COMMENT]);
 if(pei->id3info[I3I_TRACKNUM])acv_ogg_add_tag(vc,"TRACKNUM",pei->id3info[I3I_TRACKNUM]);

 sprintf(strtmp,"OggVorbis output plugin v%s for Mpxplay by PDSoft",PRG_VERSION);
 acv_ogg_add_tag(vc,"ENCODER",strtmp);
}

//-------------------------------------------------------------------------

static int ogg_write_header(struct ogg_main_info *omip)
{
 ogg_packet header;
 ogg_packet header_comm;
 ogg_packet header_code;

 srand(time(NULL));
 ogg_stream_init(&omip->os,rand());
 //ogg_stream_init(&omip->os,16); // for binary comparisons (tests)
 vorbis_analysis_headerout(&omip->vd,&omip->vc,&header,&header_comm,&header_code);
 ogg_stream_packetin(&omip->os,&header);
 ogg_stream_packetin(&omip->os,&header_comm);
 ogg_stream_packetin(&omip->os,&header_code);

 while(1){
  int result=ogg_stream_flush(&omip->os,&omip->og);
  if(result==0)
   break;
#ifdef __DOS__
  mrs->pds_dos_write(omip->filehand,omip->og.header,omip->og.header_len);
  mrs->pds_dos_write(omip->filehand,omip->og.body,omip->og.body_len);
#else
  write(omip->filehand,omip->og.header,omip->og.header_len);
  write(omip->filehand,omip->og.body,omip->og.body_len);
#endif
 }
 return 0;
}

static void OGG_writedata(struct mpxplay_audioout_info_s *aui,char *pcm_sample,unsigned long outbytes)
{
 int eos=0;
 unsigned int outsamplenum=outbytes/sizeof(MPXPLAY_PCMOUT_FLOAT_T)/aui->chan_card;
 struct ogg_main_info *omip=aui->card_private_data;

 if(!omip)
  return;

 if(outsamplenum==0){
  vorbis_analysis_wrote(&omip->vd,0);
 }else{
  OGG_DOUBLE_T **buffer=vorbis_analysis_buffer(&omip->vd,outsamplenum);
  unsigned int i,ch,outchs=aui->chan_card;
  for(ch=0;ch<outchs;ch++){
   MPXPLAY_PCMOUT_FLOAT_T *inbuf=((MPXPLAY_PCMOUT_FLOAT_T *)(pcm_sample))+ch;
   OGG_DOUBLE_T *outbuf=buffer[ch];
   for(i=0;i<outsamplenum;i++){
    MPXPLAY_PCMOUT_FLOAT_T insamp=inbuf[0];
    outbuf[0]=(OGG_DOUBLE_T)insamp;
    inbuf+=outchs;
    outbuf++;
   }
  }
  vorbis_analysis_wrote(&omip->vd,outsamplenum);
 }

 while(vorbis_analysis_blockout(&omip->vd,&omip->vb)==1){
  vorbis_analysis(&omip->vb,NULL);
  vorbis_bitrate_addblock(&omip->vb);

  while(vorbis_bitrate_flushpacket(&omip->vd,&omip->op)){
   ogg_stream_packetin(&omip->os,&omip->op);
   do{
    int writelen;
    int result=ogg_stream_pageout(&omip->os,&omip->og);
    if(result==0)
     break;

#ifdef __DOS__
    writelen=mrs->pds_dos_write(omip->filehand,omip->og.header,omip->og.header_len);
#else
    writelen=write(omip->filehand,omip->og.header,omip->og.header_len);
#endif
    if(writelen!=omip->og.header_len)
     return;

#ifdef __DOS__
    writelen=mrs->pds_dos_write(omip->filehand,omip->og.body,omip->og.body_len);
#else
    writelen=write(omip->filehand,omip->og.body,omip->og.body_len);
#endif
    if(writelen!=omip->og.body_len)
     return;

    if(ogg_page_eos(&omip->og))
     eos=1;
   }while(!eos);
  }
 }

 return;
}

static void ogg_encoder_close(struct mpxplay_audioout_info_s *aui)
{
 struct ogg_main_info *omip;

 if(aui->card_private_data){
  omip=aui->card_private_data;

  ogg_stream_clear(&omip->os);
  vorbis_block_clear(&omip->vb);
  vorbis_dsp_clear(&omip->vd);
  vorbis_comment_clear(&omip->vc);
  vorbis_info_clear(&omip->vi);

  if(omip->filehand)
#ifdef __DOS__
   mrs->pds_close(omip->filehand);
#else
   close(omip->filehand);
#endif

  free(aui->card_private_data);
  aui->card_private_data=NULL;
 }
}

static void OGG_close(struct mpxplay_audioout_info_s *aui)
{
 struct ogg_main_info *omip;

 if(aui->card_private_data){
  omip=aui->card_private_data;

  OGG_writedata(aui,NULL,0); // finish encoding (flush)

  ogg_encoder_close(aui);
 }
}

//--------------------------------------------------------------------
static int OGG_init(struct mpxplay_audioout_info_s *aui)
{
 aui->card_port=aui->card_isa_dma=aui->card_irq=aui->card_isa_hidma=aui->card_type=0;
 return 1;
}

static int OGG_detect(struct mpxplay_audioout_info_s *aui)
{
 aui->card_port=aui->card_isa_dma=aui->card_irq=aui->card_isa_hidma=aui->card_type=0;
 return 1;
}

static void OGG_card_info(struct mpxplay_audioout_info_s *aui)
{
 OGGV_PRINTF("-------------------------------------------------------------------------------");
 OGGV_PRINTF("OGGV : OGG Vorbis v1.1.2 (20050304) encoder output plugin (disk writer) v1.54.3");
 OGGV_PRINTF("");
 OGGV_PRINTF("Available options (use in the command line of Mpxplay):");
 OGGV_PRINTF(" -oggv_br NUM  : set bitrate (48-480 at 44.1kHz)(def.:128)");
 OGGV_PRINTF(" -oggv_enq NUM : set encoding quality instead of bitrate (0.0-1.0)");
 OGGV_PRINTF(" -oggv_ccp     : use channel coupling (recommended/required for low bitrates)");
 OGGV_PRINTF(" -oggv_cbr     : use constant bitrate (else VBR)(not recommended)");
 OGGV_PRINTF(" -oggv_lpf KHZ : set lowpass filter (cut off high sounds)(ie: 16.5)");
 OGGV_PRINTF(" -oggv_tn8     : disable UTF-8 tag-text encoding (enabled by default)");
 OGGV_PRINTF(" -oggv_tcp CPNAME : set codepage to UTF-8 (default:ISO-8859-2)(see charmaps.h)");
 OGGV_PRINTF("-------------------------------------------------------------------------------");
}

static void OGG_setrate(struct mpxplay_audioout_info_s *aui)
{
 OGG_close(aui);
 if(ogg_encoder_open(aui)!=0)
  ogg_encoder_close(aui);
}

one_sndcard_info OGG_sndcard_info={
 "OGG",
 SNDCARD_FLAGS_DISKWRITER,

 NULL,            // card_config
 &OGG_init,       // card_init
 &OGG_detect,     // card_detect
 &OGG_card_info,  // card_info
 NULL,            // card_start
 NULL,            // card_stop
 &OGG_close,      // card_close
 &OGG_setrate,    // card_setrate

 &OGG_writedata,  // cardbuf_writedata
 NULL,            // cardbuf_pos
 NULL,            // cardbuf_clear
 NULL,            // cardbuf_int_monitor
 NULL,            // irq_routine

 NULL,            // card_writemixer
 NULL,            // card_readmixer
 NULL             // card_mixerchans
};

static mpxplay_module_entry_s oggenc_output_module_entry={
 MPXPLAY_DLLMODULETYPE_AUCARD,
 0,
 "OGGV",
 MPXPLAY_DLLMODULEVER_AUCARD,
 (void *)&OGG_sndcard_info
};

static mpxplay_module_entry_s oggenc_cmdline_module_entry={
 MPXPLAY_DLLMODULETYPE_CONTROL_CMDLINE,
 0,
 NULL,
 MPXPLAY_DLLMODULEVER_CONTROL_CMDLINE,
 (void *)oggenc_opts
};

static mpxplay_dll_entry_s mpxplay_dll_entry_structure={
 MPXPLAY_DLLENTRY_STRUCTURE_VERSION,
 {
  &oggenc_output_module_entry,
  &oggenc_cmdline_module_entry,
  NULL
 }
};

#ifdef __DOS__

extern void dllstrtr_update_crwdata(unsigned long *cwd);

long __export mpxplay_dll_entrypoint(struct mpxplay_resource_s *p_mrs,unsigned long *crwdata_begin)
{
 mrs=p_mrs;
 dllstrtr_update_crwdata(crwdata_begin);
 return (long)(&mpxplay_dll_entry_structure);
}

#else

int matherr(struct _exception *a)
{
 a->retval=1.0;
 return 1;
}

__declspec( dllexport ) mpxplay_dll_entry_s *mpxplay_dll_entrypoint(void)
{
 return (&mpxplay_dll_entry_structure);
}

#endif
