//**************************************************************************
//*                     This file is part of the                           *
//*               FLAC/FLAKE encoder output DLL plug-in for                *
//*                      Mpxplay - audio player.                           *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2007 by PDSoft (Attila Padar)                *
//*                    http://mpxplay.cjb.net                              *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//FLAC encoding with FLAKE library

#include "mpxplay.h"
#include "newfunc\newfunc.h"
#include "control\control.h"
#include "lib_flac\flake.h"
#include <string.h>
#include <malloc.h>
#include <inttypes.h>

typedef int16_t pcmbuf_t;

typedef struct flac_main_info{
 int filehand;
 int header_len;
 unsigned long samplecount;

 unsigned long pcmbufsize;
 unsigned long pcmbuf_samples;
 pcmbuf_t *pcmbuf;
 uint8_t *framebuf;

 FlakeContext flc;

}flac_main_info;

static int opt_compr_ratio=5,opt_utf8_disable;
static char *opt_charset;

static topt flacenc_opts[] = {
{"flac_c"   ,ARG1|ARG_NUM,  &opt_compr_ratio     ,0,0},
{"flac_tn8" ,0,             &opt_utf8_disable    ,1,0},
{"flac_tcp" ,ARG2|ARG_CHAR, &opt_utf8_disable    ,0,&opt_charset},
{NULL    ,0,0,0,0}
};

#ifdef __DOS__
struct mpxplay_resource_s *mrs;
#define FLAC_PRINTF(s) mrs->pds_textdisplay_printf(s)
#else
#include <share.h>
void FLAC_PRINTF(char *s)
{
 printf("%s\n",s);
}
#endif

static int FLAC_encoder_open(struct mpxplay_audioout_info_s *aui)
{
 struct flac_main_info *fmi;
 FlakeContext *flc;
 struct playlist_entry_info *pei=aui->pei;
 char fullname[300];

 if(!pei){
  FLAC_PRINTF("FLACenc error: no aui->pei!");
  return -1;
 }

 if(opt_compr_ratio<1)
  opt_compr_ratio=1;
 if(opt_compr_ratio>12)
  opt_compr_ratio=12;

 aui->card_wave_id=0x0001; // integer pcm
 aui->bits_card=16;        // 16 bits

 aui->card_private_data=fmi=calloc(1,sizeof(struct flac_main_info));
 if(!fmi)
  return 0;

 flc=&fmi->flc;

 if(aui->card_controlbits&AUINFOS_CARDCNTRLBIT_AUTOTAGGING){
  if(!opt_utf8_disable)
   flake_comment_init_utf8(flc,opt_charset);

  if(pei->id3info[I3I_ARTIST])  flake_comment_add(flc,"artist", pei->id3info[I3I_ARTIST]);
  if(pei->id3info[I3I_TITLE])   flake_comment_add(flc,"title", pei->id3info[I3I_TITLE]);
  if(pei->id3info[I3I_ALBUM])   flake_comment_add(flc,"album", pei->id3info[I3I_ALBUM]);
  if(pei->id3info[I3I_YEAR])    flake_comment_add(flc,"date", pei->id3info[I3I_YEAR]);
  if(pei->id3info[I3I_GENRE])   flake_comment_add(flc,"genre", pei->id3info[I3I_GENRE]);
  if(pei->id3info[I3I_COMMENT]) flake_comment_add(flc,"comment", pei->id3info[I3I_COMMENT]);
  if(pei->id3info[I3I_TRACKNUM])flake_comment_add(flc,"tracknumber", pei->id3info[I3I_TRACKNUM]);
 }

 if(aui->chan_card>8)
  aui->chan_card=8;

 flc->channels=aui->chan_card;
 flc->sample_rate=aui->freq_card;
 flc->bits_per_sample=aui->bits_card;
 flc->samples=(long)((float)aui->pei->timemsec*(float)flc->sample_rate);
 flc->compression=opt_compr_ratio;
 flc->order_method=-1;
 flc->stereo_method=-1;
 flc->block_size=0;
 flc->padding_size=-1;
 flc->max_order=-1;

 fmi->header_len=flake_encode_init(flc);
 if(fmi->header_len<=0 || !flc->max_frame_size || flc->block_size<=0){
  FLAC_PRINTF("FLACenc error: cannot initialize encoder!");
  return -1;
 }

 fmi->pcmbufsize=flc->block_size*flc->channels*2;

 fmi->pcmbuf=malloc(fmi->pcmbufsize*sizeof(pcmbuf_t));
 fmi->framebuf=malloc(flc->max_frame_size);
 if(!fmi->pcmbuf|| !fmi->framebuf){
  FLAC_PRINTF("FLACenc error: cannot malloc buffers!");
  return -1;
 }

 pds_getfilename_noext_from_fullname(fullname,aui->pei->filename);
 pds_strcat(fullname,".FLA");

 if(pds_stricmp(fullname,aui->pei->filename)!=0) // input and output filename must be different
#ifdef __DOS__
  fmi->filehand=mrs->pds_open_create(fullname,O_RDWR|O_BINARY);
#else
  fmi->filehand=sopen(fullname,O_RDWR|O_BINARY|O_CREAT|O_TRUNC,SH_DENYWR,S_IREAD|S_IWRITE);
#endif
 if(!fmi->filehand){
  FLAC_PRINTF("FLACenc error: cannot open output file!");
  return -1;
 }
#ifdef __DOS__
 mrs->pds_dos_write(fmi->filehand,flc->header,fmi->header_len);
#else
 write(fmi->filehand,flc->header,fmi->header_len);
#endif

 return 0;
}

//-------------------------------------------------------------------------

static void FLAC_writedata(struct mpxplay_audioout_info_s *aui,char *pcm_sample,unsigned long outbytes)
{
 struct flac_main_info *fmi=aui->card_private_data;
 FlakeContext *flc;
 int16_t *pcm16=(int16_t *)pcm_sample;
 unsigned int n,samplenum;
 int framebytes;

 if(!fmi)
  return;
 flc=&fmi->flc;

 samplenum=outbytes/(aui->bits_card/8);

 if((fmi->pcmbuf_samples+samplenum)>fmi->pcmbufsize) // should not happen
  if(fmi->pcmbufsize>fmi->pcmbuf_samples)
   samplenum=fmi->pcmbufsize-fmi->pcmbuf_samples;
  else
   samplenum=0;

 for(n=0;n<samplenum;n++)
  fmi->pcmbuf[fmi->pcmbuf_samples+n]=pcm16[n];
 fmi->pcmbuf_samples+=samplenum;

 if(fmi->pcmbuf_samples<(flc->block_size*flc->channels))
  return;

 framebytes=flake_encode_frame(flc,fmi->framebuf,fmi->pcmbuf);

 if(framebytes>0)
#ifdef __DOS__
  mrs->pds_dos_write(fmi->filehand,fmi->framebuf,framebytes);
#else
  write(fmi->filehand,fmi->framebuf,framebytes);
#endif

 samplenum=flc->block_size*flc->channels;
 memcpy((void *)&fmi->pcmbuf[0],(void *)&fmi->pcmbuf[samplenum],samplenum*sizeof(pcmbuf_t));
 fmi->pcmbuf_samples-=samplenum;

 fmi->samplecount+=flc->block_size;

 return;
}

static long FLAC_bufpos(struct mpxplay_audioout_info_s *aui)
{
 struct flac_main_info *fmi=aui->card_private_data;

 if(!fmi)
  return (2*576*2);

 return (aui->chan_card*fmi->flc.block_size*(aui->bits_card/8)); // bytenum request
}

static void FLAC_close(struct mpxplay_audioout_info_s *aui)
{
 struct flac_main_info *fmi=aui->card_private_data;
 uint8_t sc[4];

 if(fmi){
  if(fmi->filehand){
   FlakeContext *flc=&fmi->flc;
   if(fmi->pcmbuf_samples>flc->channels){ // flush data
    int framebytes;
    flc->block_size=fmi->pcmbuf_samples/flc->channels;
    framebytes=flake_encode_frame(flc,fmi->framebuf,fmi->pcmbuf);
    if(framebytes>0)
#ifdef __DOS__
     mrs->pds_dos_write(fmi->filehand,fmi->framebuf,framebytes);
#else
     write(fmi->filehand,fmi->framebuf,framebytes);
#endif
    fmi->samplecount+=flc->block_size;
   }
  }

  flake_encode_close(&fmi->flc);

  if(fmi->filehand){
   if(fmi->samplecount){ // update header
    sc[0] = ((fmi->samplecount >> 24) & 0xFF);
    sc[1] = ((fmi->samplecount >> 16) & 0xFF);
    sc[2] = ((fmi->samplecount >>  8) & 0xFF);
    sc[3] = ( fmi->samplecount        & 0xFF);
#ifdef __DOS__
    mrs->pds_lseek(fmi->filehand,22,SEEK_SET);
    mrs->pds_dos_write(fmi->filehand,sc,4);
    mrs->pds_dos_write(fmi->filehand,fmi->flc.md5digest,16);
#else
    lseek(fmi->filehand,22,SEEK_SET);
    write(fmi->filehand,sc,4);
    write(fmi->filehand,fmi->flc.md5digest,16);
#endif
   }
#ifdef __DOS__
   mrs->pds_close(fmi->filehand);
#else
   close(fmi->filehand);
#endif
  }

  if(fmi->framebuf)
   free(fmi->framebuf);
  if(fmi->pcmbuf)
   free(fmi->pcmbuf);

  free(aui->card_private_data);
  aui->card_private_data=NULL;
 }
}

//--------------------------------------------------------------------
static int FLAC_init(struct mpxplay_audioout_info_s *aui)
{
 aui->card_port=aui->card_isa_dma=aui->card_irq=aui->card_isa_hidma=aui->card_type=0;
 return 1;
}

static int FLAC_detect(struct mpxplay_audioout_info_s *aui)
{
 aui->card_port=aui->card_isa_dma=aui->card_irq=aui->card_isa_hidma=aui->card_type=0;
 return 1;
}

static void FLAC_card_info(struct mpxplay_audioout_info_s *aui)
{
 FLAC_PRINTF("------------------------------------------------------------------------------");
 FLAC_PRINTF("FLAC : FLAKE v0.10 FLAC audio encoder output plugin (disk writer) v1.54.2");
 FLAC_PRINTF("");
 FLAC_PRINTF("Available options (use in the command line of Mpxplay):");
 FLAC_PRINTF(" -flac_c NUM     : compression ratio (1-12)(def:5)");
 FLAC_PRINTF(" -flac_tcp NAME  : codepage for utf8 text encoding (def: ISO-8859-2)");
 FLAC_PRINTF(" -flac_tn8       : disable utf8 text encoding (def: enabled)");
 FLAC_PRINTF("");
 FLAC_PRINTF("note:");
 FLAC_PRINTF("-only 16 bit is supported (8,24 are converted to this)");
 FLAC_PRINTF("------------------------------------------------------------------------------");
}

static void FLAC_setrate(struct mpxplay_audioout_info_s *aui)
{
 FLAC_close(aui);
 if(FLAC_encoder_open(aui)!=0)
  FLAC_close(aui);
}

one_sndcard_info FLAC_sndcard_info={
 "FLA",
 SNDCARD_FLAGS_DISKWRITER|SNDCARD_CARDBUF_SPACE,

 NULL,            // card_config
 &FLAC_init,      // card_init
 &FLAC_detect,    // card_detect
 &FLAC_card_info, // card_info
 NULL,            // card_start
 NULL,            // card_stop
 &FLAC_close,     // card_close
 &FLAC_setrate,   // card_setrate

 &FLAC_writedata, // cardbuf_writedata
 &FLAC_bufpos,    // cardbuf_pos
 NULL,            // cardbuf_clear
 NULL,            // cardbuf_int_monitor
 NULL,            // irq_routine

 NULL,            // card_writemixer
 NULL,            // card_readmixer
 NULL             // card_mixerchans
};

static mpxplay_module_entry_s flacenc_output_module_entry={
 MPXPLAY_DLLMODULETYPE_AUCARD,
 0,
 "FLAC",
 MPXPLAY_DLLMODULEVER_AUCARD,
 (void *)&FLAC_sndcard_info
};

static mpxplay_module_entry_s flacenc_cmdline_module_entry={
 MPXPLAY_DLLMODULETYPE_CONTROL_CMDLINE,
 0,
 NULL,
 MPXPLAY_DLLMODULEVER_CONTROL_CMDLINE,
 (void *)flacenc_opts
};

static mpxplay_dll_entry_s mpxplay_dll_entry_structure={
 MPXPLAY_DLLENTRY_STRUCTURE_VERSION,
 {
  &flacenc_output_module_entry,
  &flacenc_cmdline_module_entry,
  NULL
 }
};

#ifdef __DOS__

extern void dllstrtr_update_crwdata(unsigned long *cwd);

long __export mpxplay_dll_entrypoint(struct mpxplay_resource_s *p_mrs,unsigned long *crwdata_begin)
{
 mrs=p_mrs;
 dllstrtr_update_crwdata(crwdata_begin);
 return (long)(&mpxplay_dll_entry_structure);
}

#else

__declspec( dllexport ) mpxplay_dll_entry_s *mpxplay_dll_entrypoint(void)
{
 return (&mpxplay_dll_entry_structure);
}

#endif
