/*  findfile():
 *
 *  This routine tries to find an executable file from the argument
 *  given on the command line (which is stored in the global string
 *  'filename').
 *
 *  If the filename doesn't contain any path or drive specifiers, the 
 *  file is first searched for in the current directory, and then in 
 *  every directory in the PATH environment string.
 *
 *  If the filename doesn't have an extension, the strings ".COM",
 *  ".EXE", and ".BAT" are each in turn appended to the filename 
 *  and searched for.
 *
 *  When the first match is found, the complete file name is written to
 *  the string 'filename', the file is opened, and the function returns.
 */

int findfile(void)
{
  char  *pathString = getEnv(myEnv, "PATH"); /* points to the PATH string, if any */
  int    searchPath = pathString != 0;  /* boolean: should PATH be searched? */
  char   tempname[MAXPATH];             /* working buffer for the filename */
  int    i;
  char  *dir = strrchr(filename, '\\'); /* points to last backslash */
  char  *ext = strrchr(filename, '.');  /* points to the extension, if any */
  
  if (ext < dir) ext = 0;  /* The dot was part of a directory name */

  /* Wildcard characters are not allowed in the filename */
  if (strpbrk(filename, "?*"))
      return err_invalid_name;

  /* If path or drive specifiers are present in filename, don't search PATH */
  if (dir || filename[1] == ':')
      searchPath = 0;
   
  DosSetDTA(dta);   /* Set the DTA address */

  /* If an extension was present, check if it's COM, EXE or BAT */
  if (ext) {
      for (i = 0; i < 3; i++)
           if (!stricmp(ext, extensions[i])) {
               exeType = i;
               break;
           }
      if (i == 3)        /* if none of the three, refuse to load it */
          return err_invalid_name;
  } 

  tempname[0] = 0;      /* truncate the buffer to zero length */

  /* This loop will try the current directory the first time, 
     and then each of the PATH directories */
  do {
      size_t nlen;

      strcat(tempname, filename);
      nlen = strlen(tempname);
     
      /* this loop tries all three extensions */
      for (i = t_com; i <= t_bat; i++)
      {  
         if (!ext)
             strcpy(tempname + nlen, extensions[i]);
          
         /* Check if a file with this name exists */
         switch(rc = tryFile(tempname))
         {
            case err_dos:
            case err_open:   
               return rc;  /* these are fatal errors */       
            case err_invalid_path:
               if (!searchPath) return err_invalid_path;
               i = 3; 
               break;   /* break out from for loop - try next path */
            case OK:
               /* expand 'tempname' and copy it to 'filename' */
               expandName(filename, tempname);
               if (!ext) exeType = i;
               return OK;
             
             /* if file not found, continue searching */
         }    
          
         if (ext) break;
      }

      if (!searchPath || !pathString) break;
      
      /* copy next PATH entry to 'tempname' */
      pathString = getNextPath(pathString, tempname);
  } while (tempname[0]);

  return err_not_found;
}

/* expandName(): expand a filename
 * 
 * 'src' contains a possibly incomplete pathname, 'dest' will receive
 * the complete name, in "drive:\full_path\name.ext" form.
 */

int expandName(char *dest, char *src)
{
    char *nextPath;
    char *d = dest;

    if (src[1] == ':') {      /* does 'src' contain a drive letter? */
        *d++ = *src & ~0x20;    /* convert drive to uppercase */
        src += 2;
    }
    else *d++ = DosGetCurrDrive() + 'A';
    *d++ = ':';
    *d++ = '\\';

    if (*src == '\\') src++;  /* does 'src' start with a backslash? */
    else {
       DosGetCurrDir(*dest - '@', d); /* if not, copy the current directory to 'dest */
       if (*d) {
           d += strlen(d);
           *d++ = '\\';
       }
    }   
    
    /* copy the last part of the filename */
    strcpy(d, src);
}

/*  getEnv():
 *  
 *  Search for an environment string in the form ENTRY=VALUE,
 *  and return a pointer to VALUE, if found.
 */

char *getEnv(char *env, char *entry)
{
   size_t elen = strlen(entry);

   while (*env)
   {
     if (!strncmp(env, entry, elen))
         if (env[elen] == '=')
             return env + elen + 1;
   
     env += strlen(env) + 1;
   }

   return 0;       /* no match found */
}

/*  getNextPath():
 *
 *  This is a routine to parse the PATH environment string
 *  into individual directorys. The first directory found is
 *  copied to 'buf', and the return value is a pointer to the 
 *  next unparsed character, or 0 if the end of the string was found.
 */
 
char *getNextPath(char *str, char *buf)
{
   char *strEnd;   /* end of this directory */

   /* Skip any leading ; */
   while (*str == ';')
          str++;

   if (!(strEnd = strchr(str, ';')))
       strEnd = str + strlen(str);

   if (strEnd - str) {
       memcpy(buf, str, strEnd - str); /* copy the directory name */
       buf[strEnd - str] = '\\';       /* and append a backslash */
       buf[strEnd - str + 1] = 0;
   }
   else *buf = 0;

   return *strEnd? strEnd + 1 : 0;   /* return 0, if no more directorys */
}

/*  tryFile():
 *
 *  try to find the file 'name', using DOS 'find first' service.
 *  If it's found, try to open it.
 */

int tryFile(char *name)
{
   struct findFile *f = (void *)dta;   /* DOS writes the info to the DTA */

   /* The return value is a DOS error code, or 0 if the file was found */
   switch(DosFindFirst(name, 0))
   {
      case 0x02:    /* 'file not found' */
      case 0x12:    /* 'no more files' */
        return err_not_found;
      case 0x03:    /* 'path not found' */
        return err_invalid_path;
      case OK:
        break;
      default:
        return err_dos;   /* unknown error */
   }
   fileSize = f->size;

   /* Try to open the file, in deny-write sharing mode */
   if (!(exeHandle = DosOpen(name, 0x20)))
       return err_open;

   return OK;
}
