
/************************************************************************/
/*                                                                      */
/* FILE: library.c                                                      */
/* Library of common functions                                          */
/*                                                                      */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* This file is part of G-COM.                                          */
/* G-COM is derived from DOS-C source (GPL).                            */
/*                                                                      */
/* (C) Copyright 1999-2000  Roberto Gordo Saez   (GCOM)                 */
/* (C) Copyright 1995-1998  Pasquale J. Villani  (DOSC)                 */
/*                                                                      */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2, or (at your option)  */
/* any later version.                                                   */
/*                                                                      */
/* This program is distributed in the hope that it will be useful, but  */
/* WITHOUT ANY WARRANTY; without even the implied warranty of           */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU     */
/* General Public License for more details.                             */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to the Free Software          */
/* Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.                  */
/*                                                                      */
/************************************************************************/


/* Note that there are some differences between this functions and the  */
/* standard ones.                                                       */


#include "globals.h"


unsigned int strlen (REG char*);
unsigned int fstrlen (REG char FAR*);
void strcpy (REG char*, REG char*);
BOOL strncpy (REG char*, REG char*, unsigned int);
BOOL fstrncpy (REG char FAR*, REG char FAR*, unsigned int);
BOOL strcmp (REG char*, REG char*);
BOOL stricmp (REG char*, REG char*);
BOOL strnicmp(REG char*, REG char*, unsigned int);
BOOL fstrnicmp (REG char FAR*, REG char FAR*, unsigned int);
char toupper (char);

void printf (char*, ...);

BOOL isdigit (int);
BOOL isalpha (int);
char *itoa (int, char*, UBYTE);
char *uitoa (unsigned int, char*, UBYTE);
char *ltoa (long, char*, UBYTE);
char *ultoa (unsigned long, char*, UBYTE);
int atoi (char*);


/* -------------------------------------------------------------------- */
/* String and character manipulation functions                          */
/* -------------------------------------------------------------------- */

unsigned int strlen(REG char *s)
{
    REG unsigned int n = 0;

    while(*s++) ++n;
    return n;
}


unsigned int fstrlen(REG char FAR *s)
{
    REG unsigned int n = 0;

    while(*s++) ++n;
    return n;
}


void strcpy(REG char *d, REG char *s)
{
    while(*s)
        *d++ = *s++;
    *d = '\0';
}


BOOL strncpy(REG char *d, REG char *s, unsigned int l)
{
    unsigned int n = 1;
    while(*s && n++ < l)
        *d++ = *s++;
    *d = '\0';
    return (*s == '\0');
}


BOOL fstrncpy(REG char FAR *d, REG char FAR *s, unsigned int l)
{
    unsigned int n = 1;
    while(*s && n++ < l)
        *d++ = *s++;
    *d = '\0';
    return (*s == '\0');
}


BOOL strcmp(REG char *string1, REG char *string2)
{
    while(*string1 && *string2)
    {
        if(*string1 != *string2)
            return FALSE;
        ++string1, ++string2;
    }
    return (*string1 == *string2);
}


BOOL stricmp(REG char *string1, REG char *string2)
{
    while(*string1 && *string2)
    {
        if(toupper(*string1) != toupper(*string2))
            return FALSE;
        ++string1, ++string2;
    }
    return (*string1 == *string2);
}


BOOL strnicmp(REG char *str1, REG char *str2, unsigned int len)
{
    unsigned int c = 1;

    while(c++ < len)
    {
        if(*str1 == '\0' || *str2 == '\0')
            break;
        if(toupper(*str1) != toupper(*str2))
            return FALSE;
        ++str1;
        ++str2;
    }
    return (toupper(*str1) == toupper(*str2));
}


BOOL fstrnicmp(REG char FAR *str1, REG char FAR *str2, unsigned int len)
{
    unsigned int c = 1;

    while(c++ < len)
    {
        if(*str1 == '\0' || *str2 == '\0')
            break;
        if(toupper(*str1) != toupper(*str2))
            return FALSE;
        ++str1;
        ++str2;
    }
    return (toupper(*str1) == toupper(*str2));
}


char toupper(char c)
{
    return (c >= 'a' && c <= 'z') ? (c - 32) : c;
}


/* -------------------------------------------------------------------- */
/* Input/Output functions                                               */
/* -------------------------------------------------------------------- */

#define NONE    0
#define LEFT    1
#define RIGHT   2

void printf(char *arguments, ...)
{
    REG char *fmt = arguments;
    char **p_arg = ((char **)&arguments)+1;
    UBYTE size, slen, flag;
    long i;
    char fill;
    char s[12];

    while(*fmt != '\0')
    {
        while(*fmt && *fmt != '%')
            DosWrite(STDOUT, (char FAR *)fmt++, 1);

        if(*fmt == '\0') break;

        flag = NONE;
        size = 0;

        if(*(++fmt) == '-')
        {
            flag = LEFT;
            fmt++;
        }
        else if(*fmt == '0')
        {
            flag = RIGHT;
            fill = '0';
            fmt++;
        }

        if(*fmt == '\0') break;
        if(isdigit(*fmt))
        {
            if(flag == NONE)
            {
                flag = RIGHT;
                fill = ' ';
            }
            size = *fmt++ - '0';
            while(*fmt != '\0' && isdigit(*fmt))
            {
                size = size * 10 + (*fmt - '0');
                fmt++;
            }
        }
        if(*fmt == '\0') break;
        switch(*fmt++)
        {
        case 'c':
            DosWrite(STDOUT, (char FAR *)p_arg++, 1);
            continue;

        case 's':
            slen = strlen(*p_arg);
            DosWrite(STDOUT, (char FAR *)*p_arg, slen);

            for(i = size - slen; i > 0; i--)
                DosWrite(STDOUT, (char FAR *)" ", 1);
            ++p_arg;
            continue;

        case 'd':
            itoa(*((int *)p_arg)++, s, 10);
            break;

        case 'u':
            uitoa(*((unsigned int *)p_arg)++, s, 10);
            break;

        case 'o':
            itoa(*((int *)p_arg)++, s, 8);
            break;

        case 'x':
            itoa(*((int *)p_arg)++, s, 16);
            break;

        case 'l':
            if(*fmt == '\0') continue;
            switch(*fmt++)
            {
            case 'd':
                ltoa(*((long *)p_arg)++, s, 10);
                break;

            case 'u':
                ultoa(*((unsigned long *)p_arg)++, s, 10);
                break;

            case 'o':
                ltoa(*((long *)p_arg)++, s, 8);
                break;

            case 'x':
                ltoa(*((long *)p_arg)++, s, 16);
                break;

            default:
                continue;
            }
            break;

        default:
                continue;
        }

        slen = strlen(s);
        if(flag == RIGHT && size > slen)
        {
            for(i = size - slen; i > 0; i--)
                DosWrite(STDOUT, (char FAR *)&fill, 1);
        }
        DosWrite(STDOUT, (char FAR *)s, slen);
        if(flag == LEFT && size > slen)
        {
            for(i = size - slen; i > 0; i--)
                DosWrite(STDOUT, (char FAR *)" ", 1);
        }
    }
}


/* -------------------------------------------------------------------- */
/* Data classification and conversion functions                         */
/* -------------------------------------------------------------------- */

BOOL isdigit(int c)
{
    return (c >= '0' && c <= '9');
}


BOOL isalpha(int c)
{
    return ((c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z'));
}


char *itoa(int n, char *s, UBYTE base)
{
    return ltoa((long)n, s, base);
}


char *uitoa(unsigned int n, char *s, UBYTE base)
{
    return ultoa((unsigned long)n, s, base);
}


char *ltoa(long n, char *s, UBYTE base)
{
    if(n<0)
    {
        *s++ = '-';
        n = -n;
    }
    return ultoa(n, s, base);
}


char *ultoa(unsigned long n, char *s, UBYTE base)
{
    REG char *p, *q;
    REG char c;

    p = q = s;

    /* generate digits in reverse order */
    do *p++ = "0123456789ABCDEF"[(unsigned int)(n % base)];
    while((n /= base) > 0);

    /* terminate the string */
    *p = '\0';

    /* reverse the digits */
    while(q < --p)
    {
        c = *q;
        *q++ = *p;
        *p = c;
    }
    return s;
}


int atoi(char *s)
{
    REG char *p;
    int n, sign;

    p = skipwh(s);
    sign = (*p == '-') ? -1 : 1;

    if(*p == '+' || *p == '-')
        p++;

    for(n = 0; isdigit(*p); p++)
        n = 10 * n + (*p - '0');

    return sign * n;
}
