### Name: letterFrequency
### Title: Calculate the frequency of letters in a biological sequence, or
###   the consensus matrix of a set of sequences
### Aliases: letterFrequency alphabetFrequency
###   alphabetFrequency,XString-method alphabetFrequency,DNAString-method
###   alphabetFrequency,RNAString-method
###   alphabetFrequency,XStringSet-method
###   alphabetFrequency,DNAStringSet-method
###   alphabetFrequency,RNAStringSet-method
###   alphabetFrequency,XStringViews-method
###   alphabetFrequency,MaskedXString-method hasOnlyBaseLetters
###   hasOnlyBaseLetters,DNAString-method
###   hasOnlyBaseLetters,DNAStringSet-method
###   hasOnlyBaseLetters,RNAString-method
###   hasOnlyBaseLetters,RNAStringSet-method
###   hasOnlyBaseLetters,XStringViews-method
###   hasOnlyBaseLetters,MaskedDNAString-method
###   hasOnlyBaseLetters,MaskedRNAString-method uniqueLetters
###   uniqueLetters,XString-method uniqueLetters,XStringSet-method
###   uniqueLetters,XStringViews-method uniqueLetters,MaskedXString-method
###   consensusMatrix consensusMatrix,character-method
###   consensusMatrix,matrix-method consensusMatrix,list-method
###   consensusMatrix,XStringSet-method consensusMatrix,XStringViews-method
###   consensusString consensusString,matrix-method
###   consensusString,XStringSet-method consensusString,XStringViews-method
###   consensusString,ANY-method consmat consmat,ANY-method
### Keywords: methods manip

### ** Examples

  ## ---------------------------------------------------------------------
  ## A. BASIC alphabetFrequency() EXAMPLES
  ## ---------------------------------------------------------------------
  data(yeastSEQCHR1)
  yeast1 <- DNAString(yeastSEQCHR1)

  alphabetFrequency(yeast1)
  alphabetFrequency(yeast1, baseOnly=TRUE)
  hasOnlyBaseLetters(yeast1)
  uniqueLetters(yeast1)

  ## With input made of multiple sequences:
  library(drosophila2probe)
  probes <- DNAStringSet(drosophila2probe$sequence)
  alphabetFrequency(probes[1:50], baseOnly=TRUE)
  alphabetFrequency(probes, baseOnly=TRUE, collapse=TRUE)

  ## ---------------------------------------------------------------------
  ## B. consensus*() EXAMPLES
  ## ---------------------------------------------------------------------
  ## Read in ORF data:
  file <- system.file("extdata", "someORF.fa", package="Biostrings")
  orf <- read.DNAStringSet(file, "fasta")

  ## To illustrate, the following example assumes the ORF data
  ## to be aligned for the first 10 positions (patently false):
  orf10 <- DNAStringSet(orf, end=10)
  consensusMatrix(orf10, baseOnly=TRUE)

  ## The following example assumes the first 10 positions to be aligned
  ## after some incremental shifting to the right (patently false):
  consensusMatrix(orf10, baseOnly=TRUE, shift=0:6)
  consensusMatrix(orf10, baseOnly=TRUE, shift=0:6, width=10)

  ## For the character matrix containing the "exploded" representation
  ## of the strings, do:
  as.matrix(orf10, use.names=FALSE)

  ## consensusMatrix() can be used to just compute the alphabet frequency
  ## for each position in the input sequences:
  consensusMatrix(probes, baseOnly=TRUE)

  ## After sorting, the first 5 probes might look similar (at least on
  ## their first bases):
  consensusString(sort(probes)[1:5])

  ## ---------------------------------------------------------------------
  ## C. RELATIONSHIP BETWEEN consensusMatrix() AND coverage()
  ## ---------------------------------------------------------------------
  ## Applying colSums() on a consensus matrix gives the coverage that
  ## would be obtained by piling up (after shifting) the input sequences
  ## on top of an (imaginary) reference sequence:
  cm <- consensusMatrix(orf10, shift=0:6, width=10)
  colSums(cm)

  ## Note that this coverage can also be obtained with:
  as.integer(coverage(IRanges(rep(1, length(orf)), width(orf)), shift=0:6, width=10))



