constructors <- local({
    idTypes <- names(getSubclasses(getClass("GeneIdentifierType")))
    idTypes[idTypes != "AnnotationIdentifier"]
})

test_GeneIdentifierType_Constructors <- function() {
    ## do they exist and return the correct class?
    for (i in seq_along(constructors)) {
        res <- do.call(constructors[[i]], list())
        checkTrue(validObject(res))
        checkTrue(is(res, constructors[[i]]))
    }

    ## Required slot for AnnotationIdentifier
    res <- AnnotationIdentifier(annotation="123")
    checkTrue(validObject(res))
    checkTrue(is(res, "AnnotationIdentifier"))
    checkException(AnnotationIdentifier(), silent=TRUE) # required arg missing
    checkTrue("hgu95av2"==annotation(AnnotationIdentifier("hgu95av2")))
}

test_GeneIdentifierType_geneIdType <- function() {
    data(sample.ExpressionSet)

    gs <- GeneSet(sample.ExpressionSet[100:110],
                  setName="123", setIdentifier="456")
    suppressWarnings(geneIdType(gs) <- "EntrezIdentifier")
    checkTrue(validObject(gs, complete=TRUE))
    checkTrue(is(geneIdType(gs), "EntrezIdentifier"))

    gs <- GeneSet(sample.ExpressionSet[100:110],
                  setName="123", setIdentifier="456")
    suppressWarnings(geneIdType(gs) <- EntrezIdentifier())
    checkTrue(validObject(gs, complete=TRUE))
    checkTrue(is(geneIdType(gs), "EntrezIdentifier"))

    ## duplicate gene names exception
    gs <- GeneSet(sample.ExpressionSet[100:200],
                  setName="123", setIdentifier="456")
    opt <- options(warn=2)
    on.exit(options(opt))
    checkException(geneIdType(gs, verbose=TRUE) <- EntrezIdentifier(), silent=TRUE)
}

test_GeneIdentifierType_mapIdentifiers_toAnnotation <- function() {
    gss <- getBroadSets(system.file("extdata", "Broad.xml", package="GSEABase"))
    suppressWarnings({
        res <- mapIdentifiers(gss[[1]], AnnotationIdentifier("hgu95av2"))
    })
    detach("package:hgu95av2")
    checkTrue(validObject(res))
    checkEquals(41, length(geneIds(res)))
}

test_GeneIdentifierType_mapIdentifiers_nullAmbiguity <- function() {
    ## Original bug: 
    ##     1: Ambiguous method selection for "mapIdentifiers", target "GeneSet#AnnotationIdentifier#NullIdentifier" (the first of the signatures shown will be used)
    ##     GeneSet#AnnotationIdentifier#GeneIdentifierType
    ##     GeneSet#GeneIdentifierType#NullIdentifier
    opts <- options(warn=2)
    on.exit(options(opts))
    gs <- GeneSet(setName="123", setIdentifier="345")
    geneIdType(gs) <- AnnotationIdentifier("xyz")
    checkTrue(validObject(gs))
}

test_GeneIdentifierType_mapIdentifiers_toAnnotation_via_Dbi <- function()  {
    gss <- getBroadSets(system.file("extdata", "Broad.xml", package="GSEABase"))
    suppressMessages(suppressWarnings({
        res <- mapIdentifiers(gss[[1]], AnnotationIdentifier("hgu95av2.db"))
    }))
    detach("package:hgu95av2.db")
    checkTrue(validObject(res))
    checkEquals(41, length(geneIds(res)))
}

test_GeneIdentifierType_mapIdentifiers_verbose_warnings <- function() {
    ## duplicate gene names exception
    gs <- GeneSet(sample.ExpressionSet[100:200],
                  setName="123", setIdentifier="456")
    opt <- options(warn=2)
    on.exit(options(opt))
    checkException(mapIdentifiers(gs,  EntrezIdentifier(), verbose=TRUE), silent=TRUE)
}
