\name{DoritProliferationAssay}

\alias{readWellXML}
\alias{collectdat}
\alias{splitWellName}
\title{Specialized functions for Dorit Arlt's (DKFZ) proliferation 
assay project}
\description{Specialized functions for Dorit Arlt's (DKFZ) proliferation 
assay project}

\usage{
splitWellName(x)
readWellXML(filename, path=".", verbose=TRUE)
collectdat(thedir, verbose=TRUE)
}

\arguments{
  \item{x}{Character of length 1}
  \item{filename}{Character vector of length 1: filename}
  \item{path}{Character vector of length 1}
  \item{verbose}{Logical vector of length 1}
  \item{thedir}{Character.}
}

\details{These functions are rather ideosyncratic for the particular format in
which the EMBL's high contents screening microscope (Urban Liebel, Rainer
Pepperkok) store its data, and how that data was transfered to the DKFZ for
Dorit Arlt's proliferation assay. These functions are probably not of much
interest beyond that.

\code{readWellXML} reads an XML file with the data for one well
from Urban Liebel's (EMBL) HCS-microscope.

\code{splitWellName} splits a file name string into its components.

\code{collectdat} collects together data from the individual wells 
into one big data frame.
}

\value{Various.}

\author{Wolfgang Huber \url{http://www.dkfz.de/abt0840/whuber}}
\seealso{}
\examples{
  splitWellName("dorit10_3c_1-2--01-pDhfbr2_78k24_1E3Y1")
  sampdat <- readWellXML(file="dorit13_2c-1-2--13-Ytrail=3143937171.xml",
                   path=system.file("extdata", package="prada"))
  sampdat
}
\keyword{IO}


\eof
\name{analysePlate}
\alias{analysePlate}
\title{Apply a statistic to the data from each well in a plate}
\description{Apply a statistic to the data from each well in a plate}
\usage{
analysePlate(x, wellcol="well", wellrange, statfun, platename, plotdir=".", ...)
}
\arguments{
  \item{x}{data frame. It must contain a column whose name is the value
    of \code{wellcol}, and further columns that are needed by the function
    named by \code{stat}.}
  \item{wellcol}{character of length 1. Name of a column in \code{x} that
    contains the well ID.}
  \item{wellrange}{vector of the same type as \code{x[, wellcol]}. All
    values \code{x[, wellcol]} must be contained in \code{wellrange}.}
  \item{statfun}{character of length 1. Name of a function that can calculate
    a statistic from selected rows of \code{x}.}
  \item{platename}{character of length 1. The name or ID of this plate, which
  will be used for graphics output filenames and as the value of the column
  \code{platename} of the return value.}
  \item{plotdir}{character of length 1. The name of directory where
    diagnostic plots will be saved.}
  \item{...}{further arguments that are passed on to \code{statfun}.}
}

\details{The semantics of this function are similar to
\code{\link[base:tapply]{tapply}}, but some additional checking and 
reporting is performed, and the return value is a data frame.}

\value{A data frame with number of rows equal to
  \code{length(wellrange)}. Rows (wells) for which there is no data
  contains \code{NA}s. The columns comprise \code{platename},
  \code{well-ID} (from \code{x[, wellcol]}), and the
  results from statfun.}

\seealso{\code{\link{statWellLocfit}}}

\author{Wolfgang Huber \url{http://www.dkfz.de/abt0840/whuber}}
\keyword{manip}
\examples{##see vignette}

\eof
\name{as.all}
\alias{as.all}
\title{Coercion without introduction of NAs}
\description{Coercion without introduction of NAs}
\usage{
as.all(x, what)
}
\arguments{
  \item{x}{an object.}
  \item{what}{character of length 1.} 
}

\details{}

\value{A vector of type \code{what}: 
the result of \code{do.call(paste("as.", what, sep=""), list(x))}.}

\seealso{\code{\link[methods:as]{as}}}

\author{Wolfgang Huber \url{http://www.dkfz.de/abt0840/whuber}}
\keyword{manip}
\examples{
  as.all(runif(5)*10, "integer")
}

\eof
\name{barploterrbar}
\alias{barploterrbar}
\title{Barplot with error bars.}
\description{Barplot with error bars.}
\usage{
barploterrbar(y, yl, yh, barcol, errcol, horiz=FALSE, w=0.2, ...)
}
\arguments{
  \item{y}{Numeric vector.}
  \item{yl}{Numeric vector of same length as y.}
  \item{yh}{Numeric vector of same length as y.}
  \item{barcol}{Color of the bars.}
  \item{errcol}{Color of the error bars.}
  \item{horiz}{Logical. As in \code{\link[graphics:barplot]{barplot}}.}
  \item{w}{Size of the error bar ticks.}
  \item{...}{Further arguments that get passed on to
     \code{\link[graphics:barplot]{barplot}}.}
}

\details{The function calls \code{\link[graphics:barplot]{barplot}} with
\code{y} and decorates it with error bars according to \code{yl} and
\code{yh}.}

\value{The function is called for its side effect, producing a plot.} 

\seealso{\code{\link[graphics:barplot]{barplot}}}

\author{Wolfgang Huber \url{http://www.dkfz.de/abt0840/whuber}}
\keyword{hplot}
\examples{
  y <- matrix(runif(80), ncol=5)
  ym <- apply(y, 2, mean)
  dy <- apply(y, 2, sd)*2/sqrt(nrow(y))
  barploterrbar(ym, ym-dy, ym+dy, barcol="#0000c0", errcol="orange")
}


\eof
\name{colorramp}
\alias{colorramp}
\title{Interpolate between colors}
\description{Interpolate between colors}
\usage{colorramp(col)}
\arguments{
  \item{col}{A vector of colors, as in \code{\link[graphics:col2rgb]{col2rgb}}.}
}

\details{The function is adapated from the function
  \code{\link[geneplotter:colorRamp]{colorRamp}} in the package
  \code{geneplotter}.}

\value{A function that takes an integer and produces
  a color map which interpolates between the colors in \code{col}.}

\author{Thomas Lumley}

\examples{
  cr <- colorramp(c("green", "red"))
  cr(5)
}
\keyword{arith}


\eof
\name{estimateCrosstalk}
\alias{estimateCrosstalkWell}
\alias{estimateCrosstalkPlate}
\title{Estimate crosstalk between two color channels.}
\description{Estimate crosstalk between two color channels.}
\usage{
estimateCrosstalkPlate(x, plotfileprefix=NULL)
estimateCrosstalkWell(x, doPlot=TRUE)
}
\arguments{
  \item{x}{data frame.}
  \item{plotfileprefix}{character of length 1.}
  \item{doPlot}{logical of length 1.}
}

\details{This function is quite ideosyncratic - it was written for one
particular data sets and ought not be useful for any other purpose.}

\value{A numeric vector.}

\seealso{}
\author{Wolfgang Huber \url{http://www.dkfz.de/abt0840/whuber}}
\keyword{manip}
\examples{##see vignette}

\eof
\name{getDye}
\alias{getDye}
\title{Extract the Dye name (CFP or YFP) from the clone ID by pattern matching}
\description{Extract the Dye name (CFP or YFP) from the clone ID by pattern matching}
\usage{getDye(x)}
\arguments{\item{x}{Character vector with clone ID(s)}}

\details{The following patterns are recognised:
  \enumerate{
    \item ...[CY], ...[CY] Numbers, ...[CY] Numbers Letters, 
    \item [CY]...
    \item letter letter [CY]...
  }
See also the source code.}

\value{Character vector of the same length as \code{x}.}
\author{Wolfgang Huber \url{http://www.dkfz.de/abt0840/whuber}}
\examples{getDye("dorit10_3c_1-2--01-pDhfbr2_78k24_1E3Y1")}
\keyword{manip}


\eof
\name{getElement}
\alias{getElement}
\alias{getVector}
\title{Functions to extract atomic data types (numeric, character)
  and vectors from an XML tree}
\description{Functions to extract atomic data types (numeric, character)
  and vectors from an XML tree that
  was generated by Urban Liebel's Labview image analysis  program}
\usage{
  getElement(node, name, type="numeric")
  getVector(node, name, elementnames, elementtypes)
}
\arguments{
  \item{node}{XMLNode. A node of an XML tree}
  \item{name}{character. The expected name of the node. Maybe NULL for
    \code{getVector}, is mandatory for \code{getElement}. If
    \code{xmlName(node)} is not the same as \code{name}, an error
    message is generated.}
  \item{type}{Character. Should be either \code{numeric} or
    \code{character}.}
  \item{elementnames}{list. The list may be nested, thus describing a
    tree. This tree must correspond to the XML subtree starting at
    \code{node}. The leaves of \code{elementnames} should be of class
    \code{character} and contain the names of the leave nodes of the XML
    subtree.}
  \item{elementtypes}{list. Same form as \code{elementnames}. The leaves
    of \code{elementtypes} should be of class \code{character} and
    contain the types of the leave nodes of the XML subtree.}
}

\details{\code{getVector} descends into the XML subtree starting at
  \code{node} recursively. The atomic elements of the nested lists
  \code{elementnames} and \code{elementtypes} are used as the arguments
  \code{name} and \code{type} of the call to
  \code{getElement} for the corresponding leave nodes.}
}

\value{
  A flat list (the result of \code{unlist}) with the values from the XML
  subtree and names \code{unlist(elementnames)}.
}

\author{Wolfgang Huber \url{http://www.dkfz.de/abt0840/whuber}}
\seealso{}
\examples{ 
  exfile = file.path(.path.package("prada"), "extdata", "dorit13_2c-1-2--13-Ytrail=3143937171.xml")
  z      = xmlTreeParse(exfile)
  r      = xmlRoot(z)[[2]]
  getElement(r[[3]], name="current clone", type="character")

  thenames = list("Area (pixels)", "Perimeter", "Projection x", "Projection y",
      "Longest Segment Length",
      list("x1Left", "y1Top", "x2Right", "y2Bottom"),
      "Number of Holes", "Hole's Area (pixels)", "Dapi mean",
      "Brdu mean", "transfection mean")
  thetypes = c(rep("numeric", 5), list(rep("numeric", 4)), rep("numeric", 5))

  getVector(r[[4]][[6]], name=NULL, elementnames=thenames, elementtypes=thetypes)
  getElement(r[[4]][[6]][[3]], "Area (pixels)")
}

\keyword{programming}
\keyword{error}

\eof
\name{getPradaPar}
\alias{setPradaPars}
\alias{getPradaPar}
\title{Set and query global parameters for functions in the prada package}
\description{Set and query global parameters for functions in the prada package}
\usage{
setPradaPars(pars)
getPradaPar(parname)
}
\arguments{
\item{pars}{Named list}
\item{parname}{Character string of length 1}
}
\details{TBA}

\value{For \code{getPradaPar}, the value of the list element with name
\code{parname} of the global parameters list.
The function \code{setPradaPars} is invoked for its side effect,
which is assigning a value to the global parameters list.
It returns the value invisible(NULL).
}
\author{Wolfgang Huber \url{http://www.dkfz.de/abt0840/whuber}}
\examples{
 setPradaPars(list(tomato=1, apple="two", pear=as.integer(3)))
 getPradaPar("pear")
}
\keyword{manip}


\eof
\name{histStack}
\alias{histStack}
\title{Stacked histogram}
\description{Stacked histogram}
\usage{histStack(x, breaks, midsFun=paste, ...) }
\arguments{
  \item{x}{A list of numeric vectors.}
  \item{breaks}{Histogram breaks, as in
    \code{\link[graphics:hist]{hist}}}
  \item{midsFun}{Function, can be used to control the formatting and
    number of the bar labels on the x-axis. See example.}
  \item{...}{Further arguments that get passed to
    \code{\link[graphics:barplot]{barplot}}}
}

\details{The function calls \code{\link[graphics:hist]{hist}}
  for each element of \code{x} and plots the frequencies
  as a stacked barplot using
  \code{\link[graphics:barplot]{barplot}} with \code{beside=FALSE}.}

\value{The function is called for its side effect, producing a barplot
  on the active graphics device. It returns the result of the call to
  \code{\link[graphics:barplot]{barplot}}.}

\author{Wolfgang Huber \url{http://www.dkfz.de/abt0840/whuber}}

\examples{
  x    <- list(rnorm(42), rnorm(42)+2)
  br   <- seq(-3, 5, length=13)
  cols <- c("#1D267B", "#ceffc0")
  histStack(x, breaks=br, col=cols)

  histStack(x, breaks=br, col=cols,
            midsFun=function(z) {
              z <- paste(round(z, 3))
              z[seq(2, length(z), by=2)] <- NA
              return(z)
            } )
}
\keyword{hplot}


\eof
\name{imageMap}
\alias{imageMap}
\title{Write an HTML IMG tag together with a MAP image map.}
\description{Write an HTML IMG tag together with a MAP image map.}
\usage{
  imageMap(con, imgname, coord, tooltips, url, target="extra")
}
\arguments{
  \item{con}{Connection (see argument \code{con} of \code{\link{writeLines}}).}
  \item{imgname}{Character. Name of the image file, as it is to appear in the
  HTML output.}
  \item{coord}{Matrix with 4 columns. Each row specifies the corners of a 
  rectangle within the image.}
  \item{tooltips}{Character of length \code{nrow(coord)}.}
  \item{url}{Character of length \code{nrow(coord)}.}
  \item{target}{Character. Name of the target browser window.}
}

\details{See example.}

\value{The function is called for its side effect, which is writing text into
the connection \code{con}.}

\seealso{\code{\link[prada::plotPlate]{plotPlate}}, 
   \code{\link[base::writeLines]{writeLines}}}

\author{Wolfgang Huber \url{http://www.dkfz.de/abt0840/whuber}}
\keyword{manip}
\examples{
imageMap(stdout(), "myimage.jpg", coord=matrix(1:8,nrow=2), 
   url=c("a","b"), tooltips=c("TT1", "TT2"))
}



\eof
\name{mapSub2Plate}
\alias{mapSub2Plate}
\title{Map subplate well ID and subplate ID to 96-well plate well ID}
\description{Map subplate well ID (e.g. from chamberslides) and subplate ID to 96-plate well ID}
\usage{mapSub2Plate(sub, well)}
\arguments{
  \item{sub}{Character. Subplate identifier, see details.}
  \item{well}{Integer. Well identifier, see details.}
}
\details{\code{sub} and \code{well} must have the same length.

  For chamberslides, \code{sub} has the values
  "a", "b", "c12", or "c34". Alternatively, \code{sub} can have the values
  "1-48", "49-96".

  \code{well} must be an integer number in the appropriate range.
}

\value{An numeric vector of the same length as \code{well} with the
  plate well IDs.
}

\author{Wolfgang Huber \url{http://www.dkfz.de/abt0840/whuber}}
\seealso{}
\examples{
  plcat = function(iw, x) {
    d = numeric(96)
    d[iw] = x
    for (i in 0:7)
      cat(sapply(d[(1:12)+i*12], function(s) { sprintf('\%4s', s) }), '\n')
    cat('\n')
  }

  sub  = c(rep("a", 32), rep("b", 32), rep("c12", 16), rep("c34", 16))
  well = rep(1:32, 3)
  iw   = mapSub2Plate(sub, well)
  plcat(iw, paste(well))
  plcat(iw, sub)

  sub  = c(rep("1-48",  48), rep("49-96", 48))
  well = 1:96
  iw   = mapSub2Plate(sub, well)
  plcat(iw, paste(well))
  plcat(iw, sub)
}

\keyword{manip}


\eof
\name{openHtmlPage}
\alias{openHtmlPage}
\alias{closeHtmlPage}
\title{Open and close an HTML file for writing.}
\description{Open and close an HTML file for writing..}
\usage{
  openHtmlPage(name, title="")
  closeHtmlPage(con)
}
\arguments{
  \item{name}{Character. File name (\emph{without} the extension '.html').}
  \item{title}{Character. Argument of the \code{title} tag in the HTML header.}
  \item{con}{Connection.}
}

\details{See example.}

\value{For \code{openHtmlPage}, a \code{\link[base::connection]{connection}}. }

\seealso{}

\author{Wolfgang Huber \url{http://www.dkfz.de/abt0840/whuber}}
\keyword{IO}
\examples{
  fn <- tempfile()
  con <- openHtmlPage(fn, "My page")
  writeLines("Hello world", con)
  closeHtmlPage(con)
  readLines(paste(fn, ".html", sep=""))
}



\eof
\name{plotPlate}
\alias{plotPlate}
\title{Plot a well statistic in false color representation.}
\description{Plot a well statistic in false color representation. The
  plot is supposed to resemble the physical geometry of a microtitre plate.}
\usage{
plotPlate(x, nrow=8, ncol=12, main, xrange, col, 
  device, file, width, na.action="zero")
}
\arguments{
  \item{x}{Numeric vector of length \code{ncol*nrow}}
  \item{nrow}{Numeric of length 1. The number of rows of the plate.}
  \item{ncol}{Numeric of length 1. The number of columns of the plate.}
  \item{main}{Character of length 1. Plot title. This arguments gets passed 
     on to \code{\link[graphics:text]{text}} as the argument \code{labels}.}
  \item{xrange}{Range of \code{x} that is mapped into the color scale.}
  \item{col}{Character vector. Usually the names of two or three colors
    between which the color map is interpolated, using the function
    \code{\link{colorramp}}.}
  \item{device}{Character. Name of a valid graphics device,
    e.g. \code{x11}, \code{png}, \code{pdf}.}
  \item{file}{The \code{file} argument of the call to
    \code{device}, i.e. the name of the graphics output file.}
  \item{width}{The \code{width} argument of the call to 
    \code{device}, i.e. the width of the graphic.}
  \item{na.action}{Character. One of \code{"zero"} or \code{"omit"}.
    How should the wells for which \code{x} is NA be treated?
    For \code{"zero"}, they are plotted as if the value were 0. 
    For \code{"omit"}, they are omitted. }
}

\details{}

\value{The function produces a plot in the graphics device specified by the
\code{device} argument.  

It returns a list with two elements. The element \code{which} is a vector with
the indices of those elements in \code{x} that were plotted (see argument
\code{na.action}). The element \code{coord} is a \code{length(which)} by 4
matrix in which each row specifies the corners of a rectangle that contains a
well. It is intended to be use as an argument to a subsequent call to
\code{\link{imageMap}}.}

\seealso{\code{\link[prada::imageMap]{imageMap}}}

\author{Wolfgang Huber \url{http://www.dkfz.de/abt0840/whuber}}
\keyword{hplot}
\examples{
  plotPlate(runif(96), main="example 1", col=c("#0000e0", "#e00000"), width=7)
  plotPlate(runif(384), nrow=16, ncol=24,  
     main="example 2", col=c("#0000e0", "white", "#e00000"), width=7)
}


\eof
\name{readFCS}
\alias{readFCS}
\title{Read an FCS 3.0 file}
\description{Read an FCS 3.0 file: Data File Standard for Flow
  Cytometry, Version FCS3.0}
\usage{
  readFCS(filename)
}
\arguments{
  \item{filename}{Character of length 1: filename}
}

\details{For specifications of FACS 3.0 see
\url{http://www.isac-net.org} and the file
\url{../doc/fcs3.html} in the \code{doc} directory of the package.

The FCS 3.0 has many options that are not yet implemented in this
function. It works for the data produced by the FACS machine Calibur
from BD Bioscience together with the software Cellquest Pro at the
Division of Molecular Genome Analysis at the German Cancer Research
Center. If you need extensions, please let me know.
}

\value{A matrix with column names (one row per cell, one column per
measured parameter), and attribute \code{text}, a named
character vector with keyword-value pairs.}

\author{Wolfgang Huber \url{http://www.dkfz.de/abt0840/whuber}}
\seealso{\code{\link{readFCSdata}}}

\examples{
sampdat <- readFCS(system.file("extdata", "fas Bcl2 plate323-04-04.A01", package="prada"))
sampdat[1:5,]
dim(sampdat)
attr(sampdat, "text")[3:6]
}
\keyword{IO}


\eof
\name{readFCSaux}
\alias{readFCSdata}
\alias{readFCStext}
\alias{readFCSheader}
\alias{readFCSgetPar}
\title{Auxiliary functions for readFCS}
\description{Auxiliary functions for readFCS - not normally called by
  the user}
\usage{
  readFCSgetPar(x, pnam)
  readFCSheader(con)
  readFCStext(con, offsets, delimiter="\\")
  readFCSdata(con, offsets, x, endian="big")
}
\arguments{
  \item{x}{Named character vector.}
  \item{pnam}{Character vector, its elements must be contained in
    \code{names(x)}.}
  \item{con}{Connection.}
  \item{offsets}{Integer vector of length 6 with byte offsets of the
    header, text, and data blocks.}
  \item{delimiter}{Character.}
  \item{endian}{Character. Specifies the byte order, see \code{\link[base:readBin]{readBin}}.}
}

\details{These functions are not normally called by the user.}

\value{Various.}

\author{Wolfgang Huber \url{http://www.dkfz.de/abt0840/whuber}}
\seealso{\code{\link{readFCS}}}
\examples{}
\keyword{IO}


\eof
\name{savepng}
\alias{savepdf}
\alias{saveeps}
\alias{savepng}
\alias{savetiff}

\title{Save the contents of the current graphics device to a file}
\description{Save the contents of the current graphics device to file}

\usage{
  savepdf(fn, dir, width=6, asp=1)
  saveeps(fn, dir, width=6, asp=1) 
  savepng(fn, dir, width=480, asp=1)
  savetiff(fn, dir, density=360, keeppdf=TRUE, ...)
}

\arguments{
  \item{fn}{character: name of the output file (without extension). 
    An extension \code{.pdf}, \code{.eps}, \code{.png}, or \code{.tiff}
    will be added automatically.}
  \item{dir}{character: directory to which the file should be written.}
  \item{width}{numeric: width of the image in pixels (png) or inches
    (pdf, eps).}
  \item{asp}{numeric: aspect ratio; height=width*asp.}
  \item{density}{pixels per inch (see Details).}
  \item{keeppdf}{Should the intermediate PDF file (see Details)
    be kept? If \code{FALSE}, it is deleted before the function
    returns.}
  \item{...}{Further arguments that are passed on to \code{savepdf}
    (see Details).}
}

\details{The functions are called for their side effect, writing a
  graphics file.

  \code{savepdf}, \code{savepng}, and \code{saveeps} use the
  devices \code{\link{pdf}}, \code{\link{png}}, and
  \code{\link{postscript}}, respectively.

  There is currently no TIFF device for R, so \code{savetiff}
  works differently. It relies on the external tool \code{convert} from
  the ImageMagick software package. First, \code{savetiff} produces
  a PDF files with \code{savepdf}, then uses \code{\link{system}} to
  invoke \code{convert} with the parameter \code{density}.
  \code{savetiff} does \bold{not} check for the existence of 
  \code{convert} or the success of the system call, and returns silently
  no matter what.
}

\value{Character: name of the file that was written.}

\author{Wolfgang Huber \url{http://www.dkfz.de/abt0840/whuber}}

\seealso{\code{\link{dev.copy}}, \code{\link{pdf}}, \code{\link{png}},
  \code{\link{postscript}}}

\examples{
  x = seq(0, 20*pi, len=1000)
  plot(x*sin(x), x*cos(x), type="l")

  try({   ## on some machines, some of the devices may not be available
    c(
      savepdf("spiral", dir=tempdir()),
      savepng("spiral", dir=tempdir()),
      saveeps("spiral", dir=tempdir()),
      savetiff("spiral", dir=tempdir())
     )
  })    
}

\keyword{programming}
\keyword{error}

\eof
\name{shorth}
\alias{shorth}
\title{Midpoint of the shorth}
\description{Calculate the midpoint of the shorth}
\usage{shorth(x, na.rm=FALSE)}
\arguments{
  \item{x}{Numeric}
  \item{na.rm}{Logical}
}

\details{The shorth is the shortest interval that covers half of the
  values in \code{x}. Its midpoint is the median of the data values
  that lie in the shorth and can be used as an
  estimator of the mode of the distribution that generated \code{x}.}

\value{The midpoint of the short of \code{x}.}
\author{Wolfgang Huber \url{http://www.dkfz.de/abt0840/whuber}}

\examples{
  x <- c(rnorm(100), runif(100) * 10)
  mean(x)
  median(x)
  shorth(x)
}
\keyword{arith}


\eof
\name{sqr}
\alias{sqr}
\title{Square}
\description{Square}
\usage{sqr(x)}
\arguments{\item{x}{Numeric}}
\value{x*x.}
\author{Wolfgang Huber \url{http://www.dkfz.de/abt0840/whuber}}
\examples{sqr(2)}
\keyword{arith}


\eof
\name{statWell}
\alias{statWellLocfit}
\title{Calculate a statistic for the data from one well}
\description{Calculate a statistic for the data from one well}
\usage{
statWellLocfit(x, span,
  plotwhat="nothing", plotdir=".", plotfile, ...)
}
\arguments{
  \item{x}{data.frame with (at least) columns \code{brdu}, \code{trsf},
    \code{dapi}, \code{Field}, and \code{cloneId}.}
  \item{span}{numeric. Is used as the parameter of \code{alpha}
    in the call to the function \code{\link[locfit:locfit]{locfit}}.}
  \item{plotwhat}{character of length 1. Allowed values are
    \code{nothing}, \code{screen}, \code{figscp}, \code{mkpp}.
    The parameter is used to control the number and format of the
    diagnostic plots that are produced as a side effect of
    this function.}
  \item{plotdir}{character of length 1. The name of directory where
    diagnostic plots will be saved (only relevant if \code{plotwhat} is
    \code{figscp} or \code{mkpp}).}
  \item{plotfile}{character. The file name root for the file with
    the diagnostic plot (only relevant if \code{plotwhat} is
    \code{figscp} or \code{mkpp}). If missing, a default name is
    generated.}
  \item{...}{Further arguments that are passed on to \code{\link{plot}}.}
}

\details{The function fits a local regression model to the data. For
  details, please see the paper.}

\value{A list with elements \code{cloneId}, \code{nrcells}, \code{trsfeff},
  \code{delta}, \code{se.delta}, \code{zscore}, \code{plotfile}.}

\author{Wolfgang Huber \url{http://www.dkfz.de/abt0840/whuber}}
\keyword{manip}
\examples{##see vignette}

\eof
