\name{[-methods}
\docType{methods}

\alias{[-methods}
\alias{[}

\title{Subsetting methods for microarray objects}


\description{Subsetting methods were defined for the microarray classes,
  \code{\link{marrayInfo}},
  \code{\link{marrayLayout}},\code{\link{marrayRaw}} and
  \code{\link{marrayNorm}}. These methods create instances of the given
class, for a subset of spots and/or arrays in a batch.}

\section{Methods}{\describe{
\item{x = ANY}{generic method.}
\item{x = marrayInfo}{\code{x[i, j]} extract object of class \code{"\link{marrayInfo}"} for spots or arrays with indices i and labels with indices j.}
\item{x = marrayLayout}{\code{x[i]} extract object of class \code{"\link{marrayLayout}"} for spots with indices i.}
\item{x = marrayRaw}{\code{x[i, j]} extract object of class \code{"\link{marrayRaw}"} for spots with indices i and arrays with indices j.}
\item{x = marrayNorm}{\code{x[i, j]} extract object of class \code{"\link{marrayNorm}"} for spots with indices i and arrays with indices j.}
}}

\keyword{methods}


\eof
\name{bayesFun}
\alias{bayesFun}

\title{A function that calculates an odds ratio of each gene in a
  multi-slide microarray experiment}

\description{
  This is a wrapper function around stat.bay.est which takes
  independent, sufficient estimates of the  effect and its variance for
  each gene in a multi-slide microarray experiment and returns an odds
  ratio for each gene: log( Pr(the gene is differentially expressed) /
  Pr(the gene is not differentially expressed) ). 
}

\usage{
bayesFun(...)
}

\arguments{
\item{...}{arguments for the fuction \code{lm.series} and \code{ebayes}}
}

\details{
}

\value{
  Returns a function.
}

\references{I. Lnnstedt and T. P. Speed.  Replicated Microarray Data.
  Statistical Sinica, Accepted, see
  http://www.stat.berkeley.edu/users/terry/zarray/Html/papersindex.html} 

\author{Yee Hwa (Jean) Yang}

\seealso{\code{lm.series},\code{ebayes}}

\examples{
  bayesFun()
}

\keyword{models}

\eof
\name{boxplot}
\docType{methods}
\alias{boxplot,marrayRaw-method}
\alias{boxplot,marrayNorm-method}



\title{Boxplots for cDNA microarray spot statistics}

\description{
  The function \code{boxplot} produces boxplots of microarray spot
  statistics for the classes  \code{"\link{marrayRaw}"},
  \code{"\link{marrayNorm}"}.
  We encourage users to use \code{boxplot} rather than \code{maBoxplot}.
  The name of the arguments have changed slightly.
}

\usage{
\S4method{boxplot}{marrayRaw}(x, xvar="maPrintTip", yvar="maM", ...)
\S4method{boxplot}{marrayNorm}(x, xvar="maPrintTip", yvar="maM", ...)
}
%- maybe also `usage' for other objects documented here.

\arguments{
  \item{x}{Microarray object of class \code{"\link{marrayRaw}"}, \code{"\link{marrayNorm}"}}
  \item{xvar}{Name of accessor method for the spot statistic used to
    stratify the data, typically a slot name for the microarray layout
    object (see \code{"\link{marrayLayout}"}) such as \code{maPlate} or
    a method such as \code{maPrintTip}. If \code{x} is NULL, the data
    are not stratified.} 
  \item{yvar}{Name of accessor method for the spot statistic of
    interest, typically a slot name for the microarray object \code{m},
    such as \code{maM}.} 
  \item{\dots}{Optional graphical parameters, see \code{\link{par}}.}
}

\details{
If there are more than one array in the batch, the function produces a
boxplot for each array in the batch. Such plots are useful when
assessing the need for between array normalization, for example, to deal
with scale differences among different arrays. Default graphical
parameters are chosen for convenience using the function
\code{\link{maDefaultPar}} (e.g. color palette,  axis labels,  plot
title) but the user has the option to overwrite these parameters at any
point. 
}

\references{
S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for
exploratory analysis and normalization of cDNA microarray data. In
G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors,
\emph{The Analysis of Gene Expression Data: Methods and Software},
Springer, New York. 
}

\author{Jean Yang and Sandrine Dudoit}

\seealso{\code{\link{maBoxplot}}, \code{\link{maDefaultPar}}.}

\examples{

# To see the demo type demo(marrayPlots)

# Examples use swirl dataset, for description type ? swirl
data(swirl)

# Boxplots of pre-normalization log-ratios M for each of the 16 
# print-tip-groups for the Swirl 93 array. 
# - Default arguments
boxplot(swirl[,3])  

# All spots 
boxplot(swirl[,3], xvar=NULL, col="green")  

# Boxplots of pre-normalization red foreground intensities for each grid row
# for the Swirl 81 array. 
boxplot(swirl[,1], xvar="maGridRow", yvar = "maRf", main = "Swirl array 81: pre-normalization red foreground intensity") 

# Boxplots of pre-normalization log-ratios for each array in swirl
boxplot(swirl, main="Swirl arrays: pre-normalization log-ratios")
}
\keyword{hplot}% at least one, from doc/KEYWORDS


\eof
\name{cbind}
\alias{cbind.marrayRaw}
\alias{cbind.marrayNorm}
\alias{rbind.marrayInfo}

\title{Combine marrayRaw,  marrayNorm or marrayInfo Objects}
\description{
  Combine a series of \code{marrayRaw}, \code{marrayNorm} and
  \code{marrayInfo} objects.
}

\usage{
\method{cbind}{marrayRaw}(\dots, deparse.level=1)
\method{cbind}{marrayNorm}(\dots, deparse.level=1)
\method{rbind}{marrayInfo}(\dots, deparse.level=1)
}
\arguments{
  \item{\dots}{\code{marrayRaw} objects or \code{marrayNorm} objects}
  \item{deparse.level}{not currently used, see \code{\link[base]{cbind}} in the base package}
}
\details{
\code{cbind} combines data objects assuming the same gene lists but
different arrays. 
\code{rbind} combines data objects assuming equivalent arrays, i.e., the
same RNA targets, but different genes. 

For \code{cbind}, the matrices o f expression data from the individual
objects are cbinded. 
The data.frames of target information, if they exist, are rbinded. 
The combined data object will preserve any additional components or
attributes found in the first object to be combined. 
For \code{rbind}, the matrices of expression data are rbinded while the
target information, in any, is unchanged. 
}
\value{
}
\author{Jean Yang}
\seealso{
  \code{\link[base]{cbind}} in the base package.
}
\examples{
}
\keyword{manip}

\eof
\name{coerce-methods}
\docType{methods}
\alias{coerce-methods}
\alias{coerce}
\alias{as}

\title{Coerce an object to belong to a given microarray class}

\description{
Coercing methods were defined to convert microarray objects of one class
into objects of another class, e.g., instances of the
\code{"\link{marrayRaw}"} class into instances of the
\code{"\link{marrayNorm}"} class. 
}

\section{Methods}{\describe{
\item{from = marrayRaw, to = marrayNorm}{convert an object of class
  \code{"\link{marrayRaw}"} into an object of class
  \code{"\link{marrayNorm}"}.} 
}}

\note{Use Package convert to convert object to other data types such as
  \code{exprSet} and \code{MAList}.
}

\keyword{methods}



\eof
\name{dim}
\alias{dim.marrayRaw}
\alias{dim.marrayNorm}
\alias{dim.marrayInfo}

\title{Retrieve the Dimensions of an marrayRaw, marrayNorm or marrayInfo Object}
\description{
  Retrieve the number of rows (genes) and columns (arrays) for an
  marrayRaw, marrayNorm or marrayInfo object.
}
\usage{
\method{dim}{marrayRaw}(x)
}
\arguments{
  \item{x}{an object of class \code{marrayRaw}, \code{marrayNorm} or \code{marrayInfo}}
}
\details{
Microarray data objects share many analogies with ordinary matrices in
which the rows correspond to spots or genes and the columns to arrays. 
These methods allow one to extract the size of microarray data objects
in the same way that one would do for ordinary matrices. 

A consequence is that row and column commands \code{nrow(x)}, \code{ncol(x)} and so on also work.
}
\value{
Numeric vector of length 2.
The first element is the number of rows (genes) and the second is the number of columns (arrays).
}
\author{modified from Gordon Smyth's function}

\seealso{
  \code{\link[base]{dim}} in the base package.
}
\examples{
M <- A <- matrix(11:14,4,2)
rownames(M) <- rownames(A) <- c("a","b","c","d")
colnames(M) <- colnames(A) <- c("A1","A2")
MA <- new("marrayNorm", maM=M,maA=A)
dim(MA)
dim(M)
}
\keyword{array}

\eof
\name{findID}
\alias{findID}

\title{Find ID when given an accession number}
\description{
  Search gene ID with a vector of accession number from gene names or ID
  values.
}

\usage{
findID(text, Gnames = gnames, ID = "Name")
}

\arguments{
  \item{text}{A character strings of gene names or id names}
  \item{Gnames}{An objects of \code{marrayRaw}, \code{marrayNorm},
    \code{exprSet} or data.frame of gene names information.}
  \item{ID}{The column of ID corresponding to 'text'}
}

\value{
  A numeric vector the gene ID.
}
\author{Yee Hwa (Jean) Yang}

\seealso{\code{\link{grep}}}

\examples{
data(swirl)
findID("fb24a09", swirl, ID="ID")
findID("geno1", swirl)
}

\keyword{manip}% at least one, from doc/KEYWORDS

\eof
\name{htmlPage}
\alias{htmlPage}
\alias{table2html}

\title{Display gene list as a HTML page}
\description{
Given a set of index to a data.frame containing gene names information.
We create a web page with one element per genes that contains URLs links
to various external database links.
E.g Operon oligodatabase , Riken, GenBank and PubMed web sites. 
}

\usage{
htmlPage(genelist, filename = "GeneList.html", geneNames =
                 Gnames, mapURL = SFGL, othernames, title, table.head,
                 table.center = TRUE, disp = c("browser", "file")[1])

table2html(restable, filename = "GeneList.html", mapURL = SFGL,
                 title, table.head, table.center = TRUE, disp =
                 c("browser", "file")[1])
}

\arguments{
  \item{restable}{A data.frame that contains only the information you
    wish to display in the html file.  The rows corresponds to a
    different DNA spots.}
  \item{genelist}{A numeric vector of index to a data.frame}
  \item{filename}{The name of the file to store the HTML in. }
  \item{geneNames}{A data.frame containing the information related the
    each DNA spots. }
  \item{mapURL}{A matrix of characters containing the URL for various
    external database.  E.g \code{\link{SFGL}}.}
  \item{othernames}{A data.frame containing other information.}
  \item{title}{Title of the HTML page}
  \item{table.head}{ A character vector of column labels for the table}
  \item{table.center}{A logical indicating whether the table should be
    centered}
  \item{disp}{Either "File" or "Browser" (default is Browser). File will
    save the information in html file, while Browser will create an html
    files and display information in the user's browser.}
}
\details{
  This function is an extension to \code{ll.htmlpage}
}

\value{
  No value is return, the function produce a html file "filename" and
  output the results in a browser.
}
\author{Yee Hwa Yang }

\seealso{\code{ll.htmlpage}, \code{\link{URLstring}}, \code{\link{widget.mapGeneInfo}}}

\examples{
##library(annotate)
data(swirl)
Gnames <- maGeneTable(swirl)
swirlmap <- mapGeneInfo(Name = "none", ID="genbank")
## htmlPage(100:110, geneNames = Gnames, mapURL = swirlmap, title="Swirl")

moreinfo <- round(maM(swirl), 2)
swirlmap <- mapGeneInfo(Name = "pubmed", ID="genbank")
##htmlPage(100:110, geneNames = Gnames, mapURL = swirlmap, othernames=moreinfo, title="Swirl", disp="file")
}
\keyword{file}% at least one, from doc/KEYWORDS

\eof
\name{image}
\docType{methods}
\alias{image,marrayRaw-method}
\alias{image,marrayNorm-method}


%- Also NEED an `\alias' for EACH other topic documented here.
\title{Color image for cDNA microarray spot statistics}
\description{
  We encourage users calling "image" rather than "maImage".  The name of
  the arguments are change slightly.
  The function \code{image} creates spatial images of shades of gray or
  colors that correspond to the values of a statistic for each spot on the
  array. The statistic can be the intensity log-ratio M, a spot quality
  measure (e.g. spot size or shape), or a test statistic. This function
  can be used to explore whether there are any spatial effects in the
  data, for example, print-tip or cover-slip effects.  
}
\usage{
\S4method{image}{marrayRaw}(x, xvar = "maM", subset = TRUE, col, contours=FALSE,  bar = TRUE, overlay=NULL, ol.col=1, ...)
\S4method{image}{marrayNorm}(x, xvar = "maM", subset = TRUE, col, contours=FALSE,  bar = TRUE, overlay=NULL, ol.col=1, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{Microarray object of class \code{"\link{marrayRaw}"},
    \code{"\link{marrayNorm}"}}
  \item{xvar}{Name of accessor function for the spot statistic of interest,
    typically a slot name for the microarray object \code{x}, such as
    \code{maM}.} 
  \item{subset}{A "logical" or "numeric" vector indicating the subset of
    spots to display on the image.} 
  \item{col}{List of colors such as that generated by rainbow, heat.colors,
    topo.colors, terrain.colors, or similar
    functions. In addition to these color palette functions, a new
    function \code{\link{maPalette}} was defined to generate color
    palettes from user supplied low, middle, and high color values. } 
  \item{contours}{If \code{contours=TRUE}, contours are plotted, otherwise they are not shown.}
  \item{bar}{If \code{bar=TRUE}, a calibration color bar is shown to the right of the image.}
  \item{overlay}{A logical vector of spots to be highlighted on the image plots.}
  \item{ol.col}{Color of the overlay spots.}
  \item{\dots}{Optional graphical parameters, see \code{\link{par}}.}
}
\details{
 This function calls the general function \code{\link{maImage.func}},
 which is not specific to microarray data. If there are more than one
 array in the batch, the plot is done for the first array, by
 default. Default color palettes were set for different types of spot
 statistics using the \code{\link{maPalette}} function. When
 \code{x=c("maM", "maMloc", "maMscale")}, a green-to-red color palette
 is used. When \code{x=c("maGb", "maGf", "maLG")}, a white-to-green
 color palette is used. When \code{x=c("maRb", "maRf", "maLR")}, a
 white-to-red color palette is used. The user has the option to
 overwrite these parameters at any point.  
}

\value{
 A list with elements 
  \item{x.col}{vector of colors to be used for calibration color bar.}
  \item{x.bar}{vector of values to be used for calibration color bar.}
  \item{summary}{six number summary of the spot statistics, from the function \code{\link{summary}}.}

}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages
  for exploratory analysis and normalization of cDNA microarray data. In
  G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors,
  \emph{The Analysis of Gene Expression Data: Methods and Software},
  Springer, New York. 
}

\author{Jean Yang and Sandrine Dudoit}

\seealso{\code{\link{maImage}}, \code{\link{maImage.func}},
  \code{\link{maColorBar}}, \code{\link{maPalette}}}

\examples{
# Examples use swirl dataset, for description type ? swirl
data(swirl)

# Microarray color palettes
Gcol <- maPalette(low = "white", high = "green", k = 50)
Rcol <- maPalette(low = "white", high = "red", k = 50)
BYcol <- maPalette(low = "blue", mid="gray", high = "yellow", k = 50)

# Color images of green and red background and foreground intensities
##image(swirl[, 2], xvar ="maGb")
##image(swirl[, 2], xvar ="maGf", subset = TRUE, col = Gcol, contours = FALSE, bar = TRUE, main="Swirl array 93")
##image(swirl[, 1], xvar ="maRb", contour=TRUE)
##image(swirl[, 4], xvar ="maRf", bar=FALSE)

# Color images of pre-normalization intensity log-ratios
##image(swirl[, 1])

# Color images with overlay spots
##image(swirl[, 3], xvar = "maA", overlay = maTop(maA(swirl[, 3]), h = 0.1, l = 0.1), bar = TRUE, main = "Image of A values with \% 10 tails highlighted") 

# Color image of print-tip-group
##image(swirl[, 1],xvar = "maPrintTip")

}
\keyword{hplot}% at least one, from doc/KEYWORDS


\eof
\name{ma2D}
\alias{ma2D}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Stratified bivariate robust local regression}
\description{
This function performs robust local regression of a variable \code{z} on predictor variables \code{x} and \code{y}, separately within values of a fourth variable \code{g}. It is used by \code{\link{maNorm2D}} for 2D spatial location normalization.
}
\usage{
ma2D(x, y, z, g, w=NULL, subset=TRUE, span=0.4, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{A numeric vector of predictor variables.}
  \item{y}{A numeric vector of predictor variables.}
  \item{z}{A numeric vector of responses.}
  \item{g}{Variables used to stratify the data.}
  \item{w}{An optional numeric vector of weights.}
  \item{subset}{A "logical" or "numeric" vector indicating the subset of points used to compute the fits. }
  \item{span}{The argument \code{span} which controls the degree of
    smoothing in the  \code{\link{loess}} function.}
  \item{...}{Misc arguments}
}
\details{
\code{z} is regressed on \code{x} and \code{y}, separately within values of \code{g} using the \code{\link{loess}} function.
}
\value{
A numeric vector of fitted values.
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}


\seealso{\code{\link{maNormMain}}, \code{\link{maNorm2D}}, \code{\link{loess}}.}


\examples{
# See examples for maNormMain.
}
\keyword{smooth}% at least one, from doc/KEYWORDS


\eof
\name{maBayesian}
\alias{maBayesian}

\title{Calculates an Odds Ratio for Each Gene in a Multi-slide
  Microarray Experiment. } 

\description{This function calls the function "lm.series" and "ebayes"
  in the package "limma" to calculate the log-odds of differential
     expression by empirical Bayes shrinkage of the standard errors
     towards a common value.  }

\usage{
maBayesian(mraw, ...)
}

\arguments{
  \item{mraw}{An object of class \code{matrix}, \code{dataframe},
    \code{marrayRaw}, \code{marrayNorm} or \code{exprSet} that contains
    gene expression data.} 
  \item{...}{arguments for the fuction \code{lm.series} and \code{ebayes}}
}


\author{Yee Hwa Yang \email{jean@biostat.ucsf.edu} \cr}


\examples{
## data(swirl)
## M <- sweep(maM(swirl), 2, c(1,-1,1,-1), FUN="*")
## est <- maBayesian(M)
}

\keyword{models}









\eof
\name{maBoxplot}
\alias{maBoxplot}

%- Also NEED an `\alias' for EACH other topic documented here.
\title{Boxplots for cDNA microarray spot statistics}

\description{ The function \code{maBoxplot} produces boxplots of microarray spot
statistics for the classes  \code{\link{marrayRaw}} and
\code{\link{marrayNorm}}.We encourage users to use "boxplot"  rather than "maBoxplot".  The name of the arguments have changed.}

\usage{
maBoxplot(m, x="maPrintTip", y="maM", ...)
}
%- maybe also `usage' for other objects documented here.

\arguments{
  \item{m}{Microarray object of class \code{"\link{marrayRaw}"} and \code{"\link{marrayNorm}"}}
  \item{x}{Name of accessor method for the spot statistic used to stratify the data, typically a slot name for the microarray layout object (see \code{"\link{marrayLayout}"}) such as \code{maPlate} or a method such as \code{maPrintTip}. If \code{x} is NULL, the data are not stratified.}
  \item{y}{Name of accessor method for the spot statistic of interest, typically a slot name for the microarray object \code{m}, such as \code{maM}.}
  \item{\dots}{Optional graphical parameters, see \code{\link{par}}.}
}

\details{
If there are more than one array in the batch, the function produces a boxplot for each array in the batch. Such plots are useful when assessing the need for between array normalization, for example, to deal with scale differences among different arrays. Default graphical parameters are chosen for convenience using the function \code{\link{maDefaultPar}} (e.g. color palette,  axis labels,  plot title) but the user has the option to overwrite these parameters at any point.
}

\references{
S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{boxplot}}, \code{\link{maDefaultPar}}.}

\examples{
## see example in boxplot
}
\keyword{hplot}% at least one, from doc/KEYWORDS


\eof
\name{maColorBar}
\alias{maColorBar}

%- Also NEED an `\alias' for EACH other topic documented here.
\title{Calibration bar for color images}
\description{
This function produces a color image (color bar) which can be used for
the legend to another color image obtained from the functions
\code{\link{image}}, \code{\link{maImage}}, or
\code{\link{maImage.func}}.} 
} 
\usage{
maColorBar(x, horizontal=TRUE, col=heat.colors(50), scale=1:length(x), k=10, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{If "numeric", a vector containing the "z" values in the
    color image, i.e., the values which are represented in the color
    image. Otherwise, a "character" vector representing colors.}
  \item{horizontal}{If \code{TRUE}, the values of \code{x} are
    represented as vertical color strips in the image, else, the values
    are represented as horizontal color strips.} 
  \item{col}{Vector of colors such as that generated by
    \code{rainbow}, \code{heat.colors},
    \code{topo.colors}, \code{terrain.colors}, or similar
    functions. In addition to these color palette functions, a new
    function  \code{\link{maPalette}} was defined to generate color
    palettes from user supplied low, middle, and high color values.} 
  \item{scale}{A "numeric" vector specifying the "z" values in the color
    image. This is used when the argument \code{x} is a
    "character" vector representing color information.}
\item{k}{Object of class "numeric", for the number of labels displayed on 
    the bar.}
  \item{\dots}{Optional graphical parameters, see \code{\link{par}}.}
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}, Yee Hwa (Jean) Yang.}

\seealso{\code{\link{image}}, \code{\link{maImage}}, \code{\link{maImage.func}}, \code{\link{maPalette}}.}

\examples{

par(mfrow=c(3,1))
Rcol <- maPalette(low="white", high="red", k=10)
Gcol <- maPalette(low="white", high="green", k=50)
RGcol <- maPalette(low="green", high="red", k=100)
maColorBar(Rcol)
maColorBar(Gcol, scale=c(-5,5))
maColorBar(1:50, col=RGcol)

par(mfrow=c(1,3))
x<-seq(-1, 1, by=0.01)
maColorBar(x, col=Gcol, horizontal=FALSE, k=11)
maColorBar(x, col=Gcol, horizontal=FALSE, k=21)
maColorBar(x, col=Gcol, horizontal=FALSE, k=51)

}

\keyword{aplot}% at least one, from doc/KEYWORDS
\keyword{hplot}% __ONLY ONE__ keyword per line

\eof
\name{maCompCoord}
\alias{maCompCoord}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Generate grid and spot matrix coordinates}
\description{
This function generates grid and spot matrix coordinates from ranges of
rows and columns for the grid and spot matrices. Spots on the array are numbered consecutively starting
from the top left grid and the top left spot within each grid.
}
\usage{
maCompCoord(grows, gcols, srows, scols)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{grows}{numeric vector of grid rows.}
  \item{gcols}{numeric vector of grid columns.}
  \item{srows}{numeric vector of spot rows.}
  \item{scols}{numeric vector of spot columns.}
}

\value{

a matrix of spot four-coordinates, with rows corresponding to
  spots and columns to grid row, grid column, spot row, and spot column coordinates.


}

\author{Yee Hwa (Jean) Yang, Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{marrayLayout}}, \code{\link{maCoord2Ind}},
\code{\link{maInd2Coord}}, \code{\link{maCompInd}}.}


\examples{

maCompCoord(1:2,1,1:4,1:3)


}
\keyword{manip}
\keyword{methods}

\eof
\name{maCompInd}
\alias{maCompInd}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Generate spot indices}
\description{
This function generates spot indices from ranges of rows and columns for
the grid and spot matrices. Spots on the array are numbered consecutively starting
from the top left grid and the top left spot within each grid.

}
\usage{
maCompInd(grows, gcols, srows, scols, L)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{grows}{numeric vector of grid rows.}
  \item{gcols}{numeric vector of grid columns.}
  \item{srows}{numeric vector of spot rows.}
  \item{scols}{numeric vector of spot columns.}
  \item{L}{object of class \code{"\link{marrayLayout}"}.}
}

\value{
a numeric vector of spot indices.
}

\author{Yee Hwa (Jean) Yang, Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{marrayLayout}}, \code{\link{maCoord2Ind}},
\code{\link{maInd2Coord}}, \code{\link{maCompCoord}}.}


\examples{
L <- new("marrayLayout", maNgr=4, maNgc=4, maNsr=22, maNsc=24)
maCompInd(1:2,1,1:4,1:3,L)


}
\keyword{manip}
\keyword{methods}

\eof
\name{maCompLayout}
\alias{maCompLayout}

\title{Generate a marrayLayout object}
\description{
  Take a matrix of cooordiates and generate a marrayLayout object.
}
\usage{
maCompLayout(mat, ncolumns = 4)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{mat}{a matrix of coordinates, this can either be n by 3 matrix
    with columns (Block, Row, Column) or n by 4 matrix with columns
    (Grid.R, Grid.C, Spot.R, Spot.C)}
  \item{ncolumns}{For n by 3 matrix, the number of meta-grid columns.
    By default, it is set to 4.}
}
\details{

}
\value{
  An object of class \code{"\link{marrayLayout}"}.
}

\author{Jean Yang}

\examples{
X <- cbind(Block = c(1,1,2,2,3,3,4,4), Rows=c(1,2,1,2,1,2,1,2), Columns=rep(1,8))
maCompLayout(X, ncolumns=2)
}
\keyword{manip}% at least one, from doc/KEYWORDS

\eof
\name{maCompNormA}
\alias{maCompNormA}
\alias{maCompNormEq}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Weights for composite normalization}
\description{
This function is used for composite normalization with intensity dependent weights. The function should be used as an argument to the main normalization function \code{\link{maNormMain}}. It only applies when two normalization procedures are combined. 
}
\usage{
maCompNormA()
maCompNormEq()
}
%- maybe also `usage' for other objects documented here.

\value{
  A function which takes as arguments \code{x} and \code{n}, the spot average log-intensities A and the number of normalization procedures. This latter function returns a matrix of weights for combining two normalization procedures, rows correspond to spots and columns to normalization procedures. The weights for the first procedure are given by the empirical cumulative distribution function of the spot average log-intensities A. Note that when performing composite normalization as described in Yang et al. (2002), the first normalization procedure is the global fit and the second procedure is the within-print-tip-group fit.\cr \cr
For \code{maCompEq}, equal weights are given for each procedure.

}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.\cr \cr

Y. H. Yang, S. Dudoit, P. Luu, D. M. Lin, V. Peng, J. Ngai, and T. P. Speed (2002). Normalization for cDNA microarray data: a robust composite method addressing single and multiple slide systematic variation. \emph{Nucleic Acids Research}, Vol. 30, No. 4.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}, Yee Hwa (Jean) Yang.}

\seealso{\code{\link{maNormMain}}, \code{\link{maNormLoess}}, \code{\link{ecdf}}.}

\examples{
# See examples for maNormMain
}
\keyword{smooth}% at least one, from doc/KEYWORDS


\eof
\name{maCompPlate}
\alias{maCompPlate}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Generate plate IDs}
\description{
This function generates plate IDs from the dimensions of the grid and
spot matrices. Note that this function only applies to arrays with a
regular plate layout, where the number of spots is a multiple of the
number of wells on a plate (usually 96 or 384) and each well contributes 
exactly one spot. It should thus be used with caution.}

\usage{
maCompPlate(x, n=384)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{object of class \code{"\link{marrayLayout}"},
    \code{"\link{marrayRaw}"} and \code{"\link{marrayNorm}"}}
  \item{n}{object of class "numeric", number of wells in each plate,
    usually 384 or 96.} 
}

\details{
  Having plate IDs may be useful for the purpose of
  normalization. Normalization by plate can be done using the function
  \code{\link{maNormMain}}. 
}
\value{
a vector of plate IDs (\code{\link{factor}}).
}

\author{Yee Hwa (Jean) Yang, Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{marrayLayout}}, \code{\link{marrayRaw}},
  \code{\link{marrayNorm}}}

\examples{

L<-new("marrayLayout", maNgr=4, maNgc=4, maNsr=22, maNsc=24)
plate<-maCompPlate(L,384)
table(plate)
maPlate(L)<-plate

}
\keyword{methods}% at least one, from doc/KEYWORDS


\eof
\name{maCoord2Ind}
\alias{maCoord2Ind}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Convert grid and spot matrix coordinates to spot indices}
\description{
This functions converts grid and spot matrix coordinates (four
coordinates) to spot
indices, where spots on the array are numbered consecutively starting
from the top left grid and the top left spot within each grid.
}
\usage{
maCoord2Ind(x, L)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{a matrix of spot four-coordinates, with rows corresponding to
  spots and columns to grid row, grid column, spot row, and spot column coordinates.}
  \item{L}{an object of class \code{"\link{marrayLayout}"}.}
}

\value{
 a numeric vector of spot indices.
}

\author{Yee Hwa (Jean) Yang, Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{marrayLayout}}, \code{\link{maInd2Coord}},
\code{\link{maCompCoord}}, \code{\link{maCompInd}}.}

\examples{
L <- new("marrayLayout", maNgr=4, maNgc=4, maNsr=22, maNsc=24)
coord<-cbind(rep(2,4),rep(1,4),rep(1,4),1:4)
maCoord2Ind(coord, L)


}
\keyword{manip}
\keyword{methods}

\eof
\name{maDefaultPar}
\alias{maDefaultPar}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Default graphical parameters for microarray objects}
\description{
This function returns default graphical parameters for microarray objects. The parameters may be passed as arguments to the functions \code{maBoxplot} and \code{maPlot}.
} 

\usage{
maDefaultPar(m, x, y, z)
}
%- maybe also `usage' for other objects documented here.

\arguments{
  \item{m}{Microarray object of class \code{"\link{marrayRaw}"} and \code{"\link{marrayNorm}"}.}
  \item{x}{Name of accessor method for the abscissa spot statistic,
    typically a slot name for the microarray object \code{m}, such as
    \code{maA}.} 
  \item{y}{Name of accessor method for the ordinate spot statistic,
    typically a slot name for the microarray object \code{m}, such as
    \code{maM}.} 
  \item{z}{Name of accessor method for the spot statistic used to
    stratify the data, typically a slot name for the microarray layout
    object (see \code{"\link{marrayLayout}"}) such as \code{maPlate} or
    a method such as \code{maPrintTip}.} 
}

\value{
A list with elements
  \item{def.box}{default graphical parameters for \code{\link{maBoxplot}}.}
  \item{def.plot}{default graphical parameters for \code{\link{maPlot}}.}
  \item{def.lines}{default graphical parameters for functions such as \code{\link{maLoessLines}} used in \code{\link{maPlot}}.}  
  \item{def.legend}{default graphical parameters for functions such as \code{\link{maLegendLines}} used in \code{\link{maPlot}}.}   
  \item{def.text}{default graphical parameters for functions such as \code{\link{maText}} used in \code{\link{maPlot}}.} 

}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}


\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{maBoxplot}}, \code{\link{maPlot}},
  \code{\link{maLegendLines}}, \code{\link{maLoessLines}},
  \code{\link{maText}}, \code{\link{maDotsDefaults}}.} 

\examples{
# See examples for maPlot.

}
\keyword{iplot}% at least one, from doc/KEYWORDS
\keyword{dplot}% __ONLY ONE__ keyword per line
\keyword{color}% __ONLY ONE__ keyword per line

\eof
\name{maDiagnPlots}
\alias{maDiagnPlots}

\title{Basic diagnostic plots}
\description{
  These three functions produce sets of basic diagnostic plots for pre-
  and post-normalization cDNA microarray data. 
}
\usage{
maDiagnPlots(mraw, mNorm = NULL, save = TRUE, fname = NULL,
dev= "png", pch, col, DEBUG = FALSE, ...)
}

\arguments{
  \item{mraw}{Microarray object of class `"marrayRaw"'.
    By default, plots are done for the first array in the batch.}
  \item{mNorm}{Microarray object of class `"marrayNorm"'. By default,
    this value is NULL and the function maNorm is call to perform normalization.}
  \item{save}{If `TRUE', the figures will be saved to the file named in
          `fname'.}
  \item{fname}{A "character" string naming the output file.}
  \item{dev}{A "character" string naming the graphics device.}
  \item{pch}{point style for different control samples.}
  \item{col}{color code for different control samples.}
  \item{DEBUG}{If 'TRUE', debug statements are printed.}
  \item{\dots}{Optional graphical parameters, see `par'.}
}
\details{
  \code{maDiagnPlots} produces plots of pre- and  post-normalization cDNA
  microarray data: color images of log-ratios (M) and log-intensities
  (A) pre- and post-normalization. MA-plots of pre- and post-normalization
  log-ratios M. Log foreground to background ratios for both Cy5 and Cy3
  channels. Dot plots for replicate control spots. 
}


\author{Yee Hwa (Jean) Yang}

\note{This function may not work properly when "save=FALSE".}

\seealso{\code{\link{maPlot}}, \code{\link{maImage}}, \code{\link{maBoxplot}}}


\examples{
data(swirl)
## maDiagnPlots(swirl, dev="postscript", save=TRUE)
}

\keyword{hplot}


\eof
\name{maDiagnPlots1}
\alias{maDiagnPlots1}
\alias{maRawPlots}
\alias{maNormPlots}

%- Also NEED an `\alias' for EACH other topic documented here.
\title{Basic diagnostic plots}
\description{
  These three functions produce sets of basic diagnostic plots for pre-- and post--normalization cDNA microarray data.
}
\usage{
maDiagnPlots1(mraw, title=NULL, save=TRUE,
	fname=paste(as.character(maLabels(maTargets(mraw))[1]), ".ps", sep = ""), 
	dev=c("postscript", "jpeg"))

maRawPlots(mraw, title=NULL, save = TRUE, 
	fname = paste(as.character(maLabels(maTargets(mraw))[1]), ".ps", sep = ""), 
	dev = c("postscript", "jpeg"))


maNormPlots(mnorm, title=NULL, save = TRUE, 
	fname = paste(as.character(maLabels(maTargets(mnorm))[1]),".ps", sep = ""), 
	dev = c("postscript", "jpeg"))

}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{mraw}{Microarray object of class \code{"\link{marrayRaw}"}, for \code{maDiagnPlots1} \code{maRawPlots}. By default, plots are done for the first array in the batch.}
 \item{mnorm}{Microarray object of class \code{"\link{marrayNorm}"}, for \code{maNormPlots}. By default, plots are done for the first array in the batch.}
 \item{title}{A "character" string for the plot title. If \code{NULL} a default title is shown.}
  \item{save}{If \code{TRUE}, the figures will be saved to the file named in \code{fname}.}
  \item{fname}{A "character" string naming the output file.}
  \item{dev}{A "character" string naming the graphics device, "postscript" or "jpeg".}
}
\details{
\code{maDiagnPlots1} produces eight plots of pre-- and post--normalization cDNA microarray data: color images of Cy3 and Cy5 background intensities, and of pre-- and post--normalization log--ratios M; boxplots of pre-- and post--normalization log--ratios M by print--tip--group; MA--plots of pre-- and post--normalization log--ratios M by print--tip--group.\cr \cr

\code{maRawPlots} produces six plots of pre--normalization cDNA microarray data: color images of Cy3 and Cy5 background intensities and log--ratios M; boxplots of log--ratios M by print--tip--group and plate; MA--plot of log--ratios M by print--tip--group.\cr \cr

\code{maNormPlots} produces four plots of post--normalization cDNA microarray data: color images of location normalization values Mloc and normalized log--ratios M; boxplots of normalized log--ratios M by print--tip--group; MA--plot of normalized log--ratios M by print--tip--group.
}

\references{
S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}, Yee Hwa (Jean) Yang.}

\note{This function maynot work properly when "save=FALSE".}

\seealso{\code{\link{maPlot}}, \code{\link{maBoxplot}}, \code{\link{maImage}}.}

\examples{
# Examples use swirl dataset, for description type ? swirl
data(swirl)

maPlate(swirl)<-maCompPlate(swirl,384)

# Pre-normalization
maRawPlots(swirl)

# Post-normalization
mnorm<-maNorm(swirl[,3], norm="twoD")
maNormPlots(mnorm, title="Swirl array 93 -- 2D normalization")

}
\keyword{hplot}

\eof
\name{maDotPlots}
\alias{maDotPlots}

\title{Produce a dot plot for replicated controls genes}
\description{
  A dot plot showing the values of replicated control genes
}
\usage{
maDotPlots(data, x = list("maA"), id = "ID", pch, col, nrep = 3, ...)
}

\arguments{
  \item{data}{Microarray object of class `"marrayRaw"'. By default, plots
    are done for the first array in the batch.}
  \item{x}{Name of accessor method for the spot statistic to be display.
    E.g. maM to display log-ratios and maA to display log-intensities}
  \item{id}{Integer value specifying the index of the array.}
  \item{pch}{point style for different control samples.}
  \item{col}{color code for different control samples.}
  \item{nrep}{Integer value indicating the minimum number of replicated
    spots required.}
  \item{\dots}{Optional graphical parameters, see `par'.}
}

\value{
  Produce a dot plot showing the values of replicated control genes.
}

\examples{
  data(swirl)
  maControls(swirl) <- maGenControls(maGnames(swirl), id="Name")
  maDotPlots(swirl[,1])
}

\author{Yee Hwa (Jean) Yang}

\keyword{hplot}

\eof
\name{maDotsDefaults}
\alias{maDotsDefaults}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Replace graphical default parameters by user supplied parameters}

\description{
This function may be used to compare default graphical parameters for microarray diagnostic plots to user supplied parameters given in \code{...}. User supplied parameters overwrite the defaults. It is used in \code{\link{maBoxplot}}, \code{\link{maPlot}}, and \code{\link{maImage}}.}


\usage{
maDotsDefaults(dots, defaults)
}
%- maybe also `usage' for other objects documented here.

\arguments{
  \item{dots}{List of user supplied parameters, e.g. from \code{list(...)}.}
  \item{defaults}{List of default parameters, e.g. from the function \code{\link{maDefaultPar}}.}
}

\value{
  \item{args}{List of graphical parameters.}
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{maDefaultPar}}, \code{\link{maBoxplot}}, \code{\link{maPlot}}, \code{\link{maImage}}.}

\examples{
dots<-list(xlab="X1", ylab="Y1")
defaults<-list(xlab="X1", ylab="Y2", col=2)
pars<-maDotsDefaults(dots, defaults)

do.call("plot",c(list(x=1:10), pars))

}
\keyword{misc}% at least one, from doc/KEYWORDS


\eof
\name{maDotsMatch}
\alias{maDotsMatch}

\title{Replace default arguments of a function by user supplied values}

\description{
  This function may be used to replace default arguements for any
  functions to user supplied parameters.}


\usage{
maDotsMatch(dots, defaults)
}

\arguments{
  \item{dots}{List of user supplied argements, e.g. from \code{list(...)}.}
  \item{defaults}{List of  formal arguments of a function, e.g. from the
    function \code{\link{formals}}.} 
}

\value{
  \item{args}{List of argument of a function.}
}

\author{Jean Yee Hwa Yang}

\seealso{\code{\link{maDefaultPar}}, \code{\link{maDotsDefaults}}}

\examples{
dots<-list(x=1:10, y=11:20)
argsfun <- maDotsMatch(dots, formals(args(plot)))
do.call("plot", argsfun)
}
\keyword{misc}% at least one, from doc/KEYWORDS


\eof
\name{maGenControls}
\alias{maGenControls}
\alias{controlCode}

\title{Generating a vector recording the control status of the spotted probe sequences. }
\description{
  ControlCode is a matrix representing certain regular expression
  pattern and the control status of the spotted probe sequences.
  This function uses `grep' searches for matches to `pattern' (its first argument)
  within the character vector `x' (second argument). 
}

\usage{
maGenControls(Gnames, controlcode, id = "ID")
}

\arguments{
  \item{Gnames}{An object of class \code{matrix}, \code{data.frame} or \code{marrayInfo}
    which contains  description of spotted probe sequences.}
  \item{controlcode}{A character matrix of n by 2 columns. The first
    column contains a few regular expression of spotted probe sequences
    and the second column contains the corresponding control status.}
  \item{id}{the column number of column name in \code{Gnames} that
    contains description of each spot on the array.}
}

\value{
  A vector of characters recording the control status of the spotted probe sequences.
}

\author{Jean Yee Hwa Yang}

\seealso{\code{\link{grep}}}

\examples{
data(swirl)
maControls(swirl) <- maGenControls(maGnames(swirl), id="Name")
table(maControls(swirl))
}
\keyword{manip}% at least one, from doc/KEYWORDS


\eof
\name{maGeneTable}
\alias{maGeneTable}
%- Also NEED an `\alias' for EACH other topic documented here.

\title{Table of spot coordinates and gene names}
\description{
This function produces a table of spot coordinates and gene names for
objects of class \code{"\link{marrayRaw}"} and
\code{"\link{marrayNorm}"}.}

\usage{
maGeneTable(object)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{object}{microarray object of class \code{"\link{marrayRaw}"} and
    \code{"\link{marrayNorm}"}.}
}

\value{
  an object of class \code{\link{data.frame}}, with rows corresponding
  to spotted probe sequences. The first four columns are the grid matrix
  and spot matrix coordinates, and the remaining columns are the spot
  descriptions stored in the \code{maGnames} slot of the microarray
  object. 
}

\author{Yee Hwa (Jean) Yang}


\seealso{\code{\link{marrayInfo}}, \code{\link{marrayLayout}}, \code{\link{marrayRaw}}, \code{\link{marrayNorm}},  \code{\link{maCompCoord}}.}


\examples{
# Example uses swirl dataset, for description type ? swirl

data(swirl)

tab<-maGeneTable(swirl)
tab[1:10,]

}

\keyword{methods}% at least one, from doc/KEYWORDS


\eof
\name{maImage}
\alias{maImage}

%- Also NEED an `\alias' for EACH other topic documented here.
\title{Color image for cDNA microarray spot statistics}
\description{
  We encourage users calling "image" rather than "maImage".  The name of  the arguments are change slightly.

  The function \code{maImage} creates spatial images of shades of gray or colors that correspond to the values of a statistic for each spot on the array. The statistic can be the intensity log-ratio M, a spot quality measure (e.g. spot size or shape), or a test statistic. This function can be used to explore whether there are any spatial effects in the data, for example, print-tip or cover-slip effects. 
}
\usage{
maImage(m, x="maM", subset=TRUE, col, contours=FALSE, bar=TRUE,
overlay=NULL, ol.col=1, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{m}{Microarray object of class \code{"\link{marrayRaw}"} and \code{"\link{marrayNorm}"}.}
  \item{x}{Name of accessor function for the spot statistic of interest, typically a slot name for the microarray object \code{m}, such as \code{maM}.}
  \item{subset}{A "logical" or "numeric" vector indicating the subset of spots to display on the image.}
  \item{col}{List of colors such as that generated by rainbow, heat.colors, topo.colors, terrain.colors, or similar functions. In addition to these color palette functions, a new function \code{\link{maPalette}} was defined to generate color palettes from user supplied low, middle, and high color values. }
  \item{contours}{If \code{contours=TRUE}, contours are plotted, otherwise they are not shown.}
  \item{bar}{If \code{bar=TRUE}, a calibration color bar is shown to the
    right of the image.}
  \item{overlay}{A logical vector of spots to be highlighted on the image plots.}
  \item{ol.col}{Color of the overlay spots.}
  \item{\dots}{Optional graphical parameters, see \code{\link{par}}.}
}
\details{
 This function calls the general function \code{\link{maImage.func}}, which is not specific to microarray data. If there are more than one array in the batch, the plot is done for the first array, by default. Default color palettes were set for different types of spot statistics using the \code{\link{maPalette}} function. When \code{x=c("maM", "maMloc", "maMscale")}, a green-to-red color palette is used. When \code{x=c("maGb", "maGf", "maLG")}, a white-to-green color palette is used. When \code{x=c("maRb", "maRf", "maLR")}, a white-to-red color palette is used. The user has the option to overwrite these parameters at any point. 
}

\value{
 A list with elements 
  \item{x.col}{vector of colors to be used for calibration color bar.}
  \item{x.bar}{vector of values to be used for calibration color bar.}
  \item{summary}{six number summary of the spot statistics, from the function \code{\link{summary}}.}

}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}


\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{image}}, \code{\link{maImage.func}}, \code{\link{maColorBar}}, \code{\link{maPalette}}, \code{\link{summary}}.}

\examples{

# To see the demo type demo(marrayPlots)

# Examples use swirl dataset, for description type ? swirl
data(swirl)

# Microarray color palettes
Gcol <- maPalette(low = "white", high = "green", k = 50)
Rcol <- maPalette(low = "white", high = "red", k = 50)
RGcol <- maPalette(low = "green", high = "red", k = 50)

# Color images of green and red background and foreground intensities
maImage(swirl[, 3], x="maGb")
maImage(swirl[, 3], x = "maGf", subset = TRUE, col = Gcol, contours = FALSE, bar = TRUE, main="Swirl array 93")
maImage(swirl[, 3], x = "maRb", contour=TRUE)
maImage(swirl[, 3], x = "maRf", bar=FALSE)

# Color images of pre-normalization intensity log-ratios
maImage(swirl[, 1])
maImage(swirl[, 3], x = "maM", subset = maTop(maM(swirl[, 3]), h = 0.1, l = 0.1), col = RGcol, contours = FALSE, bar = TRUE, main = "Swirl array 93: image of pre-normalization M for \% 10 tails")

# Color image of print-tip-group
maImage(swirl[, 1],x="maPrintTip")

}
\keyword{hplot}% at least one, from doc/KEYWORDS


\eof
\name{maImage.func}
\alias{maImage.func}

%- Also NEED an `\alias' for EACH other topic documented here.
\title{Color image for cDNA microarray spot statistics}
\description{
This function creates spatial images of shades of gray or colors that correspond to the values of a statistic for each spot on the array. The statistic can be the intensity log-ratio M, a spot quality measure (e.g. spot size or shape), or a test statistic. This function can be used to explore whether there are any spatial effects in the data, for example, print-tip or cover-slip effects. This function is called by \code{\link{maImage}}.
}

\usage{
maImage.func(x, L, subset=TRUE, col=heat.colors(12), contours=FALSE, overlay=NULL, ol.col=1,  ...)
}
%- maybe also `usage' for other objects documented here.

\arguments{
  \item{x}{A "numeric" vector of spot statistics.}
  \item{L}{An object of class \code{"\link{marrayLayout}"}, if L is
    missing we will assume the dimension of x.}
  \item{subset}{A "logical" or "numeric" vector indicating the subset of spots to display on the image.}
  \item{col}{A list of colors such as that generated by rainbow, heat.colors, topo.colors, terrain.colors, or similar functions. In addition to these color palette functions, a new function  \code{\link{maPalette}} was defined to generate color palettes from user supplied low, middle, and high color values.}
  \item{contours}{If \code{contours=TRUE}, contours are plotted,
    otherwise they are not shown.}
  \item{overlay}{A logical vector of spots to be highlighted on the image plots.}
  \item{ol.col}{Color of the overlay spots.}
  \item{\dots}{Optional graphical parameters, see \code{\link{par}}.}
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}


\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{image}}, \code{\link{maImage}}, \code{\link{maColorBar}}, \code{\link{maPalette}}.}

\examples{
# See examples for image.
}
\keyword{hplot}% at least one, from doc/KEYWORDS


\eof
\name{maInd2Coord}
\alias{maInd2Coord}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Convert spot indices to grid and spot matrix coordinates}
\description{
This functions converts spot indices to grid and spot matrix coordinates (four
coordinates), where spots on the array are numbered consecutively starting
from the top left grid and the top left spot within each grid.
}
\usage{
maInd2Coord(x, L)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{a numeric vector of spot indices.}
  \item{L}{an object of class \code{"\link{marrayLayout}"}.}
}
\value{
a matrix of spot four-coordinates, with rows corresponding to
  spots and columns to grid row, grid column, spot row, and spot column coordinates.
}

\author{Yee Hwa (Jean) Yang, Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{marrayLayout}}, \code{\link{maCoord2Ind}},
\code{\link{maCompCoord}}, \code{\link{maCompInd}}.}

\examples{

L <- new("marrayLayout", maNgr=4, maNgc=4, maNsr=22, maNsc=24)
maInd2Coord(c(1:10,529:538), L)


}
\keyword{manip}
\keyword{methods}

\eof
\name{maLegendLines}
\alias{maLegendLines}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Add a legend to a plot}
\description{
This function may be used to add a legend for lines in plots such as those produced by \code{\link{plot}}, \code{\link{maPlot}}, or \code{\link{maPlot.func}}.
}

\usage{
maLegendLines(legend="", col=2, lty=1, lwd=2.5, ncol=1, ...)
}
%- maybe also `usage' for other objects documented here.

\arguments{
  \item{legend}{A vector of "character" strings to appear in the legend.}
  \item{col}{Line colors for the legend.}
  \item{lty}{Line types for the legend.}
  \item{lwd}{Line widths for the legend.}
  \item{ncol}{The number of columns in which to set the legend items (default is 1, a vertical legend).}
  \item{\dots}{Optional graphical parameters, see \code{\link{par}}.}
}


\value{
A function with bindings for \code{legend}, \code{col}, \code{lty}, \code{lwd}, \code{ncol}, and \code{\ldots}. This latter function takes as arguments \code{x} and \code{y}, the coordinates for the location of the legend on the plot, and it adds the legend to the current plot.
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}


\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{legend}}, \code{\link{maPlot}}, \code{\link{maPlot.func}}.}

\examples{
# See examples for maPlot.

}
\keyword{aplot}% at least one, from doc/KEYWORDS


\eof
\name{maLoess}
\alias{maLoess}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Stratified univariate robust local regression}
\description{
 This function performs robust local regression of a variable \code{y} on predictor variable \code{x}, separately within values of a third variable \code{z}. It is used by \code{\link{maNormLoess}} for intensity dependent location normalization.
}
\usage{
maLoess(x, y, z, w=NULL, subset=TRUE, span=0.4, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{A numeric vector of predictor variables.}
  \item{y}{A numeric vector of responses.}
  \item{z}{Variables used to stratify the data.}
  \item{w}{An optional numeric vector of weights.}
  \item{subset}{A "logical" or "numeric" vector indicating the subset of points used to compute the fits. }
  \item{span}{The argument \code{span} which controls the degree of
    smoothing in the  \code{\link{loess}} function.}
    \item{...}{Misc arguments}
}

\details{
\code{y} is regressed on \code{x}, separately within values of \code{z} using the \code{\link{loess}} function.  
}

\value{
A numeric vector of fitted values.
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.\cr \cr

Y. H. Yang, S. Dudoit, P. Luu, and T. P. Speed (2001). Normalization for cDNA microarray data. In M. L. Bittner, Y. Chen, A. N. Dorsel, and E. R. Dougherty (eds), \emph{Microarrays: Optical Technologies and Informatics}, Vol. 4266 of \emph{Proceedings of SPIE}.\cr \cr

Y. H. Yang, S. Dudoit, P. Luu, D. M. Lin, V. Peng, J. Ngai, and T. P. Speed (2002). Normalization for cDNA microarray data: a robust composite method addressing single and multiple slide systematic variation. \emph{Nucleic Acids Research}, Vol. 30, No. 4.}}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}


\seealso{\code{\link{maNormMain}}, \code{\link{maNormLoess}}, \code{\link{loess}}.}


\examples{
# See examples for maNormMain.

}
\keyword{smooth}% at least one, from doc/KEYWORDS


\eof
\name{maLoessLines}
\alias{maLoessLines}
\alias{maLowessLines}

%- Also NEED an `\alias' for EACH other topic documented here.
\title{Add smoothed fits to a plot}
\description{
This function may be used to compute and plot loess or lowess fits for an existing plot. The plot can be produced by \code{\link{plot}}, \code{\link{maPlot}}, or \code{\link{maPlot.func}}.
}

\usage{
maLoessLines(subset=TRUE, weights=NULL, loess.args=list(span = 0.4,
degree=1, family="symmetric", control=loess.control(trace.hat =
"approximate", iterations=5, surface="direct")), col=2, lty=1, lwd=2.5, ...)

maLowessLines(subset = TRUE, f = 0.3, col = 2, lty = 1, lwd = 2.5, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{subset}{A "logical" or "numeric" vector indicating the subset of points used to compute the fits. }
  \item{weights}{Optional "numeric" vector of weights -- for \code{maLoessLines} only.} 
  \item{loess.args}{List of optional arguments for the \code{\link{loess}} functions -- for \code{maLoessLines} only.}
  \item{f}{The smoother span for the \code{\link{lowess}} function -- for \code{maLowessLines} only.}
  \item{col}{The fitted line colors.}
  \item{lty}{The fitted line types.}
  \item{lwd}{The fitted line widths.}
  \item{\dots}{Optional graphical parameters, see \code{\link{par}}.}
}

\value{
A function with bindings for \code{subset}, \code{weights}, \code{loess.args},  \code{col}, \code{lty}, \code{lwd}, and \code{\ldots}. This latter function takes as arguments \code{x} and \code{y}, the abscissa and ordinates of points on the plot, and \code{z} a vector of discrete values used to stratify the points. Loess (or lowess) fits are performed separately within values of \code{z}.
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}


\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{loess}}, \code{\link{lowess}}, \code{\link{maPlot}}, \code{\link{maPlot.func}}.}

\examples{
# See examples for maPlot.

}

\keyword{aplot}% at least one, from doc/KEYWORDS
\keyword{smooth}% __ONLY ONE__ keyword per line

\eof
\name{maMAD}
\alias{maMAD}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Stratified MAD calculation}
\description{
This function computes the median absolute deviation (MAD) of values in \code{y} separately within values of \code{x}. It is used by \code{\link{maNormMAD}} for MAD scale normalization.
}
\usage{
maMAD(x, y, geo=TRUE, subset=TRUE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{Variables used to stratify the data.}
  \item{y}{A numeric vector.}
  \item{geo}{If \code{TRUE}, the MAD of each group is divided by the
  geometric mean of the MADs across groups (cf. Yang et al. (2002)). This allows observations to
  retain their original units.}
  \item{subset}{A "logical" or "numeric" vector indicating the subset of points used to compute the MAD. }
}

\value{
A numeric vector of MAD values. 
}


\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.\cr \cr

Y. H. Yang, S. Dudoit, P. Luu, and T. P. Speed (2001). Normalization for cDNA microarray data. In M. L. Bittner, Y. Chen, A. N. Dorsel, and E. R. Dougherty (eds), \emph{Microarrays: Optical Technologies and Informatics}, Vol. 4266 of \emph{Proceedings of SPIE}.\cr \cr

Y. H. Yang, S. Dudoit, P. Luu, D. M. Lin, V. Peng, J. Ngai, and T. P. Speed (2002). Normalization for cDNA microarray data: a robust composite method addressing single and multiple slide systematic variation. \emph{Nucleic Acids Research}, Vol. 30, No. 4.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}


\seealso{\code{\link{maNormMain}}, \code{\link{maNormMAD}}, \code{\link{mad}}.}

\examples{
# See examples for maNormMain.

}

\keyword{univar}% at least one, from doc/KEYWORDS
\keyword{robust}% __ONLY ONE__ keyword per line

\eof
\name{maMed}
\alias{maMed}
\title{Stratified median calculation}
\description{
This function computes the median of values in \code{y} separately within values of \code{x}. It is used by \code{\link{maNormMed}} for median location normalization.
}
\usage{
maMed(x, y, subset=TRUE)
}
\arguments{
  \item{x}{Variables used to stratify the data.}
  \item{y}{A numeric vector.}
  \item{subset}{A "logical" or "numeric" vector indicating the subset of points used to compute the median. }
}

\value{
A numeric vector of median values. 
}


\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.\cr \cr

Y. H. Yang, S. Dudoit, P. Luu, and T. P. Speed (2001). Normalization for cDNA microarray data. In M. L. Bittner, Y. Chen, A. N. Dorsel, and E. R. Dougherty (eds), \emph{Microarrays: Optical Technologies and Informatics}, Vol. 4266 of \emph{Proceedings of SPIE}.\cr \cr

Y. H. Yang, S. Dudoit, P. Luu, D. M. Lin, V. Peng, J. Ngai, and T. P. Speed (2002). Normalization for cDNA microarray data: a robust composite method addressing single and multiple slide systematic variation. \emph{Nucleic Acids Research}, Vol. 30, No. 4.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}


\seealso{\code{\link{maNormMain}}, \code{\link{maNormMed}}, \code{\link{median}}.}


\examples{
# See examples for maNormMain.

}

\keyword{univar}
\keyword{robust}

\eof
\name{na}

\alias{log.na}
\alias{sum.na}
\alias{mean.na}
\alias{var.na}
\alias{cor.na}
\alias{quantile.na}
\alias{length.na}
\alias{order.na}
\alias{scale.na}
\alias{prod.na}

\title{Basic Statistical Functions for Handling Missing Values}

\description{
Basic statistical functions for handling missing values or NA. \cr 
In \code{log.na}, \code{sum.na}, \code{mean.na} and \code{var.na},
\code{quantile.na}, \code{length.na}, missing values are omitted
from the calculation. \cr 
The function \code{cor.na} calls \code{cor} with the argument
\code{use="pairwise.complete.obs"}. \cr 
The function \code{order.na} only handles vector arguments and not
lists.  However, it gives the option of omitting the NAs
(\code{na.last=NA}), of placing the NAs at the start of the ordered
vector (\code{na.last=F}) or at the end (\code{na.last=T}). \cr 
The function \code{scale.na} is a modified version of
\code{\link{scale}} which allows NAs in the variance calculation. If
\code{scale = T}, the function \code{f} in \code{scale.na} uses
\code{var.na} to perform the variance calculation.
The function \code{prod.na} is similar to the \code{\link{prod}}
function with \code{na.rm=TRUE}. This function returns the product of
all the values present in its arguments, omitting any missing values.
}

\author{
  Yee Hwa Yang, \email{jean@biostat.berkeley.edu} \cr
}

\seealso{\code{\link{log}}, \code{\link{sum}}, \code{\link{mean}},
  \code{\link{var}}, \code{\link{cor}}, \code{\link{order}},
  \code{\link{scale}}, \code{link{prod}}.}

%%\keyword{log, sum, mean, variance, correlation, order, scale, product, missing values, NA.} 
\keyword{arith} 

\eof
\name{maNorm}
\alias{maNorm}
\title{Simple location and scale normalization function}
\description{
This function is a simple wrapper function around the main normalization function \code{\link{maNormMain}}. It allows the user to choose from a set of six basic location and scale normalization procedures. The function operates on an object of class \code{"\link{marrayRaw}"} (or possibly \code{"\link{marrayNorm}"}, if normalization is performed in several steps) and returns an object of class \code{"\link{marrayNorm}"}.
}
\usage{
maNorm(mbatch, norm=c("printTipLoess", "none", "median", "loess",
"twoD", "scalePrintTipMAD"), subset=TRUE, span=0.4, Mloc=TRUE,
Mscale=TRUE, echo=FALSE, ...)
}
\arguments{
  \item{mbatch}{Object of class \code{\link{marrayRaw}}, containing intensity 
data for the batch of arrays to be normalized. 
An object of class \code{"\link{marrayNorm}"} may also be passed if 
normalization is performed in several steps.}
\item{norm}{Character string specifying the normalization procedures: 
\describe{
\item{none}{no normalization}
\item{median}{for global median location normalization}
\item{loess}{for global intensity or A-dependent location normalization using 
the \code{\link{loess}} function}
\item{twoD}{for 2D spatial location normalization using the 
\code{\link{loess}} function}
\item{printTipLoess}{for within-print-tip-group intensity dependent location 
normalization using the \code{\link{loess}} function}
\item{scalePrintTipMAD}{for within-print-tip-group intensity dependent 
location normalization followed by within-print-tip-group scale normalization 
using the median absolute deviation (MAD). \cr
}
This argument can be specified using the first letter of each method.}}
  \item{subset}{A "logical" or "numeric" vector indicating the subset of points used to
  compute the  normalization values.}
 \item{span}{The argument \code{span} which controls the degree of smoothing in the \code{\link{loess}} function.}
\item{Mloc}{If \code{TRUE}, the location normalization values are stored in the slot \code{maMloc} of the object of class \code{"\link{marrayNorm}"} returned by the function, if \code{FALSE}, these values are not retained.}
  \item{Mscale}{If \code{TRUE}, the scale normalization values are stored in the slot \code{maMscale} of the object of class \code{"\link{marrayNorm}"} returned by the function, if \code{FALSE}, these values are not retained.}
\item{echo}{If \code{TRUE}, the index of the array currently being
  normalized is printed.}
\item{...}{Misc arguments}
}
\details{
See \code{\link{maNormMain}} for details and also more general procedures.
}
\value{
 \item{mnorm}{An object of class \code{"\link{marrayNorm}"}, containing the normalized intensity data.}
}


\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.\cr \cr

Y. H. Yang, S. Dudoit, P. Luu, and T. P. Speed (2001). Normalization for cDNA microarray data. In M. L. Bittner, Y. Chen, A. N. Dorsel, and E. R. Dougherty (eds), \emph{Microarrays: Optical Technologies and Informatics}, Vol. 4266 of \emph{Proceedings of SPIE}.\cr \cr

Y. H. Yang, S. Dudoit, P. Luu, D. M. Lin, V. Peng, J. Ngai, and T. P. Speed (2002). Normalization for cDNA microarray data: a robust composite method addressing single and multiple slide systematic variation. \emph{Nucleic Acids Research}, Vol. 30, No. 4.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{maNormMain}}, \code{\link{maNormScale}}.}

\examples{
# Examples use swirl dataset, for description type ? swirl
data(swirl)

# Global median normalization for swirl arrays 2 and 3
mnorm<-maNorm(swirl[,2:3], norm="median", echo=TRUE)

# Within-print-tip-group loess location normalization for swirl array 1
mnorm<-maNorm(swirl[,1], norm="p", span=0.45)

}




\keyword{smooth}% at least one, from doc/KEYWORDS
\keyword{methods}% __ONLY ONE__ keyword per line

\eof
\name{maNorm2D}
\alias{maNorm2D}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{2D spatial location normalization function}
\description{
This function is used for 2D spatial location normalization, using the robust local regression function \code{\link{loess}}.  It should be used as an argument to the main normalization function \code{\link{maNormMain}}.
}

\usage{
maNorm2D(x="maSpotRow", y="maSpotCol", z="maM", g="maPrintTip", w=NULL,
subset=TRUE, span=0.4, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{Name of accessor method for spot row coordinates, usually \code{maSpotRow}.}
  \item{y}{Name of accessor method for spot column coordinates, usually \code{maSpotCol}.}
  \item{z}{Name of accessor method for spot statistics, usually the log-ratio \code{maM}.}
  \item{g}{Name of accessor method for print-tip-group indices, usually \code{maPrintTip}.}
  \item{w}{An optional numeric vector of weights.}
  \item{subset}{A "logical" or "numeric" vector indicating the subset of points used to
  compute the fits. }
\item{span}{The argument \code{span} which controls the degree of
  smoothing in the  \code{\link{loess}} function.}
  \item{...}{Misc arguments}
}

\details{
The spot statistic named in \code{z} is regressed on spot row and column coordinates, separately within print-tip-group, using the \code{\link{loess}} function.
}

\value{
 A function with bindings for the above arguments. This latter function takes as argument an object of class \code{"\link{marrayRaw}"} (or possibly \code{"\link{marrayNorm}"}), and returns a vector of fitted values to be substracted from the raw log-ratios. It calls the function \code{\link{ma2D}}, which is not specific to microarray objects.
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}


\seealso{\code{\link{maNormMain}}, \code{\link{ma2D}}, \code{\link{loess}}.}

\examples{
# See examples for maNormMain.

}
\keyword{smooth}% at least one, from doc/KEYWORDS


\eof
\name{maNormLoess}
\alias{maNormLoess}
\title{Intensity dependent location normalization function}
\description{
This function is used for intensity dependent location normalization, using the robust local regression function \code{\link{loess}}.  It should be used as an argument to the main normalization function \code{\link{maNormMain}}.
}
\usage{
maNormLoess(x="maA", y="maM", z="maPrintTip", w=NULL, subset=TRUE,
span=0.4, ...)
}
\arguments{
  \item{x}{Name of accessor method for spot statistics, usually \code{maA}.}
  \item{y}{Name of accessor method for spot statistics, usually \code{maM}.}
  \item{z}{Name of accessor method for spot statistic used to stratify the data, usually a layout parameter, e.g. \code{maPrintTip} or \code{maPlate}. If \code{z} is not a character, e.g. NULL,  the data are not stratified.}
  \item{w}{An optional numeric vector of weights.}
  \item{subset}{A "logical" or "numeric" vector indicating the subset of points used to compute the fits.}
  \item{span}{The argument \code{span} which controls the degree of
    smoothing in the  \code{\link{loess}} function.}
  \item{...}{Misc arguments}
}

\value{
 A function with bindings for the above arguments. This latter function takes as argument an object of class \code{"\link{marrayRaw}"}  (or possibly \code{"\link{marrayNorm}"}), and returns a vector of fitted values to be substracted from the raw log-ratios. It calls the function \code{\link{maLoess}}, which is not specific to microarray objects.
}

\references{
S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.\cr \cr

Y. H. Yang, S. Dudoit, P. Luu, and T. P. Speed (2001). Normalization for cDNA microarray data. In M. L. Bittner, Y. Chen, A. N. Dorsel, and E. R. Dougherty (eds), \emph{Microarrays: Optical Technologies and Informatics}, Vol. 4266 of \emph{Proceedings of SPIE}.
\cr \cr

Y. H. Yang, S. Dudoit, P. Luu, D. M. Lin, V. Peng, J. Ngai, and T. P. Speed (2002). Normalization for cDNA microarray data: a robust composite method addressing single and multiple slide systematic variation. \emph{Nucleic Acids Research}, Vol. 30, No. 4.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}


\seealso{\code{\link{maNormMain}}, \code{\link{maLoess}}, \code{\link{loess}}.}

\examples{
# See examples for maNormMain.

}

\keyword{smooth}


\eof
\name{maNormMAD}
\alias{maNormMAD}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{MAD scale normalization function}
\description{
This function is used for scale normalization using the median absolute deviation (MAD) of intensity log-ratios for a group of spots. It can be used for within or between array normalization. The function should be used as an argument to the main normalization function \code{\link{maNormMain}}.
}

\usage{
maNormMAD(x=NULL, y="maM", geo=TRUE, subset=TRUE)
}
%- maybe also `usage' for other objects documented here.

\arguments{
  \item{x}{Name of accessor function for spot statistic used to stratify the data, usually a layout parameter, e.g. \code{maPrintTip} or \code{maPlate}. If \code{x} is not a character, e.g. NULL, the data are not stratified.}
  \item{y}{Name of accessor function for spot statistics, usually
  \code{maM}.}
  \item{geo}{If \code{TRUE}, the MAD of each group is divided by the
  geometric mean of the MADs across groups (cf. Yang et al. (2002)). This allows observations to
  retain their original units.}
  \item{subset}{A "logical" or "numeric" vector indicating the subset of points used to compute the scale normalization values. }
}

\value{
A function with bindings for the above arguments. This latter function takes as argument an object of class \code{"\link{marrayRaw}"}  (or possibly \code{"\link{marrayNorm}"}), and returns a vector of values used to scale the location normalized log-ratios. It calls the function \code{\link{maMAD}}, which is not specific to microarray objects.
}


\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.\cr \cr

Y. H. Yang, S. Dudoit, P. Luu, and T. P. Speed (2001). Normalization for cDNA microarray data. In M. L. Bittner, Y. Chen, A. N. Dorsel, and E. R. Dougherty (eds), \emph{Microarrays: Optical Technologies and Informatics}, Vol. 4266 of \emph{Proceedings of SPIE}.
\cr \cr

Y. H. Yang, S. Dudoit, P. Luu, D. M. Lin, V. Peng, J. Ngai, and T. P. Speed (2002). Normalization for cDNA microarray data: a robust composite method addressing single and multiple slide systematic variation. \emph{Nucleic Acids Research}, Vol. 30, No. 4.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}


\seealso{\code{\link{maNormMain}}, \code{\link{maMAD}}, \code{\link{mad}}.}

\examples{
# See examples for maNormMain.

}

\keyword{univar}% at least one, from doc/KEYWORDS
\keyword{robust}% __ONLY ONE__ keyword per line

\eof
\name{maNormMain}
\alias{maNormMain}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Main function for location and scale normalization of cDNA microarray data}
\description{
This is the main function for location and scale normalization of cDNA microarray data. Normalization is performed for a batch of arrays using location and scale normalization procedures specified by the lists of functions \code{f.loc} and \code{f.scale}. Typically, only one function is given in each list, otherwise composite normalization is performed using the weights computed by the functions \code{a.loc} and \code{a.scale}. The function operates on an object of class \code{"\link{marrayRaw}"} (or possibly \code{"\link{marrayNorm}"}, if normalization is performed in several steps) and returns an object of class \code{"\link{marrayNorm}"}. Simple wrapper functions are provided by \code{\link{maNorm}} and \code{\link{maNormScale}}.
}

\usage{
maNormMain(mbatch, f.loc=list(maNormLoess()), f.scale=NULL,
a.loc=maCompNormEq(), a.scale=maCompNormEq(), Mloc=TRUE, Mscale=TRUE, echo=FALSE)
}
%- maybe also `usage' for other objects documented here.

\arguments{
  \item{mbatch}{An object of class \code{"\link{marrayRaw}"}, containing intensity data for the batch of arrays to be normalized. An object of class \code{"\link{marrayNorm}"} may also be passed if normalization is performed in several steps.}
  \item{f.loc}{A list of location normalization functions, e.g., \code{\link{maNormLoess}}, \code{\link{maNormMed}}, or \code{\link{maNorm2D}}.}
  \item{f.scale}{A list of scale normalization functions, .e.g, \code{\link{maNormMAD}}.}
  \item{a.loc}{For composite normalization, a function for computing the weights used in combining several location normalization functions, e.g., \code{\link{maCompNormA}}.}
  \item{a.scale}{For composite normalization, a function for computing
  the weights used in combining several scale normalization functions.}
  \item{Mloc}{If \code{TRUE}, the location normalization values are stored in the slot \code{maMloc} of the object of class \code{"\link{marrayNorm}"} returned by the function, if \code{FALSE}, these values are not retained.}
  \item{Mscale}{If \code{TRUE}, the scale normalization values are stored in the slot \code{maMscale} of the object of class \code{"\link{marrayNorm}"} returned by the function, if \code{FALSE}, these values are not retained.}
  \item{echo}{If \code{TRUE}, the index of the array currently being
  normalized is printed.}
}

\details{
When both location and scale normalization functions (\code{f.loc} and \code{f.scale}) are passed, location normalization is performed before scale normalization. That is, scale values are computed for the location normalized log-rations. The same results could be obtained by two applications of the function \code{maNormMain}, first with only the location normalization function and \code{f.scale=NULL}, and second with only the scale normalization function and \code{f.loc=NULL}.
}

\value{
  \item{mnorm}{An object of class \code{"\link{marrayNorm}"}, containing the normalized intensity data.}
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.\cr \cr

Y. H. Yang, S. Dudoit, P. Luu, and T. P. Speed (2001). Normalization for cDNA microarray data. In M. L. Bittner, Y. Chen, A. N. Dorsel, and E. R. Dougherty (eds), \emph{Microarrays: Optical Technologies and Informatics}, Vol. 4266 of \emph{Proceedings of SPIE}.\cr \cr

Y. H. Yang, S. Dudoit, P. Luu, D. M. Lin, V. Peng, J. Ngai, and T. P. Speed (2002). Normalization for cDNA microarray data: a robust composite method addressing single and multiple slide systematic variation. \emph{Nucleic Acids Research}, Vol. 30, No. 4.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}


\seealso{\code{\link{maNorm}}, \code{\link{maNormScale}}, \code{\link{maNormLoess}}, \code{\link{maLoess}},
\code{\link{maNormMAD}}, \code{\link{maMAD}},
\code{\link{maNormMed}}, \code{\link{maMed}},
\code{\link{maNorm2D}}, \code{\link{ma2D}},
\code{\link{maCompNormA}}, \code{\link{maCompNormEq}}.}

\examples{
# Examples use swirl dataset, for description type ? swirl
data(swirl)

# Within-print-tip-group loess location normalization of batch swirl
# - Default normalization
swirl.norm<-maNormMain(swirl)

boxplot(swirl.norm)
boxplot(swirl.norm[,3])
plot(swirl.norm[,3])

# Global median normalization for arrays 81 and 82
swirl.norm <- maNormMain(swirl[,1:2], f.loc = list(maNormMed(x=NULL,y="maM")))

# Global loess normalization for array 81
swirl.norm <- maNormMain(swirl[,1], f.loc = list(maNormLoess(x="maA",y="maM",z=NULL)))

# Composite normalization as in Yang et al. (2002)
# No MSP controls are available here, so all spots are used for illustration
# purposes
swirl.norm <- maNormMain(swirl[,1], f.loc = list(maNormLoess(x="maA",y="maM",z=NULL),maNormLoess(x="maA",y="maM",z="maPrintTip")), a.loc=maCompNormA())
}
\keyword{smooth}% at least one, from doc/KEYWORDS
\keyword{methods}% __ONLY ONE__ keyword per line

\eof
\name{maNormMed}
\alias{maNormMed}
\title{Median location normalization function}

\description{
This function is used for location normalization using the median of 
intensity log-ratios for a group of spots. The function should be used 
as an argument to the main normalization function \code{\link{maNormMain}}.
}

\usage{
maNormMed(x=NULL, y="maM", subset=TRUE)
}

\arguments{
  \item{x}{Name of accessor method for spot statistic used to stratify the 
data, usually a layout parameter, e.g. \code{maPrintTip} or \code{maPlate}. 
If \code{x} is not a character, e.g. NULL, the data are not stratified.}
  \item{y}{Name of accessor method for spot statistics, usually \code{maM}.}
  \item{subset}{A "logical" or "numeric" vector indicating the subset of points   used to compute the location normalization values. }
}


\value{
A function with bindings for the above arguments. This latter function takes as 
argument an object of class \code{"\link{marrayRaw}"} (or possibly 
\code{"\link{marrayNorm}"}), and returns a vector of fitted values to be 
subtracted from the raw log-ratios. It calls the function \code{\link{maMed}}, 
which is not specific to microarray objects.
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for 
exploratory analysis and normalization of cDNA microarray data. 
In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, 
\emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, 
New York.\cr \cr

Y. H. Yang, S. Dudoit, P. Luu, and T. P. Speed (2001). Normalization for cDNA 
microarray data. In M. L. Bittner, Y. Chen, A. N. Dorsel, and E. R. Dougherty 
(eds), \emph{Microarrays: Optical Technologies and Informatics}, 
Vol. 4266 of \emph{Proceedings of SPIE}.\cr \cr

Y. H. Yang, S. Dudoit, P. Luu, D. M. Lin, V. Peng, J. Ngai, and T. P. Speed 
(2002). Normalization for cDNA microarray data: a robust composite method 
addressing single and multiple slide systematic variation. 
\emph{Nucleic Acids Research}, Vol. 30, No. 4.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}


\seealso{\code{\link{maNormMain}}, \code{\link{maMed}}, \code{\link{median}}.}

\examples{
# See examples for maNormMain.

}
\keyword{univar}
\keyword{robust}

\eof
\name{maNormScale}
\alias{maNormScale}
\title{Simple scale normalization function}
\description{
This function is a simple wrapper function around the main normalization
function \code{\link{maNormMain}}. It allows the user to choose from a
set of two basic scale normalization procedures. The function operates
on an object of class \code{"\link{marrayRaw}"} (or possibly
\code{"\link{marrayNorm}"}, if normalization is performed in several
steps) and returns an object of class \code{"\link{marrayNorm}"}. This
function can be used to conormalize a batch of arrays 
(\code{norm="globalMAD"} option).
}
\usage{
maNormScale(mbatch, norm=c("globalMAD", "printTipMAD"), subset=TRUE, geo=TRUE,  Mscale=TRUE, echo=FALSE)
}

\arguments{
  \item{mbatch}{An object of class \code{"\link{marrayRaw}"}, containing 
intensity data for the batch of arrays to be normalized. An object of class 
\code{\link{marrayNorm}} may also be passed if normalization is performed 
in several steps.}
\item{norm}{A character string specifying the normalization
  procedures: 
\describe{
\item{globalMAD}{for global scale
  normalization using the median absolute deviation (MAD), this allows between 
slide scale normalization}
\item{printTipMAD}{for within-print-tip-group scale normalization
  using the median absolute deviation (MAD).}
This argument can be
  specified using the first letter of each method.}}
  \item{subset}{A "logical" or "numeric" vector indicating the subset of points used to
  compute the  normalization values.}
 \item{geo}{If \code{TRUE}, the MAD of each group is divided by the
  geometric mean of the MADs across groups (cf. Yang et al. (2002)). This allows observations to
  retain their original units.}
  \item{Mscale}{If \code{TRUE}, the scale normalization values are stored in the slot \code{maMscale} of the object of class \code{"\link{marrayNorm}"} returned by the function, if \code{FALSE}, these values are not retained.}
\item{echo}{If \code{TRUE}, the index of the array currently being
  normalized is printed.}
}
\details{
See \code{\link{maNormMain}} for details and more general procedures.
}
\value{
\item{mnorm}{An object of class \code{"\link{marrayNorm}"}, containing the normalized intensity data.}
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.\cr \cr

Y. H. Yang, S. Dudoit, P. Luu, and T. P. Speed (2001). Normalization for cDNA microarray data. In M. L. Bittner, Y. Chen, A. N. Dorsel, and E. R. Dougherty (eds), \emph{Microarrays: Optical Technologies and Informatics}, Vol. 4266 of \emph{Proceedings of SPIE}.\cr \cr

Y. H. Yang, S. Dudoit, P. Luu, D. M. Lin, V. Peng, J. Ngai, and T. P. Speed (2002). Normalization for cDNA microarray data: a robust composite method addressing single and multiple slide systematic variation. \emph{Nucleic Acids Research}, Vol. 30, No. 4.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{maNormMain}}, \code{\link{maNorm}}.}


\examples{
# Examples use swirl dataset, for description type ? swirl
data(swirl)

# Global median normalization followed by global MAD normalization for 
# only arrays 2 and 3 in the batch swirl

mnorm1<-maNorm(swirl[,2:3], norm="m")
mnorm2<-maNormScale(mnorm1, norm="g")

}

\keyword{smooth}% at least one, from doc/KEYWORDS
\keyword{methods}% __ONLY ONE__ keyword per line

\eof
\name{maNum2Logic}
\alias{maNum2Logic}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Convert a numeric vector of indices to a logical vector}
\description{
This function converts a numeric vector of indices to a logical
vector. It is used for subsetting purposes.
}
\usage{
maNum2Logic(n=length(subset), subset=TRUE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{n}{the length of the logical vector to be produced.}
  \item{subset}{a numeric vector of indices. A logical vector may also
  be supplied, in which case it is also the value of the function.}
}

\value{
 a logical vector.
}


\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}


\examples{
maNum2Logic(10, 1:3)
}
\keyword{manip}% at least one, from doc/KEYWORDS

\eof
\name{maPalette}
\alias{maPalette}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Microarray color palette}
\description{
This function returns a vector of color names corresponding to a range of colors specified in the arguments.  
}
\usage{
maPalette(low = "white", high = c("green", "red"), mid=NULL, k =50)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{low}{Color for the lower end of the color palette, specified using any of the three kinds of R colors, i.e., either a color name (an element of \code{colors}), a hexadecimal string of the form \code{"#rrggbb"}, or an integer \code{i} meaning \code{palette()[i]}.}
  \item{high}{Color for the upper end of the color palette, specified
  using any of the three kinds of R colors, i.e., either a color name
  (an element of \code{colors}), a hexadecimal string of the form
  \code{"#rrggbb"}, or an integer \code{i} meaning \code{palette()[i]}.}
 \item{mid}{Color for the middle portion of the color palette, specified using any of the three kinds of R colors, i.e., either a color name (an element of \code{colors}), a hexadecimal string of the form \code{"#rrggbb"}, or an integer \code{i} meaning \code{palette()[i]}.}
  \item{k}{Number of colors in the palette.}
}

\value{
  A "character" vector of color names. This can be used to create a user-defined color palette for subsequent graphics by \code{palette}, in a \code{col=} specification in graphics functions, or in \code{\link{par}}.
}

\references{
S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}, Yee Hwa (Jean) Yang.}

\seealso{\code{\link{image}}, \code{\link{maColorBar}}, \code{\link{maImage}}, \code{\link{maImage.func}}.}

\examples{
par(mfrow=c(1,4))
pal <- maPalette(low="red", high="green")
maColorBar(seq(-2,2, 0.2), col=pal, horizontal=FALSE, k=21)
pal <- maPalette(low="red", high="green", mid="yellow")
maColorBar(seq(-2,2, 0.2), col=pal, horizontal=FALSE, k=21)
pal <- maPalette()
maColorBar(seq(-2,2, 0.2), col=pal, horizontal=FALSE, k=21)
pal <- maPalette(low="purple", high="purple",mid="white")
maColorBar(seq(-2,2, 0.2), col=pal, horizontal=FALSE, k=21)
}
\keyword{color}% at least one, from doc/KEYWORDS


\eof
\name{maPlot}
\alias{maPlot}


%- Also NEED an `\alias' for EACH other topic documented here.
\title{Scatter-plots for cDNA microarray spot statistics}
\description{The function \code{maPlot} produces scatter-plots of
  microarray spot statistics for the classes \code{"\link{marrayRaw}"} and \code{"\link{marrayNorm}"}. It also allows the user to highlight and annotate subsets of points on the plot, and display fitted curves from robust local regression or other smoothing procedures.
}

\usage{
maPlot(m, x="maA", y="maM", z="maPrintTip", lines.func, text.func, legend.func, ...)
}
%- maybe also `usage' for other objects documented here.

\arguments{
  \item{m}{Microarray object of class \code{"\link{marrayRaw}"} and \code{"\link{marrayNorm}"}.}
  \item{x}{Name of accessor function for the abscissa spot statistic, typically a slot name for the microarray object \code{m}, such as \code{maA}.}
  \item{y}{Name of accessor function for the ordinate spot statistic, typically a slot name for the microarray object \code{m}, such as \code{maM}.}
  \item{z}{Name of accessor method for the spot statistic used to stratify the data, typically a slot name for the microarray layout object (see \code{"\link{marrayLayout}"}) such as \code{maPlate} or a method such as \code{maPrintTip}. If \code{z} is NULL, the data are not stratified.}
  \item{lines.func}{Function for computing and plotting smoothed fits of \code{y} as a function of \code{x}, separately within values of \code{z}, e.g. \code{\link{maLoessLines}}. If \code{lines.func} is NULL, no fitting is performed.}
  \item{text.func}{Function for highlighting a subset of points, e.g., \code{\link{maText}}. If \code{text.func} is NULL, no points are highlighted.}
  \item{legend.func}{Function for adding a legend to the plot, e.g. \code{\link{maLegendLines}}. If \code{legend.func} is NULL, there is no legend.}
  \item{\dots}{Optional graphical parameters, see \code{\link{par}}.}
}

\details{
This function calls the general function \code{\link{maPlot.func}}, which is not specific to microarray data. If there are more than one array in the batch, the plot is done for the first array, by default. Default graphical parameters are chosen for convenience using the function \code{\link{maDefaultPar}} (e.g. color palette,  axis labels,  plot title) but the user has the option to overwrite these parameters at any point.}
}

\references{
S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{maPlot.func}}, \code{\link{maDefaultPar}}, \code{\link{maLoessLines}}, \code{\link{maLegendLines}}, \code{\link{maText}}, \code{\link{plot}}, \code{\link{lowess}}, \code{\link{loess}}, \code{\link{legend}}.}

\examples{

# To see the demo type demo(marrayPlots)

# Examples use swirl dataset, for description type ? swirl
data(swirl)

# - Default arguments
maPlot(swirl)

# Lowess fit using all spots
maPlot(swirl, z=NULL, legend.func=NULL)

# Loess fit using all spots
maPlot(swirl, z=NULL, legend.func=maLegendLines(legend="All spots",col="green"), lines.func=maLoessLines(loess.args=list(span=0.3),col="green"))


# Pre-normalization MA-plot for the Swirl 81 array, with the lowess fits for 
# individual grid columns and 1\% tails of M highlighted
defs <- maDefaultPar(swirl[, 1], x = "maA", y = "maM", z = "maGridCol")
legend.func <- do.call("maLegendLines", defs$def.legend)
lines.func <- do.call("maLowessLines", c(list(TRUE, f = 0.3), defs$def.lines))
text.func<-maText(subset=maTop(maM(swirl)[,1],h=0.01,l=0.01), labels="o", col="violet")
maPlot(swirl[, 1], x = "maA", y = "maM", z = "maGridCol", lines.func=lines.func, text.func = text.func, legend.func=legend.func, main = "Swirl array 81: pre-normalization MA-plot")

}

\keyword{hplot}% at least one, from doc/KEYWORDS


\eof
\name{maPlot.func}
\alias{maPlot.func}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Scatter-plots with fitted curves and text}
\description{
 This function produces scatter-plots of \code{x} vs. \code{y}. It also allows the user to highlight and annotate subsets of points on the plot, and display fitted curves from robust local regression or other smoothing procedures.
}

\usage{
maPlot.func(x, y, z, 
	lines.func = maLowessLines(subset = TRUE, f = 0.3, col = 1:length(unique(z)), lty = 1, lwd = 2.5),
    	text.func = maText(), 
	legend.func = maLegendLines(legend = as.character(unique(z)), col = 1:length(unique(z)), lty = 1, lwd = 2.5, ncol = 1),
	 ...)
}
%- maybe also `usage' for other objects documented here.

\arguments{
  \item{x}{A "numeric" vector for the abscissa.}
  \item{y}{A "numeric" vector for the ordinates.}
  \item{z}{A vector of statistic used to stratify the data, smoothed curves are fitted separately within values of \code{z}}
  \item{lines.func}{A function for computing and plotting smoothed fits of \code{y} as a function of \code{x}, separately within values of \code{z}, e.g. \code{\link{maLoessLines}}.}
  \item{text.func}{A function for highlighting a subset of points, e.g., \code{maText}.}
  \item{legend.func}{A function for adding a legend to the plot, e.g. \code{\link{maLegendLines}}.}
  \item{\dots}{Optional graphical parameters, see \code{\link{par}}.}
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}


\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{maPlot}}, \code{\link{maLoessLines}}, \code{\link{maLegendLines}}, \code{\link{maText}}, \code{\link{plot}}, \code{\link{lowess}}, \code{\link{loess}}, \code{\link{legend}}.}

\examples{
# See examples for maPlot.


}
\keyword{hplot}% at least one, from doc/KEYWORDS


\eof
\name{maQualityMain}
\alias{maQualityMain}
\alias{maQuality}

\title{Calculate basic array wide statistics}

\description{
  This is based on GenePix output and it calculates summary statistics
  of various signal to noise ratios.
}
\usage{
maQualityMain(mraw, path=".", fname, output = FALSE)
maQuality(mraw, path=".")
}

\arguments{
  \item{mraw}{Microarray object of class `"marrayRaw"'.
    By default, plots are done for the first array in the batch.}
  \item{path}{a character string representing the data directory. By
    default this is set to the current working directory (".").
    In the case where "fname" contains the full path name, path
    should be set to NULL. }
  \item{fname}{a vector of character strings representing the name of
    the output file. This is only used when "output" is TRUE.}
  \item{output}{if TRUE, write the results to a file.}
}

\value{
  Produce a lists of various summary statistics
}

\author{Jean Yee Hwa yang}

\seealso{\code{\link{summary}}, \code{\link{maGenControls}}}

\examples{
##   data(swirl)
##   datadir <- system.file("swirldata", package="marray")
##   maControls(swirl) <- maGenControls(maGnames(swirl), id="Name")
##   maQualityMain(swirl[,1], path=datadir)
}

\keyword{manip}% at least one, from doc/KEYWORDS


\eof
\name{maSelectGnames}
\alias{maSelectGnames}

\title{Select genes according to the values of a few different statistics}
\description{
  Select genes by considering the \code{\link{union}} or
  \code{\link{intersect}} of multiple statistics.
}

\usage{
maSelectGnames(statdata, crit1 = 50, crit2 = crit1, sub = TRUE, selectstat, operate = c("intersect", "union"))
}

\arguments{
  \item{statdata}{A numerical matrix where the rows corresponds to genes
    and the columns corresponds to various statistics corresponding to a
    particular gene.}
  \item{crit1}{The number of points to be selected.
   If crit1 < 1, the crit1*100\% spots with the smallest M values
   will be selected. If crit1 >= 1, the crit spots  with the
   smallest M values are selected.}  
 \item{crit2}{Similar to "crit1".   If crit2 < 1, the crit2*100\%
   spots with the largest M values will be selected. If crit2 >= 1,
   the crit2 spots with the largest M values are selected.}  
 \item{sub}{A "logical" or "numeric" vector indicating the subset of
   genes to be consider.}
 \item{selectstat}{A integer value indicating the statistics where the
   final ranking is based on.}
 \item{operate}{The operation used to combined different rankings}
}
\details{
  This functions calls \code{\link{stat.gnames}} to select say the 100
  most extreme genes from various statistics and combined the different
  gene lists by either union or intersection.
}
\value{
  A vector of numeric values.
}

\author{Jean Yee Hwa Yang}

\seealso{\code{\link{stat.gnames}}, \code{\link{order}}}

\examples{
X <- matrix(rnorm(1000), 100,10)
Xstat <- maStat(X, funNames=c("meanFun", "ttestFun"))
maSelectGnames(Xstat[,1:2], crit1=50)
}
\keyword{manip}% at least one, from doc/KEYWORDS


\eof
\name{maStat}
\alias{maStat}
\alias{widget.Stat}

\title{A function to calculates various statistics}
\description{
   \code{maStat} calculates various statistcs for genes in the array
   `expr' using the functions in `funNames'.It returns an matrix of
   values.
 }
\usage{
maStat(expr, funNames, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{expr}{An object of class "matrix", "dataframe",
    "marrayRaw", "marrayNorm" or "exprSet" that contains gene expression
    data.}    
  \item{funNames}{Either a character strings containning the names of
    functions or a list of statistics functions.}
  \item{\dots}{further arguments to the functions is 'funNames'}.
}

\value{
  A matrix of values.
}

\author{Yee Hwa (Jean) Yang}

\seealso{\code{\link{meanFun}}, \code{\link{ttestFun}}}

\examples{
 X <- matrix(rnorm(1000), 100,10)
 maStat(X, funNames=c("meanFun", "ttestFun"))
 maStat(X, funNames=list(mean=meanFun(), t=ttestFun()))
}
\keyword{htest}% at least one, from doc/KEYWORDS


\eof
\name{maText}
\alias{maText}


%- Also NEED an `\alias' for EACH other topic documented here.
\title{Highlight points on a plot}
\description{
This function may be used to highlight a subset of points on an existing
plot, such as a plot produced by \code{\link{plot}},
\code{\link{maPlot}}, or \code{\link{maPlot.func}}. 
}

\usage{
maText(subset=NULL, labels=as.character(1:length(subset)), ...)
}
%- maybe also `usage' for other objects documented here.

\arguments{
  \item{subset}{A "logical" or "numeric" vector indicating the subset of points to
  highlight.}
\item{labels}{One or more character strings or expressions specifying the text to be written.}
  \item{\dots}{Optional graphical parameters, see \code{\link{par}}.}
}

\value{
A function with bindings for \code{subset}, \code{labels}, and \code{\ldots}. This latter function takes as arguments \code{x} and \code{y}, the absissa and ordinates of points on the plot.
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{text}}, \code{\link{maPlot}}, \code{\link{maPlot.func}}.}

\examples{
# See examples for maPlot.


}
\keyword{aplot}% at least one, from doc/KEYWORDS


\eof
\name{maTop}
\alias{maTop}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Identify extreme values}
\description{
This function determines which values in a numeric vector are above or below user supplied cut-offs.
}
\usage{
maTop(x, h=1, l=1)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{A "numeric" vector.}
  \item{h}{A "numeric", upper cut-off.}
  \item{l}{A "numeric", lower cut-off.}
}

\value{
A "logical" vector indicating which entries are above or below the cut-offs.}
}

\references{
S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
 }


\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{maPlot}}, \code{\link{maImage}}, \code{\link{quantile}}.}

\examples{
# See examples for maPlot.

}
\keyword{misc}% at least one, from doc/KEYWORDS

\eof
\name{maTwoSamples}
\alias{maTwoSamples}
\alias{widget.TwoSamples}

\title{Changing signs for two sample analysis}
\description{
  Taking target file information and flip the dye swaps experiments.
}
\usage{
maTwoSamples(targetfile, normdata, Trt, Ctl, targetID = "TargetName", slidesID = "Slides", dyesID = "Dyes", RedID = 5, path = ".", output = TRUE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{targetfile}{A data.frame containing target samples information.}
  \item{normdata}{A R object of class 'marrayNorm'}
  \item{Trt}{A character string representing "treatment" sample.}
  \item{Ctl}{A character string representing "controls" sample.}
  \item{targetID}{A character string representing the column name
    in 'targetfile' containing target samples information.}
  \item{slidesID}{A character string representing the column name
    in 'targetfile' containing the slide label.}
  \item{dyesID}{A character string representing the column name
    in 'targetfile' containing dye labeled information.}
  \item{RedID}{The character use to represent the Cy5 dye.}
  \item{path}{A character string representing the data directory. By
    default this is set to the current working directory ("."). }
  \item{output}{Save and tab delimited file}
}

\value{
  An objects of 'marrayNorm' with the dye assignment adjusted.
}

\author{Yee Hwa (Jean) Yang}

\keyword{manip}% at least one, from doc/KEYWORDS

\eof
\name{mapGeneInfo}
\alias{mapGeneInfo}
\alias{widget.mapGeneInfo}
\alias{URLstring}
\alias{SFGL}
\alias{UCBFGL}

\title{Creating URL strings for external database links }

\description{
  These functions are used with \code{\link{htmlPage}}.
  The function \code{mapGeneInfo}, takes all the arguments and generate
  a character matrix of two columns.  The first columns representing the
  name of the argument and the second columns represents the value of an
  argument. 
  The function \code{widget.mapGeneInfo} allows the user to enter this
  information  interactively. 
}

\usage{
mapGeneInfo(widget = FALSE, Gnames, Name = "pubmed", ID =
                 "genbank", ACC = "SMDacc", ...)
widget.mapGeneInfo(Gnames) 
}

%- maybe also `usage' for other objects documented here.
\arguments{
  \item{widget}{A logical value specifying if widgets should be used.}
  \item{Name}{The external database for spot description, E.g. "pubmed".}
  \item{ID}{The external database for spot ID, E.g. "operon", "Riken", "locuslink".}
  \item{ACC}{The external database for gene accession number, E.g. "genebank".}
  \item{Gnames}{An object of class \code{matrix}, \code{data.frame} or \code{marrayInfo}
    which contains  description of spotted probe sequences.}
  \item{\dots}{Other column names}
}
\details{
  The function \code{mapGeneInfo} generates a character matrix with the
  first column representing the column headings of "Gnames" and the
  second column representing the corresponding names in the list
  \code{URLstring}.  For example, if a particular column in "Gnames"
  with column names "ID" contains genebank accession number, then the
  function \code{mapGeneInfo} generates a row containing "ID" in the
  first column and "genbank" in the second.  Examples are \code{SFGL}
  and \code{UCBFGL}.\cr
  
  \code{URLstring} is a list contains the URL to various external
  database, E.g. operon, Riken, genbank. \cr
  The current choices are: 
  "pubmed",  "locuslink",  "riken",   "SMDclid",    "SMDacc",    "operonh2",  "operonh1" ,
  "operonm2",  "operonm1" and  "genbank" .  
  "SMDclid" and "SMDacc" are links to Stanford Microarray Databases.
 }

 \author{Jean Yee Hwa Yang}

 \examples{
 mapGeneInfo(ID="genebank", ll="locuslink")
 mapGeneInfo(ID="locuslink", Sample.ID="riken")
}

\keyword{manip}% at least one, from doc/KEYWORDS



\eof
\name{Internal functions}
\alias{read.fname}
\alias{stat.gene.cluster}
\alias{plot.gene.cluster}
\alias{tablegen}
\alias{gsubAnchor}

\title{Internal marray functions}
 
\description{Internal marray functions}
 
\details{These are not to be called by the user.}

\keyword{methods}

\eof
\name{marrayInfo-class}
\docType{class}

\alias{marrayInfo-class}
\alias{marrayInfo}

\alias{maLabels}
\alias{maLabels<-}
\alias{maInfo}
\alias{maInfo<-}
\alias{maNotes}
\alias{maNotes<-}

\alias{[,marrayInfo-method}
\alias{maGnames<-,marrayRaw,marrayInfo-method}
\alias{maGnames<-,marrayNorm,marrayInfo-method}
\alias{maGnames<-,marraySpots,marrayInfo-method}
\alias{maInfo<-,marrayInfo,data.frame-method}
\alias{maInfo,marrayInfo-method}
\alias{maLabels<-,marrayInfo,character-method}
\alias{maLabels<-,marrayInfo,numeric-method}
\alias{maLabels,marrayInfo-method}
\alias{maNotes<-,marrayInfo,character-method}
\alias{maNotes,marrayInfo-method}
\alias{maTargets<-,marrayRaw,marrayInfo-method}
\alias{maTargets<-,marrayNorm,marrayInfo-method}
\alias{print,marrayInfo-method}
\alias{rbind,marrayInfo-method}
\alias{summary,marrayInfo-method}

\title{Class "marrayInfo", description of target samples or spotted probe sequences}

\description{This class is used to store information on target samples hybridized to a batch of arrays or probe sequences spotted onto these arrays. It is not specific to the microarray context.}

\section{Objects from the Class}{
Objects can be created by calls of the form \code{  new('marrayInfo',}\cr
\code{    maLabels = ...., # Object of class character}\cr
\code{    maInfo   = ...., # Object of class data.frame}\cr
\code{    maNotes  = ...., # Object of class character}\cr
\code{  )}
}

\section{Slots}{
  \describe{
    \item{\code{maLabels}:}{Object of class \code{"character"}, vector of spot or array labels.}
    \item{\code{maInfo}:}{Object of class \code{"data.frame"}. If the object of class \code{"marrayInfo"} is used to describe probe sequences, rows of \code{maInfo} correspond to spots and columns to various gene identifiers and annotations. 
If the object of class \code{"marrayInfo"} is used to describe target samples hybridized to the arrays, rows of \code{maInfo} correspond to arrays and columns to various descriptions of the hybridizations, e.g., names of Cy3 and Cy5 samples, labels for the arrays etc.}
    \item{\code{maNotes}:}{Object of class \code{"character"}, any notes on the target samples or spotted probe sequences.}
  }
}

\section{Methods}{
  \describe{
    \item{[}{\code{signature(x = "marrayInfo")}: subsetting operator for spots on the array or arrays in the batch, ensures that all slots are subset properly.}
    \item{maGnames<-}{\code{signature(object = "marrayRaw", value = "marrayInfo")}: slot assignment method.}
    \item{maGnames<-}{\code{signature(object = "marrayNorm", value = "marrayInfo")}: slot assignment method.}
    \item{maGnames<-}{\code{signature(object = "marraySpots", value = "marrayInfo")}: slot assignment method.}
    \item{maInfo}{\code{signature(object = "marrayInfo")}: slot accessor method.}
    \item{maInfo<-}{\code{signature(object = "marrayInfo", value = "data.frame")}: slot assignment method.}
    \item{maLabels}{\code{signature(object = "marrayInfo")}: slot accessor method.}
    \item{maLabels<-}{\code{signature(object = "marrayInfo", value = "character")}: slot assignment method.}
    \item{maLabels<-}{\code{signature(object = "marrayInfo", value = "numeric")}: slot assignment method.}
    \item{maNotes}{\code{signature(object = "marrayInfo")}: slot accessor method.}
    \item{maNotes<-}{\code{signature(object = "marrayInfo", value = "character")}: slot assignment method.}
    \item{maTargets<-}{\code{signature(object = "marrayRaw", value = "marrayInfo")}: slot assignment method.}
    \item{maTargets<-}{\code{signature(object = "marrayNorm", value = "marrayInfo")}: slot assignment method.}
    \item{print}{\code{signature(x = "marrayInfo")}: print method for \code{"\link{marrayInfo}"} class.}
  }
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}


\author{Jean Yang and Sandrine Dudoit}

\seealso{\code{\link{marrayLayout}}, \code{\link{marrayRaw}}, \code{\link{marrayNorm}}.}

\examples{
## See marrayRaw
}
\keyword{classes}

\eof
\name{marrayLayout-class}
\docType{class}

\alias{marrayLayout-class}
\alias{marrayLayout}

\alias{maNgr}
\alias{maNgr<-}
\alias{maNgc}
\alias{maNgc<-}
\alias{maNsr}
\alias{maNsr<-}
\alias{maNsc}
\alias{maNsc<-}
\alias{maNspots}
\alias{maNspots<-}
\alias{maSub}
\alias{maSub<-}
\alias{maPlate}
\alias{maPlate<-}
\alias{maControls}
\alias{maControls<-}

\alias{[,marrayLayout-method}
\alias{maControls<-,marrayLayout-method}
\alias{maControls,marrayLayout-method}
\alias{maGridCol,marrayLayout-method}
\alias{maGridRow,marrayLayout-method}
\alias{maLayout<-,marrayRaw,marrayLayout-method}
\alias{maLayout<-,marrayNorm,marrayLayout-method}
\alias{maNgc<-,marrayLayout,numeric-method}
\alias{maNgc,marrayLayout-method}
\alias{maNgr<-,marrayLayout,numeric-method}
\alias{maNgr,marrayLayout-method}
\alias{maNotes<-,marrayLayout,character-method}
\alias{maNotes,marrayLayout-method}
\alias{maNsc<-,marrayLayout,numeric-method}
\alias{maNsc,marrayLayout-method}
\alias{maNspots<-,marrayLayout,numeric-method}
\alias{maNspots,marrayLayout-method}
\alias{maNsr<-,marrayLayout,numeric-method}
\alias{maNsr,marrayLayout-method}
\alias{maPlate<-,marrayLayout-method}
\alias{maPlate,marrayLayout-method}
\alias{maPrintTip,marrayLayout-method}
\alias{maSpotCol,marrayLayout-method}
\alias{maSpotRow,marrayLayout-method}
\alias{maSub<-,marrayLayout,logical-method}
\alias{maSub<-,marrayLayout,numeric-method}
\alias{maSub,marrayLayout-method}
\alias{print,marrayLayout-method}
\alias{summary,marrayLayout-method}

%\alias{maNotes}

\alias{maPrintTip}
\alias{maGridRow}
\alias{maGridCol}
\alias{maSpotRow}
\alias{maSpotCol}


\title{Class "marrayLayout", classes and methods for layout parameters of cDNA microarrays}
\description{This class is used to keep track of important layout
  parameters for two-color cDNA microarrays. It contains slots for: the
  total number of spotted probe sequences on the array, the dimensions
  of the spot and grid matrices, the plate origin of the probes,
  information on spotted control sequences (e.g. probe sequences which
  should have equal abundance in the two target samples, such as
  housekeeping genes). The terms \emph{print-tip-group}, \emph{grid},
  \emph{spot matrix}, and \emph{sector} are used interchangeably and
  refer to a set of spots printed using the same print-tip.} 


\section{Objects from the Class}{
Objects can be created by calls of the form \code{  new('marrayLayout',}\cr
\code{    maNgr      = ...., # Object of class numeric}\cr
\code{    maNgc      = ...., # Object of class numeric}\cr
\code{    maNsr      = ...., # Object of class numeric}\cr
\code{    maNsc      = ...., # Object of class numeric}\cr
\code{    maNspots   = ...., # Object of class numeric}\cr
\code{    maSub      = ...., # Object of class logical}\cr
\code{    maPlate    = ...., # Object of class factor}\cr
\code{    maControls = ...., # Object of class factor}\cr
\code{    maNotes    = ...., # Object of class character}\cr
\code{  )}
}

\section{Slots}{
  \describe{
    \item{\code{maNgr}:}{Object of class \code{"numeric"}, number of rows for the grid matrix.}
    \item{\code{maNgc}:}{Object of class \code{"numeric"}, number of columns for the grid matrix.}
    \item{\code{maNsr}:}{Object of class \code{"numeric"}, number of rows for the spot matrices.}
    \item{\code{maNsc}:}{Object of class \code{"numeric"}, number of columns for the spot matrices.}
    \item{\code{maNspots}:}{Object of class \code{"numeric"}, total number of spots on the array, equal to \eqn{maNgr x maNgc x maNsr x maNsc}.}
    \item{\code{maSub}:}{Object of class \code{"logical"},  indicating which spots are currently being considered.}
    \item{\code{maPlate}:}{Object of class \code{"factor"}, recording the plate origin of the spotted probe sequences.}
    \item{\code{maControls}:}{Object of class \code{"factor"}, recording the control status of the spotted probe sequences.}
    \item{\code{maNotes}:}{Object of class \code{"character"}, any notes concerning the microarray layout, e.g., printing conditions.}
  }
}

\section{Methods}{
  \describe{
    \item{[}{\code{signature(x = "marrayLayout")}: subsetting operator for spots on the array, ensures that all slots are subset properly.}
    \item{maControls<-}{\code{signature(object = "marrayLayout")}: slot assignment method.} 
    \item{maControls}{\code{signature(object = "marrayLayout")}: slot accessor method.}
    \item{maGridCol}{\code{signature(object = "marrayLayout")}: method which computes a vector of grid column coordinates for each spot.}
    \item{maGridRow}{\code{signature(object = "marrayLayout")}: method which computes a vector of grid row coordinates for each spot.}
    \item{maLayout<-}{\code{signature(object = "marrayRaw", value = "marrayLayout")}: slot assignment method.} 
    \item{maLayout<-}{\code{signature(object = "marrayNorm", value = "marrayLayout")}: slot assignment method.} 
    \item{maNgc}{\code{signature(object = "marrayLayout")}: slot accessor method.}
 \item{maNgc<-}{\code{signature(object = "marrayLayout", value = "numeric")}: slot assignment method.}
    \item{maNgr}{\code{signature(object = "marrayLayout")}: slot accessor method.}
\item{maNgr<-}{\code{signature(object = "marrayLayout", value = "numeric")}: slot assignment method.}
    \item{maNotes}{\code{signature(object = "marrayLayout")}: slot accessor method.}
    \item{maNotes<-}{\code{signature(object = "marrayLayout", value = "character")}: slot assignment method.} 
    \item{maNsc}{\code{signature(object = "marrayLayout")}: slot accessor method.}
    \item{maNsc<-}{\code{signature(object = "marrayLayout", value = "numeric")}: slot assignment method.}
    \item{maNspots}{\code{signature(object = "marrayLayout")}: slot accessor method.}
 \item{maNspots<-}{\code{signature(object = "marrayLayout", value = "numeric")}: slot assignment method.}
    \item{maNsr}{\code{signature(object = "marrayLayout")}: slot accessor method.}
\item{maNsr<-}{\code{signature(object = "marrayLayout", value = "numeric")}: slot assignment method.}
    \item{maPlate}{\code{signature(object = "marrayLayout")}: slot accessor method.}
    \item{maPlate<-}{\code{signature(object = "marrayLayout")}: slot assignment method.} 
    \item{maPrintTip}{\code{signature(object = "marrayLayout")}: method which computes a vector of print-tip-group indices for each spot.}
    \item{maSpotCol}{\code{signature(object = "marrayLayout")}: method which computes a vector of spot column coordinates for each spot.}
    \item{maSpotRow}{\code{signature(object = "marrayLayout")}: method which computes a vector of spot row coordinates for each spot.}
    \item{maSub}{\code{signature(object = "marrayLayout")}: slot accessor method.}
    \item{maSub<-}{\code{signature(object = "marrayLayout", value = "logical")}: slot assignment method.} 
    \item{maSub<-}{\code{signature(object = "marrayLayout", value = "numeric")}: slot assignment method.} 
    \item{print}{\code{signature(x = "marrayLayout")}: print method for \code{"\link{marrayLayout}"} class.}
  }
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{marrayRaw}}, \code{\link{marrayNorm}},
  \code{\link{marrayInfo}} and \code{\link{[-methods}}.} 

\examples{
## See marrayRaw
}

\keyword{classes}

\eof
\name{marrayNorm-class}
\docType{class}

\alias{marrayNorm-class}
\alias{marrayNorm}

\alias{maA}
\alias{maA<-}
\alias{maM}
\alias{maM<-}
\alias{maMloc}
\alias{maMloc<-}
\alias{maMscale}
\alias{maMscale<-}

\alias{[,marrayNorm-method}
\alias{cbind,marrayNorm-method}
\alias{coerce,marrayRaw,marrayNorm-method}
%\alias{coerce,marrayNorm,exprSet-method}
%\alias{coerce,marrayNorm,marraySpots-method}
\alias{maA<-,marrayNorm,matrix-method}
\alias{maA,marrayNorm-method}
\alias{maControls<-,marrayNorm-method}
\alias{maControls,marrayNorm-method}
%\alias{maGnames<-,marrayNorm,marrayInfo-method}
\alias{maGnames,marrayNorm-method}
\alias{maGridCol,marrayNorm-method}
\alias{maGridRow,marrayNorm-method}
%\alias{maLayout<-,marrayNorm,marrayLayout-method}
\alias{maLayout,marrayNorm-method}
\alias{maMloc<-,marrayNorm,matrix-method}
\alias{maMloc,marrayNorm-method}
\alias{maM<-,marrayNorm,matrix-method}
\alias{maM,marrayNorm-method}
\alias{maMscale<-,marrayNorm,matrix-method}
\alias{maMscale,marrayNorm-method}
\alias{maNgc<-,marrayNorm,numeric-method}
\alias{maNgc,marrayNorm-method}
\alias{maNgr<-,marrayNorm,numeric-method}
\alias{maNgr,marrayNorm-method}
\alias{maNormCall,marrayNorm-method}
\alias{maNotes<-,marrayNorm,character-method}
\alias{maNotes,marrayNorm-method}
\alias{maNsamples,marrayNorm-method}
\alias{maNsc<-,marrayNorm,numeric-method}
\alias{maNsc,marrayNorm-method}
\alias{maNspots<-,marrayNorm,numeric-method}
\alias{maNspots,marrayNorm-method}
\alias{maNsr<-,marrayNorm,numeric-method}
\alias{maNsr,marrayNorm-method}
\alias{maPlate<-,marrayNorm-method}
\alias{maPlate,marrayNorm-method}
\alias{maPrintTip,marrayNorm-method}
\alias{maSpotCol,marrayNorm-method}
\alias{maSpotRow,marrayNorm-method}
\alias{maSub<-,marrayNorm-method}
\alias{maSub,marrayNorm-method}
%\alias{maTargets<-,marrayNorm,marrayInfo-method}
\alias{maTargets,marrayNorm-method}
\alias{maW<-,marrayNorm,matrix-method}
\alias{maW,marrayNorm-method}
\alias{print,marrayNorm-method}
\alias{show,marrayNorm-method}
\alias{summary,marrayNorm-method}
\alias{maNormCall}

\title{Class "marrayNorm", classes and methods for post-normalization cDNA microarray intensity data}


\description{
This class represents post-normalization intensity data for a batch of cDNA microarrays. A \emph{batch of arrays} consists of a collection of arrays with the same layout  (\code{"\link{marrayLayout}"}). The class contains slots for the average log-intensities A, the normalized log-ratios M, the location and scale normalization values, the layout of the arrays, and descriptions of the target samples hybridized to the arrays and probe sequences spotted onto the arrays.}


\section{Objects from the Class}{
Objects can be created by calls of the form 
\code{  new('marrayNorm',}\cr
\code{    maA        = ...., # Object of class matrix}\cr
\code{    maM        = ...., # Object of class matrix}\cr
\code{    maMloc     = ...., # Object of class matrix}\cr
\code{    maMscale   = ...., # Object of class matrix}\cr
\code{    maW        = ...., # Object of class matrix}\cr
\code{    maLayout   = ...., # Object of class marrayLayout}\cr
\code{    maGnames   = ...., # Object of class marrayInfo}\cr
\code{    maTargets  = ...., # Object of class marrayInfo}\cr
\code{    maNotes    = ...., # Object of class character}\cr
\code{    maNormCall = ...., # Object of class call}\cr
\code{  )}
}


\section{Slots}{
  \describe{
    \item{\code{maA}:}{Object of class \code{"matrix"}, average log-intensities (base 2) A, rows correspond to spotted probe sequences, columns to arrays in the batch.}
    \item{\code{maM}:}{Object of class \code{"matrix"}, intensity log-ratios (base 2) M, rows correspond to spotted probe sequences, columns to arrays in the batch.}
    \item{\code{maMloc}:}{Object of class \code{"matrix"}, location normalization values, rows correspond to spotted probe sequences, columns to arrays in the batch.}
    \item{\code{maMscale}:}{Object of class \code{"matrix"},  scale normalization values, rows correspond to spotted probe sequences, columns to arrays in the batch.}
    \item{\code{maW}:}{Object of class \code{"matrix"}, spot quality weights, rows correspond to spotted probe sequences, columns to arrays in the batch.}
    \item{\code{maLayout}:}{Object of class \code{"\link{marrayLayout}"}, layout parameters for cDNA microarrays.}
    \item{\code{maGnames}:}{Object of class \code{"\link{marrayInfo}"}, description of spotted probe sequences.}
    \item{\code{maTargets}:}{Object of class \code{"\link{marrayInfo}"}, description of target samples hybridized to the arrays.}
    \item{\code{maNotes}:}{Object of class \code{"character"}, any notes concerning the microarray experiments, e.g. hybridization or scanning conditions.}
    \item{\code{maNormCall}:}{Object of class \code{"call"}, function call for normalizing the batch of arrays.}
  }
}

\section{Methods}{
  \describe{
    \item{[}{\code{signature(x = "marrayNorm")}:  subsetting operator for spots on the array and arrays in the batch, ensures that all slots are subset properly.}
    \item{coerce}{\code{signature(from = "marrayRaw", to = "marrayNorm")}: coerce an object of class \code{"\link{marrayRaw}"} into an object of class \code{\link{marrayNorm}}. }
%    \item{coerce}{\code{signature(from = "marrayNorm", to = "exprSet")}: coerce an object of class \code{"\link{marrayNorm}"} into an object of class \code{\link{exprSet-class}}. }
% \item{coerce}{\code{signature(from = "marrayNorm", to = "marraySpots")}: coerce an object of class \code{"\link{marrayNorm}"} into an object of class \code{"\link{marraySpots}"}. }
    \item{maA}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maA<-}{\code{signature(object = "marrayNorm", value = "matrix")}: slot assignment method.} 
    \item{maControls<-}{\code{signature(object = "marrayNorm")}: slot assignment method.} 
    \item{maControls}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maGnames}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maGnames<-}{\code{signature(object = "marrayNorm", value = "marrayInfo")}: slot assignment method.} 
    \item{maGridCol}{\code{signature(object = "marrayNorm")}: method which computes a vector of grid column coordinates for each spot.}
    \item{maGridRow}{\code{signature(object = "marrayNorm")}: method which computes a vector of grid row coordinates for each spot.}
    \item{maLayout}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maLayout<-}{\code{signature(object = "marrayNorm", value = "marrayLayout")}: slot assignment method.} 
    \item{maM}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maM<-}{\code{signature(object = "marrayNorm", value = "matrix")}: slot assignment method.} 
    \item{maMloc}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maMloc<-}{\code{signature(object = "marrayNorm", value = "matrix")}: slot assignment method.} 
    \item{maMscale}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maMscale<-}{\code{signature(object = "marrayNorm", value = "matrix")}: slot assignment method.} 
    \item{maNgc}{\code{signature(object = "marrayNorm")}: slot accessor method.}
 \item{maNgc<-}{\code{signature(object = "marrayNorm", value = "numeric")}: slot assignment method.} 
    \item{maNgr}{\code{signature(object = "marrayNorm")}: slot accessor method.}
\item{maNgr<-}{\code{signature(object = "marrayNorm", value = "numeric")}: slot assignment method.} 
    \item{maNormCall}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maNotes}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maNotes<-}{\code{signature(object = "marrayNorm", value = "character")}: slot assignment method.} 
    \item{maNsamples}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maNsc}{\code{signature(object = "marrayNorm")}: slot accessor method.}
\item{maNsc<-}{\code{signature(object = "marrayNorm", value = "numeric")}: slot assignment method.} 
    \item{maNspots}{\code{signature(object = "marrayNorm")}: slot accessor method.}
\item{maNspots<-}{\code{signature(object = "marrayNorm", value = "numeric")}: slot assignment method.} 
    \item{maNsr}{\code{signature(object = "marrayNorm")}: slot accessor method.}
\item{maNsr<-}{\code{signature(object = "marrayNorm", value = "numeric")}: slot assignment method.} 
    \item{maPlate}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maPlate<-}{\code{signature(object = "marrayNorm")}: slot assignment method.} 
    \item{maPrintTip}{\code{signature(object = "marrayNorm")}: method which computes a vector of print-tip-group indices for each spot.}
    \item{maSpotCol}{\code{signature(object = "marrayNorm")}: method which computes a vector of spot column coordinates for each spot.}
    \item{maSpotRow}{\code{signature(object = "marrayNorm")}: method which computes a vector of spot row coordinates for each spot.}
    \item{maSub}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maSub<-}{\code{signature(object = "marrayNorm")}: slot assignment method.} 
    \item{maTargets}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maTargets<-}{\code{signature(object = "marrayNorm", value = "marrayInfo")}: slot assignment method.}  
    \item{maW}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maW<-}{\code{signature(object = "marrayNorm", value = "matrix")}: slot assignment method.} 
    \item{print}{\code{signature(x = "marrayNorm")}: print method for \code{"\link{marrayNorm}"} class.}
  }
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{marrayLayout}}, \code{\link{marrayRaw}},
  \code{\link{marrayInfo}}}

\examples{
# Examples use swirl dataset, for description type ? swirl

data(swirl)

# Median normalization
mnorm<-maNorm(swirl[,2:3],norm="m")

# Object of class marrayNorm for the second and third swirl arrays
mnorm

# Function call
maNormCall(mnorm)

#  Object of class marrayInfo -- Probe sequences
maGnames(mnorm)

#  Object of class marrayInfo -- Target samples
 maTargets(mnorm)

# Density plot of log-ratios M for third array
plot(density(maM(mnorm[,2])), lwd=2, col=2, main="Density plots of log-ratios M")
lines(density(maM(swirl[,3])), lwd=2)
abline(v=0)
legend(2,1,c("Pre-normalization","Post-normalization"))
}

\keyword{classes}

\eof
\name{marrayRaw-class}
\docType{class}

\alias{marrayRaw-class}
\alias{marrayRaw}

\alias{maRf}
\alias{maRf<-}
\alias{maGf}
\alias{maGf<-}
\alias{maRb}
\alias{maRb<-}
\alias{maGb}
\alias{maGb<-}
\alias{maW}
\alias{maW<-}
\alias{maLayout}
\alias{maLayout<-}

\alias{[,marrayRaw-method}
\alias{cbind,marrayRaw-method}
%\alias{coerce,marrayRaw,marrayNorm-method}
%\alias{coerce,marrayRaw,exprSet-method}
\alias{maA,marrayRaw-method}
\alias{maControls<-,marrayRaw-method}
\alias{maControls,marrayRaw-method}
\alias{maGb<-,marrayRaw,matrix-method}
\alias{maGb<-,marrayRaw,NULL-method}
\alias{maGb,marrayRaw-method}
\alias{maGf<-,marrayRaw,matrix-method}
\alias{maGf,marrayRaw-method}
%\alias{maGnames<-,marrayRaw,marrayInfo-method}
\alias{maGnames,marrayRaw-method}
\alias{maGridCol,marrayRaw-method}
\alias{maGridRow,marrayRaw-method}
%\alias{maLayout<-,marrayRaw,marrayLayout-method}
\alias{maLayout,marrayRaw-method}
\alias{maLG,marrayRaw-method}
\alias{maLR,marrayRaw-method}
\alias{maM,marrayRaw-method}
\alias{maNgc<-,marrayRaw,numeric-method}
\alias{maNgc,marrayRaw-method}
\alias{maNgr<-,marrayRaw,numeric-method}
\alias{maNgr,marrayRaw-method}
\alias{maNotes<-,marrayRaw,character-method}
\alias{maNotes,marrayRaw-method}
\alias{maNsamples,marrayRaw-method}
\alias{maNsc<-,marrayRaw,numeric-method}
\alias{maNsc,marrayRaw-method}
\alias{maNspots<-,marrayRaw,numeric-method}
\alias{maNspots,marrayRaw-method}
\alias{maNsr<-,marrayRaw,numeric-method}
\alias{maNsr,marrayRaw-method}
\alias{maPlate<-,marrayRaw-method}
\alias{maPlate,marrayRaw-method}
\alias{maPrintTip,marrayRaw-method}
\alias{maRb<-,marrayRaw,matrix-method}
\alias{maRb<-,marrayRaw,NULL-method}
\alias{maRb,marrayRaw-method}
\alias{maRf<-,marrayRaw,matrix-method}
\alias{maRf,marrayRaw-method}
\alias{maSpotCol,marrayRaw-method}
\alias{maSpotRow,marrayRaw-method}
\alias{maSub<-,marrayRaw-method}
\alias{maSub,marrayRaw-method}
%\alias{maTargets<-,marrayRaw,marrayInfo-method}
\alias{maTargets,marrayRaw-method}
\alias{maW<-,marrayRaw,matrix-method}
\alias{maW,marrayRaw-method}
\alias{print,marrayRaw-method}
\alias{show,marrayRaw-method}
\alias{summary,marrayRaw-method}

%\alias{maNgr}
%\alias{maNgc}
%\alias{maNsr}
%\alias{maNsc}
%\alias{maNspots}
%\alias{maSub}
%\alias{maPlate}
%\alias{maControls}

%\alias{maPrintTip}
%\alias{maGridRow}
%\alias{maGridCol
%\alias{maSpotRow}
%\alias{maSpotCol}

\alias{maGnames}
\alias{maGnames<-}
\alias{maTargets}
\alias{maTargets<-}
%\alias{maNotes}

\alias{maLR}
\alias{maLG}

%\alias{maA}
%\alias{maM}

\alias{maNsamples}


\title{Class "marrayRaw", classes and methods for pre-normalization cDNA microarray intensity data}

\description{This class represents pre-normalization intensity data for
  a batch of cDNA microarrays. A \emph{batch of arrays} consists of a
  collection of arrays with the same layout
  (\code{"\link{marrayLayout}"}). The class contains slots for the green
  (Cy3) and red (Cy5) foreground and background intensities, the layout
  of the arrays, and descriptions of the target samples hybridized to
  the arrays and probe sequences spotted onto the arrays.} 

\section{Objects from the Class}{
Objects can be created by calls of the form 
\code{  new('marrayRaw',}\cr
\code{    maRf      = ...., # Object of class matrix}\cr
\code{    maGf      = ...., # Object of class matrix}\cr
\code{    maRb      = ...., # Object of class matrix}\cr
\code{    maGb      = ...., # Object of class matrix}\cr
\code{    maW       = ...., # Object of class matrix}\cr
\code{    maLayout  = ...., # Object of class marrayLayout}\cr
\code{    maGnames  = ...., # Object of class marrayInfo}\cr
\code{    maTargets = ...., # Object of class marrayInfo}\cr
\code{    maNotes   = ...., # Object of class character}\cr
\code{  )}
}

\section{Slots}{
  \describe{
    \item{\code{maRf}:}{Object of class \code{"matrix"}, red foreground intensities, rows correspond to spotted probe sequences, columns to arrays in the batch.}
    \item{\code{maGf}:}{Object of class \code{"matrix"}, green foreground intensities, rows correspond to spotted probe sequences, columns to arrays in the batch. }
    \item{\code{maRb}:}{Object of class \code{"matrix"}, red background intensities, rows correspond to spotted probe sequences, columns to arrays in the batch. }
    \item{\code{maGb}:}{Object of class \code{"matrix"}, green background intensities, rows correspond to spotted probe sequences, columns to arrays in the batch. }
    \item{\code{maW}:}{Object of class \code{"matrix"}, spot quality weights, rows correspond to spotted probe sequences, columns to arrays in the batch.}
    \item{\code{maLayout}:}{Object of class \code{"\link{marrayLayout}"}, layout parameters for the cDNA microarrays.}
    \item{\code{maGnames}:}{Object of class \code{"\link{marrayInfo}"}, description of spotted probe sequences.}
    \item{\code{maTargets}:}{Object of class \code{"\link{marrayInfo}"},  description of target samples hybridized to the arrays.}
    \item{\code{maNotes}:}{Object of class \code{"character"},  any notes concerning the microarray experiments, e.g. hybridization or scanning conditions.}
  }
}

\section{Methods}{
  \describe{
    \item{[}{\code{signature(x = "marrayRaw")}: subsetting operator for spots on the array and arrays in the batch, ensures that all slots are subset properly.}
    \item{coerce}{\code{signature(from = "marrayRaw", to = "marrayNorm")}: coerce an object of class \code{"\link{marrayRaw}"} into an object of class \code{"\link{marrayNorm}"}. }
    \item{maA}{\code{signature(object = "marrayRaw")}: function which computes average log-intensities (base 2) A for an object of class \code{"\link{marrayRaw}"}.}
    \item{maControls<-}{\code{signature(object = "marrayRaw")}: slot assignment method.} 
    \item{maControls}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maGb}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maGb<-}{\code{signature(object = "marrayRaw", value = "matrix")}: slot assignment method.} 
    \item{maGb<-}{\code{signature(object = "marrayRaw", value = "NULL")}: slot assignment method.} 
    \item{maGf}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maGf<-}{\code{signature(object = "marrayRaw", value = "matrix")}: slot assignment method.}
    \item{maGnames}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maGnames<-}{\code{signature(object = "marrayRaw", value = "marrayInfo")}: slot assignment method.}
    \item{maGridCol}{\code{signature(object = "marrayRaw")}: method which computes a vector of grid column coordinates for each spot.}
    \item{maGridRow}{\code{signature(object = "marrayRaw")}: method which computes a vector of grid row coordinates for each spot.}
    \item{maLayout}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maLayout<-}{\code{signature(object = "marrayRaw", value = "marrayLayout")}: slot assignment method. }
    \item{maLG}{\code{signature(object = "marrayRaw")}: method which computes green log-intensities (base 2) for an object of class \code{"\link{marrayRaw}"}.}
    \item{maLR}{\code{signature(object = "marrayRaw")}: method  which computes red log-intensities (base 2) for an object of class \code{"\link{marrayRaw}"}.}
    \item{maM}{\code{signature(object = "marrayRaw")}: method which computes intensity log-ratios (base 2) M for an object of class \code{"\link{marrayRaw}"}.}
    \item{maNgc}{\code{signature(object = "marrayRaw")}: slot accessor method.}
 \item{maNgc<-}{\code{signature(object = "marrayRaw", value = "numeric")}:  slot assignment method.}
    \item{maNgr}{\code{signature(object = "marrayRaw")}: slot accessor method.}
 \item{maNgr<-}{\code{signature(object = "marrayRaw", value = "numeric")}:  slot assignment method.}
    \item{maNotes}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maNotes<-}{\code{signature(object = "marrayRaw", value = "character")}: slot assignment method.}
    \item{maNsamples}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maNsc}{\code{signature(object = "marrayRaw")}: slot accessor method.}
\item{maNsc<-}{\code{signature(object = "marrayRaw", value = "numeric")}:  slot assignment method.}
    \item{maNspots}{\code{signature(object = "marrayRaw")}: slot accessor method.}
\item{maNspots<-}{\code{signature(object = "marrayRaw", value = "numeric")}:  slot assignment method.}
    \item{maNsr}{\code{signature(object = "marrayRaw")}: slot accessor method.}
\item{maNsr<-}{\code{signature(object = "marrayRaw", value = "numeric")}:  slot assignment method.}
    \item{maPlate}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maPlate<-}{\code{signature(object = "marrayRaw")}: slot assignment method.}
    \item{maPrintTip}{\code{signature(object = "marrayRaw")}: method which computes a vector of print-tip-group indices for each spot.}
    \item{maRb}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maRb<-}{\code{signature(object = "marrayRaw", value = "matrix")}: slot assignment method.}
    \item{maRb<-}{\code{signature(object = "marrayRaw", value = "NULL")}: slot assignment method. }
    \item{maRf}{\code{signature(object = "marrayRaw")}: slot accessor method. }
    \item{maRf<-}{\code{signature(object = "marrayRaw", value = "matrix")}: slot assignment method. }
    \item{maSpotCol}{\code{signature(object = "marrayRaw")}: method which computes a vector of spot column coordinates for each spot.}
    \item{maSpotRow}{\code{signature(object = "marrayRaw")}: method which computes a vector of spot row coordinates for each spot.}
    \item{maSub}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maSub<-}{\code{signature(object = "marrayRaw")}: slot assignment method.}
    \item{maTargets}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maTargets<-}{\code{signature(object = "marrayRaw", value = "marrayInfo")}: slot assignment method. }
    \item{maW}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maW<-}{\code{signature(object = "marrayRaw", value = "matrix")}: slot assignment method.}
    \item{print}{\code{signature(x = "marrayRaw")}:  print method for \code{"\link{marrayRaw}"} class.}
  }
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{marrayLayout}}, \code{\link{marrayNorm}}, \code{\link{marrayInfo}}.}


\examples{
# Examples use swirl dataset, for description type ? swirl
require(limma)
data(swirl)

# Object of class marrayRaw for the 4 swirl arrays
swirl

# Object of class marrayLayout
maLayout(swirl)

# Access only the first 100 spots of the third array
swirl[1:100,3]

# Accessor methods -- How many spots on the array
maNspots(swirl)

# Density plot of log-ratios M for third array
plot(density(maM(swirl[,3])))

# Assignment methods -- Replace maNotes slot
maNotes(swirl)
maNotes(swirl)<-"This is a zebrafish microarray"
maNotes(swirl)

# Coerce methods -- Convert marrayRaw object into exprSet
## Load package convert
## library(convert)
## e<-as(swirl, "exprSet")
## e

}
\keyword{classes}

\eof
\name{marrayTools}
\alias{spotTools}
\alias{gpTools}

\title{Wrapper function}
\description{
  This functions will produce a directory of diagnostic plots, quality
  measures and normalization.
}
\usage{
spotTools(fnames, path = ".", galfile, bg = TRUE, plot = TRUE,
                 quality = TRUE, fill = TURE, raw = FALSE, echo =
                 TRUE, ...)

gpTools(fnames, path = ".", galfile, bg = TRUE, plot = TRUE, quality = TRUE, fill = TRUE, raw = FALSE, echo = TRUE, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
    \item{fnames}{a vector of character strings containing the file names of
    each spot quantification data file.  These typically end in \code{.spot}
    for the software \code{Spot} or \code{.gpr} for the software \code{GenePix}.}
  \item{path}{a character string representing the data directory. By
    default this is set to the current working directory (".").  In the
    case where \code{fnames} contains the full path name, path should be set
    to NULL.}
  \item{galfile}{a character string representing the Gal file.}
  \item{bg}{logical: if 'TRUE', background correction is performed before normalization}
  \item{plot}{logical: if 'TRUE', maDiagnPlots is call to generate basic
    diagnostic plot per array.}
  \item{quality}{logical: if 'TRUE', maQuality is call to calculate basic quality measures.}
  \item{fill}{logical: if `TRUE', `scan' will implicitly add empty fields
    to any lines with fewer fields than implied by `what'.}
  \item{raw}{logical: if 'TRUE', the microarray object class of
    'marrayRaw' is return rather than 'marrayNorm'.}
  \item{echo}{logical: if 'TRUE' (default), run-time
    information of the array currently being process is printed.}
  \item{\dots}{further arguments to \code{\link{scan}} or \code{\link{maNorm}}.}
}

\value{
  Jpeg files of diagnostic plots, excel files of quality measures, tab
  delimited files of normalized log-ratios and log-intensities values are
  saved to disk.
  In addition, an object of class \code{\link{marrayRaw}} or \code{\link{marrayNorm}} is return.
}

\author{Yee Hwa (Jean) Yang}

\seealso{\code{\link{read.marrayRaw}}}

\examples{
 data(swirl)
 datadir <- system.file("swirldata", package="marray")

## normdata <- spotTools(path=datadir, quality=FALSE)
}

\keyword{file}% at least one, from doc/KEYWORDS


\eof
\name{meanFun}
\alias{meanFun}

\title{A function that calculates the average of each gene in a
  multi-slide microarray experiment}

\description{
  Returns a function that calculate \code{mean}.
}

\usage{
meanFun(y = NULL, na.rm = TRUE)
}

\arguments{
  \item{y}{vector of class labels. Must consist of integers 1 and 2.}
  \item{na.rm}{If TRUE, the NA's are omit from the analysis.}
}

\value{
   Returns a function with bindings for 'y' (class labels) that will
   calculate average across slides.
}


\author{Yee Hwa (Jean) Yang }
\seealso{\code{\link{mean}}}

\examples{
x <- rnorm(10)
y <- meanFun()
y(x)
}

\keyword{univar}% at least one, from doc/KEYWORDS


\eof
\name{numNAFun}
\alias{numNAFun}

\title{A function that calculates the number of NA of each gene in a
  multi-slide microarray experiment. }

\description{
  Returns a function that calculates the number of NA of each gene across
  multi-slide microarray experiment.
}

\usage{
numNAFun()
}

\arguments{
}


\value{
  Returns a function.
}


\author{Yee Hwa (Jean) Yang}

\seealso{\code{\link{is.na}}}

\examples{
x <- rnorm(10)
x[8] <- NA
y <- numNAFun()
y(x)
}
\keyword{univar}% at least one, from doc/KEYWORDS



\eof
\name{opVersionID}
\alias{opVersionID}

\title{Determine the operon oligo set ID}
\description{
  This functions looks the operon ID and determine whether it belongs to 
  "Human Genome Oligo Set V1", "Human Genome Oligo Set V2", "Mouse
  Genome Oligo Set V1" or "Mouse Genome Oligo Set V2". 
}

\usage{
opVersionID(opID)
}

\arguments{
  \item{opID}{A character strings representing operon ID}
}

\value{
  A value "operonh1", "operonh2", "operonm1" or "operonm2" to represents
  "Human Genome Oligo Set V1", "Human Genome Oligo Set V2", "Mouse
   Genome Oligo Set V1" or "Mouse Genome Oligo Set V2". 
}

\references{\url{http://oparray.operon.com/}}
\author{Jean Yee Hwa Yang}

\seealso{\code{\link{URLstring}}, \code{\link{htmlPage}}}

\examples{
 opVersionID("M000205_01")
 URLstring[opVersionID("M000205_01")]
}

\keyword{manip}% at least one, from doc/KEYWORDS

\eof
\name{plot}
\docType{methods}
\alias{plot.marrayRaw}
\alias{plot.marrayNorm}
\alias{addText}
\alias{addLines}
\alias{addPoints}
\alias{text,marrayRaw-method}
\alias{text,marrayNorm-method}
\alias{lines,marrayRaw-method}
\alias{lines,marrayNorm-method}
\alias{points,marrayRaw-method}
\alias{points,marrayNorm-method}

%- Also NEED an `\alias' for EACH other topic documented here.
\title{Scatter-plots for cDNA microarray spot statistics}
\description{The function \code{maPlot} or \code{plot} produces scatter-plots of
  microarray spot statistics for the classes \code{"\link{marrayRaw}"},
  \code{"\link{marrayNorm}"}. It also allows the user to highlight and
annotate subsets of points on the plot, and display fitted curves from
robust local regression or other smoothing procedures.  }

\usage{
plot.marrayRaw(x, xvar = "maA", yvar = "maM", zvar="maPrintTip", lines.func,text.func,legend.func,...)
plot.marrayNorm(x, xvar = "maA", yvar = "maM", zvar="maPrintTip", lines.func,text.func,legend.func,...)
addText(object, xvar="maA", yvar="maM", subset=NULL, labels=as.character(1:length(subset)), ...)
addPoints(object, xvar="maA", yvar="maM", subset=TRUE, ...)
addLines(object, xvar="maA", yvar="maM", zvar="maPrintTip", subset=TRUE, ...)
\S4method{text}{marrayRaw}(x, xvar = "maA", yvar = "maM", ...)
\S4method{text}{marrayNorm}(x, xvar = "maA", yvar = "maM", ...)
\S4method{lines}{marrayRaw}(x, xvar = "maA", yvar = "maM", ...)
\S4method{lines}{marrayNorm}(x, xvar = "maA", yvar = "maM", ...)
\S4method{points}{marrayRaw}(x, xvar = "maA", yvar = "maM", ...)
\S4method{points}{marrayNorm}(x, xvar = "maA", yvar = "maM", ...)
}
%- maybe also `usage' for other objects documented here.

\arguments{
  \item{x}{Microarray object of class \code{"\link{marrayRaw}"}, \code{"\link{marrayNorm}"}.}
  \item{object}{Microarray object of class \code{"\link{marrayRaw}"}, \code{"\link{marrayNorm}"}.}
  \item{xvar}{Name of accessor function for the abscissa spot statistic, typically a slot name for the microarray object \code{x}, such as \code{maA}.}
  \item{yvar}{Name of accessor function for the ordinate spot statistic, typically a slot name for the microarray object \code{x}, such as \code{maM}.}
  \item{zvar}{Name of accessor method for the spot statistic used to stratify the data, typically a slot name for the microarray layout object (see \code{"\link{marrayLayout}"}) such as \code{maPlate} or a method such as \code{maPrintTip}. If \code{zvar} is NULL, the data are not stratified.}
  \item{lines.func}{Function for computing and plotting smoothed fits of \code{y} as a function of \code{x}, separately within values of \code{zvar}, e.g. \code{\link{maLoessLines}}. If \code{lines.func} is NULL, no fitting is performed.}
  \item{text.func}{Function for highlighting a subset of points, e.g., \code{\link{maText}}. If \code{text.func} is NULL, no points are highlighted.}
  \item{legend.func}{Function for adding a legend to the plot,  e.g. \code{\link{maLegendLines}}. If \code{legend.func} is NULL, there is no legend.}
  \item{subset}{logical vector or numeric values indicating the subset
    of points to be plotted.}
  \item{labels}{One or more character strings or expressions specifying the text to be written.}
  \item{\dots}{Optional graphical parameters, see \code{\link{par}}.}
}

\details{
This function calls the general function \code{\link{maPlot.func}},
which is not specific to microarray data. If there are more than one
array in the batch, the plot is done for the first array, by
default. Default graphical parameters are chosen for convenience using
the function \code{\link{maDefaultPar}} (e.g. color palette,  axis
labels,  plot title) but the user has the option to overwrite these
parameters at any point.} 
}

\references{
S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}

\author{Jean Yee Hwa Yang}

\seealso{\code{\link{maPlot.func}}, \code{\link{maDefaultPar}}, \code{\link{maLoessLines}}, \code{\link{maLegendLines}}, \code{\link{maText}}, \code{\link{plot}}, \code{\link{lowess}}, \code{\link{loess}}, \code{\link{legend}}.}

\examples{

# To see the demo type demo(marrayPlots)

# Examples use swirl dataset, for description type ? swirl
data(swirl)

# Pre-normalization MA-plot for the Swirl 93 array, with the lowess fits for 
# individual print-tip-groups. 
# - Default arguments
plot(swirl[,3])

# Lowess fit using all spots
plot(swirl[,3], zvar=NULL, legend.func=NULL)

# Loess fit using all spots
plot(swirl[,3], zvar=NULL, legend.func=maLegendLines(legend="All spots",col="green"), lines.func=maLoessLines(loess.args=list(span=0.3),col="green"))

}

\keyword{hplot}% at least one, from doc/KEYWORDS


\eof
\name{summary-methods}
\docType{methods}

\alias{summary-methods}
%\alias{print-methods}
%\alias{show-methods}
%\alias{print}

\title{Printing summary methods for microarray objects}

\description{
Print methods were defined for the microarray classes,
\code{"\link{marrayInfo}"}, \code{"\link{marrayLayout}"},
\code{"\link{marrayRaw}"}, \code{"\link{marrayNorm}"}. These methods
produce summaries of the intensity and textual data stored in different
classes of microarray objects. 
}

\section{Methods}{\describe{
\item{x = ANY}{generic print method}
\item{x = marrayLayout}{for an object of class \code{"\link{marrayLayout}"}, the method prints main layout parameters such as the number of spots and the dimensions of the spot and grid matrices.}
\item{x = marrayInfo}{for an object of class \code{"\link{marrayInfo}"}, the method prints the first 10 rows of the \code{"maInfo"} and \code{"maLabels"} slots.}
\item{x = marrayRaw}{for an object of class \code{"\link{marrayRaw}"}, the method prints a short description of the microarray layout \code{"maLayout"} and the target samples hybridized to the arrays \code{"maTargets"}, and a summary of the distribution of the log-ratio statistics \code{"maM"}.}
\item{x = marrayNorm}{for an object of class \code{"\link{marrayNorm}"}, the method prints a short description of the microarray layout \code{"maLayout"} and the target samples hybridized to the arrays \code{"maTargets"}, and a summary of the distribution of the log-ratio statistics \code{"maM"}.}
}}
\keyword{methods}


\eof
\name{read.Galfile}
\alias{read.Galfile}

\title{Reading GenePix Gal file}
\description{
  Reading a standard Gal file containing gene information.
}
\usage{
read.Galfile(galfile, path = ".", info.id = c("ID", "Name"),
layout.id =c(Block="Block", Row="Row", Column="Column"),
labels = "ID", notes = "", sep = "\t", skip = NULL,   ncolumns=4, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{galfile}{a character string representing the Gal file.}
  \item{path}{a character string representing the data directory. By
    default this is set to the current working directory ("."). }
  \item{info.id}{the column numbers or names in `fname' that contain the required
    information.}
  \item{layout.id}{the column names in `fname' that specified the
    printer layout information.}
  \item{labels}{the column number in \code{fname} which contains the names that the user would like to use to label spots or arrays (e.g. for default titles in \code{\link{maPlot}}.}
  \item{notes}{object of class character, vector of explanatory text}
  \item{sep}{the field separator character.  Values on each line of the
    file are separated by this character. The
    default is to read a tab delimited file.}
  \item{skip}{the number of lines of the data file to skip before
    beginning to read data.}
  \item{ncolumns}{an integer representing the number of columns of
    sub-array (print-tips) on a slides.}
  \item{\dots}{further arguments to \code{\link{scan}}.}
}

\value{
  \item{gnames}{An object of class \code{\link{marrayInfo}}.}
  \item{layout}{An object of class \code{\link{marrayLayout}}.}
}

\author{Yee Hwa (Jean) Yang}

\seealso{\code{\link{read.marrayInfo}}, \code{\link{read.marrayLayout}}}

\examples{
library(marray)
datadir <- system.file("swirldata", package="marray")
try <- read.Galfile(galfile="fish.gal", path=datadir)
names(try)
try$layout
try$gnames
}

\keyword{file}% at least one, from doc/KEYWORDS

\eof
\name{read.marrayInfo}
\alias{read.marrayInfo}
\alias{widget.marrayInfo}

\title{Create objects of class marrayInfo}
\description{
  This function creates objects of class \code{\link{marrayInfo}}. The \code{\link{marrayInfo}} class is used to store
  information regarding the target mRNA samples co-hybridized on the arrays or
  the spotted probe sequences (e.g. data frame of gene names, annotations, and other identifiers).
}
\usage{
read.marrayInfo(fname, info.id=NULL, labels=NULL, notes=fname, sep="\t",skip=0, quote="\"", ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{fname}{the name of the file that stores information on target samples or probe sequences. This is usually a file obtained from a database.}
  \item{info.id}{the column numbers in \code{fname} that contain the
    required information.}
  \item{labels}{the column number in \code{fname} which contains the names that the user would like to use to label spots or arrays (e.g. for default titles in \code{\link{maPlot}}.}
  \item{notes}{object of class character, vector of explanatory text}
    \item{sep}{the field separator character.  Values on each line of the
    file are separated by this character. The
    default is to read a tab delimited file.}
\item{skip}{the number of lines of the data file to skip before
  beginning to read data.}
\item{quote}{the set of quoting characters. By default, this is disable
  by setting `quote="\""'.}
  \item{\dots}{further arguments to \code{\link{scan}}.}
}
\details{

}
\value{
  An object of class \code{\link{marrayInfo}}.
}
\references{http://www.bioconductor.org/}
\author{Jean Yang, \email{yeehwa@stat.berkeley.edu}}

\examples{
datadir <- system.file("swirldata", package="marray")

## Reading target sample information
swirl.samples <- read.marrayInfo(file.path(datadir, "SwirlSample.txt"))

## Reading probe sequence information
swirl.gnames <- read.marrayInfo(file.path(datadir, "fish.gal"),
                                info.id=4:5, labels=5, skip=21)
}

\keyword{file}% at least one, from doc/KEYWORDS
\keyword{connection}

\eof
\name{read.marrayLayout}
\alias{read.marrayLayout}
\alias{widget.marrayLayout}

\title{Create objects of class marrayLayout}
\description{
  This function creates objects of class \code{\link{marrayLayout}} to store layout parameters for  two-color cDNA microarrays.
}
\usage{
read.marrayLayoutfname = NULL, ngr, ngc, nsr, nsc, pl.col = NULL,
                 ctl.col = NULL, sub.col = NULL, notes = fname, skip,
                 sep = "\t", quote = "\"", ...)
}

%- maybe also `usage' for other objects documented here.
\arguments{
  \item{fname}{the name of the file that stores plate and control
    information. This is usually a file obtained from a database.}
  \item{ngr}{the number of rows of grids per image.}
  \item{ngc}{the number of columns of grids per image.}
  \item{nsr}{the number of rows of spots per grid.}
  \item{nsc}{the number of columns of spots per grid.}
  \item{pl.col}{the column number in \code{fname} that contains plate information.}
  \item{ctl.col}{the column number in \code{fname} that contains control information.}
    \item{sub.col}{the column number in \code{fname} that contains full ID information.}
  \item{notes}{object of class character, vector of explanatory text.}
  \item{skip}{the number of lines of the data file to skip before
    beginning to read data.} 
  \item{sep}{the field separator character.  Values on each line of the
    file are separated by this character. The
    default is to read a tab delimited file.}
  \item{quote}{the set of quoting characters. By default, this is disable
  by setting `quote="\""'.}
  \item{\dots}{further arguments to \code{\link{scan}}.}
}

\details{}
\value{
  An object of class \code{\link{marrayLayout}}.
}

\references{http://www.bioconductor.org/ }
\author{Jean Yang \email{yeehwa@stat.berkeley.edu}}
\note{}

\examples{
datadir <- system.file("swirldata", package="marray")

### Reading in control information from file
skip <-  grep("Row", readLines(file.path(datadir,"fish.gal"), n=100)) - 1
swirl.layout <- read.marrayLayout(fname=file.path(datadir,"fish.gal"), ngr=4, ngc=4,
nsr=22, nsc=24, ctl.col=4, skip=skip)

### Setting control information.
swirl.gnames <- read.marrayInfo(file.path(datadir,"fish.gal"), info.id=4:5, labels=5, skip=21)
x <-  maInfo(swirl.gnames)[,1]
y <- rep(0, maNspots(swirl.layout))
y[x == "control"] <- 1
slot(swirl.layout, "maControls") <- as.factor(y)
}

\keyword{file}% at least one, from doc/KEYWORDS
\keyword{connection}

\eof
\name{read.marrayRaw}
\alias{read.marrayRaw}
\alias{read.Spot}
\alias{read.GenePix}
\alias{read.SMD}
\alias{read.Agilent}
\alias{widget.marrayRaw}

\title{Create objects of class "marrayRaw"}
\description{
  This function reads in cDNA microarray data from a directory and creates objects of
  class \code{"\link{marrayRaw}"} from spot quantification data files obtained from image analysis software or databases.
}

\usage{
read.marrayRaw(fnames, path=".", name.Gf=NULL, name.Gb=NULL, name.Rf=NULL,
name.Rb=NULL,name.W=NULL, layout=NULL, gnames=NULL, targets=NULL,
notes=NULL, skip=NULL, sep="	", quote="\"", DEBUG=FALSE, ...)

read.GenePix(fnames = NULL, path = NULL, name.Gf = "F532 Median",
name.Gb ="B532 Median", name.Rf = "F635 Median", name.Rb = "B635 Median",
name.W ="Flags", layout = NULL, gnames = NULL, targets = NULL,
notes = NULL, skip=NULL, sep = "	", quote = "\"", DEBUG=FALSE, ...)

read.SMD(fnames = NULL, path = NULL, name.Gf = "Ch1 Intensity (Median)",
name.Gb = "Ch1 Background (Median)", name.Rf = "Ch2 Intensity (Median)",
name.Rb = "Ch2 Background (Median)", name.W = NULL, info.id = c("Name",
"Clone ID"), layout = NULL, gnames = NULL, targets = NULL, notes = NULL, skip = NULL, sep = "\t", quote = "\"", DEBUG=FALSE, ...)

read.Spot(fnames = NULL, path = ".", name.Gf = "Gmean", name.Gb =
"morphG", name.Rf = "Rmean", name.Rb = "morphR",name.W = NULL, layout =
NULL, gnames = NULL, targets = NULL, notes = NULL, skip = NULL, sep = "	", quote = "\"", ...)

read.Agilent(fnames = NULL, path=NULL, name.Gf = "gMedianSignal", name.Gb = "gBGMedianSignal", name.Rf = "rMedianSignal", name.Rb = "rBGMedianSignal", name.W= NULL, layout = NULL, gnames = NULL, targets = NULL, notes=NULL, skip=NULL, sep="\t", quote="\"", DEBUG=FALSE, info.id=NULL, ...)

widget.marrayRaw(ext = c("spot", "xls", "gpr"), skip = 0, sep = "\t",  quote = "\"", ...)
}

\arguments{
  \item{fnames}{a vector of character strings containing the file names of
    each spot quantification data file.  These typically end in \code{.spot}
    for the software \code{Spot} or \code{.gpr} for the software \code{GenePix}.}
  \item{path}{a character string representing the data directory. By
    default this is set to the current working directory (".").  In the
    case where \code{fnames} contains the full path name, path should be set
    to NULL.}
  \item{name.Gf}{character string for the column header for green foreground intensities.}
  \item{name.Gb}{character string for the column header for green background intensities.}
  \item{name.Rf}{character string for the column header for red foreground intensities.}
  \item{name.Rb}{character string for the column header for red background intensities.}
  \item{name.W}{character string for the column header for spot quality weights.}
  \item{layout}{object of class \code{"\link{marrayLayout}"}, containing microarray layout parameters.}
  \item{gnames}{object of class \code{"\link{marrayInfo}"} containing probe sequence information.}
  \item{targets}{object of class \code{"\link{marrayInfo}"} containing target sample information.}
  \item{notes}{object of class \code{"character"}, vector of explanatory
    text.}
  \item{info.id}{object of class \code{"character"}, vector containing
    the name of the colums of the SMD file containing oligo information
    you want to retrieve. By default, this is set to read Homo sapiens
    data. You may need to modify this argument if your are working on
    another genome.}
  \item{skip}{the number of lines of the data file to skip before beginning to read in data.}
  \item{sep}{the field separator character.  Values on each line of the
    file are separated by this character. The
    default is to read a tab delimited file.}
  \item{quote}{the set of quoting characters. By default, this is
    disabled by setting \code{quote="\""}.}
  \item{ext}{a characters string representing suffix of different image
    analysis output files.}
  \item{DEBUG}{a logical value, if TRUE, a series of echo statements will be printed.}
  \item{\dots}{further arguments to \code{\link{scan}}.}
}
\details{
  
}
\value{
  An object of class \code{"\link{marrayRaw}"}.
}
\references{\url{http://www.bioconductor.org/}.}
\author{Jean Yang, \email{yeehwa@stat.berkeley.edu}}

\seealso{\code{\link{scan}}, \code{\link{read.marrayLayout}},
  \code{\link{read.marrayInfo}} }

\examples{
datadir <- system.file("swirldata", package="marray")

skip <-  grep("Row", readLines(file.path(datadir,"fish.gal"), n=100)) - 1

swirl.layout <- read.marrayLayout(ngr=4, ngc=4, nsr=22, nsc=24)

swirl.targets <- read.marrayInfo(file.path(datadir, "SwirlSample.txt"))

swirl.gnames <- read.marrayInfo(file.path(datadir, "fish.gal"),
                                info.id=4:5, labels=5, skip=skip)

x <-  maInfo(swirl.gnames)[,1]
y <- rep(0, maNspots(swirl.layout))
y[x == "control"] <- 1
slot(swirl.layout, "maControls") <- as.factor(y)

fnames <- dir(path=datadir,pattern=paste("*", "spot", sep="\."))
swirl<- read.Spot(fnames, path=datadir,
                       layout = swirl.layout,
                       gnames = swirl.gnames,
                       targets = swirl.targets)

}

\keyword{file}% at least one, from doc/KEYWORDS

\eof
\name{rm.na}
\alias{rm.na}

\title{Remove missing values}
\description{
  Remove NA's, NAN's and INF's from a vector.
}

\usage{
rm.na(x)
}

\arguments{
  \item{x}{A numeric vector}
}

\value{
  A vector with all NA's remove.
}

\author{Jean Yang}

\examples{
x <- round(rnorm(10), 2)
x[c(2,4,5)] <- NA
x
rm.na(x)
}
\keyword{manip}% at least one, from doc/KEYWORDS


\eof
\name{ShowLargeObject-class}
\docType{class}

\alias{ShowLargeObject-class}
\alias{show,ShowLargeObject-method}
%\alias{print-methods}

\title{Show Large Data Object - class}
      
\description{
A virtual class including the data classes \code{marrayRaw},
\code{marrayNorm}, \code{marrayInfo}, \code{marrayLayout},
\code{PrinterInfo}, \code{RGData} and \code{MAData}, all of which typically contain large
quantities of numerical data in vector, matrices and data.frames. 
}

\section{Methods}{
A \code{show} method is defined for objects of class
\code{ShowLargeObject} which uses \code{printHead} to print only the
leading elements or rows of components or slots which contain large
quantities of data. 
}

\author{modifid from Gordon Smyth's function}

\examples{
}

\keyword{classes}
\keyword{data}

\eof
\name{stat.confband.text}

\alias{stat.confband.text}

\title{Rank genes according to the value of a statistic.}
  
\description{
  Select values based on intensities binning.
}

\usage{
stat.confband.text(M, A, crit1=0.025, crit2=crit1, nclass=5)
}

\arguments{
 \item{A}{a vector giving the x-coordinates of the points in the scatter
          plot. In the microarray context, this could be a vector of
          average log intensities. ie A} 
 \item{M}{a vector giving the y-coordinates of the points in the scatter
   plot. In the microarray context, this could be a vector of log
   intensity ratios.} 
 \item{crit1}{The number of points to be selected.
   If crit1 < 1, the crit1*100\% spots with the smallest M values
   will be selected. If crit1 >= 1, the crit spots  with the
   smallest M values are selected.}  
 \item{crit2}{Similar to "crit1".   If crit2 < 1, the crit2*100\%
   spots with the largest M values will be selected. If crit2 >= 1,
   the crit2 spots with the largest M values are selected.}  
 \item{nclass}{A single number giving the approximate number of
   intensity depedent groups to consider.} 
% \item{labs}{ one or more character strings or expressions specifying the
%   text to be written.  If this string is not specified, by
%   default the index of the vector `M' will be used.}
% \item{output}{logical, defaulting to `FALSE'. If `TRUE' a vector
%   containning the index to the vector `M' that are  
%   highlighted.} 
}


\value{
  A vector of selected spot index.
}

\examples{
library(marray)
data(swirl)
aveA <- apply(maA(swirl), 1, mean.na)
aveM <- apply(maM(swirl), 1, mean.na)
stat.confband.text(aveM, aveA, crit1=20, crit2=50, nclass=5)
}

\seealso{ \code{\link{stat.gnames}}}

\keyword{manip}

\eof
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This file was automatically generated by Rdoc on 2001-11-08 :05:09.
% Do NOT modify this file, instead modify the source (Rarray.R).
% Rdoc is copyright (C) 2001, Henrik Bengtsson, henrikb@braju.com.
% Download the Rdoc compiler at http://www.braju.com/R/
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

\name{stat.gnames}

\alias{stat.gnames}

\title{Sort Genes According to the Value of a Statistic}

\description{
Lists genes and corresponding statistics in decreasing order of the
statistics. This function applies to any type of statistic, including
log ratios, one and two-sample t-statistics, and F-statistics. Missing
values are ignored, as in \code{\link{sort}(..., na.last=NA)}. 
}

\usage{
stat.gnames(x, gnames, crit= 50)
}

\arguments{
 \item{x}{a numeric vector containing the statistics for each
 gene. Missing values (NAs) are allowed. }
 
\item{gnames}{a character vector containing the gene names.}

 \item{crit}{specifies the number of genes to be returned. If crit <
 1, the crit*100\% genes with the largest x values are listed. If crit
 >= 1, the crit genes with the largest x values are listed. }
}

\value{
List containing the following components 
  \item{gnames}{gene names sorted in decreasing order of the
 statistics in x.}
 \item{t}{statistics sorted in decreasing order.}
}

\author{
  Yee Hwa Yang, \email{yeehwa@stat.berkeley.edu} \cr
  Sandrine Dudoit, \email{sandrine@stat.berkeley.edu} }

\seealso{\code{\link{order}}, \code{\link{sort}}.}

\examples{
data(swirl)
aveM <- apply(maM(swirl), 1, mean.na)
Gnames <- maGeneTable(swirl)

stat.gnames(abs(aveM), Gnames, crit=10)
stat.gnames(aveM, Gnames, crit=0.01)

}

\keyword{misc}

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% End of File
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

\eof
\name{swirl}
\alias{swirl}
\alias{swirl.1.spot}
\alias{swirl.2.spot}
\alias{swirl.3.spot}
\alias{swirl.4.spot}
\alias{fish.gal}
\alias{SwirlSample.txt}
\alias{SwirlSample}


\title{Gene expression data from Swirl zebrafish cDNA microarray experiment}

\description{
  The \code{swirlRaw} dataset consists of an object \code{swirl}
  of class \code{\link{marrayRaw}}, which represents
  pre-normalization intensity data for a batch of cDNA microarrays.
  
  This experiment was carried out
using zebrafish as a model organism to study early development in
vertebrates. Swirl is a point mutant in the BMP2 gene that affects the
dorsal/ventral body axis. Ventral fates such as blood are reduced, whereas
dorsal structures such as somites and notochord are expanded. A goal of
the Swirl experiment is to identify genes with altered expression in
the swirl mutant compared to wild-type zebrafish. Two sets of dye-swap experiments were performed, for a total of four replicate hybridizations. For
each of these hybridizations, target cDNA from the swirl mutant was labeled using
one of the Cy3 or Cy5 dyes and the target cDNA wild-type mutant was labeled using
the other dye. Target cDNA was hybridized to microarrays containing
8,448 cDNA probes, including 768 controls spots (e.g. negative, positive,
and normalization controls spots). Microarrays were printed using \eqn{4 \times 4} print-tips and are thus partitioned into a \eqn{4 \times 4} grid matrix. Each grid consists of a \eqn{22\times 24} spot matrix that was printed with a single print-tip. Here, spot row and plate coordinates should coincide, as each row of spots corresponds to probe sequences from the same 384 well-plate.\cr


Each of the four hybridizations produced
a pair of 16-bit images, which were processed using the image analysis software package \code{Spot}. Raw images of the Cy3 and Cy5 fluorescence intensities for all fourhybridizations are available at \url{http://fgl.lsa.berkeley.edu/Swirl/index.html}.the dataset includes four output files
\code{swirl.1.spot},
\code{swirl.2.spot},
\code{swirl.3.spot}, and
\code{swirl.4.spot}
from the \code{Spot} package.  Each of these files contains
8,448 rows and 30 columns; rows correspond to spots and columns to
different statistics from the \code{Spot} image analysis output. The file
\code{fish.gal} is a gal file generated by the \code{GenePix} program; it contains information on individual probe sequences, such as gene names, spot ID, spot coordinates.  Hybridization information for the mutant and wild-type target samples is stored in \code{SwirlSample.txt}.


}

\usage{data(swirl)}

\source{These data were provided by Katrin Wuennenberg-Stapleton from the Ngai Lab at UC Berkeley. The swirl embryos for this experiment were provided by David Kimelman and David
Raible at the University of Washington.}

\keyword{datasets}



\eof
\name{ttestFun}
\alias{ttestFun}

\title{A function that calculates the t-statistics
  of each gene in a multi-slide microarray experiment.}

\description{
  Returns a function that calculate \code{t.test}.
}
\usage{
ttestFun(y = NULL, var.equal = FALSE, alternative = c("two.sided", "less", "greater"), na.rm = TRUE)
}

\arguments{
  \item{y}{vector of class labels. Must consist of integers 1 and 2.}
  \item{var.equal}{ a logical variable indicating whether to treat the two
          variances as being equal. If `TRUE' then the pooled variance
          is used to estimate the variance otherwise the Welch
          approximation to the degrees of freedom is used.}
  \item{alternative}{ a character string specifying the alternative hypothesis,
          must be one of `"two.sided"' (default), `"greater"' or
          `"less"'.  You can specify just the initial letter.}
  \item{na.rm}{If TRUE, the NA's are omit from the analysis.}

}

\value{
  Returns a function that performs \code{t.test}.
}


\author{Yee Hwa (Jean) Yang}

\seealso{\code{\link{t.test}}}

\examples{
x <- rnorm(10)
y <- ttestFun()
y(x)
}
\keyword{univar}% at least one, from doc/KEYWORDS


\eof
\name{write.list}
\alias{write.list}

\title{Data Output}

\description{
  Writes information from a list into a text file.
}

\usage{
write.list(x, filename = "data", append = FALSE, closefile = TRUE, outfile)
}
\arguments{
  \item{x}{the list object to be written.}
  \item{filename}{a character string representing the file name.}
  \item{append}{logical; if true, the data \code{x} is appended to file
    \code{filename}.}
  \item{closefile}{logical indicating if the file connection should be closed.}
  \item{outfile}{file name or connections.}
}
\details{
  This function may be called recursively if there exists list structure
  within a list.
}

\author{Jean Yee Hwa Yang}

\seealso{\code{\link{write.table}}, \code{\link{write}}}

\examples{
data(swirl)
test <- list(A = 1:10, B= maM(swirl)[1:10,], C=list(x=1:10, y=1:4),
             D = summary(maA(swirl[,1])))
write.list(test, filename="test.txt")
}
\keyword{file}

\eof
\name{write.marray}
\alias{write.marray}

\title{Data Output}
\description{
  Calls the function write.table with predefine argument.   The entries
  in each line (row) are separated by tab. 
}

\usage{
write.marray(mraw, file="maRawResults.xls", val="maM", ...)
}

\arguments{
  \item{mraw}{the object to be written, either a \code{marrayRaw} or
    \code{marrayNorm} object.}
  \item{file}{a character string representing the file name.}
  \item{val}{a character string representing the slotNames to be written.}
  \item{\dots}{further arguments to \code{\link{write.table}}.}
}

\details{
  see \code{\link{write.table}}
}

\author{Jean Yee Hwa Yang}

\seealso{\code{\link{write.table}}, \code{\link{write.list}}}

\examples{
data(swirl)
write.marray(swirl[1:10,])
}

\keyword{file}% at least one, from doc/KEYWORDS


\eof
\name{write.xls}
\alias{write.xls}

\title{Data Output}
\description{
  Calls the function write.table with predefine argument.   The entries
  in each line (row) are separated by tab. 
}

\usage{
write.xls(res, file = "test.xls", ...)
}

\arguments{
  \item{res}{the object to be written, typically a data frame.  If not, it
          is attempted to coerce \code{x} to a data frame.}
  \item{file}{a character string representing the file name.}
  \item{\dots}{further arguments to \code{\link{write.table}}.}
}
\details{
  see \code{\link{write.table}}
}

\author{Jean Yee Hwa Yang}

\seealso{\code{\link{write.table}}, \code{\link{write.list}}}

\examples{
data(swirl)
write.xls(maM(swirl)[1:10,], "normM.xls")
}

\keyword{file}% at least one, from doc/KEYWORDS


\eof
